package Win32API::ToolHelp;

use 5.006;
use strict;
use warnings;

require Exporter;
require DynaLoader;

our @ISA = qw(Exporter DynaLoader);

our $VERSION = '0.02';

# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.

# This allows declaration	use Win32API::ToolHelp ':All';
# If you do not need this, moving things directly into @EXPORT or @EXPORT_OK
# will save memory.
our %EXPORT_TAGS = (
	'Func' => [ qw(
		CloseToolhelp32Snapshot
		CreateToolhelp32Snapshot
		GetLastToolHelpError
		Heap32First
		Heap32ListFirst
		Heap32ListNext
		Heap32Next
		Module32First
		Module32Next
		Process32First
		Process32Next
		SetLastToolHelpError
		Thread32First
		Thread32Next
		Toolhelp32ReadProcessMemory
	) ],
	'FuncA' => [ qw(
		Module32FirstA
		Module32NextA
		Process32FirstA
		Process32NextA
	) ],
	'FuncW' => [ qw(
		Module32FirstW
		Module32NextW
		Process32FirstW
		Process32NextW
	) ],
	'HF32_' => [ qw(
		HF32_DEFAULT
		HF32_SHARED
	) ],
	'LF32_' => [ qw(
		LF32_FIXED
		LF32_FREE
		LF32_MOVEABLE
	) ],
	'TH32CS_' => [ qw(
		TH32CS_INHERIT
		TH32CS_SNAPALL
		TH32CS_SNAPHEAPLIST
		TH32CS_SNAPMODULE
		TH32CS_SNAPMODULE32
		TH32CS_SNAPPROCESS
		TH32CS_SNAPTHREAD
	) ],
	'THREAD_' => [ qw(
		THREAD_PRIORITY_LOWEST
		THREAD_PRIORITY_BELOW_NORMAL
		THREAD_PRIORITY_NORMAL
		THREAD_PRIORITY_HIGHEST
		THREAD_PRIORITY_ABOVE_NORMAL
		THREAD_PRIORITY_ERROR_RETURN
		THREAD_PRIORITY_TIME_CRITICAL
		THREAD_PRIORITY_IDLE
	) ]
);

my @EXPORT_ALL = ();
foreach my $ref (values %EXPORT_TAGS) {
	push @EXPORT_ALL, @$ref;
}
$EXPORT_TAGS{'All'} = [ @EXPORT_ALL ];

our @EXPORT_OK = ( @{$EXPORT_TAGS{'All'}} );

our @EXPORT = qw();

bootstrap Win32API::ToolHelp $VERSION;

# Preloaded methods go here.

# generated by: perl -ne "print if /#define HF32_/;" <"%MSSdk%\Include\TlHelp32.h"
# and reformatted by: s/^#define\s+(\w+)\s+([^\s]+)[ \t]*.*$/sub \1\t{ \2 }/"

sub HF32_DEFAULT	{ 1 }
sub HF32_SHARED		{ 2 }

# generated by: perl -ne "print if /#define LF32_/;" <"%MSSdk%\Include\TlHelp32.h"
# and reformatted by: s/^#define\s+(\w+)\s+([^\s]+).*$/sub \1\t{ \2 }/"

sub LF32_FIXED		{ 0x00000001 }
sub LF32_FREE		{ 0x00000002 }
sub LF32_MOVEABLE	{ 0x00000004 }

# generated by: perl -ne "print if /#define TH32CS_/;" <"%MSSdk%\Include\TlHelp32.h"
# and reformatted by: s/^#define\s+(\w+)\s+([^\s]+).*$/sub \1\t{ \2 }/"

sub TH32CS_INHERIT	{ 0x80000000 }
sub TH32CS_SNAPHEAPLIST	{ 0x00000001 }
sub TH32CS_SNAPMODULE	{ 0x00000008 }
sub TH32CS_SNAPMODULE32	{ 0x00000010 }
sub TH32CS_SNAPPROCESS	{ 0x00000002 }
sub TH32CS_SNAPTHREAD	{ 0x00000004 }
sub TH32CS_SNAPALL	{ TH32CS_SNAPHEAPLIST | TH32CS_SNAPPROCESS | TH32CS_SNAPTHREAD | TH32CS_SNAPMODULE }

# generated by: perl -ne "print if /#define THREAD_BASE_/;" <"%MSSdk%\Include\WinNT.h"
# and reformatted by: s/^#define\s+(\w+)\s+([^\s]+)[ \t]*.*$/sub \1\t{ \2 }/"

sub THREAD_BASE_PRIORITY_LOWRT	{ 15 }
sub THREAD_BASE_PRIORITY_MAX	{ 2 }
sub THREAD_BASE_PRIORITY_MIN	{ -2 }
sub THREAD_BASE_PRIORITY_IDLE	{ -15 }

# generated by: perl -ne "print if /#define THREAD_/;" <"%MSSdk%\Include\WinBase.h"
# and reformatted by: s/^#define\s+(\w+)\s+([^\s]+).*$/sub \1\t{ \2 }/"

sub THREAD_PRIORITY_LOWEST		{ THREAD_BASE_PRIORITY_MIN }
sub THREAD_PRIORITY_BELOW_NORMAL	{ THREAD_PRIORITY_LOWEST + 1 }
sub THREAD_PRIORITY_NORMAL		{ 0 }
sub THREAD_PRIORITY_HIGHEST		{ THREAD_BASE_PRIORITY_MAX }
sub THREAD_PRIORITY_ABOVE_NORMAL	{ THREAD_PRIORITY_HIGHEST - 1 }
sub THREAD_PRIORITY_ERROR_RETURN	{ 0x7fffffff  } # MAXLONG
sub THREAD_PRIORITY_TIME_CRITICAL	{ THREAD_BASE_PRIORITY_LOWRT }
sub THREAD_PRIORITY_IDLE		{ THREAD_BASE_PRIORITY_IDLE }

# reformatted by: s/^\s+(\w+)A\s*$/sub \1\t{ \1A }/"

sub Module32First	{ &Module32FirstA }
sub Module32Next	{ &Module32NextA }
sub Process32First	{ &Process32FirstA }
sub Process32Next	{ &Process32NextA }

# Autoload methods go after =cut, and are processed by the autosplit program.

1;
__END__

=head1 NAME

Win32API::ToolHelp - Perl extension for obtaining information about currently
                     executing applications using the plain Win32 ToolHelp API

=head1 SYNOPSIS

  use Win32API::ToolHelp ':All';

  # --- prints names of all processes
  my $handle = CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS, 0);
  my $pe;
  Process32First($handle, $pe);
  do {
    print "$pe->{th32ProcessID}\t$pe->{szExeFile}\n";
  } while (Process32Next($handle, $pe));
  CloseToolhelp32Snapshot($handle);

  # --- prints names of the modules used by the current process
  my $handle = CreateToolhelp32Snapshot(TH32CS_SNAPMODULE, $$);
  my $me;
  Module32First($handle, $me);
  do {
    print "$me->{th32ModuleID}\t$me->{szModule}\n";
  } while (Module32Next($handle, $me));
  CloseToolhelp32Snapshot($handle);

=head1 DESCRIPTION

The functions provided by the I<ToolHelp> library make it easier for you
to obtain information about currently executing applications. These functions
are designed to streamline the creation of tools, specifically debuggers.

It covers enumerating the currently running processes and their modules,
traversing heaps, listing threads and taking snapshots of memory
of the currently running processes. The results are return in a form
as close as possible to the original Win32 API (simple types are returned
as scalars of the same type and structures are returned as references to hashes
with keys of the same names as the members in the original structures have).
There are all functions of the I<ToolHelp> library implemented in this module.

The support for the I<ToolHelp> library API is included in Windows 95
or higher or in Windows 2000 or higher. It is not included in Windows NT. See
the module C<Win32API::ProcessStatus> for the similar functionality
for Windows NT.

(Note that much of the following documentation refers to the behavior
of the underlying Win32 I<ToolHelp> API calls which may vary in its current
and future versions without any changes to this module. Therefore you should
check the Win32 I<ToolHelp> API documentation in MSDN directly when needed.)

=head2 EXPORTS

Nothing is exported by default. The following tags can be used to have sets
of symbols exported:

=over

=item :Func

The basic function names: CloseToolhelp32Snapshot CreateToolhelp32Snapshot
GetLastToolHelpError Heap32First Heap32ListFirst Heap32ListNext Heap32Next
Module32First Module32Next Process32First Process32Next SetLastToolHelpError
Thread32First Thread32Next Toolhelp32ReadProcessMemory.

=item :FuncA

The ANSI function names: Module32FirstA Module32NextA Process32FirstA
Process32NextA.

=item :FuncW

The Unicode function names: Module32FirstW Module32NextW Process32FirstW
Process32NextW.

=item :HF32_

The constants for identifying a heap type: HF32_DEFAULT HF32_SHARED.

=item :LF32_

The constants for finding a type of an allocated memory block: LF32_FIXED
LF32_FREE LF32_MOVEABLE.

=item :TH32CS_

The constants for creating snapshots of a running system: TH32CS_INHERIT
TH32CS_SNAPALL TH32CS_SNAPHEAPLIST TH32CS_SNAPMODULE TH32CS_SNAPMODULE32
TH32CS_SNAPPROCESS TH32CS_SNAPTHREAD.

=item :THREAD_

The constants defining the thread priority: THREAD_PRIORITY_LOWEST
THREAD_PRIORITY_BELOW_NORMAL THREAD_PRIORITY_NORMAL THREAD_PRIORITY_HIGHEST
THREAD_PRIORITY_ABOVE_NORMAL THREAD_PRIORITY_ERROR_RETURN
THREAD_PRIORITY_TIME_CRITICAL THREAD_PRIORITY_IDLE.

=back

=head2 CONSTANTS

The constants described here and exported by this module are used only
in its structures and functions. See the module C<Win32API::Const>
for the common Win32 constants.

C<HF32_> constants are used as flags in the heap list describing structure
C<HEAPLIST32>.

=over 14

=item HF32_DEFAULT

Process's default heap.

=item HF32_SHARED

Heap shared among processes.

=back

C<LF32_> constants are used as flags in the heap entry describing structure
C<HEAPENTRY32>.

=over 15

=item LF32_FIXED

The memory block has a fixed (unmovable) location.

=item LF32_FREE

The memory block is not used.

=item LF32_MOVEABLE

The memory block location can be moved.

=back

C<TH32CS_> constants are used by the C<CreateToolhelp32Snapshot> function
to specify portions of the system to include in the snapshot.

=over 21

=item TH32CS_INHERIT

Indicates that the snapshot handle is to be inheritable.

=item TH32CS_SNAPALL

Equivalent to specifying C<TH32CS_SNAPHEAPLIST>, C<TH32CS_SNAPMODULE>,
C<TH32CS_SNAPPROCESS>, and C<TH32CS_SNAPTHREAD>.

=item TH32CS_SNAPHEAPLIST

Includes the heap list of the specified process in the snapshot.

=item TH32CS_SNAPMODULE

Includes the module list of the specified process in the snapshot.

=item TH32CS_SNAPPROCESS

Includes the process list in the snapshot.

=item TH32CS_SNAPTHREAD

Includes the thread list in the snapshot.

=back

C<THREAD_> constants are used as flags in the thread entry describing
structure C<THREADENTRY32>.

=over 31

=item THREAD_PRIORITY_IDLE

Indicates a base priority level of 1 for C<IDLE_PRIORITY_CLASS>,
C<NORMAL_PRIORITY_CLASS>, or C<HIGH_PRIORITY_CLASS> processes, and a base
priority level of 16 for C<REALTIME_PRIORITY_CLASS> processes.

=item THREAD_PRIORITY_LOWEST

Indicates 2 points below normal priority for the priority class.

=item THREAD_PRIORITY_BELOW_NORMAL

Indicates one point below normal priority for the priority class.

=item THREAD_PRIORITY_NORMAL

Indicates normal priority for the priority class.

=item THREAD_PRIORITY_ABOVE_NORMAL

Indicates one point above normal priority for the priority class.

=item THREAD_PRIORITY_HIGHEST

Indicates two points above normal priority for the priority class.

=item THREAD_PRIORITY_TIME_CRITICAL

Indicates a base priority level of 15 for C<IDLE_PRIORITY_CLASS>,
C<NORMAL_PRIORITY_CLASS>, or C<HIGH_PRIORITY_CLASS> processes, and a base
priority level of 31 for C<REALTIME_PRIORITY_CLASS> processes.

=back

=head2 STRUCTURES

The structures that act as input and ouput parameters are handled as hashes
with keys of the same names as the members in the original structures have.
It allows those already familiar with the Win32 API to get off to a quick
start and occasionally use the original MSDN documentation to the API.

(Obviously the member C<dwSize> is omitted as there is no need to pre-allocate
the actual size for the output hash in Perl.)

=over

=item HEAPENTRY32

Describes one entry (block) of a heap that is being examined.

=over 15

=item hHandle

Handle to the heap block.

=item dwAddress

Linear address of the start of the block.

=item dwBlockSize

Size, in bytes, of the heap block.

=item dwFlags

Flags. The values available are listed in the description
of C<LF32_> constants.

=item dwLockCount

Lock count on the memory block. The lock count is incremented each time
the C<GlobalLock> or C<LocalLock> function is called on the block either
by the application or the DLL that the heap belongs to.

=item dwResvd

Reserved; do not use.

=item th32ProcessID

Identifier of the process to examine.

=item th32HeapID

Heap identifier in the owning process context. The contents of this member
has meaning only to the tool help functions. It is not a handle, nor is it
usable by functions other than the I<ToolHelp> functions.

=back

=item HEAPLIST32

Describes one entry (block) of a heap that is being examined.

=over 15

=item dwProcessID

Identifier of the process to examine.

=item th32HeapID

Heap identifier in the owning process context. The contents of this member
has meaning only to the tool help functions. It is not a handle, nor is it
usable by functions other than the I<ToolHelp> functions.

=item dwFlags

Flags. The values available are listed in the description of C<HF32_>
constants.

=back

=item MODULEENTRY32

Describes an entry from a list that enumerates the modules used
by a specified process.

=over 15

=item th32ModuleID

Module identifier in the context of the owning process. The contents
of this member has meaning only to the tool help functions. It is not
a handle, nor is it usable by functions other than the I<ToolHelp> functions.

=item th32ProcessID

Identifier of the process being examined.

=item GlblcntUsage

Global usage count on the module.

=item ProccntUsage

Module usage count in the context of the owning process.

=item modBaseAddr

Base address of the module in the context of the owning process.

=item modBaseSize

Size, in bytes, of the module.

=item hModule

Handle to the module in the context of the owning process.

=item szModule

String containing the module name.

=item szExePath

String containing the location (path) of the module.

=back

Note: The C<modBaseAddr> and C<hModule> members are valid only
in the context of the process specified by C<th32ProcessID>.

=item PROCESSENTRY32

Describes an entry from a list that enumerates the processes residing
in the system address space when a snapshot was taken.

=over 15

=item cntUsage

Number of references to the process. A process exists as long as its usage
count is nonzero. As soon as its usage count becomes zero, a process
terminates.

=item th32ProcessID

Identifier of the process.

=item th32DefaultHeapID

Identifier of the default heap for the process. The contents of this member
has meaning only to the tool help functions. It is not a handle, nor is it
usable by functions other than the I<ToolHelp> functions.

=item th32ModuleID

Module identifier of the process. The contents of this member has meaning
only to the tool help functions. It is not a handle, nor is it usable
by functions other than the I<ToolHelp> functions.

=item cntThreads

Number of execution threads started by the process.

=item th32ParentProcessID

Identifier of the process that created the process being examined.

=item pcPriClassBase

Base priority of any threads created by this process.

=item dwFlags

Reserved; do not use.

=item szExeFile

Path and filename of the executable file for the process.

=back

=item THREADENTRY32

Describes an entry from a list that enumerates the threads executing
in the system when a snapshot was taken.

=over 15

=item cntUsage

Number of references to the thread. A thread exists as long as its usage
count is nonzero. As soon as its usage count becomes zero, a thread
terminates.

=item th32ThreadID

Identifier of the thread. This identifier is compatible with the thread
identifier returned by the C<CreateProcess> function.

=item th32OwnerProcessID

Identifier of the process that created the thread.

=item tpBasePri

Initial priority level assigned to a thread. The values available are listed
in the description of C<THREAD_> constants.

=item tpDeltaPri

Change in the priority level of a thread. This value is a signed delta
from the base priority level assigned to the thread.

=item dwFlags

Reserved; do not use.

=back

=back

=head2 FUNCTIONS

I<ToolHelp> functions return either a handle to access its structures
or a boolean status of the function's result. To retrieve an extended
information about the error if it occurs use the C<GetLastToolHelpError>
function. If no error happens C<GetLastToolHelpError> still returns
the last occured error code (successful calls do not modify the last stored
error code). You can set or reset the internally stored error code
explicitely by the function C<SetLastToolHelpError>.

To use something more convenient than numbers for comparisons of return
values and error codes see the module C<Win32API::Const>.

There are couple of functions that are implemented as ANSI versions
on Windows 95 or higher and as both ANSI and Unicode versions on Windows 2000
or higher. ANSI versions are named XxxA and Unicode versions XxxW
just like the Win32 I<ToolHelp> originals. If you omit the last A/W letter
the ANSI version is used as strings are ANSI in Perl's internals. Results
of Unicode functions are converted into ANSI before returned.

=over

=item CloseToolhelp32Snapshot($hSnapshot)

Closes an open system snapshot handle.

=over

=item hSnapshot [IN]

Handle to the snapshot returned from a previous call
to the C<CreateToolhelp32Snapshot> function.

=item [RETVAL]

If the function succeeds, the return value is nonzero. If the function fails,
the return value is zero. To get extended error information, call
C<GetLastToolHelpError>.

Windows NT/2000/XP: Closing an invalid handle raises an exception
when the application is running under a debugger. This includes closing
a handle twice, for example.

=back

It invalidates the specified system snapshot handle, decrements the object's
handle count, and performs object retention checks. After the last handle
to an object is closed, the object is removed from the system.

=item CreateToolhelp32Snapshot($dwFlags, $th32ProcessID)

Takes a snapshot of the processes and the heaps, modules, and threads used
by the processes.

=over

=item dwFlags [IN]

Specifies portions of the system to include in the snapshot. The values
available are listed in the description of C<TH32CS_> constants.

=item th32ProcessID [IN]

Specifies the process identifier. This parameter can be zero to indicate
the current process. This parameter is used when the C<TH32CS_SNAPHEAPLIST>
or C<TH32CS_SNAPMODULE> value is specified. Otherwise, it is ignored.

=item [RETVAL]

Returns an open handle to the specified snapshot if successful
or 0xffffffff (C<INVALID_HANDLE_VALUE>) otherwise.

=back

The snapshot taken by this function is examined by the other I<ToolHelp>
functions to provide their results. Access to the snapshot is read only.
The snapshot handle acts like an object handle and is subject to the same
rules regarding which processes and threads it is valid in.

To retrieve an extended error status code generated by this function, use
the C<GetLastToolHelpError> function.

To destroy the snapshot, use the C<CloseToolhelp32Snapshot> function.
Alternatively you can use the general C<CloseHandle> function.

=item GetLastToolHelpError()

Retrieves the last-error code value of the I<ToolHelp> functions.
The last-error code is stored if a function fails and remembered until
another function calls when it is overwritten by the new error code.
Successful calls do not modify this internally stored last-error code value.

=over

=item [RETVAL]

The return value is the last-error code value. Functions set this value
by calling the C<SetLastToolHelpError> function if they fail.

=back

To obtain an error string for system error codes, use
the C<FormatMessage> function. For a complete list of error codes, see
the System Error Codes section in MSDN. There are pre-defined constants
for the Win32 system error codes in the module <Win32API::Const>.

You should call the C<GetLastToolHelpError> function immediately when
a function's return value indicates that such a call will return useful data.
A subsequent call to another I<ToolHelp> function could fail as well
and C<GetLastToolHelpError> would return its error code instead
of the former one.

Function failure is typically indicated by a return value such as zero,
undefined, or 1 (0xffffffff).

Error codes returned are 32-bit values with the most significant bit set
to 1 (bit 31 is the most significant bit). Zero code is C<ERROR_SUCCESS>.

=item Heap32First($lphe, $th32ProcessID, $th32HeapID)

Retrieves information about the first block of a heap that has been
allocated by a process.

=over

=item lphe [OUT]

Reference to a C<HEAPENTRY32> structure to be filled if the function
succeedes.

=item th32ProcessID [IN]

Identifier of the process context that owns the heap.

=item th32HeapID [IN]

Identifier of the heap to enumerate.

=item [RETVAL]

Returns TRUE if information for the first heap block has been copied
to the buffer or FALSE otherwise. The 18 (C<ERROR_NO_MORE_FILES>) error value
is returned by the C<GetLastToolHelpError> function if the heap is invalid
or empty.

=back

To access subsequent blocks of the same heap, use the C<Heap32Next> function.

=item Heap32ListFirst($hSnapshot, $lphl)

Retrieves information about the first heap that has been allocated
by a specified process.

=over

=item hSnapshot [IN]

Handle to the snapshot returned from a previous call
to the C<CreateToolhelp32Snapshot> function.

=item lphl [OUT]

Reference to a C<HEAPLIST32> structure to be filled if the function succeedes.

=item [RETVAL]

Returns TRUE if the first entry of the heap list has been copied to the buffer
or FALSE otherwise. The 18 (C<ERROR_NO_MORE_FILES>) error value is returned
by the C<GetLastToolHelpError> function when no heap list exists
or the snapshot does not contain heap list information.

=back

To retrieve information about other heaps in the heap list,
use the C<Heap32ListNext> function.

=item Heap32ListNext($hSnapshot, $lphl)

Retrieves information about the next heap that has been allocated
by a process.

=over

=item hSnapshot [IN]

Handle to the snapshot returned from a previous call
to the C<CreateToolhelp32Snapshot> function.

=item lphl [OUT]

Reference to a C<HEAPLIST32> structure to be filled if the function succeedes.

=item [RETVAL]

Returns TRUE if the next entry of the heap list has been copied to the buffer
or FALSE otherwise. The 18 (C<ERROR_NO_MORE_FILES>) error value is returned
by the C<GetLastToolHelpError> function when no more entries in the heap
list exist.

=back

To retrieve information about the first heap in a heap list,
use the C<Heap32ListFirst> function.

=item Heap32Next($lphe)

Retrieves information about the next block of a heap that has been
allocated by a process.

=over

=item lphe [IN/OUT]

Reference to a C<HEAPENTRY32> structure to be filled if the function
succeedes.

=item [RETVAL]

Returns TRUE if information about the next block in the heap has been copied
to the buffer or FALSE otherwise. The C<GetLastToolHelpError> function
returns 18 (C<ERROR_NO_MORE_FILES>) when no more objects in the heap exist
and 13 (C<ERROR_INVALID_DATA>) if the heap appears to be corrupt or is
modified during the walk in such a way that Heap32Next cannot continue.

=back

To retrieve information for the first block of a heap, use the C<Heap32First>
function.

=item Module32First($hSnapshot, $lpme)

Retrieves information about the first module associated with a process.

=over

=item hSnapshot [IN]

Handle to the snapshot returned from a previous call
to the C<CreateToolhelp32Snapshot> function.

=item lpme [OUT]

Reference to a C<MODULEENTRY32> structure to be filled if the function
succeedes.

=item [RETVAL]

Returns TRUE if the first entry of the module list has been copied
to the buffer or FALSE otherwise. The 18 (C<ERROR_NO_MORE_FILES>) error value
is returned by the C<GetLastToolHelpError> function if no modules exist
or the snapshot does not contain module information.

=back

This method actually calls the ANSI version C<Module32FirstA>, use
C<Module32FirstW> to call the Unicode version instead. To retrieve
information about other modules associated with the specified process,
use the C<Module32Next> function.

=item Module32Next($hSnapshot, $lpme)

Retrieves information about the next module associated with a process or thread.

=over

=item hSnapshot [IN]

Handle to the snapshot returned from a previous call
to the C<CreateToolhelp32Snapshot> function.

=item lpme [OUT]

Reference to a C<MODULEENTRY32> structure to be filled if the function
succeedes.

=item [RETVAL]

Returns TRUE if the next entry of the module list has been copied
to the buffer or FALSE otherwise. The 18 (C<ERROR_NO_MORE_FILES>) error value
is returned by the C<GetLastToolHelpError> function if no more modules exist.

=back

This method actually calls the ANSI version C<Module32NextA>, use
C<Module32NextW> to call the Unicode version instead. To retrieve information
about first module associated with a process, usethe C<Module32First>
function.

=item Process32First($hSnapshot, $lppe)

Retrieves information about the first process encountered in a system snapshot.

=over

=item hSnapshot [IN]

Handle to the snapshot returned from a previous call
to the C<CreateToolhelp32Snapshot> function.

=item lppe [OUT]

Reference to a C<PROCESSENTRY32> structure to be filled if the function
succeedes.

=item [RETVAL]

Returns TRUE if the first entry of the process list has been copied
to the buffer or FALSE otherwise. The 18 (C<ERROR_NO_MORE_FILES>) error value
is returned by the C<GetLastToolHelpError> function if no processes exist
or the snapshot does not contain process information.

=back

This method actually calls the ANSI version C<Process32FirstA>, use
C<Process32FirstW> to call the Unicode version instead. To retrieve
information about other processes recorded in the same snapshot, use
the C<Process32Next> function.

=item Process32Next($hSnapshot, $lppe)

Retrieves information about the next process recorded in a system snapshot.

=over

=item hSnapshot [IN]

Handle to the snapshot returned from a previous call
to the C<CreateToolhelp32Snapshot> function.

=item lppe [OUT]

Reference to a C<PROCESSENTRY32> structure to be filled if the function
succeedes.

=item [RETVAL]

Returns TRUE if the next entry of the process list has been copied
to the buffer or FALSE otherwise. The 18 (C<ERROR_NO_MORE_FILES>) error value
is returned by the C<GetLastToolHelpError> function if no processes exist
or the snapshot does not contain process information.

=back

This method actually calls the ANSI version C<Process32NextA>, use
C<Process32NextW> to call the Unicode version instead. To retrieve information
about the first process recorded in a snapshot, use the C<Process32First>
function.

=item SetLastToolHelpError($dwError)

Sets the last-error code value of the I<ToolHelp> functions.

=over

=item dwError [IN]

Specifies the last-error code.

=back

Error codes returned are 32-bit values with the most significant bit set
to 1 (bit 31 is the most significant bit). Zero code is C<ERROR_SUCCESS>.

Applications can retrieve the value saved by this function by using
the C<GetLastToolHelpError> function. The use of C<GetLastToolHelpError>
is optional; an application can call it to find out the specific reason
for a function failure.

=item Thread32First($hSnapshot, $lpte)

Retrieves information about the first thread of any process encountered
in a system snapshot.

=over

=item hSnapshot [IN]

Handle to the snapshot returned from a previous call
to the C<CreateToolhelp32Snapshot> function.

=item lpte [OUT]

Reference to a  THREADENTRY32 structure to be filled if the function
succeedes.

=item [RETVAL]

Returns TRUE if the first entry of the thread list has been copied
to the buffer or FALSE otherwise. The 18 (C<ERROR_NO_MORE_FILES>) error value
is returned by the C<GetLastToolHelpError> function if no threads exist
or the snapshot does not contain thread information.

=back

To retrieve information about other threads recorded in the same snapshot,
use the C<Thread32Next> function.

=item Thread32Next($hSnapshot, $lpte)

Retrieves information about the next thread of any process encountered
in the system memory snapshot.

=over

=item hSnapshot [IN]

Handle to the snapshot returned from a previous call
to the C<CreateToolhelp32Snapshot> function.

=item lpte [OUT]

Reference to a  THREADENTRY32 structure to be filled if the function
succeedes.

=item [RETVAL]

Returns TRUE if the next entry of the thread list has been copied
to the buffer or FALSE otherwise. The 18 (C<ERROR_NO_MORE_FILES>) error value
is returned by the C<GetLastToolHelpError> function if no threads exist
or the snapshot does not contain thread information.

=back

To retrieve information about the first thread recorded in a snapshot,
use the Thread32First function.

=item Toolhelp32ReadProcessMemory($th32ProcessID, $lpBaseAddress, $lpBuffer,
                                  $cbRead, $lpNumberOfBytesRead)

Copies memory allocated to another process into an application-supplied
buffer.

=over

=item th32ProcessID [IN]

Identifier of the process whose memory is being copied. This parameter can be
zero to copy the memory of the current process.

=item lpBaseAddress [IN]

Base address in the specified process to read. Before transferring any data,
the system verifies that all data in the base address and memory
of the specified size is accessible for read access. If this is the case,
the function proceeds. Otherwise, the function fails.

=item lpBuffer [OUT]

Reference to a scalar string that receives the contents of the address space
of the specified process.

=item bcRead [IN]

Number of bytes to read from the specified process.

=item lpNumberOfBytesRead [OUT]

Number of bytes copied to the specified buffer. This parameter can be omitted
if it is not needed.

=item [RETVAL]

Returns TRUE if successful.

=back

=back

=head1 AUTHOR

Ferdinand Prantl E<lt>F<prantl@host.sk>E<gt>

See F<http://prantl.host.sk/perl/modules/Win32API/ToolHelp>
for the most recent version.

=head1 COPYRIGHT

Copyright (c) 2002, Ferdinand Prantl. All rights reserved.

Permission to use, copy, modify, distribute and sell this software
and its documentation for any purpose is hereby granted without fee,
provided that the above copyright notice appear in all copies and
that both that copyright notice and this permission notice appear
in supporting documentation. Author makes no representations
about the suitability of this software for any purpose.  It is
provided "as is" without express or implied warranty.

=head1 SEE ALSO

L<Win32API::ProcessStatus>, L<Win32::Process>, L<Win32::Job>
and L<Win32API::Const>.

=cut
