/**************************************************************************
**
** Copyright (C) 2020 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of the Qt Installer Framework.
**
** $QT_BEGIN_LICENSE:GPL-EXCEPT$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3 as published by the Free Software
** Foundation with exceptions as appearing in the file LICENSE.GPL3-EXCEPT
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
**************************************************************************/

#include "../shared/packagemanager.h"

#include <updateoperations.h>
#include <packagemanagercore.h>
#include <settings.h>
#include <utils.h>

#include <QTest>

using namespace KDUpdater;
using namespace QInstaller;

class tst_deleteoperation : public QObject
{
    Q_OBJECT
private:
    void installFromCLI(const QString &repository)
    {
        QString installDir = QInstaller::generateTemporaryFileName();
        QVERIFY(QDir().mkpath(installDir));
        PackageManagerCore *core = PackageManager::getPackageManagerWithInit(installDir, repository);

        // Matches filename in component install script
        QFile file(installDir + QDir::toNativeSeparators("/test"));
        QVERIFY(file.open(QIODevice::ReadWrite));
        file.close();

        core->installDefaultComponentsSilently();
        QVERIFY(!file.exists());

        core->setPackageManager();
        core->commitSessionOperations();
        core->uninstallComponentsSilently(QStringList() << "A");
        QVERIFY(file.exists());

        QDir dir(installDir);
        QVERIFY(dir.removeRecursively());
        core->deleteLater();
    }

private slots:
    void testMissingArguments()
    {
        DeleteOperation op;

        QVERIFY(op.testOperation());
        QVERIFY(!op.performOperation());

        QCOMPARE(UpdateOperation::Error(op.error()), UpdateOperation::InvalidArguments);
        QCOMPARE(op.errorString(), QString("Invalid arguments in Delete: "
                                           "0 arguments given, exactly 1 arguments expected."));

        op.setArguments(QStringList() << "");
        QTest::ignoreMessage(QtWarningMsg, "QFile::copy: Empty or null file name");
        op.backup();

        QCOMPARE(UpdateOperation::Error(op.error()), UpdateOperation::UserDefinedError);
        QCOMPARE(op.errorString(), QString("Cannot create backup of file \"\": Unknown error"));

        // Returns true with empty file argument but does not do anything
        QVERIFY(op.performOperation());
    }

    void testDeleteRestore_data()
    {
        QTest::addColumn<QString>("path");
        QTest::newRow("relative") << "test";
        QTest::newRow("absolute") << qApp->applicationDirPath() + QDir::toNativeSeparators("/test");
    }

    void testDeleteRestore()
    {
        QFETCH(QString, path);

        QByteArray testString("Generated by QTest\n");
        QFile testFile(path);
        QVERIFY(testFile.open(QIODevice::WriteOnly | QIODevice::Text));
        QTextStream out(&testFile);
        out << testString;
        testFile.close();

        QVERIFY(QFileInfo(path).exists());
        QByteArray testFileHash = QInstaller::calculateHash(path, QCryptographicHash::Sha1);

        DeleteOperation op;
        op.setArguments(QStringList() << path);

        op.backup();
        QVERIFY(QFileInfo(op.value("backupOfExistingFile").toString()).exists());

        QVERIFY2(op.performOperation(), op.errorString().toLatin1());
        QVERIFY(!QFileInfo(path).exists());

        QVERIFY2(op.undoOperation(), op.errorString().toLatin1());
        QByteArray restoredFileHash = QInstaller::calculateHash(path, QCryptographicHash::Sha1);
        QVERIFY(testFileHash == restoredFileHash);

        QVERIFY(QFile(path).remove());
    }

    void testDeleteFromScript()
    {
        installFromCLI(":///data/repository");
    }

    void testDeleteFromXML()
    {
        installFromCLI(":///data/xmloperationrepository");
    }
};

QTEST_MAIN(tst_deleteoperation)

#include "tst_deleteoperation.moc"
