//
// MODTOOL
//
// Copyright (c) !998 David Lindauer (LADSOFT)
//
// see license.txt for licensing info
//
// ==============================================================
//
// network.cpp
//
// this is network primitives used by all the network state machines,
// e.g. connect, read, write, and an error
//
#define STRICT
#include <windows.h>
#include <winsock.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>

#include "effects.h"
#include "mailfile.h"
#include "analyze.h"
#include "profile.h"
#define NOEXTERN
#include "network.h"

int networkinuse;	// Set true if related menu items should be greyed


extern HWND hWndMain;

// ==============================================================
//
// global network callback.
// called by winproc for winsock events
//
int (* NetworkCallback)(HWND hWnd, int mode, int error);

static FILE *workfile; // work file
static int unmangle;	// state of network mangling, for a partial
			// message sequence terminated by CR.
			// e.g. beginning of next part is beginning of
			// a line

// ==============================================================
//
// network error messages
//
void NetworkMessageBox(int flag, char *fmt, ...)
{
	char string[512];
	va_list argptr;

	va_start( argptr, fmt);
  	vsprintf(string, fmt, argptr);
	va_end(argptr);
	MessageBox(0,string,"Network Error",flag);
}
void NoHost(char *server)
{
	NetworkMessageBox(MB_ICONERROR, "No DNS entry for %s\n(Unknown host)",server);
}
void NoConnect(char *server)
{
	NetworkMessageBox(MB_ICONERROR, "Can't connect to host  %s",server);
}
// ==============================================================
//
// winsock connection routine
//
// we connect asynchronously, specifying that winproc is going to
// get a WM_USER message when an event happens on the socket
//
SOCKET StreamedNetworkConnect(char *host, int port, SOCKET *rv)
{
	SOCKET S;
	struct sockaddr_in Sa;
	struct hostent *H;
	
	// get socket
	S=socket(AF_INET, SOCK_STREAM,0);
	Sa.sin_family=AF_INET;
	Sa.sin_port = htons(port);

	// get IP address of specified host
	H=gethostbyname(host);
	if (!H)
		return (SOCKET)NETWORK_NOHOST;

	Sa.sin_addr.s_addr=*((unsigned long *) H->h_addr);

	// Set up async and connect;
	WSAAsyncSelect(S,hWndMain,WM_USER,FD_READ | FD_WRITE | FD_CLOSE | FD_CONNECT);
	connect(S,(struct sockaddr *) &Sa,sizeof(Sa));

	// mark in use and return
	networkinuse = TRUE;
	*rv = S;
	return NETWORK_OK;
}
// ==============================================================
//
// disconnect from network
//

int StreamedNetworkDisconnect(SOCKET s)
{
	networkinuse = 0;
	closesocket(s);
	return 0;
}
// ==============================================================
//
// send data over network
//
int StreamedNetworkSend(SOCKET S, char *fmt,...)
{
	char string[512];

	// format the data
	va_list argptr;

	va_start( argptr, fmt);
  	vsprintf(string, fmt, argptr);
	va_end(argptr);

	// add a line terminate sequence
	strcat(string,"\r\n");

	// send the data
	if (send(S,string,strlen(string),0) == SOCKET_ERROR) {
		return NETWORK_NOWRITE;
	}
	return NETWORK_OK;
}
// ==============================================================
//
// receive data from network
//
int StreamedNetworkReceive(SOCKET S, char *string, int len)
{
	int i = recv(S,string,len,0);
	string[i] = 0;
	if (i == SOCKET_ERROR)
		return NETWORK_NOREAD;
	return NETWORK_OK;
}
// ==============================================================
//
// untangles the data blocks being received.  Mail and news messages
// come in in random chunks with no pattern to the breaks.
//
// this routine strips out the transparency dots
// flag is true if the first character in the string is also the
// first character on a line, signified by receiving a '\n' as the
// last character of the last packet
//
// this basically locates "\n." sequences and translates them to "\n"
//
int StreamedUntangle(char *string, int flag)
{
	char * offset;
	offset = string;

	// If start of line and is dot get rid of it
	if (flag && string[0] == '.')
		strcpy(offset, offset + 1);

	// now go through the buffer and translate
	while (offset = strstr(offset+1,"\n."))
		strcpy(offset+1,offset+2);

	// decide whether the end of the buffer is an end of line
	return string[strlen(string)-1] == '\n';
}
// ==============================================================
//
// This one gathers the new message into a temporary file as it comes
// in, then when the ending "\r\n\.\r\n" sequence is found we copy
// the temp file to our destination
//
// we return the status for use by the input state machines
//
int AppendToReceivedArticle(char *tempname,char *mailname,char *string, int sortable)
{
	char *offset = string,*nextline,**header;
	int len = strlen(string),count;

	// Find out if end of message and cut it out if so
	int rv = string[len-4] == '\n' && string[len-3] == '.' && string[len-2] == '\r';
	if (rv)
		string[len-3] = 0;

	// make sure work file is open 
	if (!workfile) {
		if (!(workfile = fopen(tempname,"wb"))) {
			ExtendedMessageBox("File Error",MB_ICONERROR,"Opening Temp File");
			return -1;
		}
		unmangle = 0;
	}

	// unmangle the string and stuff it in the work file
	unmangle = StreamedUntangle(string,unmangle);
	if (fputs(string,workfile) == EOF) {
		ExtendedMessageBox("File Error",MB_ICONERROR,"Writing to temp file");
		fclose(workfile);
		workfile=0;
		return -1;
	}
	// if done, move the work file to the end of the destination file
	if (rv) {
		// assume going to destination
		char *newname = mailname;
		
		// read headers and close file
		fclose(workfile);
		workfile = fopen(tempname,"r");
		if (!workfile) {
			ExtendedMessageBox("File Error",MB_ICONERROR,"Opening Temp File");
			return -1;
		}
		ReadHeaders(workfile,&count,&header,&nextline,0);
		fclose(workfile);
		workfile=0;

		// Check if sorting to inbox
		if (sortable && sortmail && !IsToGroup(count,header))
			newname = inbox;
		// free headers again
		FreeHeaders(count,header);

		// now append it to the dest file
		if (AppendNewToMailFile(newname,tempname) != FILE_OK) {
			ExtendedMessageBox("File Error",MB_ICONERROR,"Appending to mail folder %s",mailname);
			return -1;
		}
	}

	// return state for state machine
	return rv ? 0 : 1;
}	
// ==============================================================
//
// make sure the work file is closed
//
void CloseWorkFile(void)
{
	if (workfile)
		fclose(workfile);
	workfile = 0;
}