//
// MODTOOL
//
// Copyright (c) !998 David Lindauer (LADSOFT)
//
// see license.txt for licensing info
//
// ==============================================================
//
// NNTPRCV.CPP
//
// receive news from NNTP server
//
// This is a state machine.  Rather than comment every state I give a brief
// summary of transactions here:
//
// local:  connect to winsock socket and host
// remote: send either 200 (posting allowed) or 201 (posting not allowed)
// local:  send "GROUP mygroup" to select a group
// remote: send 211 along with group statistics (start and end)
//
// now we go into a loop and do the following once for each message:
//
//	local:  send "ARTICLE #" command to get a message
//	remote: sends back 423 to note cancellation of message
//		sends back 220 tp note the message will be sent
//		Note the first block of text the host sends
//		is tacked on here.  However there may be more blocks
//		so the local machine must loop receiving blocks until
//		the "\r\n.\r\n" terminating sequence is discovered
//
// local:  send "QUIT" command and disconnect
//


#define STRICT
#include <windows.h>
#include <winsock.h>
#include <stdio.h>
#include <stdarg.h>
#include <commctrl.h>
#include <time.h>

#include "progress.h"
#include "effects.h"
#include "network.h"
#include "profile.h"
#include "archive.h"

// ==============================================================
//
// states
//
#define NS_EXPCONNECT 0
#define NS_SENDGROUP 1
#define NS_RECEIVEGROUP 2
#define NS_SENDARTICLE 3
#define NS_RECEIVEARTICLE 4
#define NS_RECEIVEPARTIAL 5
#define NS_RECEIVETIMEOUT 6
#define NS_ERROR 7
#define NS_ABORT 8
#define MS_IGNORE 9

// ==============================================================
//
// winsock flags
#define FL_SEND 1
#define FL_RECEIVE 2

extern HINSTANCE hInst;
extern HWND hWndProgress,hWndMain;
extern int networkinuse;
extern int batchflags;

static SOCKET worksocket;	// work socket
static int state;		// current state
static int inprogress;		// in progress
static int flags;		// winsock flags

static char *tempname = "readnews.tmp";	// news temp file
static char *server;		// server selected

// ==============================================================
//
// state machine
//


static int callback(HWND hWnd, int mode, int error)
{
	static int oldmessage,messages,firstmessage,messageguess,total;
	static char string[10000];
	int t;

	// handle winsock errors
	if (error == SOCKET_ERROR)  {
		state = NS_ERROR;
	}

	// handle cancel button
	if (inprogress && !hWndProgress) {
		NetworkMessageBox(MB_ICONINFORMATION,"User canceled transfer");
		state = NS_ABORT;
	}
	// ignore close events
	if (mode == FD_CLOSE) {
		return 0;
	}

	// verify the connection
	if (mode == FD_CONNECT) {
		if (error == SOCKET_ERROR) {
			NoConnect(server);
			networkinuse = 0;
		}
		state = NS_EXPCONNECT;
		return 0;
	}

	// handle winsock states
	if (mode == FD_WRITE)
		flags |= FL_SEND;
	if (mode == FD_READ)
		flags |= FL_RECEIVE;

	switch(state) {
		case NS_EXPCONNECT:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			t = atoi(string);
			if (t != 200 && t != 201) {
				NetworkMessageBox(MB_ICONERROR,"NNTP host refused connection");
				goto abort;
			}
			state = NS_SENDGROUP;
		case NS_SENDGROUP:
			if (!(flags & FL_SEND))
				break;
			flags &= ~FL_SEND;
			if (StreamedNetworkSend(worksocket,"GROUP %s",groupid) != NETWORK_OK)
				goto doerror;
			state = NS_RECEIVEGROUP;
			break;
		case NS_RECEIVEGROUP:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			t = atoi(string);
			if (t != 211) {
				NetworkMessageBox(MB_ICONERROR,"NNTP host failed GROUP command, aborting");
				goto abort;
			}
			sscanf(string+4,"%ld %ld %ld",&messageguess,&firstmessage,&messages);
			messages-=firstmessage-1;
			oldmessage = ReadArchiveCountProfile();
			if (oldmessage > firstmessage) {
				messages -= oldmessage - firstmessage;
				firstmessage = oldmessage;
			}
			total = 0;
			inprogress = 1;
			MakeProgress(hWndMain,hInst,"Receiving News",messages);
			state = NS_SENDARTICLE;
		case NS_SENDARTICLE:
sendnext:
			WriteArchiveCountProfile(total + firstmessage);
			if (total > messages)
				goto abort;
			if (StreamedNetworkSend(worksocket,"ARTICLE %d",firstmessage+total++) != NETWORK_OK)
				goto doerror;
			SetProgress(total,"Messages received: %3d of %3d",total,messages+1);
			state = NS_RECEIVEARTICLE;
			break;
		case NS_RECEIVEARTICLE:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			t = atoi(string);
			if (t == 423) {
				goto sendnext;
			}
			else if (t != 220) {
				NetworkMessageBox(MB_ICONERROR,"NNTP host failed ARTICLE command, aborting");
				goto abort;
			}
			state = NS_RECEIVEPARTIAL;
			switch (AppendToReceivedArticle(tempname,GetArchiveName(),strstr(string,"\r\n")+2,FALSE)) {
				case -1:
					goto abort;
				case 0:
					goto sendnext;
			}
			break;
		case NS_RECEIVEPARTIAL:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			switch (AppendToReceivedArticle(tempname,GetArchiveName(),string,FALSE)) {
				case -1:
					goto abort;
				case 0:
					goto sendnext;
			}
			break;
doerror:
		case NS_ERROR:
			NetworkMessageBox(MB_ICONERROR,"WINSOCK error, aborting");
abort:
		case NS_ABORT:
			WriteArchiveCountProfile(total + firstmessage);
			DeleteProgress();
			inprogress = 0;
			state = MS_IGNORE;
			StreamedNetworkSend(worksocket,"QUIT");
			StreamedNetworkDisconnect(worksocket);
			CloseWorkFile();
			unlink(tempname);
			if (batchflags)
				PostQuitMessage(0);
			break;
		case MS_IGNORE:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			StreamedNetworkSend(worksocket,"QUIT");
			StreamedNetworkDisconnect(worksocket);
			break;
	}

}
// ==============================================================
//
// receive main routine
//
int ReceiveNewsMessages(void)
{
	FILE *fil;

	// check temp file
	if (!(fil = fopen(tempname,"wb"))) {
		ExtendedMessageBox("Receive News Error",MB_ICONERROR,"File %s can't be created",tempname);
		return 0;
	}
	fclose(fil);

	// set up callback
	NetworkCallback = callback;

	// select server
	if (!usenntp) 
		server = nntparchiveserver;
	else
		server = nntppostserver;

	// start connection
	switch(StreamedNetworkConnect(server,NNTPPort,&worksocket)) {
		case NETWORK_OK:
			flags = 0;
			
			break;
		case NETWORK_NOHOST:
			NoHost(server);
			break;
	}
}