//
// MODTOOL
//
// Copyright (c) !998 David Lindauer (LADSOFT)
//
// see license.txt for licensing info
//
// ==============================================================
//
// POP3.CPP
//
// Mail receiver, receives entire contents of a pop3 mailbox
//
// This is a state machine.  Rather than comment every state I give a brief
// summary of transactions here:
//
// local:  issues winsock connect command
// remote: returns +OK
// local:  sends "USER username" with the username
// remote: returns +OK
// local:  sends "PASS password" with the password
//
// at this point the state machine does a variety of things devoted
// to error recovery if the password was invalid:
//
// remote: sendse "-ERR" message to signify invalid password
// local:  sends "QUIT" message and then restarts connection from beginning
//		after prompting for new password
//
// on the other hand if the password was correct we complete the connection
//
// local:  send "STAT" command to get status
// remote: responds with the number of messages available and the total
//         transfer size
//
// now we go into a loop and do the following once for each message:
//
//	local:  send "RETR #" command to get a message
//	remote: sends back "+OK #####\r\ntext" where ##### is the size of this
//		message.  Note the first block of text the host sends
//		is tacked on here.  However there may be more blocks
//		so the local machine must loop receiving blocks until
//		the "\r\n.\r\n" terminating sequence is discovered
//
//	if we are deleting messages as they come in:
//
//	local:	send "DELE #" command to delete it
//	remote: send +OK
//
// after all messages have been gathered:
//
// local: send "QUIT"
// local: disconnect, discard final message
//
// 

#define STRICT
#include <windows.h>
#include <winsock.h>
#include <stdio.h>
#include <stdarg.h>
#include <commctrl.h>
#include <time.h>

#include "progress.h"
#include "effects.h"
#include "network.h"
#include "profile.h"
#include "dialog.h"
#include "winconst.h"

// ==============================================================
//
// states
//
#define PS_EXPCONNECT 0
#define PS_SENDUSER 1
#define PS_RECEIVEUSER 2
#define PS_SENDPASSWORD 3
#define PS_RECEIVEPASSWORD 4
#define PS_PASSFAILSENDQUIT 5
#define PS_PASSFAILRECEIVEQUIT 6
#define PS_PASSFAILRECONNECT 7
#define PS_SENDSTAT 8
#define PS_RECEIVESTAT 9
#define PS_SENDRETRIEVE 10
#define PS_RECEIVERETRIEVE 11
#define PS_RECEIVEPARTIAL 12
#define PS_SENDDELE 13
#define PS_RECEIVEDELE 14
#define PS_RECEIVETIMEOUT 15
#define PS_ERROR 16
#define PS_ABORT 17
#define PS_IGNORE 18

// ==============================================================
//
// flags for WINSOCK
#define FL_SEND 1
#define FL_RECEIVE 2

extern HINSTANCE hInst;
extern HWND hWndProgress,hWndMain;
extern int networkinuse;
extern int batchflags;

static SOCKET worksocket;	/* work socket */
static int state;		/* current state */
static int inprogress;		/* in progress marker */
static int flags;		/* WINSOCK flags */

static char *tempname = "readmail.tmp";	/* input temp file */
static char pop3name[256];	/* name of pop3 output file */
static char *server;		/* name of server we are using */
static char intermedpassword[256]; /* password so we don't clobber the ini file */

// ==============================================================
//
// dialog box to get password input
//
int FAR PASCAL POP3PasswordProc (HWND hWnd,UINT wmsg,WPARAM wparam,LPARAM lparam)
{
	switch (wmsg) {
		case WM_INITDIALOG:
			CenterWindow(hWnd);
			NewFocus(hWnd,IDC_POP3PASSWORD);
			SetEditField(hWnd,IDC_POP3PASSWORD,"");
			return TRUE;
		case WM_CLOSE:
			SendMessage(hWnd,WM_COMMAND,IDCANCEL,0);
			break;
		case WM_COMMAND:
			switch(LOWORD(wparam)) {
				case IDOK:
					GetEditField(hWnd,IDC_POP3PASSWORD,intermedpassword);
					EndDialog(hWnd,1);
					return true;
				case IDCANCEL:
					EndDialog(hWnd,0);
					return true;
			}
	}
	return FALSE;
}

// ==============================================================
//
// state machine
int ReceivePOP3Messages(int copypass);

static int callback(HWND hWnd, int mode, int error)
{
	static int messages,total,currentsize,thissize,totalsize;
	static char string[10000];

	// error from WINSOCK
	if (error == SOCKET_ERROR)  {
		state = PS_ERROR;
	}

	// cancel button pressed
	if (inprogress && !hWndProgress) {
		NetworkMessageBox(MB_ICONINFORMATION,"User canceled transfer");
		state = PS_ABORT;
	}

	// ignore close message
	if (mode == FD_CLOSE) {
		return 0;
	}

	// verify the connect
	if (mode == FD_CONNECT) {
		if (error == SOCKET_ERROR) {
			NoConnect(server);
			networkinuse = 0;
		}
		else
			state = PS_EXPCONNECT;
		return 0;
	}

	// winsock flags
	if (mode == FD_WRITE)
		flags |= FL_SEND;
	if (mode == FD_READ)
		flags |= FL_RECEIVE;

	switch(state) {
		case PS_EXPCONNECT:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			if (strncmp(string,"+OK",3)) {
				NetworkMessageBox(MB_ICONERROR,"POP3 host refused connection");
				goto abort;
			}
			state = PS_SENDUSER;
		case PS_SENDUSER:
			MakeProgress(hWndMain,hInst,"Receiving Mail",1000);
			SetProgress(0,"Logging on to user account",total,messages+1);
			inprogress = 1;
senduser:
			if (StreamedNetworkSend(worksocket,"USER %s",pop3username) != NETWORK_OK)
				goto doerror;
			state = PS_RECEIVEUSER;
			break;
		case PS_RECEIVEUSER:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_SEND;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			if (strncmp(string,"+OK",3)) {
				NetworkMessageBox(MB_ICONERROR,"POP3 USER command failed");
				goto abort;
			}
			state = PS_SENDPASSWORD;
		case PS_SENDPASSWORD:
			if (StreamedNetworkSend(worksocket,"PASS %s",intermedpassword) != NETWORK_OK)
				goto doerror;
			state = PS_RECEIVEPASSWORD;
			break;
		case PS_RECEIVEPASSWORD:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			if (strncmp(string,"+OK",3)) {
				if (DialogBox(hInst,"DLG_PASSPROMPT",hWnd,POP3PasswordProc))
					goto repass;
				else
					goto abort;
			}
			state = PS_SENDSTAT;
		case PS_SENDSTAT:
			if (StreamedNetworkSend(worksocket,"STAT") != NETWORK_OK)
				goto doerror;

			state = PS_PASSFAILRECEIVEQUIT;
			state = PS_RECEIVESTAT;
			break;
repass:
		case PS_PASSFAILSENDQUIT:
			if (StreamedNetworkSend(worksocket,"QUIT") != NETWORK_OK)
				goto doerror;

			state = PS_PASSFAILRECEIVEQUIT;
			break;
		case PS_PASSFAILRECEIVEQUIT:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_SEND;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			if (strncmp(string,"+OK",3)) {
				NetworkMessageBox(MB_ICONERROR,"POP3 QUIT command failed");
				goto abort;
			}
			state = PS_PASSFAILRECONNECT;
		case PS_PASSFAILRECONNECT:
			ReceivePOP3Messages(FALSE);
			state = PS_EXPCONNECT;
			break;
		case PS_RECEIVESTAT:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			if (strncmp(string,"+OK",3)) {
				NetworkMessageBox(MB_ICONERROR,"POP3 STAT command failed");
				goto abort;
			}
			sscanf(string+4,"%ld %ld",&messages,&totalsize);
			total = 0;
			currentsize = 0;
			state = PS_SENDRETRIEVE;
		case PS_SENDRETRIEVE:
sendnext:
			if (total >= messages)
				goto abort;
			if (StreamedNetworkSend(worksocket,"RETR %d",++total) != NETWORK_OK)
				goto doerror;
			state = PS_RECEIVERETRIEVE;
			SetProgress(1000*currentsize/totalsize,"Messages received: %3d of %3d",total,messages);
			break;
		case PS_RECEIVERETRIEVE:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			if (strncmp(string,"+OK",3)) {
				NetworkMessageBox(MB_ICONERROR,"POP3 RETR command failed");
				goto abort;
			}
			currentsize += strlen(string);
			state = PS_RECEIVEPARTIAL;
			switch (AppendToReceivedArticle(tempname,pop3name,strstr(string,"\r\n")+2,TRUE)) {
				case -1:
					goto abort;
				case 0:
					goto sendnext;
			}
			break;
		case PS_RECEIVEPARTIAL:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			currentsize += strlen(string);
			SetProgress(1000*currentsize/totalsize,"Messages received: %3d of %3d",total,messages);
			switch (AppendToReceivedArticle(tempname,pop3name,string,TRUE)) {
				case -1:
					goto abort;
				case 0:
					goto senddele;
			}
			break;
		case PS_SENDDELE:
senddele:
			if (!pop3dele)
				goto sendnext;
			if (StreamedNetworkSend(worksocket,"DELE %d",total) != NETWORK_OK)
				goto doerror;
			state = PS_RECEIVEDELE;
			break;
		case PS_RECEIVEDELE:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			if (strncmp(string,"+OK",3)) {
				NetworkMessageBox(MB_ICONERROR,"POP3 DELE command failed");
				goto abort;
			}

			state = PS_SENDRETRIEVE;
			goto sendnext;
doerror:
		case PS_ERROR:
			NetworkMessageBox(MB_ICONERROR,"WINSOCK error, aborting");
abort:
		case PS_ABORT:
			DeleteProgress();
			inprogress = 0;
			state = PS_IGNORE;
			StreamedNetworkSend(worksocket,"QUIT");
			StreamedNetworkDisconnect(worksocket);
			CloseWorkFile();
			unlink(tempname);
			if (batchflags)
				PostQuitMessage(0);
			break;
		case PS_IGNORE:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			StreamedNetworkSend(worksocket,"QUIT");
			StreamedNetworkDisconnect(worksocket);
			break;
	}

}
// ==============================================================
//
// main line for pop 3 input
int ReceivePOP3Messages(int copypass)
{
	FILE *fil;
	// verify the temp file
	if (!(fil = fopen(tempname,"wb"))) {
		ExtendedMessageBox("Receive Mail Error",MB_ICONERROR,"File %s can't be created",tempname);
		return 0;
	}
	fclose(fil);

	// set up callback
	NetworkCallback = callback;

	// set up server
	server = pop3server;

	// if this is the first time get the ini file password
	if (copypass)
		strcpy(intermedpassword,pop3password);

	// now go connect
	switch(StreamedNetworkConnect(server,POP3Port,&worksocket)) {
		case NETWORK_OK:
			flags = 0;
			strcpy(pop3name,mailfolder);
			break;
		case NETWORK_NOHOST:
			NoHost(server);
			break;
	}
}