/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.hazelcast;

import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.topic.ITopic;
import org.apache.camel.CamelExecutionException;
import org.apache.camel.builder.RouteBuilder;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mock;

import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.mockito.Mockito.atLeastOnce;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;

public class HazelcastReliableTopicProducerTest extends HazelcastCamelTestSupport {

    @Mock
    private ITopic<String> reliableTopic;

    @Override
    protected void trainHazelcastInstance(HazelcastInstance hazelcastInstance) {
        when(hazelcastInstance.<String> getReliableTopic("bar")).thenReturn(reliableTopic);
    }

    @Override
    protected void verifyHazelcastInstance(HazelcastInstance hazelcastInstance) {
        verify(hazelcastInstance, atLeastOnce()).getReliableTopic("bar");
    }

    @AfterEach
    public void verifyQueueMock() {
        verifyNoMoreInteractions(reliableTopic);
    }

    @Test
    public void testWithInvalidOperation() {
        assertThrows(CamelExecutionException.class,
                () -> template.sendBody("direct:publishInvalid", "foo"));
    }

    @Test
    public void noOperation() {
        template.sendBody("direct:no-operation", "bar");
        verify(reliableTopic).publish("bar");
    }

    @Test
    public void publish() {
        template.sendBody("direct:publish", "bar");
        verify(reliableTopic).publish("bar");
    }

    @Override
    protected RouteBuilder createRouteBuilder() throws Exception {
        return new RouteBuilder() {
            @Override
            public void configure() throws Exception {
                from("direct:no-operation").to(String.format("hazelcast-%sbar?reliable=true", HazelcastConstants.TOPIC_PREFIX));

                from("direct:publishInvalid").setHeader(HazelcastConstants.OPERATION, constant("bogus"))
                        .to(String.format("hazelcast-%sbar?reliable=true", HazelcastConstants.TOPIC_PREFIX));

                from("direct:publish").setHeader(HazelcastConstants.OPERATION, constant(HazelcastOperation.PUBLISH))
                        .to(String.format("hazelcast-%sbar?reliable=true",
                                HazelcastConstants.TOPIC_PREFIX));
            }
        };
    }

}
