/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math3.ode.nonstiff;

import org.apache.commons.math3.exception.DimensionMismatchException;
import org.apache.commons.math3.exception.MaxCountExceededException;
import org.apache.commons.math3.exception.NoBracketingException;
import org.apache.commons.math3.exception.NumberIsTooSmallException;
import org.apache.commons.math3.ode.FirstOrderDifferentialEquations;
import org.apache.commons.math3.ode.ODEIntegrator;
import org.apache.commons.math3.ode.TestProblem1;
import org.apache.commons.math3.ode.TestProblem3;
import org.apache.commons.math3.ode.TestProblem4;
import org.apache.commons.math3.ode.TestProblem5;
import org.apache.commons.math3.ode.TestProblemHandler;
import org.apache.commons.math3.ode.events.EventHandler;
import org.apache.commons.math3.ode.nonstiff.DormandPrince853Integrator;
import org.apache.commons.math3.ode.nonstiff.StepProblem;
import org.apache.commons.math3.ode.sampling.StepHandler;
import org.apache.commons.math3.ode.sampling.StepInterpolator;
import org.apache.commons.math3.util.FastMath;
import org.junit.Assert;
import org.junit.Test;

public class DormandPrince853IntegratorTest {
    @Test
    public void testMissedEndEvent() throws DimensionMismatchException, NumberIsTooSmallException, MaxCountExceededException, NoBracketingException {
        int i;
        double t0 = 1.8782503200000029E9;
        double tEvent = 1.8782503799999986E9;
        final double[] k = new double[]{1.0E-4, 1.0E-5, 1.0E-6};
        FirstOrderDifferentialEquations ode = new FirstOrderDifferentialEquations(){

            public int getDimension() {
                return k.length;
            }

            public void computeDerivatives(double t, double[] y, double[] yDot) {
                for (int i = 0; i < y.length; ++i) {
                    yDot[i] = k[i] * y[i];
                }
            }
        };
        DormandPrince853Integrator integrator = new DormandPrince853Integrator(0.0, 100.0, 1.0E-10, 1.0E-10);
        double[] y0 = new double[k.length];
        for (int i2 = 0; i2 < y0.length; ++i2) {
            y0[i2] = i2 + 1;
        }
        double[] y = new double[k.length];
        integrator.setInitialStepSize(60.0);
        double finalT = integrator.integrate(ode, 1.8782503200000029E9, y0, 1.8782503799999986E9, y);
        Assert.assertEquals((double)1.8782503799999986E9, (double)finalT, (double)5.0E-6);
        for (i = 0; i < y.length; ++i) {
            Assert.assertEquals((double)(y0[i] * FastMath.exp((double)(k[i] * (finalT - 1.8782503200000029E9)))), (double)y[i], (double)1.0E-9);
        }
        integrator.setInitialStepSize(60.0);
        integrator.addEventHandler(new EventHandler(){

            public void init(double t0, double[] y0, double t) {
            }

            public void resetState(double t, double[] y) {
            }

            public double g(double t, double[] y) {
                return t - 1.8782503799999986E9;
            }

            public EventHandler.Action eventOccurred(double t, double[] y, boolean increasing) {
                Assert.assertEquals((double)1.8782503799999986E9, (double)t, (double)5.0E-6);
                return EventHandler.Action.CONTINUE;
            }
        }, Double.POSITIVE_INFINITY, 1.0E-20, 100);
        finalT = integrator.integrate(ode, 1.8782503200000029E9, y0, 1.8782504999999986E9, y);
        Assert.assertEquals((double)1.8782504999999986E9, (double)finalT, (double)5.0E-6);
        for (i = 0; i < y.length; ++i) {
            Assert.assertEquals((double)(y0[i] * FastMath.exp((double)(k[i] * (finalT - 1.8782503200000029E9)))), (double)y[i], (double)1.0E-9);
        }
    }

    @Test(expected=DimensionMismatchException.class)
    public void testDimensionCheck() throws DimensionMismatchException, NumberIsTooSmallException, MaxCountExceededException, NoBracketingException {
        TestProblem1 pb = new TestProblem1();
        DormandPrince853Integrator integrator = new DormandPrince853Integrator(0.0, 1.0, 1.0E-10, 1.0E-10);
        integrator.integrate((FirstOrderDifferentialEquations)pb, 0.0, new double[pb.getDimension() + 10], 1.0, new double[pb.getDimension() + 10]);
        Assert.fail((String)"an exception should have been thrown");
    }

    @Test(expected=NumberIsTooSmallException.class)
    public void testNullIntervalCheck() throws DimensionMismatchException, NumberIsTooSmallException, MaxCountExceededException, NoBracketingException {
        TestProblem1 pb = new TestProblem1();
        DormandPrince853Integrator integrator = new DormandPrince853Integrator(0.0, 1.0, 1.0E-10, 1.0E-10);
        integrator.integrate((FirstOrderDifferentialEquations)pb, 0.0, new double[pb.getDimension()], 0.0, new double[pb.getDimension()]);
        Assert.fail((String)"an exception should have been thrown");
    }

    @Test(expected=NumberIsTooSmallException.class)
    public void testMinStep() throws DimensionMismatchException, NumberIsTooSmallException, MaxCountExceededException, NoBracketingException {
        TestProblem1 pb = new TestProblem1();
        double minStep = 0.1 * (pb.getFinalTime() - pb.getInitialTime());
        double maxStep = pb.getFinalTime() - pb.getInitialTime();
        double[] vecAbsoluteTolerance = new double[]{1.0E-15, 1.0E-16};
        double[] vecRelativeTolerance = new double[]{1.0E-15, 1.0E-16};
        DormandPrince853Integrator integ = new DormandPrince853Integrator(minStep, maxStep, vecAbsoluteTolerance, vecRelativeTolerance);
        TestProblemHandler handler = new TestProblemHandler(pb, (ODEIntegrator)integ);
        integ.addStepHandler((StepHandler)handler);
        integ.integrate((FirstOrderDifferentialEquations)pb, pb.getInitialTime(), pb.getInitialState(), pb.getFinalTime(), new double[pb.getDimension()]);
        Assert.fail((String)"an exception should have been thrown");
    }

    @Test
    public void testIncreasingTolerance() throws DimensionMismatchException, NumberIsTooSmallException, MaxCountExceededException, NoBracketingException {
        int previousCalls = Integer.MAX_VALUE;
        DormandPrince853Integrator integ = new DormandPrince853Integrator(0.0, Double.POSITIVE_INFINITY, Double.NaN, Double.NaN);
        for (int i = -12; i < -2; ++i) {
            TestProblem1 pb = new TestProblem1();
            double minStep = 0.0;
            double maxStep = pb.getFinalTime() - pb.getInitialTime();
            double scalAbsoluteTolerance = FastMath.pow((double)10.0, (int)i);
            double scalRelativeTolerance = 0.01 * scalAbsoluteTolerance;
            integ.setStepSizeControl(minStep, maxStep, scalAbsoluteTolerance, scalRelativeTolerance);
            TestProblemHandler handler = new TestProblemHandler(pb, (ODEIntegrator)integ);
            integ.addStepHandler((StepHandler)handler);
            integ.integrate((FirstOrderDifferentialEquations)pb, pb.getInitialTime(), pb.getInitialState(), pb.getFinalTime(), new double[pb.getDimension()]);
            Assert.assertTrue((handler.getMaximalValueError() < 1.3 * scalAbsoluteTolerance ? 1 : 0) != 0);
            Assert.assertEquals((double)0.0, (double)handler.getMaximalTimeError(), (double)1.0E-12);
            int calls = pb.getCalls();
            Assert.assertEquals((long)integ.getEvaluations(), (long)calls);
            Assert.assertTrue((calls <= previousCalls ? 1 : 0) != 0);
            previousCalls = calls;
        }
    }

    @Test
    public void testTooLargeFirstStep() throws DimensionMismatchException, NumberIsTooSmallException, MaxCountExceededException, NoBracketingException {
        DormandPrince853Integrator integ = new DormandPrince853Integrator(0.0, Double.POSITIVE_INFINITY, Double.NaN, Double.NaN);
        double start = 0.0;
        double end = 0.001;
        FirstOrderDifferentialEquations equations = new FirstOrderDifferentialEquations(){

            public int getDimension() {
                return 1;
            }

            public void computeDerivatives(double t, double[] y, double[] yDot) {
                Assert.assertTrue((t >= FastMath.nextAfter((double)0.0, (double)Double.NEGATIVE_INFINITY) ? 1 : 0) != 0);
                Assert.assertTrue((t <= FastMath.nextAfter((double)0.001, (double)Double.POSITIVE_INFINITY) ? 1 : 0) != 0);
                yDot[0] = -100.0 * y[0];
            }
        };
        integ.setStepSizeControl(0.0, 1.0, 1.0E-6, 1.0E-8);
        integ.integrate(equations, 0.0, new double[]{1.0}, 0.001, new double[1]);
    }

    @Test
    public void testBackward() throws DimensionMismatchException, NumberIsTooSmallException, MaxCountExceededException, NoBracketingException {
        TestProblem5 pb = new TestProblem5();
        double minStep = 0.0;
        double maxStep = pb.getFinalTime() - pb.getInitialTime();
        double scalAbsoluteTolerance = 1.0E-8;
        double scalRelativeTolerance = 0.01 * scalAbsoluteTolerance;
        DormandPrince853Integrator integ = new DormandPrince853Integrator(minStep, maxStep, scalAbsoluteTolerance, scalRelativeTolerance);
        TestProblemHandler handler = new TestProblemHandler(pb, (ODEIntegrator)integ);
        integ.addStepHandler((StepHandler)handler);
        integ.integrate((FirstOrderDifferentialEquations)pb, pb.getInitialTime(), pb.getInitialState(), pb.getFinalTime(), new double[pb.getDimension()]);
        Assert.assertTrue((handler.getLastError() < 1.1E-7 ? 1 : 0) != 0);
        Assert.assertTrue((handler.getMaximalValueError() < 1.1E-7 ? 1 : 0) != 0);
        Assert.assertEquals((double)0.0, (double)handler.getMaximalTimeError(), (double)1.0E-12);
        Assert.assertEquals((Object)"Dormand-Prince 8 (5, 3)", (Object)integ.getName());
    }

    @Test
    public void testEvents() throws DimensionMismatchException, NumberIsTooSmallException, MaxCountExceededException, NoBracketingException {
        TestProblem4 pb = new TestProblem4();
        double minStep = 0.0;
        double maxStep = pb.getFinalTime() - pb.getInitialTime();
        double scalAbsoluteTolerance = 1.0E-9;
        double scalRelativeTolerance = 0.01 * scalAbsoluteTolerance;
        DormandPrince853Integrator integ = new DormandPrince853Integrator(minStep, maxStep, scalAbsoluteTolerance, scalRelativeTolerance);
        TestProblemHandler handler = new TestProblemHandler(pb, (ODEIntegrator)integ);
        integ.addStepHandler((StepHandler)handler);
        EventHandler[] functions = pb.getEventsHandlers();
        double convergence = 1.0E-8 * maxStep;
        for (int l = 0; l < functions.length; ++l) {
            integ.addEventHandler(functions[l], Double.POSITIVE_INFINITY, convergence, 1000);
        }
        Assert.assertEquals((long)functions.length, (long)integ.getEventHandlers().size());
        integ.integrate((FirstOrderDifferentialEquations)pb, pb.getInitialTime(), pb.getInitialState(), pb.getFinalTime(), new double[pb.getDimension()]);
        Assert.assertEquals((double)0.0, (double)handler.getMaximalValueError(), (double)2.1E-7);
        Assert.assertEquals((double)0.0, (double)handler.getMaximalTimeError(), (double)convergence);
        Assert.assertEquals((double)12.0, (double)handler.getLastTime(), (double)convergence);
        integ.clearEventHandlers();
        Assert.assertEquals((long)0L, (long)integ.getEventHandlers().size());
    }

    @Test
    public void testKepler() throws DimensionMismatchException, NumberIsTooSmallException, MaxCountExceededException, NoBracketingException {
        double scalAbsoluteTolerance;
        TestProblem3 pb = new TestProblem3(0.9);
        double minStep = 0.0;
        double maxStep = pb.getFinalTime() - pb.getInitialTime();
        double scalRelativeTolerance = scalAbsoluteTolerance = 1.0E-8;
        DormandPrince853Integrator integ = new DormandPrince853Integrator(minStep, maxStep, scalAbsoluteTolerance, scalRelativeTolerance);
        integ.addStepHandler((StepHandler)new KeplerHandler(pb));
        integ.integrate((FirstOrderDifferentialEquations)pb, pb.getInitialTime(), pb.getInitialState(), pb.getFinalTime(), new double[pb.getDimension()]);
        Assert.assertEquals((long)integ.getEvaluations(), (long)pb.getCalls());
        Assert.assertTrue((pb.getCalls() < 3300 ? 1 : 0) != 0);
    }

    @Test
    public void testVariableSteps() throws DimensionMismatchException, NumberIsTooSmallException, MaxCountExceededException, NoBracketingException {
        double scalAbsoluteTolerance;
        TestProblem3 pb = new TestProblem3(0.9);
        double minStep = 0.0;
        double maxStep = pb.getFinalTime() - pb.getInitialTime();
        double scalRelativeTolerance = scalAbsoluteTolerance = 1.0E-8;
        DormandPrince853Integrator integ = new DormandPrince853Integrator(minStep, maxStep, scalAbsoluteTolerance, scalRelativeTolerance);
        integ.addStepHandler((StepHandler)new VariableHandler());
        double stopTime = integ.integrate((FirstOrderDifferentialEquations)pb, pb.getInitialTime(), pb.getInitialState(), pb.getFinalTime(), new double[pb.getDimension()]);
        Assert.assertEquals((double)pb.getFinalTime(), (double)stopTime, (double)1.0E-10);
        Assert.assertEquals((Object)"Dormand-Prince 8 (5, 3)", (Object)integ.getName());
    }

    @Test
    public void testUnstableDerivative() throws DimensionMismatchException, NumberIsTooSmallException, MaxCountExceededException, NoBracketingException {
        StepProblem stepProblem = new StepProblem(0.0, 1.0, 2.0);
        DormandPrince853Integrator integ = new DormandPrince853Integrator(0.1, 10.0, 1.0E-12, 0.0);
        integ.addEventHandler((EventHandler)stepProblem, 1.0, 1.0E-12, 1000);
        double[] y = new double[]{Double.NaN};
        integ.integrate((FirstOrderDifferentialEquations)stepProblem, 0.0, new double[]{0.0}, 10.0, y);
        Assert.assertEquals((double)8.0, (double)y[0], (double)1.0E-12);
    }

    @Test
    public void testEventsScheduling() {
        FirstOrderDifferentialEquations sincos = new FirstOrderDifferentialEquations(){

            public int getDimension() {
                return 2;
            }

            public void computeDerivatives(double t, double[] y, double[] yDot) {
                yDot[0] = y[1];
                yDot[1] = -y[0];
            }
        };
        SchedulingChecker sinChecker = new SchedulingChecker(0);
        SchedulingChecker cosChecker = new SchedulingChecker(1);
        DormandPrince853Integrator integ = new DormandPrince853Integrator(0.001, 1.0, 1.0E-12, 0.0);
        integ.addEventHandler((EventHandler)sinChecker, 0.01, 1.0E-7, 100);
        integ.addStepHandler((StepHandler)sinChecker);
        integ.addEventHandler((EventHandler)cosChecker, 0.01, 1.0E-7, 100);
        integ.addStepHandler((StepHandler)cosChecker);
        double t0 = 0.5;
        double[] y0 = new double[]{FastMath.sin((double)t0), FastMath.cos((double)t0)};
        double t = 10.0;
        double[] y = new double[2];
        integ.integrate(sincos, t0, y0, t, y);
    }

    private static class VariableHandler
    implements StepHandler {
        private boolean firstTime = true;
        private double minStep = 0.0;
        private double maxStep = 0.0;

        public void init(double t0, double[] y0, double t) {
            this.firstTime = true;
            this.minStep = 0.0;
            this.maxStep = 0.0;
        }

        public void handleStep(StepInterpolator interpolator, boolean isLast) {
            double step = FastMath.abs((double)(interpolator.getCurrentTime() - interpolator.getPreviousTime()));
            if (this.firstTime) {
                this.maxStep = this.minStep = FastMath.abs((double)step);
                this.firstTime = false;
            } else {
                if (step < this.minStep) {
                    this.minStep = step;
                }
                if (step > this.maxStep) {
                    this.maxStep = step;
                }
            }
            if (isLast) {
                Assert.assertTrue((this.minStep < 0.01 ? 1 : 0) != 0);
                Assert.assertTrue((this.maxStep > 0.5 ? 1 : 0) != 0);
            }
        }
    }

    private static class KeplerHandler
    implements StepHandler {
        private int nbSteps;
        private double maxError;
        private TestProblem3 pb;

        public KeplerHandler(TestProblem3 pb) {
            this.pb = pb;
        }

        public void init(double t0, double[] y0, double t) {
            this.nbSteps = 0;
            this.maxError = 0.0;
        }

        public void handleStep(StepInterpolator interpolator, boolean isLast) throws MaxCountExceededException {
            ++this.nbSteps;
            for (int a = 1; a < 10; ++a) {
                double prev = interpolator.getPreviousTime();
                double curr = interpolator.getCurrentTime();
                double interp = ((double)(10 - a) * prev + (double)a * curr) / 10.0;
                interpolator.setInterpolatedTime(interp);
                double[] interpolatedY = interpolator.getInterpolatedState();
                double[] theoreticalY = this.pb.computeTheoreticalState(interpolator.getInterpolatedTime());
                double dx = interpolatedY[0] - theoreticalY[0];
                double dy = interpolatedY[1] - theoreticalY[1];
                double error = dx * dx + dy * dy;
                if (!(error > this.maxError)) continue;
                this.maxError = error;
            }
            if (isLast) {
                Assert.assertTrue((this.maxError < 2.4E-10 ? 1 : 0) != 0);
                Assert.assertTrue((this.nbSteps < 150 ? 1 : 0) != 0);
            }
        }
    }

    private static class SchedulingChecker
    implements StepHandler,
    EventHandler {
        int index;
        double tMin;

        public SchedulingChecker(int index) {
            this.index = index;
        }

        public void init(double t0, double[] y0, double t) {
            this.tMin = t0;
        }

        public void handleStep(StepInterpolator interpolator, boolean isLast) {
            this.tMin = interpolator.getCurrentTime();
        }

        public double g(double t, double[] y) {
            Assert.assertTrue((t >= this.tMin ? 1 : 0) != 0);
            return y[this.index];
        }

        public EventHandler.Action eventOccurred(double t, double[] y, boolean increasing) {
            return EventHandler.Action.RESET_STATE;
        }

        public void resetState(double t, double[] y) {
        }
    }
}

