/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.daffodil.runtime1.processors.parsers

import java.lang.Long as JLong
import java.math.{ BigDecimal as JBigDecimal, BigInteger as JBigInteger }

import org.apache.daffodil.lib.schema.annotation.props.gen.LengthUnits
import org.apache.daffodil.lib.schema.annotation.props.gen.YesNo
import org.apache.daffodil.lib.util.DecimalUtils
import org.apache.daffodil.runtime1.processors.ElementRuntimeData
import org.apache.daffodil.runtime1.processors.Evaluatable

class IBM4690PackedDecimalKnownLengthParser(
  e: ElementRuntimeData,
  binaryDecimalVirtualPoint: Int,
  val lengthInBits: Int,
  decimalSigned: YesNo
) extends PackedBinaryDecimalBaseParser(e, binaryDecimalVirtualPoint, Some(decimalSigned))
  with HasKnownLengthInBits {

  override def toNumber(num: Array[Byte]): JBigDecimal =
    DecimalUtils.ibm4690ToBigDecimal(num, binaryDecimalVirtualPoint)

}

class IBM4690PackedDecimalRuntimeLengthParser(
  val e: ElementRuntimeData,
  binaryDecimalVirtualPoint: Int,
  val lengthEv: Evaluatable[JLong],
  val lengthUnits: LengthUnits,
  decimalSigned: YesNo
) extends PackedBinaryDecimalBaseParser(e, binaryDecimalVirtualPoint, Some(decimalSigned))
  with HasRuntimeExplicitLength {

  override def toNumber(num: Array[Byte]): JBigDecimal =
    DecimalUtils.ibm4690ToBigDecimal(num, binaryDecimalVirtualPoint)

}

class IBM4690PackedDecimalBitLimitLengthParser(
  e: ElementRuntimeData,
  binaryDecimalVirtualPoint: Int,
  decimalSigned: YesNo
) extends PackedBinaryDecimalBaseParser(e, binaryDecimalVirtualPoint, Some(decimalSigned))
  with BitLengthFromBitLimitMixin {

  override def toNumber(num: Array[Byte]): JBigDecimal =
    DecimalUtils.ibm4690ToBigDecimal(num, binaryDecimalVirtualPoint)

}

class IBM4690PackedIntegerRuntimeLengthParser(
  val e: ElementRuntimeData,
  val lengthEv: Evaluatable[JLong],
  val lengthUnits: LengthUnits
) extends PackedBinaryIntegerBaseParser(e)
  with HasRuntimeExplicitLength {

  override def toNumber(num: Array[Byte]): JBigInteger =
    DecimalUtils.ibm4690ToBigInteger(num)

}

class IBM4690PackedIntegerKnownLengthParser(
  e: ElementRuntimeData,
  val lengthInBits: Int
) extends PackedBinaryIntegerBaseParser(e)
  with HasKnownLengthInBits {

  override def toNumber(num: Array[Byte]): JBigInteger =
    DecimalUtils.ibm4690ToBigInteger(num)

}

class IBM4690PackedIntegerBitLimitLengthParser(e: ElementRuntimeData)
  extends PackedBinaryIntegerBaseParser(e)
  with BitLengthFromBitLimitMixin {

  override def toNumber(num: Array[Byte]): JBigInteger =
    DecimalUtils.ibm4690ToBigInteger(num)

}
