/* -*- mode: c++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

/*
 Copyright (C) 2010, 2014 Klaus Spanderen
 
 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it
 under the terms of the QuantLib license.  You should have received a
 copy of the license along with this program; if not, please email
 <quantlib-dev@lists.sf.net>. The license is also available online at
 <http://quantlib.org/license.shtml>.

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
*/

#include <ql/methods/finitedifferences/operators/fdmlinearoplayout.hpp>
#include <ql/methods/finitedifferences/stepconditions/fdmsimpleswingcondition.hpp>
#include <utility>

namespace QuantLib {

    FdmSimpleSwingCondition::FdmSimpleSwingCondition(
        std::vector<Time> exerciseTimes,
        ext::shared_ptr<FdmMesher> mesher,
        ext::shared_ptr<FdmInnerValueCalculator> calculator,
        Size swingDirection,
        Size minExercises)
    : exerciseTimes_(std::move(exerciseTimes)), mesher_(std::move(mesher)),
      calculator_(std::move(calculator)), minExercises_(minExercises),
      swingDirection_(swingDirection) {}

    void FdmSimpleSwingCondition::applyTo(Array& a, Time t) const {

        const auto iter
            = std::find(exerciseTimes_.begin(), exerciseTimes_.end(), t);
        const Size maxExerciseValue=mesher_->layout()->dim()[swingDirection_]-1;

        if (iter != exerciseTimes_.end()) {
            Array retVal= a;

            const Size d = std::distance(iter, exerciseTimes_.end());

            QL_REQUIRE(mesher_->layout()->size() == a.size(),
                       "inconsistent array dimensions");

            for (const auto& iter : *mesher_->layout()) {
                
                const std::vector<Size>& coor = iter.coordinates();
                
                const Size exercisesUsed = coor[swingDirection_];
                
                if (exercisesUsed < maxExerciseValue) {
                    const Real cashflow = calculator_->innerValue(iter, t);
                    const Real currentValue = a[iter.index()];
                    const Real valuePlusOneExercise
                         = a[mesher_->layout()->neighbourhood(iter, swingDirection_, 1)];
                    
                    if (   currentValue < valuePlusOneExercise + cashflow
                        || exercisesUsed + d <=  minExercises_) {
                        retVal[iter.index()] = valuePlusOneExercise + cashflow;
                    }
                }
            }
            a = retVal;
        }
    }
}
