/*******************************************************************************
 * Copyright (c) 2010-2012, Mark Czotter, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Mark Czotter - initial API and implementation
 *******************************************************************************/

package org.eclipse.incquery.patternlanguage.emf.ui.builder;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.ecore.InternalEObject;
import org.eclipse.incquery.patternlanguage.emf.util.IErrorFeedback;
import org.eclipse.incquery.patternlanguage.helper.CorePatternLanguageHelper;
import org.eclipse.incquery.patternlanguage.patternLanguage.Pattern;
import org.eclipse.incquery.patternlanguage.patternLanguage.PatternLanguagePackage;
import org.eclipse.incquery.runtime.IExtensions;
import org.eclipse.incquery.runtime.exception.IncQueryException;
import org.eclipse.incquery.tooling.core.generator.GenerateQuerySpecificationExtension;
import org.eclipse.incquery.tooling.core.generator.fragments.IGenerationFragment;
import org.eclipse.incquery.tooling.core.generator.fragments.IGenerationFragmentProvider;
import org.eclipse.incquery.tooling.core.project.ProjectGenerationHelper;
import org.eclipse.xtext.builder.EclipseResourceFileSystemAccess2;
import org.eclipse.xtext.builder.IXtextBuilderParticipant.IBuildContext;
import org.eclipse.xtext.generator.OutputConfiguration;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.resource.IResourceDescription.Delta;
import org.eclipse.xtext.xbase.lib.Pair;

import com.google.inject.Inject;
import com.google.inject.Injector;

/**
 * Clean phase support for BuilderParticipant.
 *
 * @author Mark Czotter
 *
 */
public class CleanSupport {

    @Inject
    private Injector injector;

    @Inject
    private IGenerationFragmentProvider fragmentProvider;

    @Inject
    private EclipseResourceSupport eclipseResourceSupport;

    @Inject
    private EnsurePluginSupport ensureSupport;

    @Inject
    private Logger logger;


    /**
     * Performs a full clean on the currently built project and all related fragments.
     *
     * @param context
     * @param monitor
     */
    public void fullClean(IBuildContext context, IProgressMonitor monitor) {
        try {
            internalFullClean(context, monitor);
        } catch (Exception e) {
            logger.error("Exception during Full Clean!", e);
        } finally {
            monitor.worked(1);
        }
    }

    private void internalFullClean(IBuildContext context, IProgressMonitor monitor) throws CoreException,
            IncQueryException {
        IProject modelProject = context.getBuiltProject();
        // clean all fragments
        cleanAllFragment(modelProject);
        // clean current model project
        List<Pair<String, String>> removableExtensions = new ArrayList<Pair<String, String>>();
        removableExtensions.addAll(GenerateQuerySpecificationExtension.getRemovableExtensionIdentifiers());
        ProjectGenerationHelper.removeAllExtension(modelProject, removableExtensions);
    }

    /**
     * Performs full Clean on every registered {@link IGenerationFragment}.
     *
     * @param modelProject
     * @throws CoreException
     */
    private void cleanAllFragment(IProject modelProject) throws CoreException {
        for (IGenerationFragment fragment : fragmentProvider.getAllFragments()) {
            try {
                cleanFragment(modelProject, fragment);
            } catch (Exception e) {
                logger.error("Exception during full Clean on " + fragment.getClass().getCanonicalName(), e);
            }
        }
    }

    private void cleanFragment(IProject modelProject, IGenerationFragment fragment) throws CoreException {
        IProject fragmentProject = fragmentProvider.getFragmentProject(modelProject, fragment);
        if (fragmentProject.exists() && !fragmentProject.equals(modelProject)) {
            fragmentProject.refreshLocal(IResource.DEPTH_INFINITE, null);
            // full clean on output directories
            EclipseResourceFileSystemAccess2 fsa = eclipseResourceSupport
                    .createProjectFileSystemAccess(fragmentProject);
            for (OutputConfiguration config : fsa.getOutputConfigurations().values()) {
                cleanFragmentFolder(fragmentProject, config);
            }
            // clean all removable extensions
            ProjectGenerationHelper.removeAllExtension(fragmentProject, fragment.getRemovableExtensions());
            // removing all fragment-related markers
            fragmentProject.deleteMarkers(IErrorFeedback.FRAGMENT_ERROR_TYPE, true, IResource.DEPTH_INFINITE);
        }
    }

    private void cleanFragmentFolder(IProject fragmentProject, OutputConfiguration config) throws CoreException {
        IFolder folder = fragmentProject.getFolder(config.getOutputDirectory());
        if (folder.exists()) {
            for (IResource resource : folder.members()) {
                resource.delete(IResource.KEEP_HISTORY, new NullProgressMonitor());
            }
        }
    }

    /**
     * Performs a normal Clean on the currently built project and all related fragments.
     *
     * @param context
     * @param relevantDeltas
     * @param monitor
     */
    public void normalClean(IBuildContext context, List<Delta> relevantDeltas, IProgressMonitor monitor) {
        try {
            internalNormalClean(context, relevantDeltas, monitor);
        } catch (Exception e) {
            logger.error("Exception during Normal Clean!", e);
        } finally {
            monitor.worked(1);
        }
    }

    private void internalNormalClean(IBuildContext context, List<Delta> relevantDeltas, IProgressMonitor monitor)
            throws CoreException, IncQueryException {
        for (Delta delta : relevantDeltas) {
            if (delta.getOld() != null) {
                OldVersionHelper oldVersion = injector.getInstance(OldVersionHelper.class);
                for (IEObjectDescription desc : delta.getOld().getExportedObjectsByType(PatternLanguagePackage.Literals.PATTERN)) {
                    Pattern pattern = (Pattern) desc.getEObjectOrProxy();
                    if (pattern.eIsProxy()) {
                        pattern = oldVersion.findPattern(((InternalEObject)pattern).eProxyURI());
                    }
                    final String fqn = desc.getQualifiedName().toString();
                    if (pattern == null || pattern.eIsProxy()) {
                        // Old version cannot be found, executing full clean
                        context.getBuiltProject().build(IncrementalProjectBuilder.CLEAN_BUILD, monitor);
                        return;
                    }
                    final String foundFQN = CorePatternLanguageHelper.getFullyQualifiedName(pattern);
                    if (!foundFQN.equals(fqn)){
                    	// Incorrect old version found, executing full clean
                        context.getBuiltProject().build(IncrementalProjectBuilder.CLEAN_BUILD, monitor);
                        return;
                    }
                    // clean up code and extensions in the modelProject
                    executeCleanUpOnModelProject(context.getBuiltProject(), fqn);
                    // clean up code and extensions for all fragments
                    executeCleanUpOnFragments(context.getBuiltProject(), pattern);
                }
            }
        }
    }

    /**
     * Executes Normal Build cleanUp on the current Built Project (modelProject). Removes all code generated previously
     * for the {@link Pattern}, and marks current {@link Pattern} related extensions for removal.
     *
     * @param modelProject
     * @param pattern
     * @throws CoreException
     */
    private void executeCleanUpOnModelProject(IProject modelProject, String fqn) throws CoreException {
        // only the extension id and point name is needed for removal
        String extensionId = fqn;
        ensureSupport
                .removeExtension(modelProject, Pair.of(extensionId, IExtensions.QUERY_SPECIFICATION_EXTENSION_POINT_ID));
    }

    /**
     * Executes Normal Build cleanUp on every {@link IGenerationFragment} registered to the current {@link Pattern}.
     * Marks current {@link Pattern} related extensions for removal. If the {@link IProject} related to
     * {@link IGenerationFragment} does not exist, clean up skipped for the fragment.
     *
     * @param modelProject
     * @param pattern
     * @throws CoreException
     */
    private void executeCleanUpOnFragments(IProject modelProject, Pattern pattern) throws CoreException {
        for (IGenerationFragment fragment : fragmentProvider.getAllFragments()) {
            try {
                injector.injectMembers(fragment);
                // clean if the project still exist
                IProject targetProject = fragmentProvider.getFragmentProject(modelProject, fragment);
                if (targetProject.exists()) {
                    EclipseResourceFileSystemAccess2 fsa = eclipseResourceSupport
                            .createProjectFileSystemAccess(targetProject);
                    fragment.cleanUp(pattern, fsa);
                    ensureSupport.removeAllExtension(targetProject, fragment.removeExtension(pattern));
                    // removing all fragment-related markers
                    targetProject.deleteMarkers(IErrorFeedback.FRAGMENT_ERROR_TYPE, true, IResource.DEPTH_INFINITE);
                }
            } catch (Exception e) {
                String msg = String.format("Exception when executing clean for '%s' in fragment '%s'",
                        CorePatternLanguageHelper.getFullyQualifiedName(pattern), fragment.getClass()
                                .getCanonicalName());
                logger.error(msg, e);
            }
        }
    }

}
