/**
 * Copyright (c) 2010-2012, Mark Czotter, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Mark Czotter - initial API and implementation
 */
package org.eclipse.incquery.patternlanguage.emf.util;

import com.google.common.base.Objects;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.Set;
import java.util.regex.Matcher;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.incquery.patternlanguage.emf.jvmmodel.EMFPatternLanguageJvmModelInferrer;
import org.eclipse.incquery.patternlanguage.emf.types.EMFPatternTypeProvider;
import org.eclipse.incquery.patternlanguage.emf.types.IEMFTypeProvider;
import org.eclipse.incquery.patternlanguage.emf.util.EMFJvmTypesBuilder;
import org.eclipse.incquery.patternlanguage.emf.util.IErrorFeedback;
import org.eclipse.incquery.patternlanguage.helper.CorePatternLanguageHelper;
import org.eclipse.incquery.patternlanguage.patternLanguage.Pattern;
import org.eclipse.incquery.patternlanguage.patternLanguage.PatternModel;
import org.eclipse.incquery.patternlanguage.patternLanguage.Variable;
import org.eclipse.incquery.runtime.api.impl.BaseGeneratedQuerySpecification;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.common.types.JvmGenericType;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.xbase.compiler.TypeReferenceSerializer;
import org.eclipse.xtext.xbase.compiler.output.ITreeAppendable;
import org.eclipse.xtext.xbase.jvmmodel.IJvmModelAssociations;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;
import org.eclipse.xtext.xbase.typing.ITypeProvider;

/**
 * Utility class for the EMFPatternLanguageJvmModelInferrer.
 * 
 * @author Mark Czotter
 */
@SuppressWarnings("all")
public class EMFPatternLanguageJvmModelInferrerUtil {
  @Inject
  @Extension
  private EMFJvmTypesBuilder _eMFJvmTypesBuilder;
  
  private Logger logger = Logger.getLogger(this.getClass());
  
  @Inject
  private IEMFTypeProvider emfTypeProvider;
  
  @Inject
  private TypeReferenceSerializer typeReferenceSerializer;
  
  @Inject
  private IErrorFeedback errorFeedback;
  
  @Inject
  private IJvmModelAssociations associations;
  
  /**
   * This method returns the pattern name.
   * If the pattern name contains the package (any dot),
   * then removes all segment except the last one.
   */
  public String realPatternName(final Pattern pattern) {
    String name = pattern.getName();
    boolean _contains = name.contains(".");
    if (_contains) {
      int _lastIndexOf = name.lastIndexOf(".");
      int _plus = (_lastIndexOf + 1);
      return name.substring(_plus);
    }
    return name;
  }
  
  public boolean validClassName(final String simpleName) {
    boolean _and = false;
    char _charAt = simpleName.charAt(0);
    boolean _isJavaIdentifierStart = Character.isJavaIdentifierStart(_charAt);
    if (!_isJavaIdentifierStart) {
      _and = false;
    } else {
      String[] _split = simpleName.split("\\.");
      Iterable<String> _tail = IterableExtensions.<String>tail(((Iterable<String>)Conversions.doWrapArray(_split)));
      final Function1<String, Boolean> _function = new Function1<String, Boolean>() {
        public Boolean apply(final String ch) {
          char _charAt = ch.charAt(0);
          return Boolean.valueOf(Character.isJavaIdentifierPart(_charAt));
        }
      };
      boolean _forall = IterableExtensions.<String>forall(_tail, _function);
      _and = _forall;
    }
    return _and;
  }
  
  public String modelFileName(final EObject object) {
    try {
      String _xblockexpression = null;
      {
        Resource _eResource = object.eResource();
        URI _uRI = null;
        if (_eResource!=null) {
          _uRI=_eResource.getURI();
        }
        URI _trimFileExtension = _uRI.trimFileExtension();
        final String name = _trimFileExtension.lastSegment();
        boolean _validClassName = this.validClassName(name);
        boolean _not = (!_validClassName);
        if (_not) {
          throw new IllegalAccessError((("The file name " + name) + " is not a valid Java type name. Please, rename the file!"));
        }
        _xblockexpression = name;
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Returns the QuerySpecificationClass name based on the Pattern's name
   */
  public String querySpecificationClassName(final Pattern pattern) {
    String _xblockexpression = null;
    {
      String name = pattern.getName();
      boolean _contains = name.contains(".");
      if (_contains) {
        String _realPatternName = this.realPatternName(pattern);
        name = _realPatternName;
      }
      String _firstUpper = StringExtensions.toFirstUpper(name);
      _xblockexpression = (_firstUpper + "QuerySpecification");
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the IQuerySpecificationProvider class name based on the Pattern's name
   */
  public String querySpecificationProviderClassName(final Pattern pattern) {
    return "Provider";
  }
  
  /**
   * Returns the holder class name based on the Pattern's name
   */
  public String querySpecificationHolderClassName(final Pattern pattern) {
    return "LazyHolder";
  }
  
  /**
   * Returns the MatcherClass name based on the Pattern's name
   */
  public String matcherClassName(final Pattern pattern) {
    String _xblockexpression = null;
    {
      String name = pattern.getName();
      boolean _contains = name.contains(".");
      if (_contains) {
        String _realPatternName = this.realPatternName(pattern);
        name = _realPatternName;
      }
      String _firstUpper = StringExtensions.toFirstUpper(name);
      _xblockexpression = (_firstUpper + "Matcher");
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the MatchClass name based on the Pattern's name
   */
  public String matchClassName(final Pattern pattern) {
    String _xblockexpression = null;
    {
      String name = pattern.getName();
      boolean _contains = name.contains(".");
      if (_contains) {
        String _realPatternName = this.realPatternName(pattern);
        name = _realPatternName;
      }
      String _firstUpper = StringExtensions.toFirstUpper(name);
      _xblockexpression = (_firstUpper + "Match");
    }
    return _xblockexpression;
  }
  
  public String matchImmutableInnerClassName(final Pattern pattern) {
    return "Immutable";
  }
  
  public String matchMutableInnerClassName(final Pattern pattern) {
    return "Mutable";
  }
  
  /**
   * Returns the ProcessorClass name based on the Pattern's name
   */
  public String processorClassName(final Pattern pattern) {
    String _xblockexpression = null;
    {
      String name = pattern.getName();
      boolean _contains = name.contains(".");
      if (_contains) {
        String _realPatternName = this.realPatternName(pattern);
        name = _realPatternName;
      }
      String _firstUpper = StringExtensions.toFirstUpper(name);
      _xblockexpression = (_firstUpper + "Processor");
    }
    return _xblockexpression;
  }
  
  /**
   * Returns field name for Variable
   */
  public String fieldName(final Variable variable) {
    String _name = null;
    if (variable!=null) {
      _name=variable.getName();
    }
    String _firstUpper = StringExtensions.toFirstUpper(_name);
    return ("f" + _firstUpper);
  }
  
  /**
   * Returns parameter name for Variable
   */
  public String parameterName(final Variable variable) {
    String _name = null;
    if (variable!=null) {
      _name=variable.getName();
    }
    String _firstUpper = null;
    if (_name!=null) {
      _firstUpper=StringExtensions.toFirstUpper(_name);
    }
    return ("p" + _firstUpper);
  }
  
  public String positionConstant(final Variable variable) {
    String _name = null;
    if (variable!=null) {
      _name=variable.getName();
    }
    String _upperCase = null;
    if (_name!=null) {
      _upperCase=_name.toUpperCase();
    }
    return ("POSITION_" + _upperCase);
  }
  
  /**
   * Returns correct getter method name for variable.
   * For variable with name 'class' returns getValueOfClass, otherwise returns <code>get#variable.name.toFirstUpper#</code>.
   */
  public String getterMethodName(final Variable variable) {
    String _name = variable.getName();
    boolean _equals = Objects.equal(_name, "class");
    if (_equals) {
      return "getValueOfClass";
    } else {
      String _name_1 = null;
      if (variable!=null) {
        _name_1=variable.getName();
      }
      String _firstUpper = null;
      if (_name_1!=null) {
        _firstUpper=StringExtensions.toFirstUpper(_name_1);
      }
      return ("get" + _firstUpper);
    }
  }
  
  /**
   * Returns correct setter method name for variable.
   * Currently returns <code>set#variable.name.toFirstUpper#</code>.
   */
  public String setterMethodName(final Variable variable) {
    String _name = null;
    if (variable!=null) {
      _name=variable.getName();
    }
    String _firstUpper = null;
    if (_name!=null) {
      _firstUpper=StringExtensions.toFirstUpper(_name);
    }
    return ("set" + _firstUpper);
  }
  
  /**
   * Calls the typeProvider.
   * See the XBaseUsageCrossReferencer class, possible solution for local variable usage
   * TODO: improve type calculation
   * @return JvmTypeReference pointing the EClass that defines the Variable's type.
   * @see ITypeProvider
   * @see EMFPatternTypeProvider
   */
  public JvmTypeReference calculateType(final Variable variable) {
    return this.emfTypeProvider.getVariableType(variable);
  }
  
  /**
   * Serializes the EObject into Java String variable.
   */
  public CharSequence serializeToJava(final EObject eObject) {
    final String parseString = this.serialize(eObject);
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(parseString);
    if (_isNullOrEmpty) {
      return "";
    }
    final String[] splits = parseString.split("[\r\n]+");
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("String patternString = \"\"");
    final StringConcatenation stringRep = ((StringConcatenation) _builder);
    stringRep.newLine();
    for (final String s : splits) {
      {
        stringRep.append((("+\" " + s) + " \""));
        stringRep.newLine();
      }
    }
    stringRep.append(";");
    return stringRep;
  }
  
  /**
   * Serializes the input for Javadoc
   */
  public String serializeToJavadoc(final Pattern pattern) {
    String javadocString = this.serialize(pattern);
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(javadocString);
    if (_isNullOrEmpty) {
      return "Serialization error, check Log";
    }
    String _quote = java.util.regex.Pattern.quote("\\\"");
    String _quoteReplacement = Matcher.quoteReplacement("\"");
    String _replaceAll = javadocString.replaceAll(_quote, _quoteReplacement);
    javadocString = _replaceAll;
    String _replaceAll_1 = javadocString.replaceAll("@", "{@literal @}");
    javadocString = _replaceAll_1;
    String _replaceAll_2 = javadocString.replaceAll("<", "{@literal <}");
    javadocString = _replaceAll_2;
    String _replaceAll_3 = javadocString.replaceAll(">", "{@literal >}");
    javadocString = _replaceAll_3;
    return javadocString.trim();
  }
  
  /**
   * Escapes the input to be usable in literal strings
   */
  public String escapeToQuotedString(final String inputString) {
    String string = inputString;
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(string);
    if (_isNullOrEmpty) {
      return "";
    }
    String _replace = string.replace("\\", "\\\\");
    string = _replace;
    String _replace_1 = string.replace("\n", "\\n");
    string = _replace_1;
    String _replace_2 = string.replace("\t", "\\t");
    string = _replace_2;
    String _replace_3 = string.replace("\"", "\\\"");
    string = _replace_3;
    return string.trim();
  }
  
  /**
   * Serializes EObject to a String representation. Escapes only the double qoutes.
   */
  private String serialize(final EObject eObject) {
    try {
      final ICompositeNode eObjectNode = NodeModelUtils.getNode(eObject);
      boolean _notEquals = (!Objects.equal(eObjectNode, null));
      if (_notEquals) {
        String _text = eObjectNode.getText();
        return this.escape(_text);
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        boolean _notEquals_1 = (!Objects.equal(this.logger, null));
        if (_notEquals_1) {
          EClass _eClass = eObject.eClass();
          String _name = _eClass.getName();
          String _plus = ("Error when serializing " + _name);
          this.logger.error(_plus, e);
        }
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return null;
  }
  
  private String escape(final String escapable) {
    boolean _equals = Objects.equal(escapable, null);
    if (_equals) {
      return null;
    }
    String escapedString = escapable.replaceAll("\"", "\\\\\"");
    String _replaceAll = escapedString.replaceAll("\\*+/", "");
    String _replaceAll_1 = _replaceAll.replaceAll("/*\\*", "");
    escapedString = _replaceAll_1;
    return escapedString;
  }
  
  /**
   * Returns the packageName: PatternModel.packageName or "" when nullOrEmpty.
   */
  public String getPackageName(final Pattern pattern) {
    EObject _eContainer = pattern.eContainer();
    String packageName = ((PatternModel) _eContainer).getPackageName();
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(packageName);
    if (_isNullOrEmpty) {
      packageName = "";
    }
    return packageName.toLowerCase();
  }
  
  public String getUtilPackageName(final Pattern pattern) {
    String _packageName = this.getPackageName(pattern);
    return (_packageName + ".util");
  }
  
  /**
   * Returns the packageName: PatternModel.packageName + Pattern.name, packageName is ignored, when nullOrEmpty.
   */
  public String getPackageNameOld(final Pattern pattern) {
    EObject _eContainer = pattern.eContainer();
    String packageName = ((PatternModel) _eContainer).getPackageName();
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(packageName);
    if (_isNullOrEmpty) {
      packageName = "";
    } else {
      packageName = (packageName + ".");
    }
    String _name = pattern.getName();
    String _plus = (packageName + _name);
    return _plus.toLowerCase();
  }
  
  public String getPackagePath(final Pattern pattern) {
    String _packageName = this.getPackageName(pattern);
    return _packageName.replace(".", "/");
  }
  
  /**
   * Calculates the correct package path for a selected fqn
   */
  public String getPackagePath(final String fqn) {
    String _xblockexpression = null;
    {
      Splitter _on = Splitter.on(".");
      final Iterable<String> split = _on.split(fqn);
      int _size = IterableExtensions.size(split);
      int _minus = (_size - 1);
      Iterable<String> _take = IterableExtensions.<String>take(split, _minus);
      _xblockexpression = IterableExtensions.join(_take, "/");
    }
    return _xblockexpression;
  }
  
  /**
   * This method returns the pattern name.
   * If the pattern name contains the package (any dot),
   * then removes all segment except the last one.
   */
  public String realPatternName(final String fqn) {
    Splitter _on = Splitter.on(".");
    Iterable<String> _split = _on.split(fqn);
    return IterableExtensions.<String>last(_split);
  }
  
  public ITreeAppendable referClass(final ITreeAppendable appendable, final EObject ctx, final Class<?> clazz, final JvmTypeReference... typeArgs) {
    ITreeAppendable _xblockexpression = null;
    {
      final JvmTypeReference ref = this._eMFJvmTypesBuilder.newTypeRef(ctx, clazz, typeArgs);
      ITreeAppendable _xifexpression = null;
      boolean _notEquals = (!Objects.equal(ref, null));
      if (_notEquals) {
        this.serialize(appendable, ref, ctx);
      } else {
        ITreeAppendable _xblockexpression_1 = null;
        {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Cannot resolve class ");
          String _canonicalName = clazz.getCanonicalName();
          _builder.append(_canonicalName, "");
          _builder.append(". Check project dependencies.");
          this.errorFeedback.reportError(ctx, _builder.toString(), EMFPatternLanguageJvmModelInferrer.INVALID_TYPEREF_CODE, Severity.ERROR, IErrorFeedback.JVMINFERENCE_ERROR_TYPE);
          String _canonicalName_1 = clazz.getCanonicalName();
          _xblockexpression_1 = appendable.append(_canonicalName_1);
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  public void referClass(final ITreeAppendable appendable, final EObject ctx, final JvmType type) {
    JvmTypeReference _newTypeRef = this._eMFJvmTypesBuilder.newTypeRef(type);
    this.serialize(appendable, _newTypeRef, ctx);
  }
  
  public void serialize(final ITreeAppendable appendable, final JvmTypeReference ref, final EObject ctx) {
    this.typeReferenceSerializer.serialize(ref, ctx, appendable);
  }
  
  public JvmGenericType findInferredSpecification(final Pattern pattern) {
    return this.findInferredClass(pattern, BaseGeneratedQuerySpecification.class);
  }
  
  public JvmGenericType findInferredClass(final EObject pattern, final Class clazz) {
    Set<EObject> _jvmElements = this.associations.getJvmElements(pattern);
    Iterable<JvmGenericType> _filter = Iterables.<JvmGenericType>filter(_jvmElements, JvmGenericType.class);
    final Function1<JvmGenericType, Boolean> _function = new Function1<JvmGenericType, Boolean>() {
      public Boolean apply(final JvmGenericType it) {
        EList<JvmTypeReference> _superTypes = it.getSuperTypes();
        final Function1<JvmTypeReference, Boolean> _function = new Function1<JvmTypeReference, Boolean>() {
          public Boolean apply(final JvmTypeReference it) {
            boolean _xifexpression = false;
            JvmType _type = it.getType();
            boolean _equals = Objects.equal(_type, null);
            if (_equals) {
              _xifexpression = false;
            } else {
              JvmType _type_1 = it.getType();
              String _qualifiedName = _type_1.getQualifiedName();
              String _canonicalName = clazz.getCanonicalName();
              _xifexpression = Objects.equal(_qualifiedName, _canonicalName);
            }
            return Boolean.valueOf(_xifexpression);
          }
        };
        return Boolean.valueOf(IterableExtensions.<JvmTypeReference>exists(_superTypes, _function));
      }
    };
    return IterableExtensions.<JvmGenericType>findFirst(_filter, _function);
  }
  
  public boolean isPublic(final Pattern pattern) {
    boolean _isPrivate = CorePatternLanguageHelper.isPrivate(pattern);
    return (!_isPrivate);
  }
}
