package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link ConfigBind}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableConfigBind.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableConfigBind implements ConfigBind {
  private final ConfigFile file;
  private final String configId;
  private final String configName;

  private ImmutableConfigBind(
      ConfigFile file,
      String configId,
      String configName) {
    this.file = file;
    this.configId = configId;
    this.configName = configName;
  }

  /**
   * @return The value of the {@code file} attribute
   */
  @JsonProperty("File")
  @Override
  public ConfigFile file() {
    return file;
  }

  /**
   * @return The value of the {@code configId} attribute
   */
  @JsonProperty("ConfigID")
  @Override
  public String configId() {
    return configId;
  }

  /**
   * @return The value of the {@code configName} attribute
   */
  @JsonProperty("ConfigName")
  @Override
  public String configName() {
    return configName;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ConfigBind#file() file} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for file
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableConfigBind withFile(ConfigFile value) {
    if (this.file == value) return this;
    ConfigFile newValue = Objects.requireNonNull(value, "file");
    return new ImmutableConfigBind(newValue, this.configId, this.configName);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ConfigBind#configId() configId} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for configId
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableConfigBind withConfigId(String value) {
    String newValue = Objects.requireNonNull(value, "configId");
    if (this.configId.equals(newValue)) return this;
    return new ImmutableConfigBind(this.file, newValue, this.configName);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ConfigBind#configName() configName} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for configName
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableConfigBind withConfigName(String value) {
    String newValue = Objects.requireNonNull(value, "configName");
    if (this.configName.equals(newValue)) return this;
    return new ImmutableConfigBind(this.file, this.configId, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableConfigBind} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableConfigBind
        && equalsByValue((ImmutableConfigBind) another);
  }

  private boolean equalsByValue(ImmutableConfigBind another) {
    return file.equals(another.file)
        && configId.equals(another.configId)
        && configName.equals(another.configName);
  }

  /**
   * Computes a hash code from attributes: {@code file}, {@code configId}, {@code configName}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + file.hashCode();
    h += (h << 5) + configId.hashCode();
    h += (h << 5) + configName.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code ConfigBind} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "ConfigBind{"
        + "file=" + file
        + ", configId=" + configId
        + ", configName=" + configName
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link ConfigBind} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ConfigBind instance
   */
  public static ImmutableConfigBind copyOf(ConfigBind instance) {
    if (instance instanceof ImmutableConfigBind) {
      return (ImmutableConfigBind) instance;
    }
    return ImmutableConfigBind.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableConfigBind ImmutableConfigBind}.
   * <pre>
   * ImmutableConfigBind.builder()
   *    .file(org.mandas.docker.client.messages.swarm.ConfigFile) // required {@link ConfigBind#file() file}
   *    .configId(String) // required {@link ConfigBind#configId() configId}
   *    .configName(String) // required {@link ConfigBind#configName() configName}
   *    .build();
   * </pre>
   * @return A new ImmutableConfigBind builder
   */
  public static ImmutableConfigBind.Builder builder() {
    return new ImmutableConfigBind.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableConfigBind ImmutableConfigBind}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements ConfigBind.Builder {
    private static final long INIT_BIT_FILE = 0x1L;
    private static final long INIT_BIT_CONFIG_ID = 0x2L;
    private static final long INIT_BIT_CONFIG_NAME = 0x4L;
    private long initBits = 0x7L;

    private ConfigFile file;
    private String configId;
    private String configName;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ConfigBind} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ConfigBind instance) {
      Objects.requireNonNull(instance, "instance");
      this.file(instance.file());
      this.configId(instance.configId());
      this.configName(instance.configName());
      return this;
    }

    /**
     * Initializes the value for the {@link ConfigBind#file() file} attribute.
     * @param file The value for file 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("File")
    public final Builder file(ConfigFile file) {
      this.file = Objects.requireNonNull(file, "file");
      initBits &= ~INIT_BIT_FILE;
      return this;
    }

    /**
     * Initializes the value for the {@link ConfigBind#configId() configId} attribute.
     * @param configId The value for configId 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("ConfigID")
    public final Builder configId(String configId) {
      this.configId = Objects.requireNonNull(configId, "configId");
      initBits &= ~INIT_BIT_CONFIG_ID;
      return this;
    }

    /**
     * Initializes the value for the {@link ConfigBind#configName() configName} attribute.
     * @param configName The value for configName 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("ConfigName")
    public final Builder configName(String configName) {
      this.configName = Objects.requireNonNull(configName, "configName");
      initBits &= ~INIT_BIT_CONFIG_NAME;
      return this;
    }

    /**
     * Builds a new {@link ImmutableConfigBind ImmutableConfigBind}.
     * @return An immutable instance of ConfigBind
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableConfigBind build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableConfigBind(file, configId, configName);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_FILE) != 0) attributes.add("file");
      if ((initBits & INIT_BIT_CONFIG_ID) != 0) attributes.add("configId");
      if ((initBits & INIT_BIT_CONFIG_NAME) != 0) attributes.add("configName");
      return "Cannot build ConfigBind, some of required attributes are not set " + attributes;
    }
  }
}
