"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.sortTSConfigs = exports.createTypeScriptProjectProvider = void 0;
const language_service_1 = require("@volar/language-service");
const path = require("path-browserify");
const vscode = require("vscode-languageserver");
const vscode_uri_1 = require("vscode-uri");
const isFileInDir_1 = require("../utils/isFileInDir");
const uriMap_1 = require("../utils/uriMap");
const inferredCompilerOptions_1 = require("./inferredCompilerOptions");
const simpleProjectProvider_1 = require("./simpleProjectProvider");
const typescriptProject_1 = require("./typescriptProject");
const rootTsConfigNames = ['tsconfig.json', 'jsconfig.json'];
function createTypeScriptProjectProvider(ts, tsLocalized, getLanguagePlugins) {
    let initialized = false;
    const configProjects = (0, uriMap_1.createUriMap)();
    const inferredProjects = (0, uriMap_1.createUriMap)();
    const rootTsConfigs = new Set();
    const searchedDirs = new Set();
    const projects = {
        async get(uri) {
            if (!initialized) {
                initialized = true;
                initialize(this);
            }
            const parsedUri = vscode_uri_1.URI.parse(uri);
            const tsconfig = await findMatchTSConfig(this, parsedUri);
            if (tsconfig) {
                return await getOrCreateConfiguredProject(this, tsconfig);
            }
            const workspaceFolder = (0, simpleProjectProvider_1.getWorkspaceFolder)(parsedUri, this.workspaceFolders);
            return await getOrCreateInferredProject(this, uri, workspaceFolder);
        },
        async all() {
            return await Promise.all([
                ...configProjects.values() ?? [],
                ...inferredProjects.values() ?? [],
            ]);
        },
        reload() {
            for (const project of [
                ...configProjects.values() ?? [],
                ...inferredProjects.values() ?? [],
            ]) {
                project.then(p => p.dispose());
            }
            configProjects.clear();
            inferredProjects.clear();
        },
    };
    return projects;
    function initialize(server) {
        server.onDidChangeWatchedFiles(({ changes }) => {
            const tsConfigChanges = changes.filter(change => rootTsConfigNames.includes(change.uri.substring(change.uri.lastIndexOf('/') + 1)));
            for (const change of tsConfigChanges) {
                const changeUri = vscode_uri_1.URI.parse(change.uri);
                const changeFileName = server.uriConverter.uriToFileName(change.uri, changeUri);
                if (change.type === vscode.FileChangeType.Created) {
                    rootTsConfigs.add(changeFileName);
                }
                else if ((change.type === vscode.FileChangeType.Changed || change.type === vscode.FileChangeType.Deleted) && configProjects.has(changeUri)) {
                    if (change.type === vscode.FileChangeType.Deleted) {
                        rootTsConfigs.delete(changeFileName);
                    }
                    const project = configProjects.get(changeUri);
                    configProjects.delete(changeUri);
                    project?.then(project => project.dispose());
                }
            }
            if (tsConfigChanges.length) {
                server.clearPushDiagnostics();
            }
            server.refresh(projects);
        });
    }
    async function findMatchTSConfig(server, uri) {
        const fileName = server.uriConverter.uriToFileName(uri.toString());
        let dir = path.dirname(fileName);
        while (true) {
            if (searchedDirs.has(dir)) {
                break;
            }
            searchedDirs.add(dir);
            for (const tsConfigName of rootTsConfigNames) {
                const tsconfigPath = path.join(dir, tsConfigName);
                if ((await server.fs.stat?.(server.uriConverter.fileNameToUri(tsconfigPath)))?.type === language_service_1.FileType.File) {
                    rootTsConfigs.add(tsconfigPath);
                }
            }
            dir = path.dirname(dir);
        }
        await prepareClosestootParsedCommandLine();
        return await findDirectIncludeTsconfig() ?? await findIndirectReferenceTsconfig();
        async function prepareClosestootParsedCommandLine() {
            let matches = [];
            for (const rootTsConfig of rootTsConfigs) {
                if ((0, isFileInDir_1.isFileInDir)(fileName, path.dirname(rootTsConfig))) {
                    matches.push(rootTsConfig);
                }
            }
            matches = matches.sort((a, b) => sortTSConfigs(fileName, a, b));
            if (matches.length) {
                await getParsedCommandLine(matches[0]);
            }
        }
        function findIndirectReferenceTsconfig() {
            return findTSConfig(async (tsconfig) => {
                const tsconfigUri = vscode_uri_1.URI.parse(server.uriConverter.fileNameToUri(tsconfig));
                const project = await configProjects.get(tsconfigUri);
                return project?.askedFiles.has(uri) ?? false;
            });
        }
        function findDirectIncludeTsconfig() {
            return findTSConfig(async (tsconfig) => {
                const map = (0, uriMap_1.createUriMap)();
                const parsedCommandLine = await getParsedCommandLine(tsconfig);
                for (const fileName of parsedCommandLine?.fileNames ?? []) {
                    const uri = vscode_uri_1.URI.parse(server.uriConverter.fileNameToUri(fileName));
                    map.set(uri, true);
                }
                return map.has(uri);
            });
        }
        async function findTSConfig(match) {
            const checked = new Set();
            for (const rootTsConfig of [...rootTsConfigs].sort((a, b) => sortTSConfigs(fileName, a, b))) {
                const tsconfigUri = vscode_uri_1.URI.parse(server.uriConverter.fileNameToUri(rootTsConfig));
                const project = await configProjects.get(tsconfigUri);
                if (project) {
                    let chains = await getReferencesChains(project.getParsedCommandLine(), rootTsConfig, []);
                    // This is to be consistent with tsserver behavior
                    chains = chains.reverse();
                    for (const chain of chains) {
                        for (let i = chain.length - 1; i >= 0; i--) {
                            const tsconfig = chain[i];
                            if (checked.has(tsconfig)) {
                                continue;
                            }
                            checked.add(tsconfig);
                            if (await match(tsconfig)) {
                                return tsconfig;
                            }
                        }
                    }
                }
            }
        }
        async function getReferencesChains(parsedCommandLine, tsConfig, before) {
            if (parsedCommandLine.projectReferences?.length) {
                const newChains = [];
                for (const projectReference of parsedCommandLine.projectReferences) {
                    let tsConfigPath = projectReference.path.replace(/\\/g, '/');
                    // fix https://github.com/johnsoncodehk/volar/issues/712
                    if ((await server.fs.stat?.(server.uriConverter.fileNameToUri(tsConfigPath)))?.type === language_service_1.FileType.File) {
                        const newTsConfigPath = path.join(tsConfigPath, 'tsconfig.json');
                        const newJsConfigPath = path.join(tsConfigPath, 'jsconfig.json');
                        if ((await server.fs.stat?.(server.uriConverter.fileNameToUri(newTsConfigPath)))?.type === language_service_1.FileType.File) {
                            tsConfigPath = newTsConfigPath;
                        }
                        else if ((await server.fs.stat?.(server.uriConverter.fileNameToUri(newJsConfigPath)))?.type === language_service_1.FileType.File) {
                            tsConfigPath = newJsConfigPath;
                        }
                    }
                    const beforeIndex = before.indexOf(tsConfigPath); // cycle
                    if (beforeIndex >= 0) {
                        newChains.push(before.slice(0, Math.max(beforeIndex, 1)));
                    }
                    else {
                        const referenceParsedCommandLine = await getParsedCommandLine(tsConfigPath);
                        if (referenceParsedCommandLine) {
                            for (const chain of await getReferencesChains(referenceParsedCommandLine, tsConfigPath, [...before, tsConfig])) {
                                newChains.push(chain);
                            }
                        }
                    }
                }
                return newChains;
            }
            else {
                return [[...before, tsConfig]];
            }
        }
        async function getParsedCommandLine(tsConfig) {
            const project = await getOrCreateConfiguredProject(server, tsConfig);
            return project?.getParsedCommandLine();
        }
    }
    function getOrCreateConfiguredProject(server, tsconfig) {
        tsconfig = tsconfig.replace(/\\/g, '/');
        const tsconfigUri = vscode_uri_1.URI.parse(server.uriConverter.fileNameToUri(tsconfig));
        let projectPromise = configProjects.get(tsconfigUri);
        if (!projectPromise) {
            const workspaceFolder = (0, simpleProjectProvider_1.getWorkspaceFolder)(tsconfigUri, server.workspaceFolders);
            const serviceEnv = (0, simpleProjectProvider_1.createServiceEnvironment)(server, workspaceFolder);
            projectPromise = (0, typescriptProject_1.createTypeScriptServerProject)(ts, tsLocalized, tsconfig, server, serviceEnv, getLanguagePlugins);
            configProjects.set(tsconfigUri, projectPromise);
        }
        return projectPromise;
    }
    async function getOrCreateInferredProject(server, uri, workspaceFolder) {
        if (!inferredProjects.has(workspaceFolder)) {
            inferredProjects.set(workspaceFolder, (async () => {
                const inferOptions = await (0, inferredCompilerOptions_1.getInferredCompilerOptions)(server);
                const serviceEnv = (0, simpleProjectProvider_1.createServiceEnvironment)(server, workspaceFolder);
                return (0, typescriptProject_1.createTypeScriptServerProject)(ts, tsLocalized, inferOptions, server, serviceEnv, getLanguagePlugins);
            })());
        }
        const project = await inferredProjects.get(workspaceFolder);
        project.tryAddFile(server.uriConverter.uriToFileName(uri));
        return project;
    }
}
exports.createTypeScriptProjectProvider = createTypeScriptProjectProvider;
function sortTSConfigs(file, a, b) {
    const inA = (0, isFileInDir_1.isFileInDir)(file, path.dirname(a));
    const inB = (0, isFileInDir_1.isFileInDir)(file, path.dirname(b));
    if (inA !== inB) {
        const aWeight = inA ? 1 : 0;
        const bWeight = inB ? 1 : 0;
        return bWeight - aWeight;
    }
    const aLength = a.split('/').length;
    const bLength = b.split('/').length;
    if (aLength === bLength) {
        const aWeight = path.basename(a) === 'tsconfig.json' ? 1 : 0;
        const bWeight = path.basename(b) === 'tsconfig.json' ? 1 : 0;
        return bWeight - aWeight;
    }
    return bLength - aLength;
}
exports.sortTSConfigs = sortTSConfigs;
//# sourceMappingURL=typescriptProjectProvider.js.map