/******************************** -*- C -*- ****************************
 *
 *	Binary image save/restore.
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 1988,89,90,91,92,94,95,99,2000,2001,2002
 * Free Software Foundation, Inc.
 * Written by Steve Byrne.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later 
 * version.
 * 
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 *
 ***********************************************************************/


#include "gst.h"
#include "gstpriv.h"

#include <stdio.h>

#if HAVE_INET_SOCKETS
#include <netinet/in.h>		/* for ntohl */
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if STDC_HEADERS
#include <string.h>		/* for memcpy */
#include <stdlib.h>
#endif /* STDC_HEADERS */

/* These flags help defining the flags and checking whether they are
   different between the image we are loading and our environment. */

#define MASK_ENDIANNESS_FLAG 1
#define MASK_LONG_SIZE_FLAG 2

#ifdef WORDS_BIGENDIAN
# define LOCAL_ENDIANNESS_FLAG MASK_ENDIANNESS_FLAG
#else
# define LOCAL_ENDIANNESS_FLAG 0
#endif

#if SIZEOF_LONG == 4
# define LOCAL_LONG_SIZE_FLAG 0
#else /* SIZEOF_LONG == 8 */
# define LOCAL_LONG_SIZE_FLAG MASK_LONG_SIZE_FLAG
#endif /* SIZEOF_LONG == 8 */

#define FLAG_CHANGED(flags, whichFlag)	\
  ((flags ^ LOCAL_##whichFlag) & MASK_##whichFlag)

#define FLAGS_WRITTEN		\
  (LOCAL_ENDIANNESS_FLAG | LOCAL_LONG_SIZE_FLAG)

#define VERSION_REQUIRED	\
  ((ST_MAJOR_VERSION << 16) + (ST_MINOR_VERSION << 8) + ST_EDIT_VERSION)

/* The binary image file has the following format:
	header
	complete oop table
	global oop variable data
	object data */

typedef struct save_file_header
{
  unsigned char signature[7];	/* 7+1=8 should be enough to align
				   version! */
  unsigned char flags;		/* flags for endianness and
				   sizeof(long) */
  unsigned long version;	/* the Smalltalk version that made this 
				   dump */
  unsigned long objectDataSize;	/* size of object data section in bytes 
				 */
  unsigned long oopTableSize;	/* size of the oop table at
					   dump */
  unsigned long memSpaceSize;	/* size of the semi spaces at dump time 
				 */
}
save_file_header;


/* These function respectively skip after the image-file header (because
   it is the last place to be filled) and back to the image-file header
   (at the start of the file, just before it is filled). IMAGEFD is the
   file descriptor for the image file. */
static void skip_over_header (int imageFd);
static void skip_to_header (int imageFd);

/* This function saves the object pointed to by OOP on the image-file
   whose descriptor is IMAGEFD.  The object pointers are made relative
   to the beginning of the object data-area. */
static void save_object (int imageFd,
			 OOP oop);

/* This function converts NUMFIXED absolute addresses at OBJ->data,
   which are instance variables of the object, into relative ones. */
static inline void fixup_object (mst_Object obj,
			  int numPointers);

/* This function converts NUMFIXED relative addresses at OBJ->data,
   which are instance variables of the object, into absolute ones. */
static inline void restore_object (mst_Object object,
			    int numPointers);

/* This function inverts the endianness of SIZE long-words, starting at
   BUF. */
static inline void fixup_byte_order (PTR buf,
			      int size);

/* This function converts NUMFIXED relative addresses, starting at
   OBJ->data, back to absolute form. */
static inline void restore_pointer_slots (mst_Object obj,
				          int numPointers);

/* This function loads an OOP table made of OLDSLOTSUSED slots from
   the image file stored in the file whose descriptor is IMAGEFD.
   The fixup gets handled by load_normal_oops */
static void load_ooptable (int imageFd);

/* This function loads OBJECTDATASIZE bytes of object data belonging
   to standard (i.e. non built-in OOPs) and fixes the endianness of
   the objects, as well as converting to absolute the address of
   their class.  Endianness conversion is done in two steps: first
   the non-byte objects (identified by not having the F_BYTE flag),
   including the class objects which are necessary to fix the byte
   objects, then all the byte-objects which also have instance
   variables). 
   Object data is loaded from the IMAGEFD file descriptor. */
static void load_normal_oops (int imageFd,
			      long unsigned int objectDataSize);

/* This function stores the header of the image file (specifying
   that the object data took OBJECTDATASIZE bytes) into the file
   whose descriptor is IMAGEFD. */
static void save_file_version (int imageFd,
			       long unsigned int objectDataSize);

/* This function loads into HEADERP the header of the image file
   without checking its validity.
   This data is loaded from the IMAGEFD file descriptor. */
static void load_file_version (int imageFd,
			       save_file_header * headerp);

/* This function walks the OOP table and converts all the relative
   addresses for the instance variables to absolute ones. */
static inline void restore_all_pointer_slots (void);

/* This function converts all the relative addresses for OOP's
   instance variables to absolute ones. */
static inline void restore_oop_pointer_slots (OOP oop);

/* This function prepares the OOP table to be written to the image
   file.  This contains the object sizes instead of the pointers,
   since addresses will be recalculated upon load. */
static struct OOP *make_ooptable_to_be_saved (void);

/* This function walks the OOP table and saves the OOP's whose fixed
   flag is set (or reset, according to FIXED) onto the file whose
   descriptor is IMAGEFD. */
static int save_all_objects (int imageFd,
			     mst_Boolean fixed);

/* This function reads N bytes from the file whose descriptor is FILE,
   into BUF, and switches their endianness if WRONG_ENDIANNESS is
   true. */
static inline int my_read (int file,
			   PTR buf,
			   int n);

/* This function is the heart of _gst_load_from_file, which opens
   the file and then passes the descriptor to load_snapshot into
   IMAGEFD. */
static mst_Boolean load_snapshot (int imageFd);

/* This variable says whether the image we are loading has the
   wrong endianness. */
static mst_Boolean wrong_endianness;

/* This variable contains the OOP slot index of the highest non-free
   OOP, excluding the built-in ones (i.e., it will always be <
   _gst_oop_table_size).  This is used for optimizing the size of the
   saved image, and minimizing the load time when restoring the
   system. */
static int num_used_oops = 0;

/* Convert to a relative offset from start of OOP table.  The offset
   is 0 mod pointer-size, so it still looks like a pointer to the
   IS_INT test.  */
#define OOP_RELATIVE(obj) \
  ( (OOP)((long)(obj) - (long)_gst_oop_table) )

/* Convert from relative offset to actual oop table address. */
#define OOP_ABSOLUTE(obj) \
  ( (OOP)((long)(obj) + (long)_gst_oop_table) )



mst_Boolean
_gst_save_to_file (const char *fileName)
{
  int imageFd;
  unsigned long objectDataSize;
  struct OOP *myOOPTable;

  _gst_invoke_hook ("aboutToSnapshot");

  imageFd = _gst_open_file (fileName, "w");
  if (imageFd < 0)
    return (false);

  _gst_scavenge ();		/* make sure that the world is compact */

  skip_over_header (imageFd);

  myOOPTable = make_ooptable_to_be_saved ();

  /* save up to the max oop slot in use */
  _gst_full_write (imageFd, myOOPTable,
		   sizeof (struct OOP) * num_used_oops);
  free (myOOPTable);

#ifdef SNAPSHOT_TRACE
  printf ("After saving oopt table: %d\n",
	  lseek (imageFd, 0, SEEK_CUR));
#endif /* SNAPSHOT_TRACE */

  /* Remember -- first non-fixed objects, then fixed ones */
  objectDataSize = save_all_objects (imageFd, false);

#ifdef SNAPSHOT_TRACE
  printf ("After saving non-fixed objects: %d\n",
	  lseek (imageFd, 0, SEEK_CUR));
#endif /* SNAPSHOT_TRACE */

  save_all_objects (imageFd, true);

#ifdef SNAPSHOT_TRACE
  printf ("After saving all objects: %d\n",
	  lseek (imageFd, 0, SEEK_CUR));
#endif /* SNAPSHOT_TRACE */

  skip_to_header (imageFd);
  save_file_version (imageFd, objectDataSize);

  close (imageFd);

  _gst_invoke_hook ("finishedSnapshot");

  return (true);
}


void
skip_over_header (int imageFd)
{
  lseek (imageFd, sizeof (save_file_header), SEEK_SET);
}

struct OOP *
make_ooptable_to_be_saved (void)
{
  OOP oop;
  struct OOP *myOOPTable;
  int i;

  num_used_oops = 0;

  for (oop = _gst_oop_table; oop < &_gst_oop_table[_gst_oop_table_size];
       oop++)
    if (OOP_VALID (oop))
      num_used_oops = OOP_INDEX (oop) + 1;

#ifdef SNAPSHOT_TRACE
  printf ("there are %d free oops out of %d oops, leaving %d\n",
	  _gst_num_free_oops, _gst_oop_table_size,
	  _gst_oop_table_size - _gst_num_free_oops);
  printf ("%d slots will be saved\n", num_used_oops);
#endif /* SNAPSHOT_TRACE */

  myOOPTable = malloc (sizeof (struct OOP) * num_used_oops);

  for (i = 0, oop = _gst_oop_table; i < num_used_oops; oop++, i++)
    {
      myOOPTable[i].flags = oop->flags;
      if (OOP_VALID (oop))
	{
	  int numPointers;
	  numPointers = NUM_OOPS (oop->object);

	  /* Cache the number of indexed instance variables.  We prefer
	     to do more work upon saving (done once) than upon loading
	     (done many times). */
	  if (numPointers < (F_COUNT >> F_COUNT_SHIFT))
	    myOOPTable[i].flags |= numPointers << F_COUNT_SHIFT;
	  else
	    myOOPTable[i].flags |= F_COUNT;

	  myOOPTable[i].object = (mst_Object) TO_INT (oop->object->objSize);
	}
    }
  return (myOOPTable);
}

int
save_all_objects (int imageFd,
		  mst_Boolean fixed)
{
  long objectStart, objectEnd;
  OOP oop;

  objectStart = lseek (imageFd, 0, SEEK_CUR);

  for (oop = _gst_oop_table; oop < &_gst_oop_table[num_used_oops];
       oop++)
    {
      if (!OOP_VALID (oop))
	continue;

      if (!(oop->flags & F_FIXED) ^ fixed)
	save_object (imageFd, oop);
    }

  objectEnd = lseek (imageFd, 0, SEEK_CUR);

  return (objectEnd - objectStart);
}

void
save_object (int imageFd,
	     OOP oop)
{
  mst_Object object;
  int numPointers;

  object = OOP_TO_OBJ (oop);
  numPointers = NUM_OOPS (object);

  fixup_object (object, numPointers);
  _gst_full_write (imageFd, object, sizeof (OOP) * TO_INT (object->objSize));
  restore_object (object, numPointers);
}

void
skip_to_header (int imageFd)
{
  lseek (imageFd, 0, SEEK_SET);
}

void
save_file_version (int imageFd,
		   long unsigned int objectDataSize)
{
  save_file_header header;

  memcpy (header.signature, "GSTIm\0\0", 7);
  header.flags = FLAGS_WRITTEN;
  header.version = VERSION_REQUIRED;
  header.objectDataSize = objectDataSize;
  header.oopTableSize = num_used_oops;
  header.memSpaceSize =
    MAX (_gst_mem_space.totalSize, objectDataSize * 2);

  _gst_full_write (imageFd, &header, sizeof (save_file_header));
}



/***********************************************************************
 *
 *	Binary loading routines.
 *
 ***********************************************************************/


mst_Boolean
_gst_load_from_file (const char *fileName)
{
  mst_Boolean loaded;
  int imageFd;

  imageFd = _gst_open_file (fileName, "r");
  loaded = (imageFd >= 0) && load_snapshot (imageFd);

  close (imageFd);
  return (loaded);
}

mst_Boolean
load_snapshot (int imageFd)
{
  save_file_header header;

  load_file_version (imageFd, &header);
  if (strcmp (header.signature, "GSTIm\0\0"))
    return (false);

  if UNCOMMON ((wrong_endianness =
         FLAG_CHANGED (header.flags, ENDIANNESS_FLAG)))
    {
      header.objectDataSize = BYTE_INVERT (header.objectDataSize);
      header.oopTableSize =
	BYTE_INVERT (header.oopTableSize);
      header.memSpaceSize = BYTE_INVERT (header.memSpaceSize);
      header.version = BYTE_INVERT (header.version);
    }

  /* different sizeof(long) not supported */
  if (FLAG_CHANGED (header.flags, LONG_SIZE_FLAG))
    return (false);

  /* check for version mismatch; if so this image file is invalid */
  if (header.version != VERSION_REQUIRED)
    return (false);

  if (header.memSpaceSize > _gst_mem_space.totalSize)
    _gst_grow_memory_to (header.memSpaceSize);

  _gst_init_ooptable (MAX (header.oopTableSize * 2,
		           INITIAL_OOP_TABLE_SIZE));

  num_used_oops = header.oopTableSize;
  load_ooptable (imageFd);

#ifdef SNAPSHOT_TRACE
  printf ("After loading oopt table: %d\n",
	  lseek (imageFd, 0, SEEK_CUR));
#endif /* SNAPSHOT_TRACE */

  load_normal_oops (imageFd, header.objectDataSize);

#ifdef SNAPSHOT_TRACE
  printf ("After loading objects: %d\n", lseek (imageFd, 0, SEEK_CUR));
#endif /* SNAPSHOT_TRACE */

  restore_all_pointer_slots ();
  _gst_refresh_oop_free_list ();

  return (_gst_init_dictionary_on_image_load (num_used_oops));
}

void
load_file_version (int imageFd,
		   save_file_header * headerp)
{
  read (imageFd, headerp, sizeof (save_file_header));
}

void
load_ooptable (int imageFd)
{
  OOP oop;

  /* load in the valid OOP slots from previous dump */
  my_read (imageFd, _gst_oop_table, sizeof (struct OOP) * num_used_oops);

  /* mark the remaining ones as available */
  for (oop = &_gst_oop_table[num_used_oops];
       oop < &_gst_oop_table[_gst_oop_table_size]; oop++)
    {
      oop->flags = F_FREE;
      oop++;
    }

  _gst_num_free_oops = _gst_oop_table_size - num_used_oops;
}


void
load_normal_oops (int imageFd,
		  long unsigned int objectDataSize)
{
  OOP oop;
  mst_Object object;
  OOP *objPtr;

  /* First load the main heap. Not everything must be byte-inverted in
     this part of the file, so read everything through good old read
     and fix later. */

  objPtr = (OOP *) _gst_cur_space_addr ();
  read (imageFd, objPtr, objectDataSize);

  /* Now walk the oop table.  Start fixing the byte order, and get to
     the end of the image file by loading fixed objects. */

  _gst_num_free_oops = 0;
  for (oop = _gst_oop_table; oop < &_gst_oop_table[num_used_oops];
       oop++)
    {
      if (!OOP_VALID (oop))
	{
	  _gst_num_free_oops++;
	  oop->flags = F_FREE;	/* just free */
	  continue;
	}

      if (oop->flags & F_FIXED)
	{
	  /* oops... this still is to be read... */
	  long size;
	  size = (long) oop->object;

	  object = (mst_Object) xmalloc (size * SIZEOF_LONG);
	  read (imageFd, object, SIZEOF_LONG * size);
	}
      else
	object = (mst_Object) objPtr;

      if (oop->flags & F_CONTEXT)
	{
	  /* this is another quirk; this is not the best place to do
	     it. We have to reset the nativeIPs so that we can find
	     restarted processes and recompile their methods. */
	  gst_method_context context = (gst_method_context) object;
	  context->native_ip = DUMMY_NATIVE_IP;
	}

      oop->object = object;
      if UNCOMMON (wrong_endianness)
	{
	  OOP size = (OOP) BYTE_INVERT ((long) oop->object->objSize);
	  fixup_byte_order (object, 
			    (oop->flags & F_BYTE)
			      ? OBJ_HEADER_SIZE_WORDS : TO_INT (size));
	}

      object->objClass = OOP_ABSOLUTE (object->objClass);

      if (!(oop->flags & F_FIXED))
	objPtr += TO_INT (object->objSize);

      /* Remove flags that are invalid after an image has been loaded. */
      oop->flags &= ~F_RUNTIME;
    }

  /* NUM_OOPS requires access to the instance spec in the class
     objects. So we start by fixing the endianness of NON-BYTE objects
     (including classes!), for which we can do without NUM_OOPS, then
     do another pass here and fix the byte objects using the now
     correct class objects. */
  if UNCOMMON (wrong_endianness)
    for (oop = _gst_oop_table; oop < &_gst_oop_table[num_used_oops];
         oop++)
      if (oop->flags & F_BYTE)
	{
	  object = OOP_TO_OBJ (oop);
	  fixup_byte_order (object->data, NUM_OOPS (object));
	}

  _gst_set_space_info (objectDataSize);
}


/* Routines to convert to/from relative pointers, shared by
   loading and saving */

void
fixup_object (mst_Object obj,
	      int numPointers)
{
  OOP instOOP, class_oop, *i;

  class_oop = obj->objClass;
  obj->objClass = OOP_RELATIVE (class_oop);

  for (i = obj->data; numPointers; i++, numPointers--)
    {
      instOOP = *i;
      if (IS_OOP (instOOP))
	*i = OOP_RELATIVE (instOOP);
    }
}

void
restore_object (mst_Object object,
		     int numPointers)
{
  object->objClass = OOP_ABSOLUTE (object->objClass);
  restore_pointer_slots (object, numPointers);
}

void
restore_pointer_slots (mst_Object obj,
		       int numPointers)
{
  OOP instOOP, *i;

  i = obj->data;
  while (numPointers--)
    {
      instOOP = *i;
      if (IS_OOP (instOOP))
	*i = OOP_ABSOLUTE (instOOP);

      i++;
    }
}

void
restore_all_pointer_slots ()
{
  OOP oop;

  for (oop = _gst_oop_table; oop < &_gst_oop_table[num_used_oops];
       oop++)
    if (OOP_VALID (oop))
      restore_oop_pointer_slots (oop);
}

void
restore_oop_pointer_slots (OOP oop)
{
  int numPointers;
  mst_Object object;

  object = OOP_TO_OBJ (oop);
  if UNCOMMON ((oop->flags & F_COUNT) == F_COUNT)
    numPointers = NUM_OOPS (object);
  else
    numPointers = oop->flags >> F_COUNT_SHIFT;

  restore_pointer_slots (object, numPointers);
}




void
fixup_byte_order (PTR buf,
		  int size)
{
  long unsigned int *p = (long unsigned int *) buf;
  for (; size--; p++)
    *p = BYTE_INVERT (*p);
}

int
my_read (int file,
	 PTR buf,
	 int n)
{
  int result;
  long unsigned int *p = (long unsigned int *) buf;

  result = read (file, p, n);
  if UNCOMMON (wrong_endianness)
    fixup_byte_order (p, result / sizeof (long));

  return (result);
}
