/*\
||| Written By Fredrik Hbinette <hubbe@lysator.liu.se>
||| and Louis Bavoil <bavoil@enseirb.fr>
||| All rights reserved. No warrenties, use at your own risk.
\*/

#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <sysexits.h>
#include <signal.h>
#include <stdarg.h>
#include <npapi.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <errno.h>
#include <sys/time.h>
#include <X11/X.h>
#include <X11/Xos.h>
#include <X11/Intrinsic.h>
#include <X11/Xatom.h>
#include <ctype.h>
#include <sys/socket.h>

#include "mozplugger.h"

#define WINDOW ((Window) THIS->windata.window)

struct data
{
  Display *display;
  char *displayname;
  int pid1;
  int pid2;

  NPWindow windata;

  char *mimetype;
  int repeats;

  int flags;
  char *command;
  char *winname;
};

#define THIS (*(struct data **) &(instance->pdata))
#define FEND(X,Y) (strchr(X,Y)?strchr(X,Y):(X)+strlen(X))
#define xalloc malloc
#define xfree free


/*
 * Helper functions
 */

/* Check weather the first word in 'file' is somewhere to be found in
 * the users PATH. If so, return 1 otherwise 0.
 */
static int inpath(char *file)
{
  struct stat buf;
  char tmp[16384];
  char *path=getenv("PATH");
  char *pos;
  if(!path) return 0;

  D("inpath(%s)\n",file);
  if(file[0]=='/')
    return !stat(file, &buf);

  if(!strncmp(file,"internal:",9)) return 1;

  /* Ugly ugly ugly */
  if(file[0]=='i' && file[1]=='f' && isspace(file[2]))
    file+=3;

  /* Insert magical shell-script tests here */

  D("Hmm? PATH=%s\n",path);
  pos=path;
  while(1)
  {
    char *next;
    next=FEND(pos,':');
    if(next!=pos)
    {
      char *ptr=tmp;

      memcpy(ptr=tmp,pos,next-pos);
      ptr+=next-pos;
      *(ptr++)='/';
      memcpy(ptr,
	     file,
	     FEND(file,' ')-file);
      ptr+=FEND(file,' ')-file;
      *ptr=0;
      D("stat(%s)\n",tmp);
      if(!stat(tmp, &buf)) return 1;
      D("nope\n");
    }
    if(!*next) return 0;
    pos=next+1;
  }
  D("GAAHHH!\n");
}

#ifndef MAX_FD
#define MAX_FD 1024
#endif

static void handler(int sig)
{
  int status;
  wait(&status);
}

/* Shell for fork which handles some magic needed to prevent
 * netscape from freaking out. It also prevents interferrance
 * from signals.
 */

static int my_fork(NPP instance)
{
  int pid;
  sigset_t set,oset;
  sigfillset(& set);
  D(">>>>>>>>Forking<<<<<<<<,\n");
  sigprocmask(SIG_SETMASK,&set,&oset);
  signal(SIGCHLD, &handler);
  pid=fork();
  if(pid==-1) return pid;
  if(!pid)
  {
    int signum;
    alarm(0);
    if(!(THIS->flags & H_DAEMON))
      setsid();

    for(signum=0;signum<NSIG;signum++)
      signal(signum, SIG_DFL);

#ifdef DEBUG
    debug_output=0;
    D("Chilid re-opened ndebug\n");
#endif
    THIS->display=0;
  }else{
    D("Child running with pid=%d\n",pid);
  }
  sigprocmask(SIG_SETMASK,&oset,&set);
  return pid;
}

static char *helper_fname=NULL;

static void run(NPP instance, const char *file)
{
  char buffer[1024];

  sprintf(buffer,"%d,%d,%d,%d,%d,%d,%d",
	  THIS->flags,
	  THIS->repeats,
	  (int)THIS->windata.window,
	  (int)THIS->windata.x,
	  (int)THIS->windata.y,
	  (int)THIS->windata.width,
	  (int)THIS->windata.height);
  D("Executing helper: %s %s %s %s %s %s %s\n",
    helper_fname,
    buffer,
    file,
    THIS->winname,
    THIS->displayname,
    THIS->command,
    THIS->mimetype);

  execlp(helper_fname,
	 helper_fname,
	 buffer,
	 file ? file : "",
	 THIS->winname ? THIS->winname : "",
	 THIS->displayname ? THIS->displayname : "",
	 THIS->command,
	 THIS->mimetype,
	 0);

  D("EXECLP FAILED!\n");
	 
  _exit(69); /* EX_UNAVAILABLE */
}


/* 
 * Functions below
 * are called from inside netscape/mozilla/opera
 */

struct mimetype
{
  struct mimetype *next;
  char *line;
};

struct command
{
  struct command *next;
  int flags;
  char *cmd;
  char *winname;
};

struct handle
{
  struct handle *next;
  struct mimetype *types;
  struct command *commands;
};

static struct handle *first_handle=NULL;

static int gobble(char *line, char *kw)
{
  return !strncasecmp(line,kw,strlen(kw)) && !isalnum(line[strlen(kw)]);
}

/* read configuration file into memory */
static void read_config(FILE *f)
{
  struct handle **handlep;
  struct command **commandp=0;
  struct mimetype **typep=0;
  char buffer[16384];
  int have_commands=1;
  D("read_config\n");

  handlep=&first_handle;

  while(!feof(f))
  {
    fgets(buffer,sizeof(buffer),f);
    D("::: %s",buffer);
    if(buffer[0]=='#' || !buffer[0] || buffer[0]=='\n') continue;

    if(buffer[strlen(buffer)-1]=='\n')
      buffer[strlen(buffer)-1]=0;

    if(isspace(buffer[0]))
    {
      char *x=buffer+1;
      while(isspace(*x)) x++;
      if(!*x)
      {
	D("Empty line.\n");
	continue;
      }

      D("New command\n");

      *commandp=(struct command *)xalloc(sizeof(struct command));
      if(!*commandp)
      {
	D("xalloc failed\n");
	return;
      }
      (*commandp)->flags=0;
      (*commandp)->cmd=0;
      (*commandp)->winname=0;
      (*commandp)->next=0;

      /* Command */
      have_commands++;
      while(*x!=':' && *x)
      {
/* 	D("Parsing %s\n",x); */
	switch(*x)
	{
	  case ',':
	  case ' ':
	  case '\t':
	    x++;
	    break;

	  default:
#define GOBBLE(X,Y) \
  if(gobble(x,X)) { x+=strlen(X); (*commandp)->flags|=Y; break; }
	    GOBBLE("repeat",H_REPEATCOUNT);
	    GOBBLE("loop",H_LOOP);
	    GOBBLE("stream",H_STREAM);
	    GOBBLE("preload",H_PRELOAD);
	    GOBBLE("many",H_MANY);
	    GOBBLE("ignore_errors",H_IGNORE_ERRORS);
	    GOBBLE("exits",H_EXITS);
	    GOBBLE("nokill",H_DAEMON);

	    GOBBLE("maxaspect",H_MAXASPECT);
	    GOBBLE("fill",H_FILL);
#ifdef H_NOISY
	    GOBBLE("noisy",H_NOISY);
#endif
	    if(gobble(x,"swallow"))
	    {
	      char *end;
	      (*commandp)->flags|=H_SWALLOW;
	      x+=7;
	      while(isspace(*x)) x++;
	      if(*x != '(')
	      {
		fprintf(stderr,"Expected '(' after 'swallow'\n");
		continue;
	      }
	      x++;
	      end=strchr(x,')');
	      if(end)
	      {
		(*commandp)->winname=xalloc(end - x +1);
		memcpy((*commandp)->winname, x, end-x);
		(*commandp)->winname[end-x]=0;
	      }
	      x=end+1;
	      break;
	    }
	    D("Unknown directive: %s\n",x);

	    /* Unknown directive */
	    fprintf(stderr,"Unknown directive: %s\n",x);
	    x+=strlen(x);
	}
      }
      if(*x==':')
      {
	x++;
	while(isspace(*x)) x++;
	(*commandp)->cmd=strdup(x);
      }else{
	D("No colon? (%s)\n",x);
      }
      if(!(*commandp)->cmd)
      {
	xfree(*commandp);
	*commandp=0;
	D("strdup failed %s\n",x);
	return;
      }
      commandp=&((*commandp)->next);
    }else{
      /* Mime type */

      if(have_commands)
      {
  	D("New handle\n");
	if(commandp)
	  D("Commandp=%p\n",*commandp);
	*handlep=(struct handle *)xalloc(sizeof(struct handle));
	if(!*handlep)
	  return;
	
	(*handlep)->commands=0;
	(*handlep)->types=0;
	(*handlep)->next=0;
	commandp=&((*handlep)->commands);
	typep=&((*handlep)->types);
	handlep=&((*handlep)->next);
	have_commands=0;
      }

      D("New mimetype\n");
      *typep=(struct mimetype *)xalloc(sizeof(struct mimetype));
      if(!*typep)
	return;

      (*typep)->next=0;
      (*typep)->line=strdup(buffer);

      if(!(*typep)->line)
      {
	xfree(*typep);
	*typep=0;
	return;
      }
      typep=&((*typep)->next);
    }
  }
}


int find_helper_file(char *basename,
		     int (*cb)(char *,void *data),
		     void *data)
{
  static char fname[8192];
  char *tmp;
  D("find_helper_file '%s'\n",basename);
  if((tmp=getenv("HOME")) && strlen(tmp)<8000)
  {
    sprintf(fname,"%s/.netscape/%s",tmp,basename);
    if(cb(fname,data)) return 1;
    sprintf(fname,"%s/.mozilla/%s",tmp,basename);
    if(cb(fname,data)) return 1;
    sprintf(fname,"%s/.opera/%s",tmp,basename);
    if(cb(fname,data)) return 1;
  }

  if((tmp=getenv("MOZILLA_HOME")) && strlen(tmp)<8000)
  {
    sprintf(fname,"%s/%s",tmp,basename);
    if(cb(fname, data)) return 1;
  }

  if((tmp=getenv("OPERA_DIR")) && strlen(tmp)<8000)
  {
    sprintf(fname,"%s/%s",tmp,basename);
    if(cb(fname, data)) return 1;
  }

  sprintf(fname,"/usr/local/netscape/%s",basename);
  if(cb(fname, data)) return 1;
  sprintf(fname,"/etc/%s",basename);
  if(cb(fname, data)) return 1;
  sprintf(fname,"/usr/etc/%s",basename);
  if(cb(fname, data)) return 1;
  sprintf(fname,"/usr/local/etc/%s",basename);
  if(cb(fname, data)) return 1;
  if(cb(basename, data)) return 1;
  
  return 0;
}

static int read_config_cb(char *fname, void *data)
{
  FILE *f=fopen(fname,"r");
  if(!f) return 0;
  read_config(f);
  fclose(f);
  return 1;
}

static int find_plugger_helper_cb(char *fname, void *data)
{
  struct stat buf;
  if(stat(fname, &buf)) return 0;
  helper_fname=strdup(fname);
  return 1;
}

/* Find configuration file and read it into memory */
static void do_read_config(void)
{
  if(first_handle) return;
  D("do_read_config\n");
  
  if(!find_helper_file("mozpluggerrc-" VERSION,read_config_cb,0) &&
     !find_helper_file("mozpluggerrc",read_config_cb,0))
  {
    fprintf(stderr,"MozPlugger: Unable to find mozpluggerrc file!\n");
    return;
  }
  D("do_read_config done\n");

  find_helper_file("mozplugger-" VERSION, find_plugger_helper_cb, 0);
  if(inpath("mozplugger-" VERSION))
    helper_fname="mozplugger-" VERSION;

  if(!helper_fname)
    fprintf(stderr,"MozPlugger: Unable to find mozplugger-" VERSION "\n");
}

/* Construct a MIME Description string for netscape
 * so that netscape shall know when to call us back.
 */
char *NPP_GetMIMEDescription(void)
{
  char *x,*y;
  struct handle *h;
  struct mimetype *m;
  int size_required;

  do_read_config();

  D("Getmimedescription\n");

  size_required=0;
  for(h=first_handle;h;h=h->next)
    for(m=h->types;m;m=m->next)
      size_required+=strlen(m->line)+1;

  D("Size required=%d\n",size_required);
  x=(char *)xalloc(size_required+1);
  if(!x) return 0;

  D("Malloc did not fail\n");

  y=x;

  for(h=first_handle;h;h=h->next)
  {
    D("Foo: %p\n",h->commands);
    for(m=h->types;m;m=m->next)
    {
/*       D("appending: %s\n",m->line); */
      memcpy(y,m->line,strlen(m->line));
      y+=strlen(m->line);
      *(y++)=';';
    }
  }
  if(x!=y) y--;
  *(y++)=0;
  D("Getmimedescription done: %s\n",x);
  return x;
}

/* Go through the commands in the config file 
 * and find one that fits our needs.
 */
static int find_command(NPP instance,
			int streaming)
{
  struct handle *h;
  struct mimetype *m;
  struct command *c;

  D("find_command\n");

  do_read_config();

  D("find_command...\n");

  for(h=first_handle;h;h=h->next)
  {
    D("commands for this handle at (%p)\n",h->commands);
    for(m=h->types;m;m=m->next)
    {
      char *tmp1=FEND(m->line,':');
      char tmp2;
      int tmp3;
      D("Checking '%s'\n",m->line);

      while(tmp1>m->line && isspace(tmp1[-1]))
	tmp1--;

      D("tmp1=%s\n",tmp1);

      tmp2=*tmp1;
      *tmp1=0;
      D("Checking '%s' ?= '%s'\n",m->line,THIS->mimetype);
      tmp3=strcasecmp(THIS->mimetype, m->line);
      *tmp1=tmp2;
      if(!tmp3)
      {
	D("Match found!\n");
	break;
      }else{
	D("Not same.\n");
      }
    }

    if(m)
    {
      for(c=h->commands;c;c=c->next)
      {
	D("Checking command: %s\n",c->cmd);
	if((c->flags & H_LOOP) && THIS->repeats != MAXINT) continue;
	if( (!!streaming) != (!!(c->flags & H_STREAM))) continue;
	if(!inpath(c->cmd)) continue;
	D("Match found!\n");
	THIS->command=c->cmd;
	THIS->flags=c->flags;
	THIS->winname=c->winname;
	return 1;
      }
    }
  }
  D("No match found\n");
  return 0;
}

/* Let netscape know things about mozplugger */
NPError NPP_GetValue(void *future, NPPVariable variable, void *value)
{
  NPError err = NPERR_NO_ERROR;

  D("Getvalue %d\n",variable);
  
  switch (variable)
  {
  case NPPVpluginNameString:
    D("GET Plugin name\n");
    *((char **)value) = "MozPlugger "
      VERSION
      ;

  break;

  case NPPVpluginDescriptionString:
    D("GET Plugin description\n");
    *((char **)value) =
      "MozPlugger version "
      VERSION
      ", written by "
      "<a href=http://fredrik.hubbe.net/>Fredrik Hbinette</a> "
      "<a href=mailto:hubbe@hubbe.net>&lt;hubbe@hubbe.net&gt</a> "
      "and Louis Bavoil "
      "<a href=mailto:bavoil@enseirb.fr>&lt;bavoil@enseirb.fr&gt</a>.<br>"
      "For documentation on how to configure mozplugger, "
      "check the man page. (type <tt>man&nbsp;mozplugger</tt>)"
      ;
  break;

  default:
    err = NPERR_GENERIC_ERROR;
  }
  return err;
}

/* Initiate another instance of mozplugger, it is important to know
 * that there might be several instances going at one time.
 */
NPError  NPP_New(NPMIMEType pluginType,
		 NPP instance,
		 uint16 mode,
		 int16 argc,
		 char* argn[],
		 char* argv[],
		 NPSavedData* saved)
{
  int e;

  D("NEW (%s)\n",pluginType);

  if (!instance)
    return NPERR_INVALID_INSTANCE_ERROR;

  THIS = NPN_MemAlloc(sizeof(struct data));
  if(!THIS) return NPERR_OUT_OF_MEMORY_ERROR;
  memset((char *)THIS, 0, sizeof(*THIS));

  /* Change default to repeat once? */
  THIS->repeats=0x7fffffff;
  THIS->pid1=-1;
  THIS->pid2=-1;
  WINDOW=0;
  THIS->displayname=0;
  THIS->display=0;
  
  if(!pluginType) return NPERR_GENERIC_ERROR;

  for(e=0;e<argc;e++)
  {
    if(!strcasecmp("loop",argn[e]))
    {
      switch(argv[e][0])
      {
      case 't': case 'T': case 'y': case 'Y':
	THIS->repeats=MAXINT;
	break;
      case 'f': case 'F': case 'n': case 'N':
	THIS->repeats=1;
	break;
      case '0': case '1': case '2': case '3': case '4':
      case '5': case '6': case '7': case '8': case '9':
	THIS->repeats=atoi(argv[e]);
      }
    }
  }

  return NPERR_NO_ERROR;
}

/* Free data, kill processes, it is time for this instance to die */
NPError NPP_Destroy(NPP instance, NPSavedData** save)
{
  if (!instance)
    return NPERR_INVALID_INSTANCE_ERROR;

  D("Destroy\n");

  if (THIS)
  {
    if(THIS->pid1 > 0)
    {
      kill(-THIS->pid1, SIGKILL);
      kill(THIS->pid1, SIGKILL);
    }
    
    if(THIS->pid2 > 0)
    {
      kill(-THIS->pid2, SIGKILL);
      kill(THIS->pid2, SIGKILL);
    }

    D("Freeing memory %p\n",THIS->mimetype);
    
    if(THIS->mimetype)
    {
      NPN_MemFree(THIS->mimetype);
      THIS->mimetype=0;
    }
  }
  D("Destroy finished\n");
  
  return NPERR_NO_ERROR;
}

static char tempfile;

/* Open a new stream,
 * each instance can only handle one stream at a time.
 */
NPError NPP_NewStream(NPP instance,
		      NPMIMEType type,
		      NPStream *stream, 
		      NPBool seekable,
		      uint16 *stype)
{
  D("Newstream ... \n");

  if (instance == NULL)
    return NPERR_INVALID_INSTANCE_ERROR;

  /* This is a stupid special case and should be coded into
   * mozpluggerc instead...
   */
  if(!strncasecmp("image/",type,6) ||
     !strncasecmp("x-image/",type,6))
    THIS->repeats=1;

  D("Mime type %s\n",type);

  if(THIS->mimetype)
  {
    NPN_MemFree(THIS->mimetype);
    THIS->mimetype=0;
  }
  THIS->mimetype = NPN_MemAlloc(strlen(type)+1);
  if(!THIS->mimetype) return NPERR_OUT_OF_MEMORY_ERROR;
  strcpy(THIS->mimetype, type);

  D("Url is %s (seekable=%d)\n",stream->url,seekable);
  
  if(!find_command(instance, 1))
  {
    if(!find_command(instance, 0))
    {
      if(!inpath("xmessage"))
      {
	NPN_Status(instance, "No approperiate application found!");
	return NPERR_GENERIC_ERROR;
      }else{
	D("Using xmessage!!!\n");
	/* Hard-coded default */
	THIS->command="xmessage -buttons '' \"MozPlugger: No approperiate application for type $mimetype found!\"";
	THIS->flags = H_REPEATCOUNT | H_FILL | H_SWALLOW | H_IGNORE_ERRORS;
	THIS->winname="Xmessage";
      }
    }
  }

  if((THIS->flags & H_STREAM) && strncasecmp(stream->url,"file:",5))
  {
    *stype = NP_NORMAL;
    tempfile = 0;

    THIS->pid1=my_fork(instance);
    if(THIS->pid1==-1)
    {
      NPN_Status(instance, "Streamer: My_Fork failed!");
      return NPERR_GENERIC_ERROR;
    }
    
    if(!THIS->pid1)
    {
      THIS->repeats=1;
      D("CHILD RUNNING run() [1]\n");
      run(instance, stream->url);
      exit(69);
    } else {
      if(THIS->repeats < MAXINT)
	THIS->repeats--;
    }
    
    D("Ok\n");
  } else {
    *stype = NP_ASFILEONLY;
    tempfile = 1;
  }
  return NPERR_NO_ERROR;
}

int32 NPP_WriteReady(NPP instance, NPStream *stream)
{
  if (!instance)
    return 0;
  if (!tempfile)
    {
      NPP_DestroyStream(instance, stream, NPERR_NO_ERROR);
      return 0;
    }
  return 0x7fffffff;
}

int32 NPP_Write(NPP instance,
		NPStream *stream,
		int32 offset,
		int32 len,
		void *buf)
{
  return len;
}

NPError NPP_DestroyStream(NPP instance, NPStream *stream, NPError reason)
{
  D("Destroystream\n");
  return NPERR_NO_ERROR;
}

void NPP_StreamAsFile(NPP instance,
		      NPStream *stream,
		      const char* fname)
{
  D("Streamasfile\n");

  if (!tempfile) return;
  if (!fname) return;
  if (instance == NULL) return;

  D("......going to fork......\n");
  THIS->pid2=my_fork(instance);
  if(THIS->pid2==-1) return;
    
  if(!THIS->pid2)
    {
      D("CHILD RUNNING run() [7]\n");

      if(!find_command(instance, 0))
	{
	  if(inpath("xmessage"))
	    {
	      D("Using xmessage!!!\n");
	      /* Hard-coded default */
	      THIS->command="xmessage -buttons '' \"MozPlugger: No approperiate application for type $mimetype found!\"";
	      THIS->flags = H_REPEATCOUNT | H_FILL | H_SWALLOW | H_IGNORE_ERRORS;
	      THIS->winname="Xmessage";
	    }else{
	      exit(EX_UNAVAILABLE);
	    }
	}

      D("CHILD RUNNING run() [2]\n");
      run(instance, fname);
      exit(69);
    }
}

NPError NPP_Initialize(void)
{
  D("init\n");
  return NPERR_NO_ERROR;
}
jref NPP_GetJavaClass()
{
  D("Getjavaclass\n");
  return NULL;
}
void NPP_Shutdown(void)
{
  D("Shutdown\n");
}

NPError NPP_SetWindow(NPP instance, NPWindow* window)
{
  D("SETWINDOW\n");

  if (!instance)
    return NPERR_INVALID_INSTANCE_ERROR;
  
  if (!window)
    return NPERR_NO_ERROR;
  
  if (!window->window)
    return (NPERR_NO_ERROR);

  if (!window->ws_info)
    return (NPERR_NO_ERROR);

  THIS->display= ((NPSetWindowCallbackStruct *)window->ws_info)->display;
  THIS->displayname=XDisplayName( DisplayString(THIS->display));
  THIS->windata = *window;
  D("Displayname=%s Window=%x %d %d %d %d\n",
    THIS->displayname,
    WINDOW,
    window->x,window->y,
    window->width, window->height);

  XResizeWindow(THIS->display,
		WINDOW,
		window->width,
		window->height);

  XSync (THIS->display, FALSE);

  if(THIS->pid1 != -1)
  {
    D("Sending SIGWINCH to pid1 (%d)\n",THIS->pid1);
    kill(THIS->pid1, SIGWINCH);
  }
  if(THIS->pid2 != -1)
  {
    D("Sending SIGWINCH to pid2 (%d)\n",THIS->pid2);
    kill(THIS->pid2, SIGWINCH);
  }

  usleep(4000);

  return NPERR_NO_ERROR;
}

void NPP_Print(NPP instance, NPPrint* printInfo)
{
  D("PRINT\n");
  if(printInfo == NULL)
    return;
  
  if (instance != NULL)
  {
    if (printInfo->mode == NP_FULL) {
      void* platformPrint =
	printInfo->print.fullPrint.platformPrint;
      NPBool printOne =
	printInfo->print.fullPrint.printOne;
      
      /* Do the default*/
      printInfo->print.fullPrint.pluginPrinted = FALSE;
    }
    else
    {	/* If not fullscreen, we must be embedded */
      NPWindow* printWindow =
	&(printInfo->print.embedPrint.window);
      void* platformPrint =
	printInfo->print.embedPrint.platformPrint;
    }
  }
}
