"""
This file is part of the 'Elements' Project
Elements is a 2D Physics API for Python (supporting Box2D2)

Copyright (C) 2008, The Elements Team, <elements@linuxuser.at>

Home:  http://wiki.laptop.org/go/Elements
IRC:   #elements on irc.freenode.org

Code:  http://www.assembla.com/wiki/show/elements
       svn co http://svn2.assembla.com/svn/elements                     

License:  GPLv3 | See LICENSE for the full text
This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.              
"""
from functools import partial

from math import fabs, sqrt, atan, degrees

# Some Hex Tools
def hex2dec(hex): return int(hex, 16)
def hex2rgb(hex): 
    if hex[0:1] == '#': hex = hex[1:]; 
    return (hex2dec(hex[:2]), hex2dec(hex[2:4]), hex2dec(hex[4:6]))


def calc_center(points):
    """Calculate the center of a polygon
    
    :return: The center (x,y)
    """
    tot_x, tot_y = 0,0
    for p in points:
        tot_x += p[0]
        tot_y += p[1]
    n = len(points)
    return (tot_x/n, tot_y/n)
    
def poly_center_vertices(pointlist):
    """Rearranges vectors around the center
    
    :return: pointlist ([(x, y), ...])
    """    
    poly_points_center = []
    center = cx, cy = calc_center(pointlist)
    
    for p in pointlist:
        x = p[0] - cx
        y = cy - p[1]
        poly_points_center.append((x, y))
    
    return poly_points_center
    
def is_line(vertices, tolerance=0.20):
    # Step 1: Points -> Vectors
    p_old = vertices[0]
    vectors = []
    alpha_min = 180
    alpha_max = -180
    
    for p in vertices[1:]:
        x1, y1 = p_old
        x2, y2 = p
        p_old = p
        
        vx, vy = (x2-x1, y2-y1)
        l = sqrt((vx*vx) + (vy*vy))
        
        # normalize vector
        print l
        if l == 0.0:
            return False

        vx /= l
        vy /= l

        if vx == 0:
            alpha = 90
        else:
            alpha = degrees(atan(vy / vx))
 
        if alpha > 180:
            alpha = 360 - alpha
        
        alpha = fabs(alpha)
        
        if alpha > alpha_max:
            alpha_max = alpha
        if alpha < alpha_min:
            alpha_min = alpha
            
        vectors.append((vx, vy))
        print vectors[-1], "alpha=", alpha
#        print alpha
        
#    print ">> ", x_sum, y_sum
    alpha_min = fabs(alpha_min)
    alpha_max = fabs(alpha_max)
    print ">> alpha min, max:", alpha_min, alpha_max
    # get diff

    if alpha_max > alpha_min:    
        alpha_diff = fabs(alpha_max - alpha_min)
    else:
        alpha_diff = fabs(alpha_min - alpha_max)
    print ">> alpha_diff:", alpha_diff
    
    if alpha_diff < 25.0:
        return True
    else:
        return False
    
# The following functions is_left, reduce_poly and convex_hull are 
# from the pymunk project (http://code.google.com/p/pymunk/)
def is_left(p0, p1, p2):
    """Test if p2 is left, on or right of the (infinite) line (p0,p1).
    
    :return: > 0 for p2 left of the line through p0 and p1
        = 0 for p2 on the line
        < 0 for p2 right of the line
    """
    sorting = (p1[0] - p0[0])*(p2[1]-p0[1]) - (p2[0]-p0[0])*(p1[1]-p0[1])
    if sorting > 0: return 1
    elif sorting < 0: return -1 
    else: return 0


def reduce_poly(points, tolerance=50):
    """Remove close points to simplify a polyline
    tolerance is the min distance between two points squared.
    
    :return: The reduced polygon as a list of (x,y)
    """
    curr_p = points[0]
    reduced_ps = [points[0]]
    
    for p in points[1:]:
    	x1, y1 = curr_p
    	x2, y2 = p
    	dx = fabs(x2 - x1)
    	dy = fabs(y2 - y1)
    	l = sqrt((dx*dx) + (dy*dy))
#    	print dx, dy, l
        if l > tolerance:
            curr_p = p
            reduced_ps.append(p)
            
    return reduced_ps

def convex_hull(points):
    """Create a convex hull from a list of points.
    This function uses the Graham Scan Algorithm.
    
    :return: Convex hull as a list of (x,y)
    """
    ### Find lowest rightmost point
    p0 = points[0]
    for p in points[1:]:
        if p[1] < p0[1]:
            p0 = p
        elif p[1] == p0[1] and p[0] > p0[0]:
            p0 = p
    points.remove(p0)
    
    ### Sort the points angularly about p0 as center
    f = partial(is_left, p0)
    points.sort(cmp = f)
    points.reverse()
    points.insert(0, p0)
    
    ### Find the hull points
    hull = [p0, points[1]]
    
    for p in points[2:]:
        
        pt1 = hull[-1]
        pt2 = hull[-2]
        l = is_left(pt2, pt1, p) 
        if l > 0:
            hull.append(p)
        else:
            while l <= 0 and len(hull) > 2:
                hull.pop()
                pt1 = hull[-1]
                pt2 = hull[-2]
                l = is_left(pt2, pt1, p)
            hull.append(p)
    return hull     
    
