/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>

#include "cmp.h"
#include "kalc.h"
#include "realfunctions.h"


void f_add(void)
{
  /*
   * This function calls the _f_add function through the wrapper.
   */

  run2_1_Function(_f_add, "+");
}


void f_and(void)
{
  /*
   * This function calls the _f_and function through the wrapper.
   */

  run2_1_Function(_f_and, "and");
}


void f_or(void)
{
  /*
   * This function calls the _f_or function through the wrapper.
   */

  run2_1_Function(_f_or, "or");
}


void f_xor(void)
{
  /*
   * This function calls the _f_xor function through the wrapper.
   */

  run2_1_Function(_f_xor, "xor");
}


void f_not(void)
{
  /*
   * This function calls the _f_not function through the wrapper.
   */

  run1_1_Function(_f_not, "not");
}


void f_neg(void)
{
  /*
   * This function calls the _f_neg function through the wrapper.
   */

  run1_1_Function(_f_neg, "neg");
}


Object _f_add(Object n, Object p, int *err)
{
  /*
   * This function does string concatenation if at least of of its
   * arguments is a string, or arithmetic sum of real or complex numbers,
   * and hex strings.
   */

  if (type(n) == TYPE_STR || type(p) == TYPE_STR)
    n = _f_strcat(n, p, err);
  else
    switch (type(n)) {
    case TYPE_REAL:
      switch (type(p)) {
      case TYPE_REAL:
	*err = ERR_NOERR;
	n.value.real += p.value.real;
	break;
      case TYPE_CMP:
	*err = ERR_NOERR;
	__f_reTOc(&n);
	n.value.cmp = cmp_add(n.value.cmp, p.value.cmp);
	break;
      case TYPE_HXS:
	*err = ERR_NOERR;
	__f_rTOb(&n);
	n.value.h = hxs_add(n.value.h, p.value.h);
	break;
      default:
	*err = ERR_BADARGUMENTTYPE;
	break;
      }
      break;

    case TYPE_CMP:
      switch (type(p)) {
      case TYPE_REAL:
	__f_reTOc(&p); /* Fall through */
      case TYPE_CMP:
	*err = ERR_NOERR;
	n.value.cmp = cmp_add(n.value.cmp, p.value.cmp);
	break;
      default:
	*err = ERR_BADARGUMENTTYPE;
	break;
      }
      break;

    case TYPE_HXS:
      switch (type(p)) {
      case TYPE_REAL:
	__f_rTOb(&p); /* Fall through */
      case TYPE_HXS:
	*err = ERR_NOERR;
	n.value.h = hxs_add(n.value.h, p.value.h);
	break;
      default:
	*err = ERR_BADARGUMENTTYPE;
	break;
      }
      break;

    default:
      *err = ERR_BADARGUMENTTYPE;
      break;
    }

  return n;
}


Object _f_and(Object n, Object p, int *err)
{
  /*
   * For real arguments:
   * This function returns 1 if the two arguments are TRUE (non-zero),
   * or 0 otherwise.
   *
   * For hxs arguments:
   * This function returns the bitwise AND function of its arguments.
   *
   * For string arguments:
   * This function does a bitwise AND between the character of the
   * strings, which must be of equal length.
   */
  size_t len;
  Object r;
  
  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
    *err = ERR_NOERR;
    r.type = TYPE_REAL;
    r.value.real = n.value.real && p.value.real;
  } else if (type(n) == TYPE_HXS && type(p) == TYPE_HXS) {
    *err = ERR_NOERR;
    r.type = TYPE_HXS;
    r.value.h = normalize(n.value.h & p.value.h);
  } else if (type(n) == TYPE_STR && type(p) == TYPE_STR) {
    if ((len = strlen(n.value.str)) == strlen(p.value.str)) {
      register int i;

      *err = ERR_NOERR;
      r.type = TYPE_STR;
      r.value.str = (char *) malloc(len + 1);
      for (i = 0; i < len; i++)
	r.value.str[i] = n.value.str[i] & p.value.str[i];
    } else
      *err = ERR_BADARGUMENTVALUE;
  } else
    *err = ERR_BADARGUMENTTYPE;

  return r;
}


Object _f_or(Object n, Object p, int *err)
{
  /*
   * For real arguments:
   * This function returns 1 if at least argument is TRUE (non-zero),
   * or 0 otherwise.
   *
   * For hxs arguments:
   * This function returns the bitwise OR function of its arguments.
   *
   * For string arguments:
   * This function does a bitwise OR between the character of the
   * strings, which must be of equal length.
   */
  size_t len;
  Object r;
  
  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
    *err = ERR_NOERR;
    r.type = TYPE_REAL;
    r.value.real = n.value.real || p.value.real;
  } else if (type(n) == TYPE_HXS && type(p) == TYPE_HXS) {
    *err = ERR_NOERR;
    r.type = TYPE_HXS;
    r.value.h = normalize(n.value.h | p.value.h);
  } else if (type(n) == TYPE_STR && type(p) == TYPE_STR) {
    if ((len = strlen(n.value.str)) == strlen(p.value.str)) {
      register int i;
      
      *err = ERR_NOERR;
      r.type = TYPE_STR;
      r.value.str = (char *) malloc(len + 1);
      for (i = 0; i < len; i++)
	r.value.str[i] = n.value.str[i] | p.value.str[i];
    } else
      *err = ERR_BADARGUMENTVALUE;
  } else
    *err = ERR_BADARGUMENTTYPE;

  return r;
}


Object _f_xor(Object n, Object p, int *err)
{
  /*
   * For real arguments:
   * This function returns 1 if one and only one of the arguments is
   * TRUE (non-zero), or 0 otherwise.
   *
   * For hxs arguments:
   * This function returns the bitwise XOR function of its arguments.
   *
   * For string arguments:
   * This function does a bitwise XOR between the character of the
   * strings, which must be of equal length.
   */
  size_t len;
  Object r;
  
  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
    *err = ERR_NOERR;
    r.type = TYPE_REAL;
    r.value.real = ((n.value.real || p.value.real)
		    && !(n.value.real && p.value.real));
  } else if (type(n) == TYPE_HXS && type(p) == TYPE_HXS) {
    *err = ERR_NOERR;
    r.type = TYPE_HXS;
    r.value.h = normalize(n.value.h ^ p.value.h);
  } else if (type(n) == TYPE_STR && type(p) == TYPE_STR) {
    if ((len = strlen(n.value.str)) == strlen(p.value.str)) {
      register int i;
      
      *err = ERR_NOERR;
      r.type = TYPE_STR;
      r.value.str = (char *) malloc(len + 1);
      for (i = 0; i < len; i++)
	r.value.str[i] = n.value.str[i] ^ p.value.str[i];
    } else
      *err = ERR_BADARGUMENTVALUE;
  } else
    *err = ERR_BADARGUMENTTYPE;
  return r;
}


Object _f_not(Object n, int *err)
{
  /*
   * For real arguments:
   * This function returns 1 if its argument is FALSE (zero) or 0
   * if it is TRUE (non-zero).
   *
   * For hxs arguments:
   * This function returns the bitwise one's complement of its argument.
   */
  Object r;

  if (type(n) == TYPE_REAL) {
    *err = ERR_NOERR;
    r.type = TYPE_REAL;
    r.value.real = !n.value.real;
  } else if (type(n) == TYPE_HXS) {
    *err = ERR_NOERR;
    r.type = TYPE_HXS;
    r.value.h = normalize(~n.value.h);
  } else if (type(n) == TYPE_STR) {
    size_t len = strlen(n.value.str);
    register int i;
    *err = ERR_NOERR;
    r.type = TYPE_STR;
    r.value.str = (char *) malloc(len + 1);

    for (i = 0; i < len; i++)
      r.value.str[i] = ~n.value.str[i];
  } else
    *err = ERR_BADARGUMENTTYPE;

  return r;
}


Object _f_neg(Object n, int *err)
{
  /*
   * For real or complex arguments:
   * This function returns its argument with changed sign.
   *
   * For hxs arguments:
   * This function returns the bitwise two's complement of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    n.value.real = re_chs(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_chs(n.value.cmp);
    break;

  case TYPE_HXS:
    *err = ERR_NOERR;
    n.value.h = normalize(~n.value.h + 1);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}
