/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/

#ifndef __CMP_H
#define __CMP_H 1

typedef struct {
  double re, im;
} Complex;


/*********************************************************************
 * NOTE:                                                             *
 *   In the function definitions in this file, z is a complex number *
 * so that z = x + y*i, and that its magnitude is r and its argument *
 * is theta.                                                         *
 *********************************************************************/


/*********************
 * General functions *
 *********************/
/*
 * Absolute value of a complex number.
 *
 * Definition:
 *    abs(z) = sqrt(x^2 + y^2)
 */
#define cmp_abs(n) (hypot((n).re, (n).im))

/*
 * Argument of a complex number.
 * Definition:
 *    arg(z) = atan(y/x)
 */
#define cmp_arg(n) (atan2((n).im, (n).re))

/*
 * Real part, from polar form.
 *
 * Definition:
 *    re(z) = r*cos(theta)
 */
#define cmp_pTOr_re(n) ((n).re*cos((n).im))

/*
 * Iaginary part, from polar form.
 *
 * Definition:
 *    im(z) = r*sin(theta)
 */
#define cmp_pTOr_im(n) ((n).re*sin((n).im))


Complex cmp_conj(Complex n);
Complex cmp_sign(Complex n);



/************************
 * Arithmetic functions *
 ************************/
/*
 * Quotient of two complex numbers.
 *
 * Definition:
 *    (a + bi)/(c - di) = (a + bi)*(c - di)^-1
 */
#define cmp_div(n, p) (cmp_mul(n, cmp_inv(p)))

Complex cmp_add(Complex n, Complex p);
Complex cmp_sub(Complex n, Complex p);
Complex cmp_mul(Complex n, Complex p);
Complex cmp_inv(Complex n);
Complex cmp_chs(Complex n);


/*****************************************
 * Exponential and logarithmic functions *
 *****************************************/
#define cmp_sq(n) (cmp_mul(n, n))

Complex cmp_exp(Complex n);
Complex cmp_ln(Complex n);
Complex cmp_pow(Complex n, Complex p);
Complex cmp_sqrt(Complex n);
Complex cmp_log(Complex n);
Complex cmp_alog(Complex n);
Complex cmp_cis(double n);


/***************************
 * Trigonometric functions *
 ***************************/
/*
 * Tangent of a complex number.
 *
 * Definition:
 *    tan(z) = sin(z)/cos(z)   (As for real numbers)
 */
#define cmp_tan(n) (cmp_div(cmp_sin(n), cmp_cos(n)))
/*
 * Secant of o complex number.
 *
 * Definition:
 *    sec(z) = 1/cos(z)        (As for real numbers)
 */
#define cmp_sec(n) (cmp_inv(cmp_cos(n)))
/*
 * Co-secant of a complex number.
 *
 * Definition:
 *    csc(z) = 1/sin(z)        (As for real numbers)
 */
#define cmp_csc(n) (cmp_inv(cmp_sin(n)))
/*
 * Co-tangent of a complex number.
 *
 * Definition:
 *    cot(z) = cos(z)/sin(z)   (As for real numbers)
 */
#define cmp_cot(n) (cmp_div(cmp_cos(n), cmp_sin(n)))
/*
 * Arc secant of o complex number.
 *
 * Definition:
 *    asec(z) = acos(1/z)      (As for real numbers)
 */
#define cmp_asec(n) (cmp_acos(cmp_inv(n)))
/*
 * Arc co-secant of a complex number.
 *
 * Definition:
 *    acsc(z) = asin(1/z)      (As for real numbers)
 */
#define cmp_acsc(n) (cmp_asin(cmp_inv(n)))
/*
 * Arc co-tangent of a complex number.
 *
 * Definition:
 *    acot(z) = atan(1/z)      (As for real numbers)
 */
#define cmp_acot(n) (cmp_atan(cmp_inv(n)))


Complex cmp_sin(Complex n);
Complex cmp_cos(Complex n);
Complex cmp_asin(Complex n);
Complex cmp_acos(Complex n);
Complex cmp_atan(Complex n);
Complex cmp_vers(Complex n);
Complex cmp_hav(Complex n);


/************************
 * Hyperbolic functions *
 ************************/
/*
 * Hyperbolic tangent of a complex number.
 *
 * Definition:
 *    tanh(z) = sinh(z) / cosh(z)   (As for real numbers)
 */
#define cmp_tanh(n) (cmp_div(cmp_sinh(n), cmp_cosh(n)))
/*
 * Hyperbolic secant of a complex number.
 *
 * Definition:
 *    sech(z) = 1 / cosh(z)         (As for real numbers)
 */
#define cmp_sech(n) (cmp_inv(cmp_cosh(n)))
/*
 * Hyperbolic co-secant of a complex number.
 *
 * Definition:
 *    csch(z) = 1 / senh(z)         (As for real numbers)
 */
#define cmp_csch(n) (cmp_inv(cmp_sinh(n)))
/*
 * Hyperbolic co-tangent of a complex number.
 *
 * Definition:
 *    coth(z) = cosh(z) / senh(z)   (As for real numbers)
 */
#define cmp_coth(n) (cmp_div(cmp_cosh(n), cmp_sinh(n)))
/*
 * Hyperbolic arc secant of a complex number.
 *
 * Definition:
 *    asech(z) = acosh(1/z)         (As for real numbers)
 */
#define cmp_asech(n) (cmp_acosh(cmp_inv(n)))
/*
 * Hyperbolic arc co-secant of a complex number.
 *
 * Definition:
 *    acsch(z) = asenh(1/z)         (As for real numbers)
 */
#define cmp_acsch(n) (cmp_asinh(cmp_inv(n)))
/*
 * Hyperbolic arc co-tangent of a complex number.
 *
 * Definition:
 *    acoth(z) = atanh(1/z)         (As for real numbers)
 */
#define cmp_acoth(n) (cmp_atanh(cmp_inv(n)))
/*
 * Inverse Gudermannian function of a complex number.
 *
 * Definition:
 *    invgd(z) = ln(sec(z) + tan(z))
 */
#define cmp_invgd(z) (cmp_ln(cmp_add(cmp_sec(z), cmp_tan(z))))

Complex cmp_sinh(Complex n);
Complex cmp_cosh(Complex n);
Complex cmp_asinh(Complex n);
Complex cmp_acosh(Complex n);
Complex cmp_atanh(Complex n);
Complex cmp_gd(Complex n);

#endif /* __CMP_H */
