/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <setjmp.h>
#include <math.h>

#include "cmp.h"
#include "realfunctions.h"
#include "kalc.h"


char *decomp(Object n, int numberFormat, int precision) 
{
  /*
   * This function decompiles the object into its string representation
   * using the given number-format and precision (only used for numbers).
   *
   * This function should not be called directly. Instead the
   * editDecomp and stackDecomp (which currently are just macros) should
   * be called. The difference is that editDecomp will use STD mode, and
   * stackDecomp uses the current number mode.
   *
   * The return value is malloc()'ed, so you should free it after using.
   */

  char *returnStr, *tempStr, *delim;

  switch (type(n)) {
  case TYPE_REAL:
    returnStr = realDecomp(n.value.real, numberFormat, precision);
    break;

  case TYPE_CMP:
    returnStr = cmpDecomp(n.value.cmp, numberFormat, precision);
    break;

  case TYPE_HXS:
    returnStr = hxsDecomp(n.value.h);
    break;

  case TYPE_STR:
  case TYPE_ID:
  case TYPE_UNQUOTEDID:
    returnStr = (char *) malloc(strlen(n.value.str) + 3);
    delim = (type(n) == TYPE_STR ? "\"" : (type(n) == TYPE_ID ? "\'" : ""));
    sprintf(returnStr, "%s%s%s", delim, n.value.str, delim);
    break;

  case TYPE_TAGGED:
    tempStr = stackDecomp(*n.value.tagged.obj);
    returnStr = (char *) malloc(strlen(tempStr)
				+ strlen(n.value.tagged.tag) + 4);
    if (!returnStr) {
      free(tempStr);
      doError("Fatal", ERR_NOTENOUGHMEMORY);
    }
    sprintf(returnStr, ":%s: %s", n.value.tagged.tag, tempStr);
    free(tempStr);
    break;

  case TYPE_FUNC:
    tempStr = getFuncName(n.value.func);
    if (tempStr != NULL) {
      returnStr = strdup(tempStr);
      break;
    }

  case TYPE_PROG:
    returnStr = progDecomp(n.value.comp);
    break;

  case TYPE_NOTANOBJECT:
    returnStr = strdup("Internal Error: NaO found");
    break;
    
  default:
    returnStr = strdup("External");
    break;
  }

  return returnStr;
}


char *stackPrepare(Object n, int level) 
{
  /*
   * This function returns a string that is the representation of a
   * stack level (specified by the int level) with the object n.
   *
   * The return value is malloc()'ed.
   */

  char *returnStr, *objectStr;
  int padding = re_xpon(level) + 1, maxlen = userOptions.width - 2 - padding;

  objectStr = stackDecomp(n);
  if (type(n) == TYPE_TAGGED)  /* Remove the first colon */
    objectStr++;
  
  if ((type(n) == TYPE_STR || type(n) == TYPE_ID
       || type(n) == TYPE_TAGGED || type(n) == TYPE_PROG)
      && needsWrapping(objectStr, maxlen)) {
    char *wrapped = wordWrap(objectStr, userOptions.width, padding + 2);
    returnStr = (char *) malloc(strlen(wrapped) + padding + 3);
    if (!returnStr)
      doError("Fatal", ERR_NOTENOUGHMEMORY);
    sprintf(returnStr, "%d: %s", level, wrapped);
    free(wrapped);
  } else if (type(n) == TYPE_CMP && strlen(objectStr) > maxlen) {
    char *tmp;
    register int i;

    returnStr = (char *) malloc(strlen(objectStr) + 2*padding + 6);

    tmp = strtok(objectStr, ",");
    sprintf(returnStr, "%d: %s,\n", level, tmp);

    tmp = returnStr + strlen(returnStr);
    for (i = 0; i < padding + 2; i++)
      tmp[i] = ' ';
    tmp[i] = '\0';
    
    tmp = strtok(NULL, ",");
    strcat(returnStr, tmp);
  } else {
    /* If the object fits in one line, our job is easy. */
    returnStr = (char *) malloc(userOptions.width + 1);
    sprintf(returnStr, "%d: %*s", level, maxlen, objectStr);
  }

  if (type(n) == TYPE_TAGGED)  /* We must get back or the free() */
    objectStr--;               /* will fail */

  free(objectStr);
  
  return returnStr;
}


int needsWrapping(char *str, int maxlen) 
{
  if (strlen(str) > maxlen)
    return 1;
  if (strchr(str, '\n'))
    return 1;

  return 0;
} 


char *wordWrap(char *str, int maxlen, int padding) 
{
  /*
   * This function word-wrapps str up to maxlen chars. At the beggining
   * of each line (except the first), padding blank chars are added.
   *
   * The result is malloc()'ed.
   */

  int currLen = 0, currLineLen = 0, currMaxLen, wordLen;
  register int i;
  char *buffer;
  char *nextWord;
  char *pos;

  /* Guesstimate the number of characters necessary for the return buffer.
     The formula is somewhat conservative, but this prevents repetitive
     calls to realloc(). */
  currMaxLen = (((strlen(str) / (maxlen - 7)) + 3) * (maxlen + 3));
  buffer = (char *) malloc(currMaxLen);
  if (!buffer)
    doError("fatal", ERR_NOTENOUGHMEMORY);
  
  while (*str) {
    pos = strchr(str, '\n');
    if (pos == NULL)
      wordLen = maxlen + 20;
    else
      wordLen = pos - str;
    if (wordLen < maxlen - currLineLen) {
      memcpy(buffer + currLen, str, wordLen);
      currLen += wordLen;
      *(buffer + currLen++) = '\n';
      for (i = 0; i < padding; i++)
	*(buffer + currLen++) = ' ';
      currLineLen = padding;
      str = pos + 1;
    } else {
      nextWord = getNextWord(str);
      wordLen = strlen(nextWord);
      if (wordLen < maxlen - currLineLen) {
	currLineLen += wordLen;
      } else {
	currLineLen = padding + wordLen;
	*(buffer + currLen++) = '\n';
	for (i = 0; i < padding; i++)
	  *(buffer + currLen++) = ' ';
      }
      memcpy(buffer + currLen, nextWord, wordLen);
      free(nextWord);
      currLen += wordLen;
      str += wordLen;
    }

    if (currLen > currMaxLen - 30) {
      currMaxLen *= 2;
      buffer = (char *) realloc(buffer, currMaxLen);
      if (!buffer)
	doError("Fatal", ERR_NOTENOUGHMEMORY);
    }
  }

  *(buffer + currLen) = '\0';
  
  return buffer;
}


char *getNextWord(char *str) 
{
  /*
   * This function returns everything from the start of the str string
   * to the next blank.
   */
  int currMaxSize = 80;
  char *returnBuffer;
  register int i = 0;

  if ((returnBuffer = (char *) malloc(currMaxSize)) == NULL)
    doError("Fatal", ERR_NOTENOUGHMEMORY);

  while (!isspace(*str) && !ispunct(*str)) {
    returnBuffer[i++] = *str++;
    if (i >= currMaxSize) {
      currMaxSize *= 2;
      if ((returnBuffer = (char *) realloc(returnBuffer, currMaxSize)) == NULL)
	doError("fatal", ERR_NOTENOUGHMEMORY);
    }
  }

  returnBuffer[i++] = *str;
  returnBuffer[i] = '\0';
  return returnBuffer;
}


char *realDecomp(double n, int numberFormat, int decimalPlaces)
{
  /*
   * This function decompiles a real object into its string representation
   * using the given settings for number format and decimal places.
   *
   * The return string is malloc()'ed.
   */

  long xpon = (long) re_xpon(n);
  double mant = re_mant(n);
  char *strObject = (char *) malloc(30 * sizeof(char));

  if (!strObject)
    doError("Fatal", ERR_NOTENOUGHMEMORY);

  if (isinf(n) || isnan(n))
    numberFormat = FMT_STD;

  switch(numberFormat) {
  case FMT_STD:
    sprintf(strObject, "%.15g", n);
    break;

  case FMT_FIX: 
    if (xpon < 16 && labs(xpon) < decimalPlaces)
      sprintf(strObject, "%#.*f",
	      xpon >= 0
	      ? decimalPlaces - (int) labs(xpon) - 1 : decimalPlaces - 1,
	      n);
    else
      sprintf(strObject, "%#.*e", decimalPlaces, n);

    break;

  case FMT_SCI:
    sprintf(strObject, "%#.*e", decimalPlaces, n);
    break;

  case FMT_ENG:
    {
      int modulus = re_mod((double)xpon, 3.0);
      int placesInMantissa;

      placesInMantissa = decimalPlaces - modulus;
      if (placesInMantissa < 0)
	placesInMantissa = 0;

      xpon -= modulus;
      
      sprintf(strObject, "%#.*fe%+03ld",
	      placesInMantissa,
	      mant * re_alog(modulus), xpon);
    }
  }

  return strObject;
}


char *cmpDecomp(Complex n, int numberFormat, int decimalPlaces)
{
  /*
   * This function decompiles a complex number into its string
   * representation using the given settings.
   *
   * The return string is malloc()'ed.
   */

  char *strObject = (char *) malloc(60);
  char *reStr, *imStr;
  char angleDelim[2] = "\0";

  if (userOptions.coordinate == COORD_CYLIN) {
    double r, theta;

    r = cmp_abs(n);
    theta = cmp_arg(n);

    n.re = r;
    n.im = theta;
    if (userOptions.angleMode == ANGLE_DEG)
      n.im *= 180/PI;
    angleDelim[0] = '<';
  }

  reStr = realDecomp(n.re, numberFormat, decimalPlaces);
  imStr = realDecomp(n.im, numberFormat, decimalPlaces);

  sprintf(strObject, "(%s,%s%s)", reStr, angleDelim, imStr);

  free(reStr);
  free(imStr);
  
  return strObject;
}


char *hxsDecomp(hxs h) 
{
  /*
   * This function decompiles the hex string h, using the current base
   * and wordsize.
   * The return is malloc()'ed.
   */

  char *decompiled = (char *) malloc(userOptions.wordSize + 6);
  char baseStr[4];
  register int i = 2;
  register int r;

  h = normalize(h);

  strcpy(decompiled, "# ");

  if (h) {
    while (h) {
      r = h % userOptions.base;
      decompiled[i++] = forDigit(r, userOptions.base);
      h /= userOptions.base;
    }
    decompiled[i] = '\0';
    reverseStr(decompiled, 2);
  } else
    strcat(decompiled, "0");
  

  switch (userOptions.base) {
  case 2:
    strcat(decompiled, "b");
    break;

  case 8:
    strcat(decompiled, "o");
    break;

  case 10:
    strcat(decompiled, "d");
    break;

  case 16:
    strcat(decompiled, "h");
    break;

  default:
    sprintf(baseStr, "@%u", userOptions.base);
    strcat(decompiled, baseStr);
    break;
  }

  return decompiled;
}


char forDigit(int d, int base) 
{
  /*
   * This function returns the character that represents the algarism d
   * in base base.
   */

  char c;

  if (d >= base)
    return '\0';
  
  if (d < 10)
    c = (char) ('0' + d);
  else
    c = (char) ('A' + d - 10);

  return c;
}
  

char *reverseStr(char *str, int start) 
{
  /*
   * This function reverses the given string, starting at the given
   * position. str is returned.
   */

  register int i, j;
  char temp;

  for (i = start, j = strlen(str) - 1; i < j; i++, j--) {
    temp = str[i];
    str[i] = str[j];
    str[j] = temp;
  }

  return str;
}


char *
progDecomp(Composite *comp)
{
  /*
   * This function decompiles a program, which starts with the given
   * object. The return is malloc()'ed.
   */

  char *returnStr, *objStr;
  int currSize = 0, currCap = 250;

  returnStr = (char *) malloc(currCap);
  strcpy(returnStr, "<<");

  while (comp) {
    objStr = stackDecomp(*comp->obj);
    if (currSize + 1 + strlen(objStr) >= currCap - 1) {
      char *newP = (char *) realloc(returnStr, currCap *= 2);
      if (!newP)
	doError("Fatal", ERR_NOTENOUGHMEMORY);
      returnStr = newP;
    }
    strcat(returnStr, " ");
    strcat(returnStr, objStr);
    currSize += strlen(objStr) + 1;
    free(objStr);
    comp = comp->next;
  }

  if (currSize + 3 >= currCap - 1) {
    char *newP = (char *) realloc(returnStr, currCap += 5);
    if (!newP)
      doError("Fatal", ERR_NOTENOUGHMEMORY);
    returnStr = newP;
  }

  strcat(returnStr, " >>");

  return returnStr;
}
