/* procfs.c -  crea un "fichero" en /proc 
 * Copyright (C) 1998-1999 by Ori Pomerantz
 */


/* Los ficheros de cabeceras necesarios */

/* Estndar en los mdulos del ncleo */
#include <linux/kernel.h>   /* Estamos haciendo trabajo del ncleo */
#include <linux/module.h>   /* Especficamente, un mdulo */

/* Distribuido con CONFIG_MODVERSIONS */
#if CONFIG_MODVERSIONS==1
#define MODVERSIONS
#include <linux/modversions.h>
#endif        


/* Necesario porque usamos el sistema de ficheros proc */
#include <linux/proc_fs.h>



/* En 2.2.3 /usr/include/linux/version.h se incluye
 * una macro para eso, pero 2.0.35 no lo hace - por lo
 * tanto lo aado aqu si es necesario */
#ifndef KERNEL_VERSION
#define KERNEL_VERSION(a,b,c) ((a)*65536+(b)*256+(c))
#endif



/* Ponemos datos en el fichero del sistema de fichero proc.

   Argumentos
   ==========
   1. El buffer donde los datos van a ser insertados, si
      decides usarlo.
   2. Un puntero a un puntero de caracteres. Esto es til si
      no quieres usar el buffer asignado por el ncleo.
   3. La posicin actual en el fichero.
   4. El tamao del buffer en el primer argumento.
   5. Cero (para uso futuro?).

   Uso y Valores de Retorno
   ========================
   Si utilizas tu propio buffer, como yo, pon su situacin
   en el segundo argumento y retorna el nmero de bytes
   usados en el buffer.

   Un valor de retorno de cero significa que actualmente no 
   tienes ms informacin (final del fichero). Un valor negativo
   es una condicin de error.

   Para Ms Informacin
   ====================
   La forma en la que descubr qu hacer con esta funcin 
   no fue leyendo documentacin, sino que fue leyendo el cdigo
   que lo utiliza. Justamente mir para ver para qu usa el campo
   get_info de la struct proc_dir_entry (Us una combinacin 
   de find y grep, por si ests interesado), y vi que se usa en
   <directorio del cdigo del ncleo>/fs/proc/array.c.

   Si algo no es conocido sobre el ncleo, esta es la forma
   habitual de hacerlo. En Linux tenemos la gran ventaja
   de tener el cdigo fuente del ncleo gratis - salo.
 */
int procfile_read(char *buffer, 
		  char **buffer_location, 
		  off_t offset, 
		  int buffer_length, 
		  int zero)
{
  int len;  /* El nmero de bytes usados realmente */

  /* Esto es static, por lo tanto permanecer en 
   * memoria cuando abandonemos esta funcin */
  static char my_buffer[80];  

  static int count = 1;

  /* Damos toda nuestra informacin de una vez, por lo tanto
   * si el usuario nos pregunta si tenemos ms informacin
   * la respuesta debera de ser no.
   *
   * Esto es importante porque la funcin estndar de lectura
   * de la librera debera continuar emitiendo la
   * llamada al sistema read hasta que el ncleo responda
   * que no hay ms informacin, o hasta que el buffer est
   * lleno.
   */
  if (offset > 0)
    return 0;

  /* Rellenamos el buffer y cogemos su longitud */
  len = sprintf(my_buffer, 
    "Para la vez %d%s, vete!\n", count,
    (count % 100 > 10 && count % 100 < 14) ? "th" : 
      (count % 10 == 1) ? "st" :
        (count % 10 == 2) ? "nd" :
          (count % 10 == 3) ? "rd" : "th" );
  count++;

  /* Dice a la funcin que llamamos dnde est el buffer */
  *buffer_location = my_buffer;

  /* Devolvemos la longitud */
  return len;
}


struct proc_dir_entry Our_Proc_File = 
  {
    0, /* Nmero de Inodo - ignralo, ser rellenado por 
        * proc_register[_dynamic] */
    4, /* Longitud del nombre del fichero */
    "test", /* El nombre del fichero */
    S_IFREG | S_IRUGO, /* Modo del fichero - este es un fichero 
                        * regular que puede ser ledo por su 
                        * dueo, por su grupo, y por todo el mundo */ 
    1,	/* Nmero de enlaces (directorios donde el 
         * fichero est referenciado) */
    0, 0,  /* El uid y gid para el fichero - se lo damos 
            * a root */
    80, /* El tamao del fichero devuelto por ls. */
    NULL, /* funciones que pueden ser realizadas en el inodo 
           * (enlazado, borrado, etc.) - no soportamos 
           * ninguna. */
    procfile_read, /* La funcin read para este fichero, 
                    * la funcin llamada cuando alguien 
                    * intenta leer algo de el. */
    NULL /* Podemos tener aqu un funcin que rellene el
          * inodo del fichero, para habilitarnos el jugar
          * con los permisos, dueo, etc. */ 
  }; 





/* Inicializa el mdulo - registra el fichero proc */
int init_module()
{
  /* Tiene xito si proc_register[_dynamic] tiene xito,
   * falla en otro caso. */
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,2,0)
  /* En la versin 2.2, proc_register asigna un nmero
   * de inodo automticamente si hay cero en la estructura,
   * por lo tanto no necesitamos nada ms para 
   * proc_register_dynamic
   */


  return proc_register(&proc_root, &Our_Proc_File);
#else
  return proc_register_dynamic(&proc_root, &Our_Proc_File);
#endif
 
  /* proc_root es el directorio raiz para el sistema de ficheros
   * proc (/proc). Aqu es dnde queremos que nuestro fichero est
   * localizado. 
   */
}


/* Limpieza - liberamos nuestro fichero de /proc */
void cleanup_module()
{
  proc_unregister(&proc_root, Our_Proc_File.low_ino);
}  



