#ifdef ALLOW_CHANGE_PASSWORD
/* fork a child process to exec passwd and write to its
* tty to change a users password. This is running as the
* user who is attempting to change the password.
*/

/* 
 * This code was copied/borrowed and stolen from various sources.
 * The primary source was the poppasswd.c from the authors of POPMail. This software
 * was included as a client to change passwords using the 'passwd' program
 * on the remote machine.
 *
 * It has been dissected and recreated in a form that allows passwd changing for
 * a valid user (user is already validated BEFORE this routine is run). It compares
 * the user name to those in the LOCAL password file and, if they are found, it 
 * runs 'passwd.'  If not, however, it assumes, since the person has already been 
 * validated, that they are an NIS user and it uses 'yppasswd.' 
 *
 * This routine is called by set_user_password() in password.c only if ALLOW_PASSWORD_CHANGE
 * is defined in the compiler directives located in the Makefile.
 *
 *
 * This code has been hacked by Bob Nance (nance@niehs.nih.gov) and Evan Patterson
 * (patters2@niehs.nih.gov) at the National Institute of Environmental Health Sciences
 * and rights to modify, distribute or incorporate this change to the CAP suite or
 * using it for any other reason are granted, so long as this disclaimer is left intact.
 */

#include "includes.h"
#include "loadparm.h"

#define MINPASSWDLENGTH 5
#define BUFSIZE 512

extern int DEBUGLEVEL;

static char *P1[] =
{
  "*old*password*",
  ""
};

static char *P2[] =
{
  "*new*password*",
  ""
};

static char *P3[] =
{
  "*new*password*",
  ""
};

static char *P4[] =
{
  "\n",
  "*entry changed*",
  "*passwd changed*",
  "*password changed*",
  ""
};

static int findpty (char **slave);
static int talktochild(int master,char *oldpass,char *newpass,char *emess);
static int dochild(int master,char *slavedev,char *name,char *oldpass,char *newpass);
static int expect(int master,char **expected,char *buf);
static void writestring (int fd,char *s);

BOOL chgpasswd(name,oldpass,newpass)
     char *name, *oldpass, *newpass;
{
  char emess[255];
  char *slavedev;
  struct passwd *getpwnam();
  int master;
  pid_t pid, wpid;
  int wstat;
  BOOL chstat;
  int putpwent();
  
  strlower(name); 
  DEBUG(3,("Password change for user: %s\n",name));
#if DEBUG_PASSWORD
  DEBUG(100,("Passwords: old=%s new=%s\n",oldpass,newpass)); 
#endif
  
  /* Take the passed information and test it for minimum criteria */
  /* Minimum password length */
  if (strlen(newpass) < MINPASSWDLENGTH) /* too short, must be at least MINPASSWDLENGTH */ 
    {
      DEBUG(2,("Password Change: %s, New password is shorter than MINPASSWDLENGTH\n",name));
      return (False);		/* inform the user */
    }
  
  /* Password is same as old password */
  if (strncmp(oldpass,newpass,8) == 0) /* don't allow same password */
    {
      DEBUG(2,("Password Change: %s, New password is same as old\n",name)); /* log the attempt */
      return (False);		/* inform the user */
    }

  /* That done, let's attempt to actually change the password */
  /* allocate a pseudo-terminal device */
  if ((master = findpty (&slavedev)) < 0)
    {
      DEBUG(3,("Cannot Allocate pty for password change: %s",name));
      return(False);
    }

  if ((pid = fork()) < 0)
    {
      DEBUG(3,("Cannot fork() child for password change: %s",name));
      return(False);
    }

  /* we now have a pty */
  if (pid > 0){			/* This is the parent process */
    if ((chstat = talktochild (master, oldpass, newpass, emess)) == False)
      {
	DEBUG(3,("Child failed to change password: %s\n",name));
	kill(pid, SIGKILL); /* be sure to end this process */
	return(False);
      }
    if ((wpid = waitpid (pid, &wstat, 0)) < 0) {
      DEBUG(3,("The process is no longer waiting!\n\n"));
      return(False);
    }
    if (pid != wpid) {
      DEBUG(3,("We were waiting for the wrong process ID\n"));	
      return(False);
    }
    if (WIFEXITED (wstat) == 0) {
      DEBUG(3,("The process exited while we were waiting\n"));
      return(False);
    }
    if (WEXITSTATUS (wstat) != 0) {
      DEBUG(3,("The status of the process exiting was %d\n", wstat));
      return(False);
    }

  }
  else				/* CHILD */
    {
      struct passwd *pass = Get_Pwnam(name);
      int gid = pass->pw_gid;
      int uid = pass->pw_uid;

      /* make us completely into the right uid */
#ifdef USE_SETRES
      setresgid(0,0,0);
      setresuid(0,0,0);
      setresgid(gid,gid,gid);
      setresuid(uid,uid,uid);      
#else      
      setuid(0);
      seteuid(0);
      setgid(gid);
      setegid(gid);
      setuid(uid);
      seteuid(uid);
#endif

      /* make sure it doesn't freeze */
      alarm(10);

      DEBUG(3,("Dochild for user %s (uid=%d,gid=%d)\n",name,getuid(),getgid()));
      chstat = dochild (master, slavedev, name, oldpass, newpass);
    }
  DEBUG(3,("Password change %ssuccessful for user %s\n", (chstat?"":"un"), name));
  return (chstat);
}

static int dochild (master, slavedev, name, oldpass, newpass)
     int master;
     char *slavedev, *name, *oldpass, *newpass;
{
  int slave;
  struct termios stermios;
  string passwordprogram;
  char *shortname="passwd";
	
  strcpy(passwordprogram,lp_passwd_program());

  string_sub(passwordprogram,"%u",name);

  /* Start new session - gets rid of controlling terminal. */
  if (setsid() < 0) {
    DEBUG(3,("Weirdness, couldn't let go of controlling terminal\n"));
    return(False);
  }

  /* Open slave pty and acquire as new controlling terminal. */
  if ((slave = open(slavedev, O_RDWR)) < 0) {
    DEBUG(3,("More weirdness, could not read/write to new pty\n"));
    return(False);
  }
  if (ioctl(slave,TIOCSCTTY,0) <0) {
     DEBUG(3,("Error in ioctl call for slave pty\n"));
     /* return(False); */
  }

  /* Close master. */
  close(master);

  /* Make slave stdin/out/err of child. */

  if (dup2(slave, STDIN_FILENO) != STDIN_FILENO) {
    DEBUG(3,("Could not re-direct stdin\n"));
    return(False);
  }
  if (dup2(slave, STDOUT_FILENO) != STDOUT_FILENO) {
    DEBUG(3,("Could not re-direct stdout\n"));
    return(False);
  }
  if (dup2(slave, STDERR_FILENO) != STDERR_FILENO) {
    DEBUG(3,("Could not re-direct stderr\n"));
    return(False);
  }
  if (slave > 2) close(slave);

  /* Set proper terminal attributes - no echo, canonical input processing,
     no map NL to CR/NL on output. */

  if (tcgetattr(0, &stermios) < 0) {
    DEBUG(3,("could not read default terminal attributes on pty\n"));
    return(False);
  }
  stermios.c_lflag &= ~(ECHO | ECHOE | ECHOK | ECHONL);
  stermios.c_lflag |= ICANON;
  stermios.c_oflag &= ~(ONLCR);
  if (tcsetattr(0, TCSANOW, &stermios) < 0) {
    DEBUG(3,("could not set attributes of pty\n"));
    return(False);
  }

  /* execl() password-change application */
  if (execl("/bin/sh","sh","-c",passwordprogram,NULL) < 0) {
    DEBUG(3,("Bad status returned from %s\n",passwordprogram));
    return(False);
  }
  return(True);
}

static int talktochild (master, oldpass, newpass, emess)
     int master;
     char *oldpass, *newpass, *emess;
{
  char buf[BUFSIZE];
  char pswd[BUFSIZE+1];
  
  *buf = 0;
  *pswd = 0;
  *emess = 0;
  sleep(1);
  
  if (!expect(master, P1, buf)) /* Compare what is on the pty with what is stored in P1 */
    {
      DEBUG(3,("Response 1 not correct: %s\n",buf));
      return (False);
    }

  sprintf(pswd, "%s\n", oldpass);
  writestring(master, pswd);

  if (!expect(master, P2, buf)) 
    {
      DEBUG(3,("Response 2 not correct: %s\n",buf));
      return(False);
    }

  sprintf(pswd,"%s\n", newpass);
  writestring(master, pswd);

  if (!expect(master, P3, buf)) 
    {
      DEBUG(3,("Response 3 not correct: %s\n",buf));
      return(False);
    }

  sprintf(pswd,"%s\n", newpass);
  writestring(master, pswd);

  if (!expect(master, P4, buf)) 
    {
      DEBUG(3,("Response 4 not correct: %s\n",buf));
      return(False);
    }

  return (True);
}

static int expect(master, expected, buf)
     int master;
     char **expected;
     char *buf;
{
  int n, m;
  char **s;
 
  n = 0;
  buf[0] = 0;
  while (1) {
    if (n >= BUFSIZE-1) {
      return False;
    }

    /* allow 2 seconds for some output to appear */
    m = read_with_timeout(master, buf+n, 1, BUFSIZE-1-n, 2000, True);
    if (m < 0) return False;

    n += m;
    buf[n] = 0;

    for (s = expected; **s != 0; s++) {
      {
	string s1,s2;
	strcpy(s1,buf);
	strcpy(s2,*s);
	if (do_match(s1, s2, False))
	  return(True);
      }
    }
  }
}


static void writestring (int fd,char *s)
{
  int l;
  
  l = strlen (s);
  write (fd, s, l);
}

static int findpty (char **slave)
{
  int master;
  static char line[12] = "/dev/ptyXX";
  void *dirp;
  char *dpname;
  
  dirp = OpenDir("/dev");
  if (!dirp) return(-1);
  while ((dpname = ReadDirName(dirp)) != NULL) {
    if (strncmp(dpname, "pty", 3) == 0 && strlen(dpname) == 5) {
      line[8] = dpname[3];
      line[9] = dpname[4];
      if ((master = open(line, O_RDWR)) >= 0) {
	line[5] = 't';
	*slave = line;
	CloseDir(dirp);
	return (master);
      }
    }
  }
  CloseDir(dirp);
  return (-1);
}
#else
int dummy_chgpasswd(void)
{
  return(0);
}
#endif
