/* ====================================================================
 * Copyright (c) 1998 Ralf S. Engelschall. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by 
 *     Ralf S. Engelschall <rse@engelschall.com> for use in the
 *     mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * 4. The names "mod_ssl" must not be used to endorse or promote
 *    products derived from this software without prior written
 *    permission. For written permission, please contact
 *    rse@engelschall.com.
 *
 * 5. Products derived from this software may not be called "mod_ssl"
 *    nor may "mod_ssl" appear in their names without prior
 *    written permission of Ralf S. Engelschall.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by 
 *     Ralf S. Engelschall <rse@engelschall.com> for use in the
 *     mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * THIS SOFTWARE IS PROVIDED BY RALF S. ENGELSCHALL ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RALF S. ENGELSCHALL OR
 * HIS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 * This product includes software developed by Ben Laurie
 * for use in the Apache-SSL HTTP server project.
 */

/* ====================================================================
 * Copyright (c) 1995, 1996, 1997, 1998 Ben Laurie.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by Ben Laurie
 *    for use in the Apache-SSL HTTP server project."
 *
 * 4. The name "Apache-SSL Server" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission.
 *
 * 5. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Ben Laurie
 *    for use in the Apache-SSL HTTP server project."
 *
 * THIS SOFTWARE IS PROVIDED BY BEN LAURIE ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL BEN LAURIE OR
 * HIS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 */


/* OS headers */
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/un.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>

/* Apache headers */
#include "httpd.h"

/* mod_ssl headers */
#include "ssl_gcache.h"

typedef union {
    struct sockaddr_in in;
    struct sockaddr_un un;
} SockAddr;

#if defined(NEED_STRERROR)
extern char *const sys_errlist[];
char *strerror(int err)
{
    char *p;

    p = sys_errlist[err];
    return (p);
}
#endif

static void ssl_gcache_HandleAdd(int nFD)
{
    UCHAR *aucKey;
    int nKey;
    Cache *p;

    aucKey = ssl_gcache_ReadThing(nFD, &nKey);
    if (aucKey == NULL)
        return;
    if ((p = ssl_gcache_LocalFind(aucKey, nKey)) != NULL) {
        free(p->aucData);
        free(aucKey);
    }
    else {
        p = ssl_gcache_LocalAdd(aucKey, nKey, 0);
        if (p == NULL)
            return;
    }
    p->aucData = ssl_gcache_ReadThing(nFD, &p->nData);
    read(nFD, &p->tExpiresAt, sizeof(p->tExpiresAt));
}

static void ssl_gcache_HandleGet(int nFD)
{
    UCHAR *aucKey;
    int nKey;
    Cache *p;

    aucKey = ssl_gcache_ReadThing(nFD, &nKey);
    if (aucKey == NULL)
        return;
    if ((p = ssl_gcache_LocalFind(aucKey, nKey)) == NULL) { 
        ssl_gcache_WriteThing(nFD, NULL, 0);
    }
    else {
        ssl_gcache_WriteThing(nFD, p->aucData, p->nData);
        write(nFD, &p->tExpiresAt, sizeof(p->tExpiresAt));
    }
    return;
}

static void ssl_gcache_HandleClient(int nFD)
{
    static char c;

    if (read(nFD, &c, 1) != 1)
        return;
    switch (c) {
        case ADD_TO_CACHE:
            ssl_gcache_HandleAdd(nFD);
            break;
        case GET_FROM_CACHE:
            ssl_gcache_HandleGet(nFD);
            break;
        default:
            fprintf(stderr, "[%s] ssl_gcache: invalid cache request\n", ssl_gcache_get_time());
            break;
    }
    return;
}

static void signal_handler(int sig)
{
    fprintf(stderr, "[%s] ssl_gcache: terminated\n", ssl_gcache_get_time());
    sleep(1);
    exit(0);
}

int main(int argc, char **argv)
{
    const char *szPort;
    int nUserID;
    int nPort;
    int nSocket;
    SockAddr saServer;
    int one = 1;
    int nSize;
    int nFD;
    int nLen;
    SockAddr saClient;

    if (argc != 3) {
        fprintf(stderr, "%s <user id> <port_or_file>\n", argv[0]);
        exit(1);
    }

    fprintf(stderr, "[%s] ssl_gcache: started\n", ssl_gcache_get_time());

    signal(SIGINT,  signal_handler);
    signal(SIGQUIT, signal_handler);
    signal(SIGTERM, signal_handler);
#ifdef SIGHUP
    signal(SIGHUP,  SIG_IGN);
#endif
    signal(SIGPIPE, SIG_IGN);

    nUserID = atoi(argv[1]);
    szPort  = argv[2];
    nPort   = atoi(szPort);

    if (nPort)
        nSocket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    else
        nSocket = socket(AF_UNIX, SOCK_STREAM, 0);
    if (nSocket < 0) {
        fprintf(stderr, "[%s] ssl_gcache: socket(): %s\n", 
                ssl_gcache_get_time(), strerror(errno));
        exit(2);
    }

    if (nPort
        && setsockopt(nSocket, SOL_SOCKET, SO_REUSEADDR, (char *) &one, sizeof one) < 0) {
        fprintf(stderr, "[%s] ssl_gcache: setsockopt(): %s\n", 
                ssl_gcache_get_time(), strerror(errno));
        exit(7);
    }

    memset(&saServer, 0, sizeof saServer);
    if (nPort) {
        saServer.in.sin_family = AF_INET;
        saServer.in.sin_port = htons(nPort);
        nSize = sizeof saServer.in;
    }
    else {
        if (strlen(szPort) >= sizeof(saServer.un.sun_path))
            exit(6);
        saServer.un.sun_family = AF_UNIX;
        unlink(szPort);
        strcpy(saServer.un.sun_path, szPort);
        nSize = sizeof(saServer.un);
    }

    if (bind(nSocket, (struct sockaddr *)&saServer, nSize) < 0) {
        fprintf(stderr, "[%s] ssl_gcache: bind(): %s\n", 
                ssl_gcache_get_time(), strerror(errno));
        exit(3);
    }

    if (listen(nSocket, 512) < 0) {
        fprintf(stderr, "[%s] ssl_gcache: listen(): %s\n", 
                ssl_gcache_get_time(), strerror(errno));
        exit(4);
    }

    if (!nPort) {
        chmod(szPort, 0700);
        chown(szPort, nUserID, -1);
    }

    if (!geteuid() && setuid(nUserID) == -1) {
        fprintf(stderr, "[%s] ssl_gcache: setuid(%d): %s\n", 
                ssl_gcache_get_time(), nUserID, strerror(errno));
        exit(5);
    }

    for (;;) {
        nLen = nSize;
        nFD = accept(nSocket, (struct sockaddr *) &saClient, &nLen);
        if (nFD < 0) {
            fprintf(stderr, "[%s] ssl_gcache: accept(): %s\n", 
                    ssl_gcache_get_time(), strerror(errno));
            continue;
        }
        if (!nPort || ntohl(saClient.in.sin_addr.s_addr) == 0x7f000001)
            ssl_gcache_HandleClient(nFD);
        else
            fprintf(stderr,
                    "[%s] ssl_gcache: unexpected connect from %s - ignored\n",
                    ssl_gcache_get_time(), inet_ntoa(saClient.in.sin_addr));
        close(nFD);
    }
    return (0);
}

