/* ====================================================================
 * Copyright (c) 1998 Ralf S. Engelschall. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by 
 *     Ralf S. Engelschall <rse@engelschall.com> for use in the
 *     mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * 4. The names "mod_ssl" must not be used to endorse or promote
 *    products derived from this software without prior written
 *    permission. For written permission, please contact
 *    rse@engelschall.com.
 *
 * 5. Products derived from this software may not be called "mod_ssl"
 *    nor may "mod_ssl" appear in their names without prior
 *    written permission of Ralf S. Engelschall.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by 
 *     Ralf S. Engelschall <rse@engelschall.com> for use in the
 *     mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * THIS SOFTWARE IS PROVIDED BY RALF S. ENGELSCHALL ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RALF S. ENGELSCHALL OR
 * HIS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 */


#include "mod_ssl.h"


/*  _________________________________________________________________
**
**  Backward Compatibility
**  _________________________________________________________________
*/

/*
 * The mapping of obsolete directives to official ones...
 */

static struct {
    enum { CRM_END, CRM_TXTSUB, CRM_TXTMSG, CRM_PATFCT } nType;
    char *cpMatch;
    void *vpSubst;
} ssl_cmd_rewrite_map[] = {
    /* 
     * Apache-SSL 1.x & mod_ssl 2.0.x backward compatibility
     */
    { CRM_TXTSUB, "SSLEnable",                   "SSLEngine on"                },
    { CRM_TXTSUB, "SSLDisable",                  "SSLEngine off"               },
    { CRM_TXTSUB, "SSLRequiredCiphers",          "SSLCipherSuite"              },
    { CRM_TXTMSG, "SSLFakeBasicAuth",            "SSLOptions +FakeBasicAuth"   },
    { CRM_TXTMSG, "SSLExportClientCertificates", "SSLOptions +ExportCertData"  },
    { CRM_TXTMSG, "SSLCacheServerPath",          "Use SSLSessionCache instead" },
    { CRM_TXTMSG, "SSLCacheServerPort",          "Use SSLSessionCache instead" },
    { CRM_TXTMSG, "SSLCacheServerRunDir",        "Not needed inside mod_ssl"   },
    /* 
     * Sioux 1.x backward compatibility
     */
    { CRM_TXTSUB, "SSL_CertFile",           "SSLCertificateFile"               },
    { CRM_TXTSUB, "SSL_KeyFile",            "SSLCertificateKeyFile"            },
    { CRM_TXTSUB, "SSL_CipherSuite",        "SSLCipherSuite"                   },
    { CRM_TXTSUB, "SSL_X509VerifyDir",      "SSLCACertificatePath"             },
    { CRM_TXTSUB, "SSL_Log",                "SSLLogFile"                       },
    { CRM_TXTSUB, "SSL_Connect"             "SSLEngine"                        },
    { CRM_TXTMSG, "SSL_ClientAuth"          "SSLVerifyClient"                  },
    { CRM_TXTMSG, "SSL_X509VerifyDepth"     "SSLVerifyDepth"                   },
    { CRM_TXTMSG, "SSL_FetchKeyPhraseFrom", "Use SSLPassPhraseDialog instead"  },
    { CRM_TXTMSG, "SSL_SessionDir",         "Use SSLSessionCache instead"      },
    { CRM_TXTMSG, "SSL_Require",            "Use SSLRequire instead, but be careful about syntax" },
    { CRM_TXTMSG, "SSL_CertFileType",       "Not supported"                    },
    { CRM_TXTMSG, "SSL_KeyFileType",        "Not supported"                    },
    { CRM_TXTMSG, "SSL_X509VerifyPolicy",   "Not supported"                    },
    { CRM_TXTMSG, "SSL_LogX509Attributes",  "Not supported"                    },
    /*
     * End of map.
     */
    { CRM_END, NULL, NULL }
};

char *ssl_compat_directive(server_rec *s, pool *p, const char *oline)
{
    int i;
    char *line;
    char *cp;

    line = NULL;
    for (i = 0; ssl_cmd_rewrite_map[i].nType != CRM_END; i++) {
        if (ssl_cmd_rewrite_map[i].nType == CRM_TXTSUB) {
            if ((cp = strstr(oline, ssl_cmd_rewrite_map[i].cpMatch)) != NULL) {
                line = ssl_util_ptxtsub(p, oline, ssl_cmd_rewrite_map[i].cpMatch,
                                        (char *)(ssl_cmd_rewrite_map[i].vpSubst));
                break;
            }
        }
        if (ssl_cmd_rewrite_map[i].nType == CRM_TXTMSG) {
            if ((cp = strstr(oline, ssl_cmd_rewrite_map[i].cpMatch)) != NULL) {
                ap_log_error(APLOG_MARK, APLOG_WARNING|APLOG_NOERRNO, s, 
                             "mod_ssl: Obsolete/unsupported directive: %s",
                             oline);
                ap_log_error(APLOG_MARK, APLOG_WARNING|APLOG_NOERRNO, s, 
                             "mod_ssl: Hint: %s",
                             (char *)(ssl_cmd_rewrite_map[i].vpSubst));
                line = "";
                break;
            }
        }
        else if (ssl_cmd_rewrite_map[i].nType == CRM_PATFCT) {
            if (ap_fnmatch(ssl_cmd_rewrite_map[i].cpMatch, oline, 0) == 0) {
                line = ((char *(*)(pool *, const char *))
                        (ssl_cmd_rewrite_map[i].vpSubst))(p, oline);
                break;
            }
        }
    }
    return line;
}

/*
 * The mapping of obsolete environment variables to official ones...
 */

static struct {
    char *cpOld;
    char *cpNew;
} ssl_env_var_map[] = {
    /* 
     * Apache-SSL 1.x & mod_ssl 2.0.x backward compatibility
     */
    { "HTTPS",                 "HTTPS" },
    /*
     * End of map.
     */
    { NULL, NULL }
};

void ssl_compat_variables(request_rec *r)
{
    char *cpOld;
    char *cpNew;
    char *cpVal;
    int i;

    for (i = 0; ssl_env_var_map[i].cpOld != NULL; i++) {
        cpOld = ssl_env_var_map[i].cpOld;
        cpNew = ssl_env_var_map[i].cpNew;
        cpVal = ssl_var_lookup(r->pool, r->server, r->connection, r, cpNew);
        if (!strIsEmpty(cpVal))
            ap_table_set(r->subprocess_env, cpOld, cpVal);
    }
    return;
}

