/******************************************************************************
 jUNIXUtil.cc

	Routines specific to UNIX.

	Copyright  1997-98 John Lindal. All rights reserved.

 ******************************************************************************/

#include <jUNIXUtil.h>
#include <JString.h>
#include <JStdError.h>
#include <jErrno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <grp.h>
#include <pwd.h>
#include <jAssert.h>

/******************************************************************************
 JGetHostName

	Returns the name of the machine.

 ******************************************************************************/

JString
JGetHostName()
{
	JCharacter hostName[255];
	const int result = gethostname(hostName, 255);
	assert( result == 0 );
	return JString(hostName);
}

/******************************************************************************
 JGetUserName

	Returns the name of the specified user.

 ******************************************************************************/

JString
JGetUserName()
{
	return JGetUserName(getuid());
}

JString
JGetUserName
	(
	const uid_t uid
	)
{
	struct passwd* pwbuf = getpwuid(uid);
	if (pwbuf != NULL)
		{
		return JString(pwbuf->pw_name);
		}
	else
		{
		return JString(uid, 0, JString::kForceNoExponent);
		}
}

/******************************************************************************
 JGetUserRealWorldName

	Returns the "real world" name of the specified user.

 ******************************************************************************/

JString
JGetUserRealWorldName()
{
	return JGetUserRealWorldName(getuid());
}

JString
JGetUserRealWorldName
	(
	const uid_t uid
	)
{
	struct passwd* pwbuf = getpwuid(uid);
	if (pwbuf != NULL)
		{
		return JString(pwbuf->pw_gecos);
		}
	else
		{
		return JGetUserName(uid);
		}
}

/******************************************************************************
 JGetGroupName

	Returns the name of the specified user.

 ******************************************************************************/

JString
JGetGroupName()
{
	return JGetGroupName(getgid());
}

JString
JGetGroupName
	(
	const gid_t gid
	)
{
	struct group* grbuf = getgrgid(gid);
	if (grbuf != NULL)
		{
		return JString(grbuf->gr_name);
		}
	else
		{
		return JString(gid, 0, JString::kForceNoExponent);
		}
}

/******************************************************************************
 JUNIXSocketExists

	Returns kTrue if pathName is the pathname of an existing unix socket,
	kFalse otherwise.

 *****************************************************************************/

JBoolean
JUNIXSocketExists
	(
	const JCharacter* pathName
	)
{
	struct stat info;

	return JConvertToBoolean(
			lstat(pathName, &info) == 0 &&
			stat( pathName, &info) == 0 &&
#ifdef sun
			S_ISFIFO(info.st_mode)
#else
			S_ISSOCK(info.st_mode)
#endif
			);
}

/******************************************************************************
 JCreatePipe

	Opens a pair of file descriptors.  fd[0] is for reading.  fd[1] is for
	writing.

 *****************************************************************************/

JError
JCreatePipe
	(
	int fd[2]
	)
{
	jclear_errno();
	if (ACE_OS::pipe(fd) == 0)
		{
		return JNoError();
		}

	const int err = jerrno();
	if (err == EFAULT)
		{
		return JSegFault();
		}
	else if (err == EMFILE || err == ENFILE)
		{
		return JTooManyDescriptorsOpen();
		}
	else
		{
		return JUnexpectedError(err);
		}
}
