/****************************************************************************/
/*   xmObject.C                                                             */
/****************************************************************************/
/*                                                                          */
/*   Copyright (c) 1992, 1993, 1994 Bernhard Strassl                        */
/*       Vienna User Interface Group                                        */
/*       Institute for Applied Computer Science and Information Systems     */
/*       University of Vienna, Austria                                      */
/*                                                                          */
/*   See file COPYRIGHT in this directory for details. If this file is      */
/*   missing please mail to xmplus@ani.univie.ac.at                         */
/****************************************************************************/

// #include <stdlib.h>
// #include <unistd.h>
// #include <sys/wait.h>

#include <stdarg.h>
#include <stream.h>
#include <strstream.h>

#include "xmObject.h"

extern "C"
{
void XtMoveWidget(Widget, int, int);
void XtResizeWidget(Widget, int, int, int);
void XtConfigureWidget(Widget, int, int, int, int, int);
void exit(int);
}

#undef cb
#undef display
#undef screen
#undef mainWindow

char _xm_object_tmpstrstrbuf[4096];
#define TMPSTMRBUF _xm_object_tmpstrstrbuf,4096

#define XmSDCS XmSTRING_DEFAULT_CHARSET

// Application /////////////////////////////////////////////

XmApp* App;

void (*noX )() = NULL;

#define xmplus_license_string "NN0000NNN090794XMPLUS"

XmApp::XmApp(int argc, char** argv)
{
	if(argc == 2 && !strcmp(argv[1], "xmplus_license"))
	{	cerr << xmplus_license_string << "\n";
		exit(0);
	}
	App = this;
	appArgc = argc;
	appArgv = argv;

	XtToolkitInitialize();
	appShell = NULL;
	toplevelShells = new XmObject*[100];
	numShells = 0;
	appContext = XtCreateApplicationContext();
#ifdef X11R4
	display = XtOpenDisplay(appContext, NULL, NULL, "XmApp", NULL, 0, (Cardinal* )&appArgc, appArgv);
#else
	display = XtOpenDisplay(appContext, NULL, NULL, "XmApp", NULL, 0, &appArgc, appArgv);
#endif
	screen = NULL;
	rscDb = XtDatabase(display);
#ifdef XAW
	unitType = 0;	// unused...
#else
	unitType = XmPIXELS;
#endif
	useWindows = FALSE;
	active = FALSE;
	initialize();
	run();
}

// #include <MainW.h>

extern char* _control_current_label_buffer;
extern char* _control_get_text_buffer;
extern char* _edit_get_text_buffer;
extern char** _listbox_items_ptr;
extern int* _listbox_indices_ptr;

XmApp::~XmApp()
{
	delete toplevelShells;

	XtFree(_control_current_label_buffer);
	XtFree(_control_get_text_buffer);
	XtFree(_edit_get_text_buffer);

	if(_listbox_items_ptr)
	{	for(int i = 0; _listbox_items_ptr[i]; i++)
			XtFree(_listbox_items_ptr[i]);
		delete _listbox_items_ptr;
	}
	delete _listbox_indices_ptr;

	exit(1);
}

void XmApp::run()
{
	if(appShell)
	{	// appShell->realize();
		// screen = XtScreen(appShell->handle());
		active = TRUE;
		XtAppMainLoop(appContext);
	}
	cerr << "xmApp: Error - no application window created, exiting...\n";
}

void XmApp::quit(bool force)
{
#ifndef __GNUG__
	if(!force)
	{	for(int i = 0; i < numShells; i++)
			if(!((XmUserDialog*)toplevelShells[i])->queryClose())
				return;
	}
#endif
	active = FALSE;

	for(int j = 0; j < numShells; j++)
		delete toplevelShells[j];
	delete this;
}

char* XmApp::getCmdArg(int n)
{
	return(n < appArgc ? appArgv[n] : NULL);
}

#include <Shell.h>
#ifdef XAW
extern "C" {
#include <Dialog.h>
}
#else
#include <DialogS.h>
#endif

XmObject* XmApp::findObject(char*)
{
	return(NULL);
}

bool XmApp::setMainObject(XmObject* obj)
{
	for(int i = 0; i < numShells; i++)
	{	if(toplevelShells[i] == obj)
		{	appShell = obj;
			return(TRUE);
		}
	}
	return(FALSE);
}

Widget XmApp::getMainWindow()
{
	return(appShell ? appShell->handle() : NULL);
}

void XmApp::setScreenUnitType(unsigned char u)
{
	if(!active)
		unitType = u;
	else
		cerr << "xmApp: Warning - cannot change unit type of running application!\n";
}

Widget XmApp::newMainWindow(XmObject* anObj, char* name, int width, int height, xmStyle s)
{
	Widget w;
	Arg a[10];
	int argcount = 0;
	char* wn = name ? name : "XmApp";

	if(!display)
	{	cerr << "xmApp: cannot run without valid X Windows display, terminating.\n";
		exit(-1);
	}
#ifndef XAW
	if(s)
	{	long func, decor;
		translateWindowStyles(s, func, decor);
		if(func)
			XtSetArg(a[argcount], XmNmwmFunctions, func); argcount++;	// macro, can't use a[argcount++]...
		if(decor)
			XtSetArg(a[argcount], XmNmwmDecorations, decor); argcount++;
	}
#endif
	if(width || height)
	{	XtSetArg(a[argcount], XtNwidth, (width) ? width : 300); argcount++;
		XtSetArg(a[argcount], XtNheight, (height) ? height : 200); argcount++;
		w = XtAppCreateShell(NULL, wn, applicationShellWidgetClass, display, a, argcount);
	}
	else
		w = XtAppCreateShell(NULL, wn, applicationShellWidgetClass, display, a, argcount);

	if(!appShell)
		appShell = anObj;	// set the default application shell
	if(numShells < 100)
		toplevelShells[numShells++] = anObj;
	return(w);
}

void XmApp::removeMainWindow(XmObject* obj)
{
	if(!active)
		return;

	for(int i = 0; i < numShells; i++)
	{	if(toplevelShells[i] == obj)
		{	for(int j = i; j < numShells; j++)
				toplevelShells[j] = toplevelShells[j + 1];
			numShells--;
			break;
		}
	}
	if(!numShells)
	{	delete this;
		return;
	}
	if(obj == appShell)
		appShell = toplevelShells[0];
}

main(int argc, char** argv)
{
	new XmApp(argc, argv);
}

// Manager //////////////////////////////////////////////////

// will provide support for a more sufficient application
// architecture, not really implemented yet....

XmManager::~XmManager()
{
	if(objects)
	{	if(objectCount)
			cerr << "xmManager: Warning - deleting manager with " << objectCount << " references!\n";
	}
}

bool XmManager::addObject(XmObject* anObject, objType t)
{
	XmObject** newArray;

	if(t == WindowObj || t == UserDialog || t == DialogWindow)
	{	if(objectCount == maxObjects)
		{	newArray = new XmObject*[objectCount ? objectCount * 2 : 5];
			for(int i = 0; i < objectCount; i++)
				newArray[i] = objects[i];
			delete objects;
			objects = newArray;
		}
		objects[objectCount++] = anObject;
		return(TRUE);
	}
	return(FALSE);
}

bool XmManager::removeObject(XmObject* anObject)
{
	for(int i = 0; i < objectCount; i++)
	{	if(objects[i] == anObject)
		{	--objectCount;
			for(int j = i; j < objectCount; j++)
				objects[j] = objects[j + 1];
			return(TRUE);
		}
	}
	return(FALSE);
}

XmObject* XmManager::findObject(char* n, objType t)
{
	for(int i = 0; i < objectCount; i++)
	{	if(!strcmp(objects[i]->getName(), n) && objects[i]->objectType() == t)
			return(objects[i]);
	}
	return(NULL);
}

void XmManager::changed(char* txt, XmObject* except)
{
	for(int i = 0; i < objectCount; i++)
	{	// cerr << objects[i]->getName() << ", ";
		if(objects[i] != except)
			objects[i]->update(txt);
	}
	if(manager)
		manager->changed(txt);
}


// Object //////////////////////////////////////////////////


void XmObject::init(char* n, XmObject* p)
{
	magic = MAGIC_OBJECT_CONST;
	basePtr = this;
	parent = p; 
    	wid = (Widget )NULL;
    	win = (Window )NULL;
	if(!n)
		n = "unnamedObject";
	wname =  strcpy(new char[strlen(n) + 1], n);
    	wargcount = 0;
	gc = (GC )NULL;
    	inputMask = 0;
#ifndef XAW
	addArg(XmNunitType, XmPIXELS);
#endif
	cdptrs = new cb_data*[100];
	numcdptrs = 0;
	destroyWidget = FALSE;	// avoids overhead in all cases where a widget tree is destroyed
}

XmObject::~XmObject()
{
	if(magic)
	{	magic = 0L;
		delete wname;
		for(int i = 0; i < numcdptrs; i++)
			delete cdptrs[i];
		delete cdptrs;
		if(destroyWidget)
		{	if(wid)
				XtVaSetValues(wid, XtNdestroyCallback, NULL, NULL);
			XmObject::destroy();	// can't call virtual here...
		}
	}
	else
		cerr << "xmObject: Warning - deleting invalid object pointer!\n";
}

void genericCallback(Widget w, XtPointer cl_data, XtPointer ca_data)
{
	cb_data* d = (cb_data* )cl_data;

//	if(((XmObject*)d->object)->valid())			// ensure that object has not been deleted...
	{	void* answer = NULL;

		if(d->client_data == CB_XM_DATA)
			answer = ca_data;
		else if(d->client_data == CB_XM_HANDLE)
			answer = w;
		else if(d->client_data == CB_OBJ_NAME)
			answer = XtName(w);
		else if(d->client_data > CB_OBJ_NAME)
			answer = d->client_data;

		((d->object)->*(d->callback))(answer);
	}
}

char XmObject::parse(char* str)		// caution: parses string in place!
{
	char c;

	for(int i = 0; c = str[i]; i++)
	{	if(c == '&')
		{	for(int j = i; str[j]; j++)
				str[j] = str[j + 1];
			return(str[i]);
		}
	}
	return(0);	// str[0]);
}

bool XmObject::setCbFor(Widget w, char* evtName, XmObject* obj, cbProc cb, bool set, void* data)
{
	cb_data* d;
	XtCallbackList cbks;

	if(!wid)
		return(FALSE);

	if(set)
	{	d = new cb_data;
		cdptrs[numcdptrs++] = d;
		d->callback = cb;
		d->object = obj->getBase();
		d->client_data = (XtPointer )data;
		XtAddCallback(w, evtName, genericCallback, (XtPointer )d);
		return(TRUE);
	}
	else
	{	if(XtHasCallbacks(w, evtName) == XtCallbackHasSome)
		{	XtVaGetValues(w, evtName, &cbks, NULL);
			for(int i = 0; cbks[i].callback == NULL; i++)
			{	if(d = (cb_data* )cbks[i]. /* client_data */ closure)
				{	if(d->object == obj && d->callback == cb)
					{	XtRemoveCallback(w, evtName, genericCallback, (XtPointer )d);
						// delete d;  // done by destructor...
						return(TRUE);
					}
				}
			}
		}
	}
	return(FALSE);
}

// A comment to the usage of the XmObject::XXXarg() functions: use them to collect
// a unknown number of Args in several functions, avoid them in one shot operations
// (use a local arg array instead, there may be some pending things in the wargs
// array which should take affect later).

int XmObject::argIndex(String n)
{
	for(int i = 0; i < wargcount; i++)
	{	if(!strcmp(wargs[i].name, n))
			return(i);
	}
	return(-1);
}

void XmObject::setArg(String n, XtArgVal  v)
{
	addArg(n, v);
	if(wid)
	{	XtSetValues(wid, wargs, wargcount);
		wargcount = 0;
	}
}

void XmObject::changeArg(String n, XtArgVal  v)
{
	int ndx;

	if((ndx = argIndex(n)) != -1)
		wargs[ndx].value = v;			// dont use this func with in-place-args!!
	else
		addArg(n, v);
}

void XmObject::forceArg(String n, XtArgVal  v)
{
	Arg a;

	if(wid)
	{	if(n)
		{	XtSetArg(a, n, v);
			XtSetValues(wid, &a, 1);
		}
		else
		{	XtSetValues(wid, wargs, wargcount);
			wargcount = 0;
		}
	}
	else
		cerr << "xmObject: Warning - forceArg " << n << " ignored: object not created.\n";
}

void XmObject::syscall(char*)	// callback name, not implemented yet...
{
}

char RPB[500];

char* XmObject::getResourcePath(bool append)
{
	int olen, alen, clen;

	olen = append ? strlen(RPB) : 0;
	alen = parent ? 0 : strlen(App->getName());
	clen = strlen(wname) + alen;

	char* tmp = strcpy(new char[strlen(RPB) + 1], RPB);
	strncpy(&RPB[clen + 1], tmp, olen);
	delete tmp;
	RPB[clen + olen + 1] = '\0';
	RPB[alen] = '*';
	strncpy(&RPB[alen + 1], wname, strlen(wname));

	if(parent)
		return(parent->getResourcePath(TRUE));

	strncpy(RPB, App->getName(), alen);
	return(RPB);
}

bool XmObject::checkResource(char* aName)
{
	XrmValue dum;
	char* ddum;
	ostrstream rstr(TMPSTMRBUF);

	rstr << getResourcePath() << "." << aName; rstr.put('\0');
	return(XrmGetResource(App->getResourceDatabase(), rstr.str(), rstr.str(), &ddum, &dum) ? TRUE : FALSE);
}

bool XmObject::realize()
{
	if(!wid)
	{	cerr << "xmObject: error - cannot realize object (no widget created).\n";
		return(FALSE);
	}
	if(App->isActive())
	{	if(parent && !XtIsRealized(parent->handle()))
		{	cerr << "xmObject: error - cannot realize object (parent not realized).\n";
			return(FALSE);
		}
		XtRealizeWidget(parent ? wid : XtParent(wid));
		return(TRUE);
	}
	if(!parent)
	{	XtRealizeWidget(XtParent(wid));
		return(TRUE);
	}
	return(FALSE);
}

bool XmObject::destroy()
{
//cerr << "XmObject::destroy() " << this << "\n";
	if(wid)				// may have been destroyed already in subclass
	{	Widget w = wid;
		bool isToplevel = FALSE;
//cerr 	<< "XtUnmanageChild " << XtName(wid) << " "
//	<< (XtParent(wid) ? XtName(XtParent(wid)) : "*no parent*") << "\n";
		XtUnmanageChild(wid);
/*
		if(parent || !XtParent(wid))
			w = wid;
		else                    // no Xm++ parent object - parent widget
*/
		if(XtParent(wid) && XtIsSubclass(XtParent(wid), shellWidgetClass))
		{	w = XtParent(wid);
			isToplevel = TRUE;
		}
		wid = NULL;
//cerr << "XtDestroyWidget " << XtName(w) << "\n";
		XtDestroyWidget(w);
		if(isToplevel)
			App->removeMainWindow(this);
		return(TRUE);
	}
	return(FALSE);
}

bool XmObject::hide()
{
	if(isVisible())
	{
#ifdef XAW
		objType t = objectType();
		if(parent && (t != UserDialog && t != DialogWindow && t != WindowObj))
#else
		if(parent)
#endif
			XtUnmanageChild(topHandle());
		else
			//XtUnrealizeWidget(topHandle());
			XtUnmapWidget(XtParent(topHandle()));
		return(TRUE);
	}
	return(FALSE);
}

bool XmObject::show()
{
	if(wid && !isVisible())
#ifdef SUN
	;
#endif
	{
#ifdef XAW
		objType t = objectType();
		if(parent && (t != UserDialog && t != DialogWindow && t != WindowObj))
#else
		if(parent)
#endif
			XtManageChild(topHandle());
		else
			//XtRealizeWidget(topHandle());
			XtMapWidget(XtParent(topHandle()));
		return(TRUE);
	}
	return(FALSE);
}

bool XmObject::isRealized()
{
	return(wid && XtIsRealized(wid) ? TRUE : FALSE);
}

bool XmObject::isVisible()
{
	if(isRealized())
	{	if(parent)
			return(XtIsManaged(wid) ? TRUE : FALSE);
		XWindowAttributes a;
		XGetWindowAttributes(App->getDisplay(), XtWindow(wid), &a);
		return(a.map_state == IsUnmapped ? FALSE : TRUE);
	}
	return(FALSE);
}

bool XmObject::move(int nx, int ny)
{
	if(wid)
	{	XtMoveWidget(topHandle(), nx, ny);
		return(TRUE);
	}
	return(FALSE);
}

bool XmObject::resize(int nw, int nh)
{
	if(wid)
	{	Arg a;
		Dimension bw = 0;
		XtSetArg(a, XtNborderWidth, &bw);
		XtGetValues(wid, &a, 1);
		XtResizeWidget(topHandle(), nw, nh, bw);
		return(TRUE);
	}
	return(FALSE);
}

bool XmObject::reframe(int nx, int ny, int nw, int nh)
{
	if(wid)
	{	Arg a;
		Dimension bw = 0;
		XtSetArg(a, XtNborderWidth, &bw);
		XtGetValues(wid, &a, 1);
		XtConfigureWidget(topHandle(), nx, ny, nw, nh, bw);
		return(TRUE);
	}
	return(FALSE);
}

bool XmObject::getFrame(int& fx, int& fy, int& fw, int& fh)
{
	if(wid)
	{	Arg a[4];
		Position x, y;
		Dimension w, h;
		XtSetArg(a[0], XtNx, &x);
		XtSetArg(a[1], XtNy, &y);
		XtSetArg(a[2], XtNwidth, &w);
		XtSetArg(a[3], XtNheight, &h);
		XtGetValues(wid, a, 4);
		fx = x; fy = y; fw = w; fh = h;
		return(TRUE);
	}
	return(FALSE);
}

bool XmObject::setBackgroundColor(char* n)
{
	addArg(XtNbackground, (XtArgVal )n);
	if(wid)
		forceArg(NULL, (XtArgVal )NULL);
	return(TRUE);
}

bool XmObject::setForegroundColor(char* n)
{
	addArg(XtNforeground, (XtArgVal )n);
	if(wid)
		forceArg(NULL, (XtArgVal )NULL);
	return(TRUE);
}

#include <cursorfont.h>

char* _object_cursor_names[] = {
	"default", "busy", "left-arrow",
	"up-arrow", "cross", "i-beam", NULL
};

int _object_cursor_constants[] = {
	XC_X_cursor, XC_watch, XC_arrow,
	XC_center_ptr, XC_crosshair, XC_xterm
};

bool XmObject::setCursor(char* n)
{
	for(int i = 0; _object_cursor_names[i]; i++)
	{	if(!strcmp(_object_cursor_names[i], n))
			return(setCursor(i));
	}
	return(FALSE);
}

bool XmObject::setCursor(int n)
{
	if(!(wid && (win || (win = XtWindow(wid)))))
		return(FALSE);
	XDefineCursor(App->getDisplay(), win, (n == XC_X_cursor ? None :
	  XCreateFontCursor(App->getDisplay(), _object_cursor_constants[n])));
	return(TRUE);
}


// Menu base class //////////////////////////////////////////////////////////////

#ifdef XAW
extern "C" {
#include <Box.h>
#include <MenuButton.h>
#include <SimpleMenu.h>
#include <Sme.h>
#include <SmeBSB.h>
#include <SmeLine.h>
}
#else
#include <RowColumn.h>
#include <LabelG.h>
#include <PushBG.h>
#include <ToggleBG.h>
#include <CascadeB.h>
#include <SeparatoG.h>
#endif

int* locateItemPtr = NULL;

int locateItem(Widget)
{
	return(locateItemPtr ? *locateItemPtr : 0);
}

XmMenu::XmMenu(char* name, XmObject* par) : XmObject(name, par)
{
	nextItemPos = -1;
	numSubmenues = 0;
}

XmMenu::~XmMenu()
{
}

XmObject* XmMenu::parentOfMenu()	// find the root in a submenue hierarchy
{
	if(parent->objectType() == Menu)
		return(((XmMenu* )parent)->parentOfMenu());
	return(parent);
}

bool XmMenu::checkMenuResource(char* anItem)
{
	XrmValue dum;
	char* ddum;
	ostrstream rstr(TMPSTMRBUF);

	rstr << getResourcePath() << "*" << anItem << ".labelString"; rstr.put('\0');
	return(XrmGetResource(App->getResourceDatabase(), rstr.str(), rstr.str(), &ddum, &dum) ? TRUE : FALSE);
}

bool XmMenu::addSeparator(char* name)
{
	Arg a;
	char* n = name;
	bool locate = FALSE;

	if(nextItemPos != -1)
	{	XtSetArg(a, XtNinsertPosition, locateItem);
		XtSetValues(getCurrentLabel(), &a, 1);
		locateItemPtr = &nextItemPos;
		locate = TRUE;
	}

	if(!n)
		n = "Separator";
#ifdef XAW
	XtManageChild(XtCreateManagedWidget(n, smeLineObjectClass, getCurrentLabel(), NULL, 0));
#else
	XtManageChild(XmCreateSeparatorGadget(getCurrentLabel(), n, NULL, 0));
#endif
	if(locate)
	{	nextItemPos = -1;
		locateItemPtr = NULL;
		XtSetArg(a, XtNinsertPosition, NULL);
		XtSetValues(getCurrentLabel(), &a, 1);
	}
	return(TRUE);
}

bool XmMenu::removeSeparator(char* name)
{
	char* n = name;
	Widget w;

	if(!n)
		n = "Separator";
	if(!(w = XtNameToWidget(wid, n)))
		return(FALSE);
	XtDestroyWidget(w);
	return(TRUE);
}

void menuCallback(Widget w, XtPointer cl_data, XtPointer)
{
	cb_data* d = (cb_data* )cl_data;
	char* aStr = NULL;

	if(d->client_data)
	{	Arg a;
		XmString lStr = NULL;
#ifdef XAW
		a.name = XtNlabel;
#else
		a.name = XmNlabelString;
#endif
		a.value = (XtArgVal )&lStr;
		XtGetValues(w, &a, 1);

		if(!lStr)
			cerr << "xmMenu (menuCallback): Internal error - cannot get menu label...\n";
#ifdef XAW
		aStr = lStr;
#else
		if(!XmStringGetLtoR(lStr, XmSDCS, &aStr))
			cerr << "xmMenu (menuCallback): Internal error - cannot extract menu label text...\n";
#endif
	}
	else
		aStr = XtName(w);
	((d->object)->*(d->callback))(aStr);
}

#ifdef NO_VA_OBJS
VA_OBJ_SUBST_IMPL(XmMenu, addItem, addItems)
#define NO_VARARGS
#endif

#define MAXITEMS 20

bool XmMenu::addItems(Entry entry, ...)
{
	Arg a;
	int argno = 0;
	char* names[MAXITEMS], * aName;
	char* items[MAXITEMS], * anItem;
	XmManager* receivers[MAXITEMS];
	cbProc procs[MAXITEMS];
	void* data[MAXITEMS];
	bool flags1[MAXITEMS];
	bool flags2[MAXITEMS];
	Entry anEntry;
	bool locate = FALSE;

	names[argno] = entry.name;
	items[argno] = entry.text;
	receivers[argno] = entry.receiver;
	procs[argno] = entry.callback;
	data[argno] = entry.client_data;
	flags1[argno] = entry.checkable;
	flags2[argno++] = entry.radio;
#ifndef NO_VARARGS
	va_list ap;
	va_start(ap, entry);
	while(1)
	{	anEntry = va_arg(ap, Entry);
		if(!(names[argno] = anEntry.name))
			break;
		items[argno] = anEntry.text;
		receivers[argno] = anEntry.receiver;
		procs[argno] = anEntry.callback;
		data[argno] = anEntry.client_data;
		flags1[argno] = anEntry.checkable;
		flags2[argno++] = anEntry.radio;
	}
	va_end(ap);
#else
	names[argno] = NULL;
#endif
	if(nextItemPos != -1)
	{	XtSetArg(a, XtNinsertPosition, locateItem);
		XtSetValues(getCurrentLabel(), &a, 1);
		locateItemPtr = &nextItemPos;
		locate = TRUE;
	}
	for(int i = 0; aName = names[i]; i++)
	{	char tmpstrbuf[1024];
		ostrstream tmpstrm(tmpstrbuf, 1024);
		tmpstrm << items[i]; tmpstrm.put('\0');
		anItem = tmpstrm.str();
		char mm = parse(anItem);
		if(names[i] == items[i])  // name and text are identical...
			aName = anItem;
		Widget w;
		cb_data* d = new cb_data;
		cdptrs[numcdptrs++] = d;
		if(!checkMenuResource(anItem))
#ifdef XAW
			addArg(XtNlabel, (XtArgVal )anItem);
		XtManageChild(w = XtCreateManagedWidget(aName, smeBSBObjectClass, getCurrentLabel(), wargs, wargcount));
#else
		{	addArg(XmNlabelString, (XtArgVal )XmStringCreateLtoR(anItem, XmSDCS));
			addArg(XmNmnemonic, mm);
		}
		if(flags1[i]) // checkable...
		{	if(flags2[i]) // radio...
				addArg(XmNindicatorType, XmONE_OF_MANY);
			XtManageChild(w = XmCreateToggleButtonGadget(getCurrentLabel(), aName, wargs, wargcount));
		}
		else
			XtManageChild(w = XmCreatePushButtonGadget(getCurrentLabel(), aName, wargs, wargcount));
#endif
		wargcount = 0;
		d->callback = procs[i];
		d->object = receivers[i] ? receivers[i] : parentOfMenu()->getBase();
		d->client_data = data[i];
#ifdef XAW
		XtAddCallback(w, XtNcallback, menuCallback, (XtPointer )d);
#else
		if(flags1[i]) // checkable...
			XtAddCallback(w, XmNvalueChangedCallback, menuCallback, (XtPointer )d);
		else
			XtAddCallback(w, XmNactivateCallback, menuCallback, (XtPointer )d);
#endif
		if(locate)
			nextItemPos++;
	}
	if(locate)
	{	nextItemPos = -1;
		locateItemPtr = NULL;
		XtSetArg(a, XtNinsertPosition, NULL);
		XtSetValues(getCurrentLabel(), &a, 1);
	}
	return(TRUE);
}

#ifdef NO_VA_OBJS
#undef NO_VARARGS
#endif

bool XmMenu::removeItems(char* first, ...)
{
	bool ret = TRUE;
	Widget w;
#ifndef NO_VARARGS
	char* item = first;
	va_list ap;

	va_start(ap, first);
	do
	{	if(!(w = XtNameToWidget(getCurrentLabel(), item)))
		{	ret = FALSE;
			continue;
		}
		XtDestroyWidget(w);
	} while(item = va_arg(ap, char*));
	va_end(ap);
#else
	if(!(w = XtNameToWidget(getCurrentLabel(), first)))
		return(FALSE);
	XtDestroyWidget(w);
#endif
	return(ret);

}

bool XmMenu::enableItems(char* first, ...)
{
	bool ret = TRUE;
	Widget w;
	Arg a;

	XtSetArg(a, XtNsensitive, TRUE);
#ifndef NO_VARARGS
	char* item = first;
	va_list ap;
	va_start(ap, first);
	do
	{	if(!(w = XtNameToWidget(getCurrentLabel(), item)))
		{	ret = FALSE;
			continue;
		}
		XtSetValues(w, &a, 1);
	} while(item = va_arg(ap, char*));
	va_end(ap);
#else
	if(!(w = XtNameToWidget(getCurrentLabel(), first)))
		return(FALSE);
	XtSetValues(w, &a, 1);
#endif
	return(ret);
}

bool XmMenu::disableItems(char* first, ...)
{
	bool ret = TRUE;
	Widget w;
	Arg a;

	XtSetArg(a, XtNsensitive, FALSE);
#ifndef NO_VARARGS
	char* item = first;
	va_list ap;
	va_start(ap, first);
	do
	{	if(!(w = XtNameToWidget(getCurrentLabel(), item)))
		{	ret = FALSE;
			continue;
		}
		XtSetValues(w, &a, 1);
	} while(item = va_arg(ap, char*));
	va_end(ap);
#else
	if(!(w = XtNameToWidget(getCurrentLabel(), first)))
		return(FALSE);
	XtSetValues(w, &a, 1);
#endif
	return(ret);
}

#ifndef XAW
#include <CascadeBG.h>
#endif

XmSubMenu* XmMenu::addSubmenu(Entry)
{
	return(NULL);
}

XmSubMenu* XmMenu::addSubmenu(char* xn)
{
#ifdef XAW
	return(NULL);	// not implemented yet...
#else
	XmSubMenu* aMenu;
	bool locate = FALSE;
	Widget w;
	Arg a;

	if(numSubmenues == MAX_SUB_MENUES)
		return(NULL);
	if(nextItemPos != -1)
	{	XtSetArg(a, XmNinsertPosition, locateItem);
		XtSetValues(getCurrentLabel(), &a, 1);
		locateItemPtr = &nextItemPos;
		locate = TRUE;
	}

	char tmpstrbuf[1024];
	ostrstream tmpstrm(tmpstrbuf, 1024);
	tmpstrm << xn; tmpstrm.put('\0');
	char* n = tmpstrm.str();
	char mm = parse(n);
	aMenu = new XmSubMenu(n, this);
	if(!checkMenuResource(n))
	{	addArg(XmNlabelString, (XtArgVal )XmStringCreateLtoR(n, XmSDCS));
		addArg(XmNmnemonic, mm);
	}
	addArg(XmNsubMenuId, (XtArgVal )aMenu->handle());
	XtManageChild(w = XmCreateCascadeButtonGadget(getCurrentLabel(), n, wargs, wargcount));
	wargcount = 0;
	if(locate)
	{	nextItemPos = -1;
		locateItemPtr = NULL;
		XtSetArg(a, XmNinsertPosition, NULL);
		XtSetValues(getCurrentLabel(), &a, 1);
	}
	submenues[numSubmenues] = aMenu;
	subHandles[numSubmenues++] = w;
	return(aMenu);
#endif
}

XmSubMenu* XmMenu::submenuAt(char* xn)
{
#ifdef XAW
	return(NULL);
#else
	char tmpstrbuf[1024];
	ostrstream tmpstrm(tmpstrbuf, 1024);
	tmpstrm << xn; tmpstrm.put('\0');
	char* n = tmpstrm.str();
		
	parse(n);
	for(int i = 0; i < numSubmenues; i++)
	{	if(!strcmp(XtName(subHandles[i]), n))
			return(submenues[i]);
	}
	return(NULL);
#endif
}

bool XmMenu::removeSubmenu(char* n)
{
#ifdef XAW
	return(FALSE);
#else
	int ndx = -1;
	for(int i = 0; i < numSubmenues; i++)
	{	if(!strcmp(XtName(subHandles[i]), n))
			break;
	}
	if(ndx < 0)
		return(FALSE);
	delete submenues[ndx];
	XtDestroyWidget(subHandles[ndx]);
	for(int j = ndx; j < numSubmenues; j++)
	{	submenues[j] = submenues[j + 1];
		subHandles[j] = subHandles[j + 1];
	}
	numSubmenues--;
	return(TRUE);
#endif
}

bool XmMenu::changeItemText(char* itemName, char* xnewText)
{
	Widget w;
	Arg a[2];
	char tmpstrbuf[1024];
	ostrstream tmpstrm(tmpstrbuf, 1024);
	tmpstrm << xnewText; tmpstrm.put('\0');
	char* newText = tmpstrm.str();
	char mm = parse(newText);
		
	if(!(w = XtNameToWidget(getCurrentLabel(), itemName)))
		return(FALSE);
#ifdef XAW
	XtSetArg(a[0], XtNlabel, newText);
	XtSetValues(w, a, 1);
#else
	XtSetArg(a[0], XmNlabelString, XmStringCreateLtoR(newText, XmSDCS));
	XtSetArg(a[1], XmNmnemonic, mm);
	XtSetValues(w, a, 2);
#endif
	return(TRUE);
}

bool XmMenu::getItemStatus(char* itemName)
{
	Widget w;
#ifdef XAW
	return(FALSE);
#else
	if(!(w = XtNameToWidget(getCurrentLabel(), itemName)) ||
	    XtClass(w) != xmToggleButtonGadgetClass)
		return(FALSE);
	return(XmToggleButtonGadgetGetState(w) ? TRUE : FALSE);
#endif
}

bool XmMenu::setItemStatus(char* itemName, bool newStatus)
{
	Widget w;
#ifdef XAW
	return(FALSE);
#else
	if(!(w = XtNameToWidget(getCurrentLabel(), itemName)) ||
	    XtClass(w) != xmToggleButtonGadgetClass)
		return(FALSE);
	XmToggleButtonGadgetSetState(w, newStatus, FALSE);
	return(TRUE);
#endif
}

// Popup menu /////////////////////////////////////////////////////////////////

void postPopup(Widget, XtPointer popup, XEvent* e, Boolean*)
{
	if(e->xbutton.button == Button3)
#ifdef XAW
//	{	XtCallActionProc((Widget )popup, "XawPositionSimpleMenu", e, NULL, 0);
	{	Arg a[2];
		XtSetArg(a[0], XtNx, e->xbutton.x);
		XtSetArg(a[1], XtNy, e->xbutton.y);
		XtSetValues((Widget )popup, a, 2);
#else
	{	XmMenuPosition((Widget )popup, &(e->xbutton));
#endif
		XtManageChild((Widget )popup);
	}
}

XmPopupMenu::XmPopupMenu(char* xname, XmObject* par, bool defaultMenu) : XmMenu(xname, par)
{
	Arg a;
	Widget w[2];
	char acc[20], tmpstrbuf[1024];
	ostrstream tmpstrm(tmpstrbuf, 1024);
	tmpstrm << xname; tmpstrm.put('\0');
	char* name = tmpstrm.str();
	char mn = parse(name);

	strcpy(acc, "Ctrl <Key> p");
#ifdef XAW
    if(wid = XtCreatePopupShell(name, simpleMenuWidgetClass, par->handle(), &a, 1))
#else
	acc[11] = mn;
	XtSetArg(a, XmNmenuAccelerator, acc);
	if(wid = XmCreatePopupMenu(par->handle(), name, &a, 1))
#endif
	{	bool extrnl;
		if(!(extrnl = checkMenuResource(name)))
#ifdef XAW
		{	XtSetArg(a, XtNlabel, name);
			XtSetValues(wid, &a, 1);
		}
#else
			XtSetArg(a, XmNlabelString, XmStringCreateLtoR(name, XmSDCS));
		w[0] = XmCreateLabelGadget(wid, "popup-label", &a, extrnl ? 0 : 1);
		w[1] = XmCreateSeparatorGadget(wid, "popup-seperator", NULL, 0);
		XtManageChildren(w, 2);
#endif
		if(defaultMenu)
			makeDefault(TRUE);
	}
}

void XmPopupMenu::makeDefault(bool sr)
{
	if(sr)
		XtAddEventHandler(parent->handle(), ButtonPressMask, FALSE, postPopup, (XtPointer )wid);
	else
		XtRemoveEventHandler(parent->handle(), ButtonPressMask, FALSE, postPopup, (XtPointer )wid);
}

bool XmPopupMenu::setLabel(char* l)
{
	Arg a;
	Widget w;

#ifdef XAW
	w = wid;
	XtSetArg(a, XtNlabel, l);
#else
	if(!(w = XtNameToWidget(wid, "popup-label")))
		return(FALSE);
	XtSetArg(a, XmNlabelString, XmStringCreateLtoR(l, XmSDCS));
#endif
	XtSetValues(w, &a, 1);
	return(TRUE);
}

bool XmPopupMenu::popup()
{
	Window root, child;
	int rx, wx, ry, wy;
	unsigned int kb;

	if(!wid)
		return(FALSE);
	XQueryPointer(App->getDisplay(), RootWindow(App->getDisplay(), 0), &root, &child, &rx, &ry, &wx, &wy, &kb);
	XtVaSetValues(wid, XtNx, rx, XtNy, ry, NULL);
	XtManageChild(wid);
	return(TRUE);
}


// Dropdown menu /////////////////////////////////////////////////////////////////

XmDropdownMenu::XmDropdownMenu(char* name, XmObject* par) : XmMenu(name, par)
{
	numDropdowns = 0;
	nextLabelPos = -1;
	currentLabel = (Widget )NULL;

#ifdef XAW
	addArg(XtNleft, XtChainLeft);
	addArg(XtNright, XtChainRight);
	addArg(XtNtop, XtChainTop);
	addArg(XtNbottom, XtChainTop);
	addArg(XtNorientation, XtorientHorizontal);
	addArg(XtNborderWidth, 1);
	wid = XtCreateWidget(name, boxWidgetClass, par->handle(), wargs, wargcount);
#else
	addArg(XmNtopAttachment, XmATTACH_FORM);
	addArg(XmNrightAttachment, XmATTACH_FORM);
	addArg(XmNleftAttachment, XmATTACH_FORM);
	wid = XmCreateMenuBar(par->handle(), name, wargs, wargcount);
#endif
	XtManageChild(wid);
	wargcount = 0;
}

int* locateLabelPtr = NULL;

int locateLabel(Widget)
{
	return(locateLabelPtr ? *locateLabelPtr : 0);
}

#define DROPDOWN_PANE_POSTFIX "-menu-pane"

#ifdef NO_VA_OBJS
VA_OBJ_SUBST_IMPL(XmDropdownMenu, addLabel, addLabels)
#define NO_VARARGS
#endif

bool XmDropdownMenu::addLabels(Entry first, ...)
{
	return(FALSE);
}

#ifdef NO_VA_OBJS
#undef NO_VARARGS
#endif


bool XmDropdownMenu::addLabels(char* first, ...)
{
	int argno = 0;
	char* items[MAX_DROPDOWN_LABELS], * xitem, * item;
	bool locate = FALSE;

	items[argno++] = first;

#ifndef NO_VARARGS
	va_list ap;
	va_start(ap, first);
	while((items[argno++] = va_arg(ap, char *)) != (char *)0);
	va_end(ap);
#endif
	items[argno] = (char* )NULL;
	if(nextLabelPos != -1)
	{	forceArg(XtNinsertPosition, (XtArgVal )locateLabel);
		locateLabelPtr = &nextLabelPos;
		locate = TRUE;
	}
	for(int i = 0; xitem = items[i]; i++)
	{	char tmpstrbuf1[1024], tmpstrbuf2[1024];
		ostrstream tmpstrm(tmpstrbuf1, 1024);
		tmpstrm << xitem; tmpstrm.put('\0');
		item = tmpstrm.str();
		char mm = parse(item);
		ostrstream strm(tmpstrbuf2, 1024);
		Widget w;

		strm << item << DROPDOWN_PANE_POSTFIX; strm.put('\0');
#ifdef XAW
		char* menuName = strcpy(new char[strm.pcount() + 1], strm.str());
		currentLabel = XtCreatePopupShell(menuName, simpleMenuWidgetClass, wid, NULL, 0);
		if(!checkMenuResource(item))
			addArg(XtNlabel, (XtArgVal )item);
		addArg(XtNmenuName, (XtArgVal )menuName);
		w = XtCreateWidget(item, menuButtonWidgetClass, wid, wargs, wargcount);
#else
		currentLabel = XmCreatePulldownMenu(wid, strm.str(), NULL, 0);
		addArg(XmNsubMenuId, (XtArgVal )currentLabel);
		if(!checkMenuResource(item))
		{	addArg(XmNlabelString, (XtArgVal )XmStringCreateLtoR(item, XmSDCS));
			addArg(XmNmnemonic, mm);
		}
		w = XmCreateCascadeButton(wid, item, wargs, wargcount);
#endif
		XtManageChild(w);
		wargcount = 0;
		if(locate)
			nextLabelPos++;
		dropdowns[numDropdowns++] = currentLabel;
	}
	if(locate)
	{	nextLabelPos = -1;
		locateLabelPtr = NULL;
		forceArg(XtNinsertPosition, (XtArgVal )NULL);
	}
	return(TRUE);
}

bool XmDropdownMenu::removeLabels(char* first, ...)
{
	int argno = 0;
	char* items[MAX_DROPDOWN_LABELS], * xitem, * item;
	bool ret = TRUE;

	items[argno++] = first;

#ifndef NO_VARARGS
	va_list ap;
	va_start(ap, first);
	while((items[argno++] = va_arg(ap, char *)) != (char *)0);
	va_end(ap);
#endif
	items[argno] = (char* )NULL;

	for(int i = 0; xitem = items[i]; i++)
	{	Widget w1, w2;
		char tmpstrbuf1[1024], tmpstrbuf2[1024];
		ostrstream strm(tmpstrbuf1, 1024);
		ostrstream tmpstrm(tmpstrbuf2, 1024);
		tmpstrm << xitem; tmpstrm.put('\0');
		item = tmpstrm.str();
		parse(item);
		strm << item << DROPDOWN_PANE_POSTFIX; strm.put('\0');

		if(!((w1 = XtNameToWidget(wid, strm.str())) && (w2 = XtNameToWidget(wid, item))))
		{	ret = FALSE;
			continue;
		}
		XtDestroyWidget(w1);
		XtDestroyWidget(w2);
	}
	return(ret);
}

bool XmDropdownMenu::setCurrentLabel(char* n)
{
	Widget lastLabel = currentLabel;
	char tmpstrbuf[1024];
	ostrstream lbn(tmpstrbuf, 1024);

	lbn << n << DROPDOWN_PANE_POSTFIX; lbn.put('\0');

	for(int i = 0; i < numDropdowns; i++)
	{	if(!strcmp(XtName(currentLabel = dropdowns[i]), lbn.str()))
			return(TRUE);
	}
	cerr << "xmDropdownMenu: Warning - cannot find label " << n << " using: " << hex(long(lastLabel)) << "\n";
	currentLabel = lastLabel;
	return(FALSE);
}

bool XmDropdownMenu::changeLabelText(char*, char*)
{
	return(FALSE);
}

// subMenu /////////////////////////////////////////////////////////////////////

XmSubMenu::XmSubMenu(char* n, XmMenu* par) : XmMenu(n, par)
{
#ifndef XAW
	wid = XmCreatePulldownMenu(par->getCurrentLabel(), n, NULL, 0);
//	XtManageChild(wid);
#endif
	wargcount = 0;
}

XmSubMenu::~XmSubMenu()
{
}

#ifndef XAW
#include <Xm/MwmUtil.h>
#endif

// Dialog base class //////////////////////////////////////////////////////////

XmDialog::XmDialog(char* n, XmObject* par, xmStyle s) : XmObject(n, par)
{
	basePtr = (XmObject* )((void* )this);
	dlgStyle = (s & XmSdlgWinModal || s & XmSdlgAppModal || s & XmSdlgSysModal || s & XmSpopup) ? s : XmSdlgModeless;
}

XmDialog::~XmDialog()
{
}

#ifndef XAW
#define mwmMode(s) 	(s & XmSdlgModeless ? MWM_INPUT_MODELESS : \
					(s & XmSdlgWinModal ? MWM_INPUT_PRIMARY_APPLICATION_MODAL : \
					(s & XmSdlgAppModal ? MWM_INPUT_FULL_APPLICATION_MODAL : \
										  MWM_INPUT_SYSTEM_MODAL)))
#endif

bool XmDialog::setLabelAndMode(char* label)
{
	Widget sh;

	if(!wid || !(sh = XtParent(wid)))
		return(FALSE);
#ifdef XAW
	addArg(XtNlabel, (XtArgVal )label);
#else
	addArg(XmNtitle, (XtArgVal )label);
	addArg(XmNmwmInputMode, (XtArgVal )mwmMode(dlgStyle));
#endif
	XtSetValues(XtParent(wid), wargs, wargcount);
	wargcount = 0;
	return(TRUE);
}

bool XmDialog::run()
{
	if(dlgStyle & XmSdlgModeless)
	{	cerr << "xmDialog: Error - cannot run a modeless dialog.\n";
		return(FALSE);
	}
	completed = returned = FALSE;
	XtManageChild(wid);
#ifdef XAW
	XtRealizeWidget(XtParent(wid));
#endif
	while(!completed)
		XtAppProcessEvent(App->getAppContext(), XtIMAll);
	return(returned);
}

void XmDialog::ok(void*)
{
	returned = completed = TRUE;
}

void XmDialog::cancel(void*)
{
	completed = TRUE;
}

void XmDialog::close(void*)
{
	delete this;
}

// Predefined system dialogs //////////////////////////////////////////////////

XmSystemDialog::XmSystemDialog(char* n, XmObject* par, xmStyle s) : XmDialog(n, par, s)
{
	destroyOnDelete();
}

#ifdef XAW
#include <Dialog.h>
#else
#include <MessageB.h>
#endif

typedef Widget (*DlgCreateFunc )(Widget, char*, Arg*, unsigned int);

XmMsgBox::XmMsgBox(char* text, char* label, xmStyle s, XmObject* rec, cbProc okCb) : XmSystemDialog("msg", NULL, s ? s : XmSmsgOkCancel | XmSdlgAppModal)
{
	Widget aParent, aShell = NULL;

#ifdef XAW
	if(FALSE && rec && rec->handle())	// unsolved transient shell problem...
	{	addArg(XtNwidth, 200);
		addArg(XtNheight, 100);
		addArg(XtNtransientFor, (XtArgVal )rec->handle());
		aShell = XtCreateWidget(label, transientShellWidgetClass, rec->topHandle(), wargs, wargcount);
		wargcount = 0;
	}
	else
		aShell = App->newMainWindow(this, label, 200, 100, s);

	addArg(XtNallowShellResize, TRUE);
	addArg(XtNtitle, (XtArgVal )label);
	XtSetValues(aShell, wargs, wargcount);
	wargcount = 0;
	addArg(XtNlabel, (XtArgVal )text);
/*													// not implemented yet...
	addArg(XtNicon,
			(s & XmSmsgMemo ? XmDIALOG_MESSAGE :
			(s & XmSmsgInfo ? XmDIALOG_INFORMATION :
			(s & XmSmsgQuestion ? XmDIALOG_QUESTION :
			(s & XmSmsgWorking ? XmDIALOG_WORKING :
			(s & XmSmsgWarning ? XmDIALOG_WARNING :
			(s & XmSmsgError ? XmDIALOG_ERROR : XmDIALOG_MESSAGE)))))));
*/
	addArg(XtNresizable, TRUE);
	char tmpstrbuf[1024];
	ostrstream dlgName(tmpstrbuf, 1024);
	dlgName << label << "-dlg"; dlgName.put('\0');
	XtManageChild(wid = XtCreateWidget(dlgName.str(), dialogWidgetClass, aShell, wargs, wargcount));
	wargcount = 0;

	if(dlgStyle & XmSmsgOk || dlgStyle & XmSmsgOkCancel || dlgStyle & XmSmsgOkCancelHelp)
		XawDialogAddButton(wid, "OK", NULL, NULL);
	if(dlgStyle & XmSmsgOkCancel || dlgStyle & XmSmsgOkCancelHelp)
		XawDialogAddButton(wid, "Cancel", NULL, NULL);
	if(dlgStyle & XmSmsgOkCancelHelp)
		XawDialogAddButton(wid, "Help", NULL, NULL);
	if(dlgStyle & XmSmsgYesNo)
	{	XawDialogAddButton(wid, "Yes", NULL, NULL);
		XawDialogAddButton(wid, "No", NULL, NULL);
	}
#else
	if(!(aParent = (rec && rec->handle()) ? rec->handle() : App->getMainWindow()))
	{	cerr << "creating dlg shell...\n";
		aShell = App->newMainWindow(this, label, 10, 10, s);
	}

	DlgCreateFunc mfunc;

	if(aShell)
	{	mfunc = XmCreateMessageBox;
		addArg(XmNdialogType,
			(s & XmSmsgMemo ? XmDIALOG_MESSAGE :
			(s & XmSmsgInfo ? XmDIALOG_INFORMATION :
			(s & XmSmsgQuestion ? XmDIALOG_QUESTION :
			(s & XmSmsgWorking ? XmDIALOG_WORKING :
			(s & XmSmsgWarning ? XmDIALOG_WARNING :
			(s & XmSmsgError ? XmDIALOG_ERROR : XmDIALOG_MESSAGE)))))));
	}
	else
		mfunc = (s & XmSmsgMemo ? XmCreateMessageDialog :
				(s & XmSmsgInfo ? XmCreateInformationDialog :
				(s & XmSmsgQuestion ? XmCreateQuestionDialog :
				(s & XmSmsgWorking ? XmCreateWorkingDialog :
				(s & XmSmsgWarning ? XmCreateWarningDialog :
				(s & XmSmsgError ? XmCreateErrorDialog : XmCreateMessageDialog))))));

	addArg(XmNmessageString, (XtArgVal )XmStringCreateLtoR(text, XmSDCS));
	addArg(XmNmessageAlignment, (dlgStyle & XmSleft ? XmALIGNMENT_BEGINNING : (dlgStyle & XmSright ? XmALIGNMENT_END : XmALIGNMENT_CENTER)));
	if(dlgStyle & XmSmsgYesNo || dlgStyle & XmSmsgYesNoCancel)
	{	addArg(XmNokLabelString, (XtArgVal )XmStringCreateLtoR("Yes", XmSDCS));
		addArg(XmNcancelLabelString, (XtArgVal )XmStringCreateLtoR("No", XmSDCS));
		addArg(XmNhelpLabelString, (XtArgVal )XmStringCreateLtoR("Cancel", XmSDCS));
	}
	wid = (*mfunc )(aShell ? aShell : aParent, label, wargs, wargcount);
	wargcount = 0;

	if(dlgStyle & XmSmsgOk || dlgStyle & XmSmsgOkCancel || dlgStyle & XmSmsgYesNo)
		XtUnmanageChild(XmMessageBoxGetChild(wid, XmDIALOG_HELP_BUTTON));
	if(dlgStyle & XmSmsgOk)
		XtUnmanageChild(XmMessageBoxGetChild(wid, XmDIALOG_CANCEL_BUTTON));
#endif
	setDefaultButtonProcs(rec, okCb);

	if(dlgStyle & XmSmsgQuestion)
		dlgStyle |= XmSdlgWinModal;
	if(dlgStyle & XmSmsgWarning)
		dlgStyle |= XmSdlgAppModal;
	if(dlgStyle & XmSmsgError)
		dlgStyle |= XmSdlgSysModal;

	setLabelAndMode(label);
	if(dlgStyle & XmSdlgModeless)
	{	XtManageChild(wid);
#ifdef XAW
		XtRealizeWidget(XtParent(wid));
#endif
	}
}


XmMsgBox::XmMsgBox(char* n, XmObject* par, xmStyle s) : XmSystemDialog(n, par, (s ? s : XmSmsgOkCancel | XmSdlgAppModal))
{
}

void XmMsgBox::setDefaultButtonProcs(XmObject* rec, cbProc okCb)
{
	if(!(dlgStyle & XmSdlgModeless) && validProc(okCb))
	{	cerr << "xmMsgBox: Warning - cannot use callback in modal dialog.\n";
		rec = NULL;
	}
#ifdef XAW
	Widget dlgWid = wid;
	wid = XtNameToWidget(dlgWid, "OK");
	if(wid || (wid = XtNameToWidget(dlgWid, "Yes")))
	{	if(rec && validProc(okCb))
			setCallback(XtNcallback, rec, okCb);
		else
			setCallback(XtNcallback, this, CBK(XmDialog, ok));
	}
	else
		cerr << "XmMsgBox: Error - no OK or Yes button found...\n";
	wid = XtNameToWidget(dlgWid, "Cancel");
	if(wid || (wid = XtNameToWidget(dlgWid, "No")))
		setCallback(XtNcallback, this, CBK(XmDialog, cancel));
	wid = dlgWid;
#else
	if(rec && validProc(okCb))
		setCallback(XmNokCallback, rec, okCb);
	else
		setCallback(XmNokCallback, this, CBK(XmDialog, ok));
	setCallback(XmNcancelCallback, this, CBK(XmDialog, cancel));
#endif
}

bool XmMsgBox::showMsg()
{
	bool rcode = run();
	delete this;
	return(rcode);
}

bool XmMsgBox::setButtonText(msgButton b, char* t)
{
	if(!wid)
		return(FALSE);
#ifdef XAW
	return(FALSE);
#else
	String bnames[HELP + 1];
	Arg a;

	bnames[OK] 		= XmNokLabelString;
	bnames[CANCEL] 	= XmNcancelLabelString;
	bnames[HELP] 	= XmNhelpLabelString;
	a.name = bnames[b];
	a.value = (XtArgVal )XmStringCreateLtoR(t, XmSDCS);
	XtSetValues(wid, &a, 1);
	return(TRUE);
#endif
}

#ifndef XAW
#include <SelectioB.h>
#include <Text.h>
#endif

XmPrompter::XmPrompter(char* n, XmObject* par, xmStyle s) : XmMsgBox(n, par, s)
{
}

#ifdef XAW
XmPrompter::XmPrompter(char* prompt, char* label, char* def, xmStyle s, XmObject* rec, cbProc okCb) :XmMsgBox(prompt, label, s, rec, okCb)
{
	forceArg(XtNvalue, (XtArgVal )(def ? def : ""));
	if((dlgStyle = s) & XmSdlgModeless)
	{	XtManageChild(wid);
		XtRealizeWidget(XtParent(wid));
	}
}
#else
XmPrompter::XmPrompter(char* prompt, char* label, char* def, xmStyle s, XmObject* rec, cbProc okCb) : XmMsgBox("prompter", (XmObject* )NULL, s)
{
	// setText(def);
	addArg(XmNselectionLabelString, (XtArgVal )XmStringCreateLtoR(prompt, XmSDCS));
	wid = XmCreatePromptDialog((rec && rec->handle()) ? rec->handle() : App->getMainWindow(), label, wargs, wargcount);
	wargcount = 0;

	setText(def);	// does not work otherwise...

	if(dlgStyle & XmSmsgOkCancel)
		XtUnmanageChild(XmSelectionBoxGetChild(wid, XmDIALOG_HELP_BUTTON));

	setDefaultButtonProcs(rec, okCb);
	setLabelAndMode(label);
	if(dlgStyle & XmSdlgModeless)
		XtManageChild(wid);
}
#endif

char* XmPrompter::prompt()
{
	char* txt = NULL;

	if(run())
		txt = getText();
	delete this;
	return(txt);
}

bool XmPrompter::setText(char* txt)
{
#if defined( LINUX ) && ! defined( XAW )	
	if(wid)
	{	XmTextSetString(XmSelectionBoxGetChild(wid, XmDIALOG_TEXT), txt);
		return(TRUE);
	}
	return(FALSE);
#else
#ifdef XAW
	addArg(XtNvalue, (XtArgVal )txt);
#else
	addArg(XmNtextString, (XtArgVal )XmStringCreateLtoR(txt, XmSDCS));
#endif
	if(wid)
	{	XtSetValues(wid, wargs, wargcount);
		wargcount = 0;
	}
	return(TRUE);
#endif
}

char* XmPrompter::getText()
{
	char* txt;

	if(!wid)
		return((char* )NULL);
#ifdef XAW
	txt = XawDialogGetValueString(wid);
#else
	delete _edit_get_text_buffer;
	_edit_get_text_buffer = NULL;

	getArg(XmNtextString);
	XtGetValues(wid, wargs, wargcount);
	XmStringGetLtoR((XmString )wargs[0].value, XmSDCS, &txt);
	_edit_get_text_buffer = txt;
#endif
	wargcount = 0;
	return(txt);
}

#ifdef XAW
XmListPrompter::XmListPrompter(char** list, char* prompt, char* label, char* def, xmStyle s, XmObject* rec, cbProc okCb) : XmPrompter("Sorry, list selection not implemented.", label, list[0], s, rec, okCb)
{
	if((dlgStyle = s) & XmSdlgModeless)
	{	XtManageChild(wid);
		XtRealizeWidget(XtParent(wid));
	}
}
#else
XmListPrompter::XmListPrompter(char** list, char* prompt, char* label, char* def, xmStyle s, XmObject* rec, cbProc okCb) : XmPrompter("listPrompter", NULL, s)
{
	char** aStr = list;
	XmString* items;
	int count = 0;

	while(*aStr++ && count < 100)
		count++;

	items = new XmString[count];
	for(int i = 0; i < count; i++)
		items[i] = XmStringCreateLtoR(list[i], XmSDCS);

	setText(def);
	addArg(XmNlistItems, (XtArgVal )items);
	addArg(XmNlistItemCount, count);
	addArg(XmNselectionLabelString, (XtArgVal )XmStringCreateLtoR(prompt, XmSDCS));
	wid = XmCreateSelectionDialog((rec && rec->handle()) ? rec->handle() : App->getMainWindow(), label, wargs, wargcount);
	wargcount = 0;
	delete items;

	XtUnmanageChild(XmSelectionBoxGetChild(wid, XmDIALOG_APPLY_BUTTON));
	if(dlgStyle & XmSmsgOkCancel)
		XtUnmanageChild(XmSelectionBoxGetChild(wid, XmDIALOG_HELP_BUTTON));

	setDefaultButtonProcs(rec, okCb);
	setLabelAndMode(label);
	if(dlgStyle & XmSdlgModeless)
		XtManageChild(wid);
}
#endif

int XmListPrompter::promptIndex()
{
	int ndx = -1;

	if(run())
		ndx = getIndex();
	delete this;
	return(ndx);
}

XmListPrompter* XmListPrompter::selectText(char*)
{
	return(this);
}

XmListPrompter* XmListPrompter::selectIndex(int)
{
	return(this);
}

int XmListPrompter::getIndex()
{
	return(0);
}

#ifndef XAW
#include <FileSB.h>
#endif

#ifdef XAW
XmFileSelector::XmFileSelector(char* path, char* label, char* deflt, xmStyle s, XmObject*rec, cbProc okCb) : XmPrompter("Enter filename:", label ? label : "File Selector", path, s, rec, okCb)
{
	if((dlgStyle = s) & XmSdlgModeless)
	{	XtManageChild(wid);
		XtRealizeWidget(XtParent(wid));
	}
}
#else
XmFileSelector::XmFileSelector(char* path, char* label, char* deflt, xmStyle s, XmObject*rec, cbProc okCb) : XmPrompter("fileSelector", NULL, s)
{
	char tmpdeflt[1024], * tmplnk;

	addArg(XmNdirMask, (XtArgVal )XmStringCreateLtoR(path, XmSDCS));
	if(deflt && *deflt)
	{	strcpy(tmpdeflt, path);
		if(tmplnk = strrchr(tmpdeflt, '/'))
			*(tmplnk + 1) = 0;
		else
			*tmpdeflt = 0;
		strcat(tmpdeflt, deflt);
		addArg(XmNdirSpec, (XtArgVal )XmStringCreateLtoR(tmpdeflt, XmSDCS));
	}
	wid = XmCreateFileSelectionDialog((rec && rec->handle()) ? rec->handle() : App->getMainWindow(), "fileSelector", wargs, wargcount);
	wargcount = 0;
	if(deflt && *deflt)
		XmTextSetString(XmFileSelectionBoxGetChild(wid, XmDIALOG_TEXT), tmpdeflt);
	Widget helpButton = XmMessageBoxGetChild(wid, XmDIALOG_HELP_BUTTON);
	if(dlgStyle & XmSmsgOkCancel && helpButton)
		XtUnmanageChild(helpButton);

	setDefaultButtonProcs(rec, okCb);
	setLabelAndMode(label ? label : "File Selector");

	if(dlgStyle & XmSdlgModeless)
		XtManageChild(wid);
}
#endif

char* XmFileSelector::promptFile()
{
	char* txt;

	if(!run())
	{	delete this;
		return(NULL);
	}
	txt = getPath();
	delete this;
	return(txt);
}

bool XmFileSelector::setPath(char* txt)
{
#ifdef XAW
	return(FALSE);
#else
	addArg(XmNdirSpec, (XtArgVal )XmStringCreateLtoR(txt, XmSDCS));
	addArg(XmNdirMask, (XtArgVal )XmStringCreateLtoR(txt, XmSDCS));
	if(wid)
	{	XtSetValues(wid, wargs, wargcount);
		wargcount = 0;
	}
	return(TRUE);
#endif
}

char* XmFileSelector::getPath()
{
	char* txt;

	if(!wid)
		return((char* )NULL);
#ifdef XAW
	return(getText());
#else
	delete _edit_get_text_buffer;
	_edit_get_text_buffer = NULL;

	getArg(XmNdirSpec);
	XtGetValues(wid, wargs, wargcount);
	wargcount = 0;
	XmStringGetLtoR((XmString )wargs[0].value, XmSDCS, &txt);
	_edit_get_text_buffer = txt;
	return(txt);
#endif
}


// Window classes /////////////////////////////////////////////////////////////

#ifndef XAW
#include <MwmUtil.h>

void translateWindowStyles(xmStyle s, long& f, long& d)
{
	f = d = 0;

	if(s & XmSmaximize)
	{	f |= MWM_FUNC_MAXIMIZE;
		d |= MWM_DECOR_MAXIMIZE;
	}
	if(s & XmSminimize)
	{	f |= MWM_FUNC_MINIMIZE;
		d |= MWM_DECOR_MINIMIZE;
	}
	if(s & XmSbordered)
		d |= MWM_DECOR_BORDER;
	if(s & XmSmoveable)
	{	f |= MWM_FUNC_MOVE;
		d |= MWM_DECOR_TITLE;
	}
	if(s & XmSresizeable)
	{	f |= MWM_FUNC_RESIZE;
		d |= MWM_DECOR_RESIZEH;
	}
	if(s & XmScloseable)
		f |= MWM_FUNC_CLOSE;
	if(s & XmStitled)
		d |= MWM_DECOR_TITLE;
	if(s & XmSsysMenu)
		d |= MWM_DECOR_MENU;
}

#endif

XmDialogWindow::XmDialogWindow(char* n, XmObject* par, xmStyle s) : XmUserDialog(n, par, s)
{
	style = s;
	dropdown = NULL;
	popup = NULL;
}

XmDialogWindow::XmDialogWindow(char* n, XmObject* par, XmManager* mgr, xmStyle s) : XmUserDialog(n, par, mgr, s)
{
	style = s;
	dropdown = NULL;
	popup = NULL;
}

XmDialogWindow::~XmDialogWindow()
{
	delete dropdown;
	delete popup;
//	if(manager)
//		manager->removeObject(this);
}

#ifdef XAW
#define DEF_DLGW_WIDTH 300
#define DEF_DLGW_HEIGHT 200

inline void resizeBoxChild(Widget wid, Dimension w, Dimension h, Dimension b)
{
	XtUnmanageChild(wid);
	XtResizeWidget(wid, w, h, b);
	XtManageChild(wid);
}

#endif

bool XmDialogWindow::realize()
{
	if(dropdown)
		setInitWindowSize(ix, iy, iw, ih + 35);
#ifdef XAW
	if(wid)
	{	int x = 0, y = 0, w = 0, h = 0;
		if(initWindowSize(x, y, w, h))
		{	ix += x; iy += y; iw += w; ih += h;
		}
		if(ix) addArg(XtNx, ix);
		if(iy) addArg(XtNy, iy);
		addArg(XtNwidth, iw ? iw : (iw = DEF_DLGW_WIDTH));
		addArg(XtNheight, ih ? ih : (ih = DEF_DLGW_HEIGHT));
		XtSetValues(XtParent(wid), wargs, wargcount);
		wargcount = 0;
		XtSetMappedWhenManaged(wid, FALSE);
		XtRealizeWidget(XtParent(wid));
		if(dropdown)
		{	Arg a;
			Dimension ddh;
			XtSetArg(a, XtNheight, &ddh);
			XtGetValues(dropdown->handle(), &a, 1);
			resizeBoxChild(dropdown->handle(), iw - 2, ddh, 1);
		}
		XtSetMappedWhenManaged(wid, TRUE);
		return(TRUE);
	}
	return(FALSE);
#else
	return(XmUserDialog::realize());
#endif
}

XmControl* XmDialogWindow::findControl(ctrlType t, char* n, bool w)
{
	return(XmDialogPane::findControl(t, n, w));
}

bool XmDialogWindow::add(XmControl* c, XmObject* rec, cbProc cb, void* cd)	
{
	if(dropdown)
	{	Arg a;
		Dimension h = 0;
		XtSetArg(a, XtNheight, (XtArgVal )&h);
		XtGetValues(dropdown->handle(), &a, 1);
#ifdef XAW
		c->cy += h;
#else
		c->cy += 35; 	// did'nt figure out how to get correct motif menu widget size...
#endif
		c->move(c->cx, c->cy);
	}
	return(XmDialogPane::add(c, rec, cb, cd));
}

void XmWindow::initWindowContents(char* n)
{
	char tmpstrbuf[1024];
	ostrstream pn(tmpstrbuf, 1024);

	pn << n << "Panes"; pn.put('\0');
#ifdef XAW
	panes = new XmPaneArea(pn.str(), 0, 0, 1, 1);
	add(panes);
	panes->addArg(XtNtop, XtChainTop);
	panes->addArg(XtNleft, XtChainLeft);
	panes->addArg(XtNright, XtChainRight);
	panes->addArg(XtNbottom, XtChainBottom);
	panes->forceArg(NULL, (XtArgVal )NULL);
#else
	panes = new XmPaneArea(pn.str(), 0, 30, 100, 100);
	panes->addArg(XmNtopAttachment, XmATTACH_FORM);
	panes->addArg(XmNleftAttachment, XmATTACH_FORM);
	panes->addArg(XmNrightAttachment, XmATTACH_FORM);
	panes->addArg(XmNbottomAttachment, XmATTACH_FORM);
	add(panes);
#endif

	toolbars[0] = toolbars[1] = toolbars[2] = toolbars[3] = NULL;
	statusbar = NULL;
}

void XmWindow::menuSet()
{
	checkLayout();
}

bool XmWindow::realize()
{
#ifdef XAW
	if(wid)
	{	XtSetMappedWhenManaged(wid, FALSE);
		int x = 0, y = 0, w = 0, h = 0;
		if(initWindowSize(x, y, w, h))
		{	ix = x; iy = y; iw = w; ih = h;
		}
		if(ix) addArg(XtNx, ix);
		if(iy) addArg(XtNy, iy);
		addArg(XtNwidth, iw ? iw : (iw = DEF_DLGW_WIDTH));
		addArg(XtNheight, ih ? ih : (ih = DEF_DLGW_HEIGHT));
		XtSetValues(XtParent(wid), wargs, wargcount);
		wargcount = 0;
		XtRealizeWidget(XtParent(wid));
		Dimension ddh = 0;
		if(dropdown)
		{	Arg a;
			XtSetArg(a, XtNheight, &ddh);
			XtGetValues(dropdown->handle(), &a, 1);
			resizeBoxChild(dropdown->handle(), iw - 2, ddh, 1);
		}
		Dimension tbw[4], tbh[4];
		for(int i = 0; i < 4; i++)
		{	if(toolbars[i])
			{	toolbars[i]->updateDimensions();
				tbw[i] = toolbars[i]->cw;
				tbh[i] = toolbars[i]->ch;
			}
			else
				tbw[i] = tbh[i] = 0;
		}
		Dimension sth = 0;
		if(statusbar)
		{	statusbar->updateDimensions();
			sth = statusbar->ch;
		}
		Dimension h1 = ih - ddh - tbh[2] - tbh[3] - sth - 4;
		if(toolbars[0])
			resizeBoxChild(toolbars[0]->handle(), tbw[0], h1, 1);
		if(toolbars[1])
			resizeBoxChild(toolbars[1]->handle(), tbw[1], h1, 1);
		if(toolbars[2])
			resizeBoxChild(toolbars[2]->handle(), iw, tbh[2], 1);
		if(toolbars[3])
			resizeBoxChild(toolbars[3]->handle(), iw, tbh[3], 1);
		if(statusbar)
			resizeBoxChild(statusbar->handle(), iw, sth, 1);
		Dimension w1 = iw - tbw[0] - tbw[1] - 4;
		resizeBoxChild(panes->handle(), w1, h1, 1);
		XtSetMappedWhenManaged(wid, TRUE);
		return(TRUE);
	}
	return(FALSE);
#else
	return(XmUserDialog::realize());
#endif
}

inline int orNdx(xmStyle s)
{
	return(s == XmSleft ? 0 : (s == XmSright ? 1 : (s == XmSbottom ? 3 : 2)));
}

#ifdef XAW

char* attachmentNames[] =	{ XtNleft, XtNright, XtNtop, XtNbottom };
char* widgetNames[] =		{ XtNfromHoriz, XtNfromHoriz, XtNfromVert, XtNfromVert };

void XmWindow::checkLayout()
{
	if(toolbars[2])
	{	if(dropdown)
			toolbars[2]->addArg(XtNfromVert, (XtArgVal )dropdown->handle());
		else
			toolbars[2]->addArg(XtNfromVert, (XtArgVal )NULL);
		toolbars[2]->addArg(XtNtop, XtChainTop);
		toolbars[2]->addArg(XtNbottom, XtChainTop);
		toolbars[2]->addArg(XtNleft, XtChainLeft);
		toolbars[2]->addArg(XtNright, XtChainRight);
		toolbars[2]->forceArg(NULL, (XtArgVal )NULL);
	}
	if(toolbars[3])
	{	toolbars[3]->addArg(XtNfromVert, (XtArgVal )panes->handle());
		toolbars[3]->addArg(XtNleft, XtChainLeft);
		toolbars[3]->addArg(XtNright, XtChainRight);
		toolbars[3]->addArg(XtNtop, XtChainBottom);
		toolbars[3]->addArg(XtNbottom, XtChainBottom);
		toolbars[3]->forceArg(NULL, (XtArgVal )NULL);
	}
	if(toolbars[0])
	{	if(toolbars[2])
			toolbars[0]->addArg(XtNfromVert, (XtArgVal )toolbars[2]->handle());
		else if(dropdown)
			toolbars[0]->addArg(XtNfromVert, (XtArgVal )dropdown->handle());
		else
			toolbars[0]->addArg(XtNfromVert, (XtArgVal )NULL);
		toolbars[0]->addArg(XtNtop, XtChainTop);
		toolbars[0]->addArg(XtNbottom, XtChainBottom);
		toolbars[0]->addArg(XtNleft, XtChainLeft);
		toolbars[0]->addArg(XtNright, XtChainLeft);
		toolbars[0]->forceArg(NULL, (XtArgVal )NULL);
	}
	if(toolbars[1])
	{	if(toolbars[2])
			toolbars[1]->addArg(XtNfromVert, (XtArgVal )toolbars[2]->handle());
		else if(dropdown)
			toolbars[1]->addArg(XtNfromVert, (XtArgVal )dropdown->handle());
		else
			toolbars[1]->addArg(XtNfromVert, (XtArgVal )NULL);
		toolbars[1]->addArg(XtNtop, XtChainTop);
		toolbars[1]->addArg(XtNfromHoriz, (XtArgVal )panes->handle());
		toolbars[1]->addArg(XtNleft, XtChainRight);
		toolbars[1]->addArg(XtNright, XtChainRight);
		toolbars[1]->addArg(XtNbottom, XtChainBottom);
		toolbars[1]->forceArg(NULL, (XtArgVal )NULL);
	}
	if(statusbar)
	{	if(toolbars[3])
			statusbar->addArg(XtNfromVert, (XtArgVal )toolbars[3]->handle());
		else
			statusbar->addArg(XtNfromVert, (XtArgVal )panes->handle());
		statusbar->addArg(XtNleft, XtChainLeft);
		statusbar->addArg(XtNright, XtChainRight);
		statusbar->addArg(XtNtop, XtChainBottom);
		statusbar->addArg(XtNbottom, XtChainBottom);
		statusbar->forceArg(NULL, (XtArgVal )NULL);
	}
	if(toolbars[2])
		panes->addArg(XtNfromVert, (XtArgVal )toolbars[2]->handle());
	else if(dropdown)
		panes->addArg(XtNfromVert, (XtArgVal )dropdown->handle());
	else
		panes->addArg(XtNfromVert, (XtArgVal )NULL);
	if(toolbars[0])
		panes->addArg(XtNfromHoriz, (XtArgVal )toolbars[0]->handle());
	else
		panes->addArg(XtNfromHoriz, (XtArgVal )NULL);
	panes->addArg(XtNleft, XtChainLeft);
	panes->addArg(XtNright, XtChainRight);
	panes->addArg(XtNtop, XtChainTop);
	panes->addArg(XtNbottom, XtChainBottom);
	panes->forceArg(NULL, (XtArgVal )NULL);
}

#else

char* attachmentNames[] =	{ XmNleftAttachment, XmNrightAttachment, XmNtopAttachment, XmNbottomAttachment };
char* widgetNames[] =		{ XmNleftWidget, XmNrightWidget, XmNtopWidget, XmNbottomWidget };

void XmWindow::checkLayout()
{
	if(toolbars[2])
	{	toolbars[2]->addArg(XmNtopAttachment, dropdown ? XmATTACH_WIDGET : XmATTACH_FORM);
		if(dropdown)
			toolbars[2]->addArg(XmNtopWidget, (XtArgVal )dropdown->handle());
		toolbars[2]->addArg(XmNleftAttachment, XmATTACH_FORM);
		toolbars[2]->addArg(XmNrightAttachment, XmATTACH_FORM);
		toolbars[2]->addArg(XmNbottomAttachment, XmATTACH_NONE);
		toolbars[2]->addArg(XmNtopOffset, 0);	// why...?
		toolbars[2]->forceArg(NULL, (XtArgVal )NULL);
	}
	if(toolbars[3])
	{	toolbars[3]->addArg(XmNbottomAttachment, statusbar ? XmATTACH_WIDGET : XmATTACH_FORM);
		if(statusbar)
			toolbars[3]->addArg(XmNbottomWidget, (XtArgVal )statusbar->handle());
		toolbars[3]->addArg(XmNleftAttachment, XmATTACH_FORM);
		toolbars[3]->addArg(XmNrightAttachment, XmATTACH_FORM);
		toolbars[3]->addArg(XmNtopAttachment, XmATTACH_NONE);
		toolbars[3]->forceArg(NULL, (XtArgVal )NULL);
	}
	if(toolbars[0])
	{	toolbars[0]->addArg(XmNtopAttachment, ((toolbars[2] || dropdown) ? XmATTACH_WIDGET : XmATTACH_FORM));
		if((toolbars[2]) || dropdown)
			toolbars[0]->addArg(XmNtopWidget, (XtArgVal )(toolbars[2] ? toolbars[2]->handle() : dropdown->handle()));
		toolbars[0]->addArg(XmNbottomAttachment, (toolbars[3] || statusbar) ? XmATTACH_WIDGET : XmATTACH_FORM);
		if(toolbars[3] || statusbar)
			toolbars[0]->addArg(XmNbottomWidget, (XtArgVal )(toolbars[3] ? toolbars[3]->handle() : statusbar->handle()));
		toolbars[0]->addArg(XmNleftAttachment, XmATTACH_FORM);
		toolbars[0]->addArg(XmNrightAttachment, XmATTACH_NONE);
		toolbars[0]->addArg(XmNtopOffset, 0);	// why...?
		toolbars[0]->forceArg(NULL, (XtArgVal )NULL);
	}
	if(toolbars[1])
	{	toolbars[1]->addArg(XmNtopAttachment, (toolbars[2] || dropdown) ? XmATTACH_WIDGET : XmATTACH_FORM);
		if(toolbars[2] || dropdown)
			toolbars[1]->addArg(XmNtopWidget, (XtArgVal )(toolbars[2] ? toolbars[2]->handle() : dropdown->handle()));
		toolbars[1]->addArg(XmNbottomAttachment, (toolbars[3] || statusbar) ? XmATTACH_WIDGET : XmATTACH_FORM);
		if(toolbars[3] || statusbar)
			toolbars[1]->addArg(XmNbottomWidget, (XtArgVal )(toolbars[3] ? toolbars[3]->handle() : statusbar->handle()));
		toolbars[1]->addArg(XmNrightAttachment, XmATTACH_FORM);
		toolbars[1]->addArg(XmNleftAttachment, XmATTACH_NONE);
		toolbars[1]->addArg(XmNtopOffset, 0);	// why...?
		toolbars[1]->forceArg(NULL, (XtArgVal )NULL);
	}
	panes->addArg(XmNtopAttachment, (toolbars[2] || dropdown) ? XmATTACH_WIDGET : XmATTACH_FORM);
	if(toolbars[2] || dropdown)
		panes->addArg(XmNtopWidget, (XtArgVal )(toolbars[2] ? toolbars[2]->handle() : dropdown->handle()));
	panes->addArg(XmNbottomAttachment, (toolbars[3] || statusbar) ? XmATTACH_WIDGET : XmATTACH_FORM);
	if(toolbars[3] || statusbar)
		panes->addArg(XmNbottomWidget, (XtArgVal )(toolbars[3] ? toolbars[3]->handle() : statusbar->handle()));
	panes->addArg(XmNleftAttachment, (toolbars[0]) ? XmATTACH_WIDGET : XmATTACH_FORM);
	if(toolbars[0])
		panes->addArg(XmNleftWidget, (XtArgVal )toolbars[0]->handle());
	panes->addArg(XmNrightAttachment, (toolbars[1]) ? XmATTACH_WIDGET : XmATTACH_FORM);
	if(toolbars[1])
		panes->addArg(XmNrightWidget, (XtArgVal )toolbars[1]->handle());
	panes->forceArg(NULL, (XtArgVal )NULL);
}

#endif

#include "xmDrawing.h"

bool XmWindow::addSubpane(ctrlType ct, char* n, char* l, int y, int h, XmObject* rec, cbProc cb)
{
	XmControl* aControl;

	switch(ct)
	{	case Edit:
		aControl = new XmEdit(n, 0, y, 10, h, XmSautovscroll | XmSautohscroll);
		break;
		case ListBox:
		aControl = new XmListBox(n, 0, y, 10, h, XmSautovscroll | XmSautohscroll);
		break;
		case Drawing:
		aControl = new XmDrawing(n, 0, y, 10, h, XmSautovscroll | XmSautohscroll);
		break;
		case GroupBox: case StaticText: case StaticImage: case PushButton:
		case CheckBox: case RadioButton: case ScrollBar: case ComboBox: case Pane: 
		case PaneArea: case ToolBar:
		cerr << "xmWindow: Warning - control type " << ct << " not supported as subpane, ignoring...\n"; 
		return(FALSE);
	}
	return(addSubpane(aControl, l, rec, cb));
}

bool XmWindow::addSubpane(XmControl* c, char* l, XmObject* rec, cbProc cbk)
{
	if(!panes->numSubpanes())
		checkLayout();

	return(panes->addSubpane(c, l, rec, cbk));
}

XmToolBar* XmWindow::addToolbar(xmStyle s)
{
	char tmpstrbuf[1024];
	ostrstream tbName(tmpstrbuf, 1024);
	int ndx = orNdx(s);
	XmToolBar* aToolbar;

	if(toolbars[ndx])
		return(NULL);
	tbName << wname << "-Toolbar-" << ndx; tbName.put('\0');
	toolbars[ndx] = aToolbar = new XmToolBar(tbName.str(), (s == XmSleft || s == XmSright) ? XmSvertical : XmShorizontal);
	add(aToolbar);
	checkLayout();
	return(aToolbar);
}

bool XmWindow::removeToolbar(XmToolBar* tb)
{
	int ndx = -1;

	for(int i = 0; i < 4; i++)
	{	if(toolbars[i] == tb)
		{	ndx = i;
			break;
		}
	}
	if(ndx == -1)
		return(FALSE);
	toolbars[ndx] = NULL;
	checkLayout();
	remove((char* )tb->getName());
	return(TRUE);
}

XmToolBar* XmWindow::toolbar(xmStyle s)
{
	return(toolbars[orNdx(s)]);
}

XmStatusBar* XmWindow::createStatusBar()
{
	char tmpstrbuf[1024];
	ostrstream stbName(tmpstrbuf, 1024);

	if(statusbar)
		return(NULL);

	stbName << wname << "-Statusbar"; stbName.put('\0');
	statusbar = new XmStatusBar(stbName.str());
#ifdef XAW
	statusbar->addArg(XtNleft, XtChainLeft);
	statusbar->addArg(XtNright, XtChainRight);
	statusbar->addArg(XtNbottom, XtChainBottom);
#else
	statusbar->addArg(XmNleftAttachment, XmATTACH_FORM);
	statusbar->addArg(XmNrightAttachment, XmATTACH_FORM);
	statusbar->addArg(XmNbottomAttachment, XmATTACH_FORM);
#endif
	add(statusbar);
	checkLayout();
	return(statusbar);
}

#ifndef __GNUG__

#define tbWinStyle (XmSdlgModeless | XmSbordered | XmSmoveable | XmScloseable | XmStitled | XmSsysMenu)

XmToolBox::XmToolBox(char* n, XmObject* par, xmStyle s, int r) : XmDialogWindow(n, par, tbWinStyle), XmToolBar(n, s)
{
	setRows(r);
	XmDialogWindow::add(this);
}

XmToolBox::XmToolBox(char* n, XmObject* par, XmManager* m, xmStyle s, int r) : XmDialogWindow(n, par, m, tbWinStyle), XmToolBar(n, s)
{
	setRows(r);
	XmDialogWindow::add(this);
}

#endif
