/* ************************************************************************ 
 *         The Amulet User Interface Development Environment              *
 * ************************************************************************
 * This code was written as part of the Amulet project at                 *
 * Carnegie Mellon University, and has been placed in the public          *
 * domain.  If you are using this code or any part of Amulet,             *
 * please contact amulet@cs.cmu.edu to be put on the mailing list.        *
 * ************************************************************************/

extern "C" {
#include <stdlib.h>
}

#include <am_inc.h>

#include FORMULA_ADVANCED__H
#include STANDARD_SLOTS__H

#include FORMULA__H
#include STDVALUE__H
#include REGISTRY__H

typedef union {
  Am_FProc_Wrapper* f_wrapper;
  Am_FProc_Void*    f_void;
  Am_FProc_Int*     f_int;
  Am_FProc_Long*    f_long;
  Am_FProc_Bool*    f_bool;
  Am_FProc_Float*   f_float;
  Am_FProc_Double*  f_double;
  Am_FProc_Char*    f_char;
  Am_FProc_String*  f_string;
  Am_FProc_Proc*    f_proc;
  Am_FProc_Method*  f_method;
  Am_FProc_Value*   f_value;
  Am_FProc_Value_P* f_value_p;
} F_Type;

typedef enum { F_WRAPPER, F_VOID, F_INT, F_LONG, F_BOOL, F_FLOAT, F_DOUBLE,
	       F_CHAR, F_STRING, F_PROC, F_METHOD, F_VALUE, F_VALUE_P
} Formula_Type;

class Formula_Constraint;

#ifdef MEMORY
#include "dynarray.h"
#endif

class Dependency {
#ifdef MEMORY
 public:
  void* operator new (size_t)
  {
     return memory.New ();
  }
  void operator delete (void* ptr, size_t)
  {
    memory.Delete (ptr);
  }
  static Dyn_Memory_Manager memory;
#endif
 public:
  Am_Slot depended;
  Am_Constraint_Tag dependency_tag;
  Dependency* next;

  void Move_Back_Dep (Am_Slot& dest_slot, Am_Constraint* context);
  void Remove ();
};

#ifdef MEMORY
Dyn_Memory_Manager Dependency::memory (sizeof (Dependency));
#endif

void Dependency::Move_Back_Dep (Am_Slot& dest_slot,
				Am_Constraint* context)
{
  Remove ();
  depended = dest_slot;
  dependency_tag = dest_slot.Add_Dependency (context);
}

void Dependency::Remove ()
{
  if (depended)
    depended.Remove_Dependency (dependency_tag);
}

class Formula_Context : public Am_Constraint_Context {
 public:
  Am_ID_Tag ID ();
  Am_Slot Get (const Am_Object_Advanced& object, Am_Slot_Key key);
  Am_Object Get_Owner (Am_Object_Advanced object);
  Am_Object Get_Part (Am_Object_Advanced object, Am_Slot_Key key);
  void Set (Am_Object_Advanced object, Am_Slot_Key key,
            const Am_Value& new_value);
  void Note_Changed (Am_Object_Advanced, Am_Slot_Key)
  { changed = true; }
  void Note_Unchanged (Am_Object_Advanced, Am_Slot_Key)
  { changed = false; }
  void Note_Uninitialized ()
  { inited = false; }
  void* Get_Data ();
  void Set_Data (void* data);
  
  Am_Slot depender;
  Dependency *last_position;
  Dependency *current_position;
  Formula_Constraint* context;
  bool changed, inited;
  bool return_value_from_get;

  Formula_Context (Formula_Constraint* context, const Am_Slot& in_depender,
		   Dependency* depends_on);
  void Clean_Up (Dependency **depends_on);
};

typedef struct {
  void* data;
  Am_Data_Release* proc;
} Data_Store;

class Am_Formula_Data {
 public:
  Am_Formula_Data (Formula_Type in_type, const char* in_name)
  {
    type = in_type;
    name = in_name;
  }

  F_Type form_data;
  Formula_Type type;
  const char* name;
};

class Formula_Constraint : public Am_Formula_Advanced {
#ifdef MEMORY
 public:
  void* operator new (size_t)
  {
     return memory.New ();
  }
  void operator delete (void* ptr, size_t)
  {
    memory.Delete (ptr);
  }
  static Dyn_Memory_Manager memory;
#endif
 public:
  Formula_Constraint (Am_Formula_Data* in_data);
  Formula_Constraint (Formula_Constraint* proto_formula);

  ~Formula_Constraint ()
  {
    if (data) {
      if (data->proc)
        data->proc (data->data);
      delete data;
    }
  }

  bool Get (const Am_Slot& fetching_slot, Am_Value& value, bool& changed);
  
  void Invalidated (const Am_Slot& slot_invalidated,
		    Am_Constraint* invalidating_constraint,
		    const Am_Value& value);
  void Changed (const Am_Slot& slot_changed,
		Am_Constraint* changing_constraint,
		const Am_Value& prev_value, const Am_Value& new_value);
  void Changed (const Am_Slot& slot_changed,
		Am_Constraint* changing_constraint);
  void Slot_Event (Am_Object_Context* oc, const Am_Slot& slot);
  Am_Constraint* Get_Prototype ();
  bool Is_Instance_Of (Am_Constraint* proto);
  
  Am_Constraint* Constraint_Added (const Am_Slot& adding_slot);
  Am_Constraint* Dependency_Added (const Am_Slot& adding_slot);
  void Constraint_Removed (const Am_Slot& removing_slot);
  void Dependency_Removed (const Am_Slot& removing_slot);

  Am_Constraint* Create (const Am_Slot& current_slot,
			 const Am_Slot& new_slot);
  Am_Constraint* Copy (const Am_Slot& current_slot,
		       const Am_Slot& new_slot);
  Am_ID_Tag ID () const;
  const char* Get_Name();

  Am_Formula_Mode Get_Mode ();
  void Set_Mode (Am_Formula_Mode mode);
  Am_Slot Get_Context ();
  void Set_Data (void* data, Am_Data_Release* release_proc);
  
  static Am_ID_Tag id;
  
  Formula_Type type;
  bool internal_remove;
  unsigned short flags;
  Am_Slot context;
  F_Type formula;
  Dependency* depends_on;
  Data_Store* data;
  Formula_Constraint* prototype;
  Formula_Constraint* first_instance;
  Formula_Constraint* next_instance;
#ifdef DEBUG
  const char* formula_name;
#endif
  };

#ifdef MEMORY
Dyn_Memory_Manager Formula_Constraint::memory (sizeof (Formula_Constraint));
#endif

Am_Depends_Iterator::Am_Depends_Iterator ()
{
  context = NULL;
  current = NULL;
}

Am_Depends_Iterator::Am_Depends_Iterator (const Am_Formula_Advanced* formula)
{
  context = formula;
  current = NULL;
}

Am_Depends_Iterator& Am_Depends_Iterator::operator=
    (const Am_Formula_Advanced* formula)
{
  context = formula;
  current = NULL;
  return *this;
}

unsigned short Am_Depends_Iterator::Length () const
{
  if (context) {
    Dependency* curr;
    unsigned short count = 0;
    for (curr = ((Formula_Constraint*)context)->depends_on; curr != NULL;
	 curr = curr->next)
      ++count;
    return count;
  }
  else
    return 0;
}

void Am_Depends_Iterator::Start ()
{
  if (context)
    current = (Am_Dependency_Data*)((Formula_Constraint*)context)->depends_on;
}
  
void Am_Depends_Iterator::Next ()
{
  if (current)
    current = (Am_Dependency_Data*)((Dependency*)current)->next;
  else if (context)
    current = (Am_Dependency_Data*)((Formula_Constraint*)context)->depends_on;
  while (current && !(((Dependency*)current)->depended))
    current = (Am_Dependency_Data*)((Dependency*)current)->next;
}
				    
bool Am_Depends_Iterator::Last () const
{
  return !current;
}

Am_Slot Am_Depends_Iterator::Get () const
{
  if (current)
    return ((Dependency*)current)->depended;
  else
    return NULL;
}

Formula_Context::Formula_Context (Formula_Constraint* in_context,
				  const Am_Slot& in_depender,
				  Dependency* depends_on)
  : depender (in_depender)
{
  inited = true;
  context = in_context;
  last_position = NULL;
  current_position = depends_on;
  changed = false;
  return_value_from_get = true;
}

void Formula_Context::Clean_Up (Dependency **depends_on)
{
  if (current_position != NULL) {
    if (last_position)
      last_position->next = NULL;
    else
      *depends_on = NULL;
    Dependency* current = current_position;
    Dependency* free_node;
    context->internal_remove = true;
    while (current) {
      free_node = current;
      current = current->next;
      free_node->Remove ();
      free_node->depended = Am_Slot(NULL);
      free_node->next = NULL;
      delete free_node;
    }
    context->internal_remove = false;
  }
}

Am_ID_Tag Formula_Constraint::id =
    Am_Get_Unique_ID_Tag("Formula_Constraint", Am_CONSTRAINT);

const char* Formula_Constraint::Get_Name () {
#ifdef DEBUG
  Formula_Constraint* me = (Formula_Constraint*)this;
  return me->formula_name;
#else
  return NULL; //no name if not debugging
#endif  
}

// Primary constructor for Formula_Constraint, makes one from a
// procedure and a type, and an optional name
Formula_Constraint::Formula_Constraint (Am_Formula_Data* in_data)
    : context (Am_NULL_SLOT)
{
  formula = in_data->form_data;
  type = in_data->type;
#ifdef DEBUG
  formula_name = in_data->name;
#endif  
  
  flags = Am_FORMULA_ONE_PER_SLOT;
  //  context = NULL;
  data = NULL;
  depends_on = NULL;
  prototype = NULL;
  first_instance = NULL;
  next_instance = NULL;
  internal_remove = false;
}

Formula_Constraint::Formula_Constraint (Formula_Constraint* proto_formula)
{
  formula = proto_formula->formula;
  type = proto_formula->type;
  flags = proto_formula->flags;
#ifdef DEBUG
  formula_name = proto_formula->formula_name;
#endif  
  
  if (proto_formula->data) {
    data = new Data_Store;
    data->data = proto_formula->data->data;
    data->proc = proto_formula->data->proc;
  }
  else
    data = NULL;
  depends_on = NULL;
  prototype = NULL;
  first_instance = NULL;
  next_instance = NULL;
  internal_remove = false;
}

bool Formula_Constraint::Get (const Am_Slot& fetching_slot, Am_Value& value,
			      bool& changed)
{
  Formula_Context fc (this, fetching_slot, depends_on);

  Am_Object owner;
  owner = context.Get_Owner ();
  switch (type) {
  case F_WRAPPER:
    value = formula.f_wrapper (fc, owner);
    break;
  case F_VOID:
    value = formula.f_void (fc, owner);
    break;
  case F_INT:
    value = formula.f_int (fc, owner);
    break;
  case F_LONG:
    value = formula.f_long (fc, owner);
    break;
  case F_BOOL:
    value = formula.f_bool (fc, owner);
    break;
  case F_FLOAT:
    value = formula.f_float (fc, owner);
    break;
  case F_DOUBLE:
  {
    double hold = formula.f_double (fc, owner);
    if (!fc.changed)
      value = hold;
    break;
  }
  case F_CHAR:
    value = formula.f_char (fc, owner);
    break;
  case F_STRING:
  {
    const char* hold = formula.f_string (fc, owner);
    if (!fc.changed)
      value = hold;
    break;
  }
  case F_PROC:
    value = formula.f_proc (fc, owner);
    break;
  case F_METHOD:
    value = formula.f_method (fc, owner);
    break;
  case F_VALUE:
    value = formula.f_value (fc, owner);
    break;
  case F_VALUE_P:
    formula.f_value_p (fc, owner, value);
  }
  fc.Clean_Up (&depends_on);
  changed = fc.changed;
  if (!fc.inited)
    value.Set_Uninitialized ();
  return fc.return_value_from_get;
}
     
Am_Constraint* Formula_Constraint::Get_Prototype ()
{
  return prototype;
}

bool Formula_Constraint::Is_Instance_Of (Am_Constraint* proto)
{
  Formula_Constraint* current = this;
  while (current) {
    if (current == proto)
      return true;
    current = current->prototype;
  }
  return false;
}

void Formula_Constraint::Invalidated (const Am_Slot&, Am_Constraint*,
				      const Am_Value&)
{
  if (!(flags & Am_FORMULA_DO_NOT_PROPAGATE) && context)
   context.Invalidate (this);
}
     
void Formula_Constraint::Changed (const Am_Slot&, Am_Constraint* /*cause*/,
				  const Am_Value&, const Am_Value&)
{
  if (!(flags & Am_FORMULA_DO_NOT_PROPAGATE) && context /*&& !cause*/)
    context.Invalidate (this);
}

void Formula_Constraint::Changed (const Am_Slot&, Am_Constraint* /*cause*/)
{
  if (!(flags & Am_FORMULA_DO_NOT_PROPAGATE) && context /*&& !cause*/)
    context.Invalidate (this);
}

void Formula_Constraint::Slot_Event (Am_Object_Context* oc,
				     const Am_Slot&)
{
  context.Event (oc);
}

Am_Constraint* Formula_Constraint::Constraint_Added
     (const Am_Slot& adding_slot)
{
  if (flags & Am_FORMULA_KEEP_SOLITARY) {
    Am_Constraint_Iterator iter = adding_slot;
    Am_Constraint* constraint;
    Am_Constraint_Tag tag;
    iter.Start ();
    while (!iter.Last ()) {
      constraint = iter.Get ();
      tag = iter.Get_Tag ();
      iter.Next ();
      if (constraint != this)
        adding_slot.Remove_Constraint (tag);
    }
  }
  context = adding_slot;
  if (flags & Am_FORMULA_ONE_PER_SLOT) {
    Am_Constraint_Iterator iter = adding_slot;
    iter.Start ();
    Am_Constraint* constraint;
    Am_Constraint_Tag tag;
    Formula_Constraint* current;
    while (!iter.Last ()) {
      constraint = iter.Get ();
      if ((constraint->ID () == id) && (constraint != this)) {
        current = (Formula_Constraint*)constraint;
        if (current->prototype == prototype) {
          Formula_Constraint* curr_con = current->first_instance;
          Formula_Constraint* next_con;
          while (curr_con) {
            next_con = curr_con->next_instance;
            curr_con->prototype = this;
            curr_con->next_instance = first_instance;
            first_instance = curr_con;
            curr_con = next_con;
          }
          if (prototype) {
            Formula_Constraint* prev_con = NULL;
            curr_con = prototype->first_instance;
            while (curr_con) {
              if (curr_con == current) {
                if (prev_con)
                  prev_con->next_instance = curr_con->next_instance;
                else
                  prototype->first_instance = curr_con->next_instance;
                break;
              }
              prev_con = curr_con;
              curr_con = curr_con->next_instance;
            }
          }
          current->prototype = NULL;
          current->first_instance = NULL;
          current->next_instance = NULL;
          tag = iter.Get_Tag ();
          adding_slot.Remove_Constraint (tag);
          break;
        }
      }
      iter.Next ();
    }
  }
  return this;
}

Am_Constraint* Formula_Constraint::Dependency_Added (const Am_Slot&)
{
  return this;
}

void Formula_Constraint::Constraint_Removed (const Am_Slot&)
{
  Dependency* current;
  Dependency* free_node;

  context = Am_Slot(NULL);
  current = depends_on;
  internal_remove = true;
  while (current != NULL) {
    free_node = current;
    current = current->next;
    free_node->next = NULL;
    free_node->Remove ();
    delete free_node;
  }
  depends_on = NULL;
  internal_remove = false;
  if (prototype) {
    Formula_Constraint* prev_con = NULL;
    Formula_Constraint* curr_con = prototype->first_instance;
    while (curr_con) {
      if (curr_con == this) {
        if (prev_con)
          prev_con->next_instance = curr_con->next_instance;
        else
          prototype->first_instance = curr_con->next_instance;
	curr_con->next_instance = NULL;
        break;
      }
      prev_con = curr_con;
      curr_con = curr_con->next_instance;
    }
  }
  prototype = NULL;
  Formula_Constraint* curr_con = first_instance;
  Formula_Constraint* next_con;
  while (curr_con) {
    next_con = curr_con->next_instance;
    curr_con->prototype = NULL;
    curr_con->next_instance = NULL;
    curr_con = next_con;
  }
  first_instance = NULL;
  delete this;
}

void Formula_Constraint::Dependency_Removed (const Am_Slot& removing_slot)
{
  if (!internal_remove) {
    Dependency* current;
    for ( current = depends_on; current != NULL; current = current->next) {
      if (current->depended == removing_slot) {
        current->depended = Am_Slot(NULL);
        break;
      }
    }
  }
}

Am_Constraint* Formula_Constraint::Create (const Am_Slot&,
					   const Am_Slot&)
{
  Formula_Constraint* new_formula = new Formula_Constraint (this);
  new_formula->next_instance = first_instance;
  first_instance = new_formula;
  new_formula->prototype = this;
  return new_formula;
}

Am_Constraint* Formula_Constraint::Copy (const Am_Slot&, const Am_Slot&)
{
  Formula_Constraint* new_formula = new Formula_Constraint (this);
  if (prototype) {
    new_formula->next_instance = prototype->first_instance;
    prototype->first_instance = new_formula;
    new_formula->prototype = prototype;
  }
  return new_formula;
}

Am_ID_Tag Formula_Constraint::ID () const
{
  return id;
}

Am_Formula_Mode Formula_Constraint::Get_Mode ()
{
  return flags;
}

void Formula_Constraint::Set_Mode (Am_Formula_Mode mode)
{
  flags = mode;
}

Am_Slot Formula_Constraint::Get_Context ()
{
  return context;
}

void Formula_Constraint::Set_Data (void* in_data, Am_Data_Release* in_proc)
{
  if (!data)
    data = new Data_Store;
  data->data = in_data;
  data->proc = in_proc;
}

Am_Formula_Advanced* Am_Formula_Advanced::Narrow (Am_Constraint* value)
{
  if (Formula_Constraint::id == value->ID ())
    return (Am_Formula_Advanced*)value;
  else {
    Am_Error ("Constraint narrowed to Formula type is not a Formula.");
    return NULL;
  }
}

bool Am_Formula_Advanced::Test (Am_Constraint* value)
{
  if (Formula_Constraint::id == value->ID ())
    return true;
  else return false;
}

Am_Formula::Am_Formula (Am_FProc_Wrapper* formula, const char* name)
{
  data = new Am_Formula_Data (F_WRAPPER, name);
  data->form_data.f_wrapper = formula;
}

Am_Formula::Am_Formula (Am_FProc_Void* formula, const char* name)
{
  data = new Am_Formula_Data (F_VOID, name);
  data->form_data.f_void = formula;
}

Am_Formula::Am_Formula (Am_FProc_Int* formula, const char* name)
{
  data = new Am_Formula_Data (F_INT, name);
  data->form_data.f_int = formula;
}

Am_Formula::Am_Formula (Am_FProc_Long* formula, const char* name)
{
  data = new Am_Formula_Data (F_LONG, name);
  data->form_data.f_long = formula;
}

#if !defined(NEED_BOOL)
Am_Formula::Am_Formula (Am_FProc_Bool* formula, const char* name)
{
  data = new Am_Formula_Data (F_BOOL, name);
  data->form_data.f_bool = formula;
}
#endif

Am_Formula::Am_Formula (Am_FProc_Float* formula, const char* name)
{
  data = new Am_Formula_Data (F_FLOAT, name);
  data->form_data.f_float = formula;
}

Am_Formula::Am_Formula (Am_FProc_Double* formula, const char* name)
{
  data = new Am_Formula_Data (F_DOUBLE, name);
  data->form_data.f_double = formula;
}

Am_Formula::Am_Formula (Am_FProc_Char* formula, const char* name)
{
  data = new Am_Formula_Data (F_CHAR, name);
  data->form_data.f_char = formula;
}

Am_Formula::Am_Formula (Am_FProc_String* formula, const char* name)
{
  data = new Am_Formula_Data (F_STRING, name);
  data->form_data.f_string = formula;
}

Am_Formula::Am_Formula (Am_FProc_Proc* formula, const char* name)
{
  data = new Am_Formula_Data (F_PROC, name);
  data->form_data.f_proc = formula;
}

Am_Formula::Am_Formula (Am_FProc_Method* formula, const char* name)
{
  data = new Am_Formula_Data (F_METHOD, name);
  data->form_data.f_method = formula;
}

Am_Formula::Am_Formula (Am_FProc_Value* formula, const char* name)
{
  data = new Am_Formula_Data (F_VALUE, name);
  data->form_data.f_value = formula;
}

Am_Formula::Am_Formula (Am_FProc_Value_P* formula, const char* name)
{
  data = new Am_Formula_Data (F_VALUE_P, name);
  data->form_data.f_value_p = formula;
}

Am_Formula::~Am_Formula ()
{
  delete data;
}

const Am_Value Am_Formula::operator () (Am_Constraint_Context& cc,
					Am_Object& context) const
{
  Am_Value temp;
  switch (data->type) {
  case F_WRAPPER:
    temp = data->form_data.f_wrapper (cc, context);
    break;
  case F_VOID:
    temp = data->form_data.f_void (cc, context);
    break;
  case F_INT:
    temp = data->form_data.f_int (cc, context);
    break;
  case F_LONG:
    temp = data->form_data.f_long (cc, context);
    break;
  case F_BOOL:
    temp = data->form_data.f_bool (cc, context);
    break;
  case F_FLOAT:
    temp = data->form_data.f_float (cc, context);
    break;
  case F_DOUBLE:
    temp = data->form_data.f_double (cc, context);
    break;
  case F_CHAR:
    temp = data->form_data.f_char (cc, context);
    break;
  case F_STRING:
    temp = data->form_data.f_string (cc, context);
    break;
  case F_PROC:
    temp = data->form_data.f_proc (cc, context);
    break;
  case F_METHOD:
    temp = data->form_data.f_method (cc, context);
    break;
  case F_VALUE:
    temp = data->form_data.f_value (cc, context);
    break;
  case F_VALUE_P:
    data->form_data.f_value_p (cc, context, temp);
    break;
  }
  return temp;
}

Am_Formula::operator Am_Constraint* ()
{
  return new Formula_Constraint (data);
}

Am_ID_Tag Am_Formula::ID () const
{
  return Formula_Constraint::id;
}

Am_ID_Tag Formula_Context::ID ()
{
  return Formula_Constraint::id;
}

Am_Slot Formula_Context::Get (const Am_Object_Advanced& object, Am_Slot_Key key)
{
  Am_Slot slot;
  
  if (current_position) {
    slot = current_position->depended;
    if (slot && (slot.Get_Owner () == object) && (slot.Get_Key () == key)) {
      last_position = current_position;
      current_position = current_position->next;
      return slot;
    }
  }
  slot = object.Get_Slot (key);
  if (current_position) {
    context->internal_remove = true;
    current_position->Move_Back_Dep (slot, context);
    context->internal_remove = false;
    last_position = current_position;
    current_position = current_position->next;
  }
  else {
    Dependency* new_dep = new Dependency ();

    new_dep->depended = slot;
    new_dep->dependency_tag = slot.Add_Dependency (context);
    new_dep->next = NULL;
    if (last_position)
      last_position->next = new_dep;
    else
      context->depends_on = new_dep;
    last_position = new_dep;
  }
  return slot;
}

Am_Object Formula_Context::Get_Owner (Am_Object_Advanced object)
{
  Am_Slot slot;

  if (current_position) {
    slot = current_position->depended;
    if (slot && (slot.Get_Owner () == object) &&
        (slot.Get_Key () == Am_OWNER)) {
      last_position = current_position;
      current_position = current_position->next;
      return slot.Get ();
    }
  }
  slot = object.Get_Owner_Slot ();
  if (current_position) {
    context->internal_remove = true;
    current_position->Move_Back_Dep (slot, context);
    context->internal_remove = false;
    last_position = current_position;
    current_position = current_position->next;
  }
  else {
    Dependency* new_dep = new Dependency;

    new_dep->depended = slot;
    new_dep->dependency_tag = slot.Add_Dependency (context);
    new_dep->next = NULL;
    if (last_position)
      last_position->next = new_dep;
    else
      context->depends_on = new_dep;
    last_position = new_dep;
  }
  return object.Get_Owner ();
}

Am_Object Formula_Context::Get_Part (Am_Object_Advanced object,
				      Am_Slot_Key key)
{
  Am_Slot slot;

  if (current_position) {
    slot = current_position->depended;
    if (slot && (slot.Get_Owner () == object) && (slot.Get_Key () == key)) {
      last_position = current_position;
      current_position = current_position->next;
      if (slot.Get_Type () == Am_OBJECT)
	return slot.Get ();
      else
	return 0;
    }
  }
  Am_Object part_obj = object.Get_Part (key);
  Am_Object_Advanced part = (Am_Object_Advanced&)part_obj;
  if (part.Valid ())
    slot = part.Get_Part_Slot ();
  else
    slot = object.Get_Slot (key);
  if (current_position) {
    context->internal_remove = true;
    current_position->Move_Back_Dep (slot, context);
    context->internal_remove = false;
    last_position = current_position;
    current_position = current_position->next;
  }
  else {
    Dependency* new_dep = new Dependency;

    new_dep->depended = slot;
    new_dep->dependency_tag = slot.Add_Dependency (context);
    new_dep->next = NULL;
    if (last_position)
      last_position->next = new_dep;
    else
      context->depends_on = new_dep;
    last_position = new_dep;
  }
  return part;
}

void Formula_Context::Set (Am_Object_Advanced object, Am_Slot_Key key,
			   const Am_Value& value)
{
  if ((object == depender.Get_Owner ()) && (key == depender.Get_Key ())) {
    changed = true;
    return_value_from_get = false;
    depender.Set (value, context);
  }
  else {
    Am_Error ("** SV called in a formula where the destination is not the "
	      "slot the formula is contained in.");
  }
}
           
void* Formula_Context::Get_Data ()
{
  if (context->data)
    return context->data->data;
  else
    return NULL;
}

void Formula_Context::Set_Data (void* data)
{
  if (!context->data)
    context->data = new Data_Store;
  context->data->data = data;
}
