/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include <assert.h>
#include <cstring>

#define _TOAD_PRIVATE

#include <toad/toadbase.hh>
#include <toad/pen.hh>
#include <toad/window.hh>
#include <toad/region.hh>
#include <iostream>

// Copy TPoint array to XPoint array and add translation in (_dx, _dy).
#define TPOINT_2_XPOINT(SRC, DST, SIZE) \
	XPoint DST[n]; \
	const TPoint *sp = SRC; \
	const TPoint *se = SRC+SIZE; \
	XPoint *dp = DST; \
	while(sp!=se) { \
		dp->x = sp->x + _dx; \
		dp->y = sp->y + _dy; \
		dp++; \
		sp++; \
	}

// Adjust (width, height) to raster coordinates for XDraw... operations.
#define XDRAW_RASTER_COORD(w,h) \
	if (!w || !h) \
		return; \
	if (w<0) { \
		w=-w; \
		x-=w-1; \
	} \
	if (h<0) { \
		h=-h; \
		y-=h-1; \
	} \
	w--; \
	h--;

// Adjust (width, height) to pixel coordinates for XDraw... operations.
#define XDRAW_PIXEL_COORD(w,h) \
	if (w<0) { \
		w=-w; \
		x-=w; \
	} \
	if (h<0) { \
		h=-h; \
		y-=h; \
	}

// point
//----------------------------------------------------------------------------
void TPen::DrawPoint(int x, int y) const
{
  XDrawPoint(x11display, x11drawable, o_gc, x+_dx, y+_dy);
}

// line
//----------------------------------------------------------------------------
void TPen::DrawLine(int x1, int y1, int x2, int y2) const
{
  XDrawLine(x11display, x11drawable, o_gc, x1+_dx,y1+_dy, x2+_dx,y2+_dy);
}

void TPen::DrawLines(const TPoint *s, int n) const
{
	TPOINT_2_XPOINT(s,d,n)
	XDrawLines(x11display, x11drawable, o_gc, d, n, CoordModeOrigin);
}

// rectangle
//----------------------------------------------------------------------------
void TPen::DrawRectangle(int x, int y, int w, int h) const
{
	XDRAW_RASTER_COORD(w,h)
	if (w==0 || h==0) {
		XDrawLine(x11display, x11drawable, o_gc, x+_dx, y+_dy, x+_dx+w,y+_dy+h);
		return;
	}
  XDrawRectangle(x11display, x11drawable, o_gc, x+_dx,y+_dy,w,h);
}

void TPen::DrawRectanglePC(int x, int y, int w, int h) const
{
	XDRAW_PIXEL_COORD(w,h)
	XDrawRectangle(x11display, x11drawable, o_gc, x+_dx,y+_dy,w,h);
}

void TPen::FillRectangle(int x, int y, int w, int h) const
{
	XDRAW_RASTER_COORD(w,h)
  if (two_colors) {
    XFillRectangle(x11display, x11drawable, f_gc, x+_dx, y+_dy,w,h);
    XDrawRectangle(x11display, x11drawable, o_gc, x+_dx, y+_dy,w,h);
  } else {
    XFillRectangle(x11display, x11drawable, o_gc, x+_dx,y+_dy,w+1,h+1);
  }
}

void TPen::FillRectanglePC(int x, int y, int w, int h) const
{
	XDRAW_PIXEL_COORD(w,h)
  if (two_colors) {
    XFillRectangle(x11display, x11drawable, f_gc, x+_dx, y+_dy,w,h);
    XDrawRectangle(x11display, x11drawable, o_gc, x+_dx, y+_dy,w,h);
  } else {
    XFillRectangle(x11display, x11drawable, o_gc, x+_dx,y+_dy,w+1,h+1);
  }
}

// circle
//----------------------------------------------------------------------------
void TPen::DrawCircle(int x, int y, int w, int h) const
{
	XDRAW_RASTER_COORD(w,h)
	if (w==0 || h==0) {
		XDrawLine(x11display, x11drawable, o_gc, x+_dx, y+_dy, x+_dx+w,y+_dy+h);
		return;
	}
  XDrawArc(x11display, x11drawable, o_gc, x+_dx,y+_dy,w,h, 0,360*64);
}

void TPen::DrawCirclePC(int x, int y, int w, int h) const
{
	XDRAW_PIXEL_COORD(w,h)
	XDrawArc(x11display, x11drawable, o_gc, x+_dx,y+_dy,w,h, 0,360*64);
}

void TPen::FillCircle(int x, int y, int w, int h) const
{
	XDRAW_RASTER_COORD(w,h)
	XDRAW_PIXEL_COORD(w,h)
	XFillArc(x11display, x11drawable, two_colors ? f_gc : o_gc, x+_dx, y+_dy,w,h, 0,360*64);
	XDrawArc(x11display, x11drawable, o_gc, x+_dx, y+_dy,w,h, 0,360*64);
}

void TPen::FillCirclePC(int x, int y, int w, int h) const
{
	XDRAW_PIXEL_COORD(w,h)
	XFillArc(x11display, x11drawable, two_colors ? f_gc : o_gc, x+_dx, y+_dy,w,h, 0,360*64);
	XDrawArc(x11display, x11drawable, o_gc, x+_dx, y+_dy,w,h, 0,360*64);
}

// arc
//----------------------------------------------------------------------------
void TPen::DrawArc(int x, int y, int w, int h, double r1, double r2) const
{
	XDRAW_RASTER_COORD(w,h)
	if (w==0 || h==0) {
		XDrawLine(x11display, x11drawable, o_gc, x+_dx, y+_dy, x+_dx+w,y+_dy+h);
		return;
	}
  XDrawArc(x11display, x11drawable, o_gc, x+_dx,y+_dy,w,h, (int)(r1*64.0),(int)(r2*64.0));
}

void TPen::DrawArcPC(int x, int y, int w, int h, double r1, double r2) const
{
	XDRAW_PIXEL_COORD(w,h)
	XDrawArc(x11display, x11drawable, o_gc, x+_dx,y+_dy,w,h, (int)(r1*64.0),(int)(r2*64.0));
}

void TPen::FillArc(int x, int y, int w, int h, double r1, double r2) const
{
	XDRAW_RASTER_COORD(w,h)
	XDRAW_PIXEL_COORD(w,h)
	int i1=(int)(r1*64.0);
	int i2=(int)(r2*64.0);
	XFillArc(x11display, x11drawable, two_colors ? f_gc : o_gc, x+_dx, y+_dy,w,h, i1,i2);
	XDrawArc(x11display, x11drawable, o_gc, x+_dx, y+_dy,w,h, i1,i2);
}

void TPen::FillArcPC(int x, int y, int w, int h, double r1, double r2) const
{
	XDRAW_PIXEL_COORD(w,h)
	int i1=(int)(r1*64.0);
	int i2=(int)(r2*64.0);
	XFillArc(x11display, x11drawable, two_colors ? f_gc : o_gc, x+_dx, y+_dy,w,h, i1,i2);
	XDrawArc(x11display, x11drawable, o_gc, x+_dx, y+_dy,w,h, i1,i2);
}

// polygon
//----------------------------------------------------------------------------
void TPen::DrawPolygon(const TPoint points[], int n) const
{
  DrawLines(points, n);
  DrawLine(points[0].x,points[0].y,points[n-1].x,points[n-1].y);
}

void TPen::FillPolygon(const TPoint s[], int n) const
{
   TPOINT_2_XPOINT(s,d,n);
   XFillPolygon(x11display, x11drawable, two_colors? f_gc : o_gc, 
	 d, n, Nonconvex, CoordModeOrigin);
   XDrawLines(x11display, x11drawable, o_gc, 
       d, n, CoordModeOrigin);
   XDrawLine(x11display, x11drawable, o_gc,
      s[0].x,s[0].y,
      s[n-1].x,s[n-1].y);
}

// bitmap
//----------------------------------------------------------------------------
void TPen::DrawBitmap(int x, int y, const TBitmap* bmp) const
{
  bmp->DrawBitmap(this, x+_dx, y+_dy);
}

void TPen::DrawBitmap(int x, int y, const TBitmap& bmp) const
{
  bmp.DrawBitmap(this, x+_dx, y+_dy);
}

void TPen::DrawBitmap(int x, int y, const TBitmap* bmp, int ax, int ay, int aw, int ah) const
{
  bmp->DrawBitmap(this, x+_dx, y+_dy, ax,ay,aw,ah);
}

void TPen::DrawBitmap(int x, int y, const TBitmap& bmp, int ax, int ay, int aw, int ah) const
{
  bmp.DrawBitmap(this, x+_dx, y+_dy, ax,ay,aw,ah);
}

// 3D rectangle
//----------------------------------------------------------------------------
void TPen::Draw3DRectangle(int x, int y, int w, int h)
{
	TColor saved_color = o_color;
	
	TPoint p[3];
	SetColor(255,255,255);
	p[0].Set(x+1	,y+h-1);
	p[1].Set(x+w-1,y+h-1);
	p[2].Set(x+w-1,y);
	DrawLines(p,3);
	
	SetColor(192,192,192);
	p[0].Set(x+2	,y+h-2);
	p[1].Set(x+w-2,y+h-2);
	p[2].Set(x+w-2,y+1);
	DrawLines(p,3);

	SetColor(128,128,128);
	p[0].Set(x	  ,y+h-1);
	p[1].Set(x    ,y);
	p[2].Set(x+w-1,y);
	DrawLines(p,3);
	
	SetColor(0,0,0);
	p[0].Set(x+1  ,y+h-2);
	p[1].Set(x+1  ,y+1);
	p[2].Set(x+w-2,y+1);
	DrawLines(p,3);
	
	SetColor(saved_color);
}

// text string
//----------------------------------------------------------------------------
int TPen::TextWidth(const string &str) const
{
  return font->TextWidth(str.c_str());
}

int TPen::TextWidth(const char *str) const
{
  return font->TextWidth(str);
}

//. Width of 'str' when printed with the current font.
int TPen::TextWidth(const char *str, int len) const
{
  return font->TextWidth(str,len);
}

//. Ascent of the current font.
int TPen::Ascent() const
{
  return font->Ascent();
}

//. Descent of the current font.
int TPen::Descent() const
{
  return font->Descent();
}

//. Height of the current font.
int TPen::Height() const
{
  return font->Height();
}

//. Draw string `str'. <VAR>x</VAR>, <VAR>y</VAR> is the upper left
//. coordinate of the string.<BR>
//. DrawString is a little bit slower than FillString.
void TPen::DrawString(int x,int y, const string &str) const
{
  TPen::DrawString(x,y,str.c_str(),(int)str.size());
}

void TPen::DrawString(int x,int y, const char *str, int strlen) const
{
  if (!str)
    return;
  XDrawString(x11display, x11drawable, o_gc, x+_dx,y+_dy+Ascent(), str, strlen);
}

//. <B>Experimental</B><BR>
//. FillString will fill the background with the current back color when
//. drawing the string.<BR>
//. The back color can be set with SetBackColor.<BR>
//. Please note that FillString doesn't support color dithering and will
//. use the nearest color TOAD was able to allocate.<BR>
//. Maybe i'm going to rename this method into `PrintString' since
//. `FillString' is really a very idiotic name.
void TPen::FillString(int x,int y, const string &str) const
{
  TPen::FillString(x,y,str.c_str(),(int)str.size());
}

void TPen::FillString(int x,int y, const char *str, int strlen) const
{
  if (!str)
    return;
  XDrawImageString(x11display, x11drawable, o_gc, x+_dx,y+_dy+Ascent(), str, strlen);
}

// DrawTextWidth
//-------------------------------------------------------------------------
//. Draw string 'str' in multiple lines, reduce spaces between words to one 
//. an break lines to fit width. 'str' can contain '\n'.
int TPen::DrawTextWidth(int x,int y,const string &str, unsigned width) const
{
  x+=_dx;
  y+=_dy;
  const char* text=str.c_str();
  
  unsigned i;
  
  // 1st step: count words and lines
  unsigned word_count, min_lines;
  font->count_words_and_lines(text, &word_count, &min_lines);
  if (!word_count) return 0;
  
  // 2nd step: create a word list
  TFont::TWord* word = font->make_wordlist(text, word_count);
  
  // 3rd step: output
  unsigned blank_width = TextWidth(" ",1);
  unsigned line_len = 0;
  unsigned word_of_line = 1;
  
  for(i=0; i<word_count; i++)
    {
      if ((line_len+word[i].len>width && i!=0) || word[i].linefeeds)
	{
	  if (word[i].linefeeds)
	    y+=Height()*word[i].linefeeds;
	  else
	    y+=Height();
	  line_len = 0;
	  word_of_line = 0;
	}
      DrawString(x+line_len,y, word[i].pos, word[i].bytes);
      line_len+=word[i].len+blank_width;
      word_of_line++;
    }
  
  delete[] word;
  return y+Height();
}
