/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

#include <toad/toadbase.hh>
#include <toad/pen.hh>
#include <toad/window.hh>
#include <toad/radiobuttonbase.hh>

//! TRadioState
//. Coordinates several <A HREF="TRadioButton.html">radio buttons</A>.
//. <P>
//. A method being connected to <CODE>sigValueChanged</CODE> should use
//. a pointer to the radio state to distinguish between different buttons.<BR>
//. E.g.:
//. <PRE>
//. TMyWindow::TMyWindow() {
//.   TRadioState *state = new TRadioState();
//.   state->sigValueChanged.Add(this, action, state);
//.   new TRadioButton(this, "One",   state, 1);
//.   new TRadioButton(this, "Two",   state, 2);
//.   new TRadioButton(this, "Three", state, 3);
//. }
//.
//. void TMyWindow::action(TRadioState *state) {
//.   int id = state->Current()->Value();
//.   ...
//. }
//. </PRE>
//. <P>
//. Another way is to use the signals <CODE>sigActivate</CODE> and 
//. <CODE>sigDeactivate</CODE> of the buttons themselves:
//. <PRE>
//. TMyWindow::TMyWindow() {
//.   TRadioState *state = new TRadioState();
//.   TRadioButton *btn;
//.   btn = new TRadioButton(this, "One",   state);
//.   btn->sigActivate.Add(this, action, 1);
//.   btn = new TRadioButton(this, "Two",   state);
//.   btn->sigActivate.Add(this, action, 2);
//.   btn = new TRadioButton(this, "Three", state);
//.   btn->sigActivate.Add(this, action, 3);
//. }
//.
//. void TMyWindow::action(int id) {
//.   ...
//. }
//. </PRE>

TRadioState::TRadioState()
{
	_current = NULL;
}

void TRadioState::Register(TRadioButtonBase *rb)
{
	listener.push_back(rb);
}

void TRadioState::Unregister(TRadioButtonBase *rb)
{
//	TListenerBuffer::iterator p = listener.find(rb);
	TListenerBuffer::iterator p,e;
	p = listener.begin();
	e = listener.end();
	while(p!=e) {
		if (*p==rb) {
			listener.erase(p);
			return;
		}
		p++;
	}
}

void TRadioState::SetCurrent(TRadioButtonBase *btn)
{
	if (btn==_current)
		return;
	TRadioButtonBase *old = _current;
	_current = btn;
	if (old)
		old->_SetDown(false);
	if (_current)
		_current->_SetDown(true);
	sigValueChanged();
}

void TRadioState::SetValue(int value)
{
	TListenerBuffer::iterator p,e;
	p = listener.begin();
	e = listener.end();
	while(p!=e) {
		if ((*p)->Value()==value) {
			SetCurrent(*p);
			return;
		}
		p++;
	}
	cerr << "TRadioState: `" << value << "' no such value" << endl;
}

int TRadioState::Value()
{
	return _current ? _current->Value() : 0;
}

//! TRadioButtonBase
//. The base class for all radio buttons.
//. <P>
//. See <A HREF="TRadioState.html">TRadioState</A> for code examples.

// ClearGroupMember
//---------------------------------------------------------------------------
void TRadioButtonBase::_ClearGroupMember()
{
	TWindow *ptr=Parent();
	if (!ptr) return;
	ptr=ptr->FirstChild();
	while(ptr) {
		if(ptr!=this)	{
			TRadioButtonBase *rb = dynamic_cast<TRadioButtonBase*>(ptr);
			if (rb)	{
				if (rb->Group()==Group() && rb->bDown) {
					rb->SetDown(false);
					return;
				}
			}
		}
		ptr=TWindow::NextSibling(ptr);
	}
}

// mouseLDown
//---------------------------------------------------------------------------
void TRadioButtonBase::mouseLDown(int,int,unsigned)
{
	SetDown(true);
	SetFocus();
}

//. <I>Note: Only 'true' is implemented yet, which is the default value</I>
//---------------------------------------------------------------------------
void TRadioButtonBase::SetDown(bool down)
{
	SetFocus();
	if (_state) {
		_state->SetCurrent(this);
	} else {
		if (!IsDown()) {
			_ClearGroupMember();
			super::SetDown(true);
			sigActivate();
		}	else {
			super::SetDown(down);
		}
	}
}

void TRadioButtonBase::_SetDown(bool down)
{
	if (IsDown()==down)
		return;
	super::SetDown(down);
	if (down)
		sigActivate();
}

// keyDown
//---------------------------------------------------------------------------
void TRadioButtonBase::keyDown(TKey key, char* str, unsigned modifier)
{
	if (!bDown && modifier==0 && (key==TK_RETURN || *str==' ')) {
		_ClearGroupMember();
		SetDown(true);
		sigActivate();
	}
}

// SetValue
//----------------------------------------------------------------------------
//. <B>obsolete</B>
void TRadioButtonBase::SetValue(int v)
{
	TWindow *ptr = Parent();
	if (!ptr) return;
	ptr=ptr->FirstChild();
	while(ptr) {
		TRadioButtonBase *rb = dynamic_cast<TRadioButtonBase*>(ptr);
		if (rb && rb->Group()==Group())	{
			if (rb->IsDown() && rb->_value!=v) {
				rb->SetDown(false);
				rb->UpdateWindow(true);
			}	
			else if (!rb->IsDown() && rb->_value==v)
			{
				rb->SetDown(true);
				rb->UpdateWindow(true);
			}
		}
		ptr=NextSibling(ptr);
	}
}

// Value
//----------------------------------------------------------------------------
//. <B>obsolete</B>
int TRadioButtonBase::Value()
{
	TWindow *ptr = Parent();
	if (!ptr) return _value;
	ptr=ptr->FirstChild();
	while(ptr) {
		TRadioButtonBase *rb = dynamic_cast<TRadioButtonBase*>(ptr);
		if (rb && rb->Group()==Group() && rb->IsDown())	{
			return rb->_value;
		}
		ptr=NextSibling(ptr);
	}
	return _value;
}
