/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996,97 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

/*
 * TRegion violates the good programming style by accessing Xlibs' internal
 * Region structure for performance reasons.
 * The file "toad/X11/region.h" was part of XFree86 3.2 (X11R6.1) but
 * seems to do well with X11R5 on SUN also.
 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>

#include <toad/toadbase.hh>
#include <toad/region.hh>
#include <toad/X11/region.h>

TRegion::TRegion()
{
	TOAD_XLIB_MTLOCK();
	x11region = XCreateRegion();
//cout << "creating X11 region " << x11region << endl;
	TOAD_XLIB_MTUNLOCK();
}

void TRegion::Clear()
{
#warning	"'void TRegion::Clear()' is a dummy"
/*
	XDestroyRegion(x11region);
	region = XCreateRegion();
*/
	AddRect(0,0,4096,4096);	
}

TRegion::~TRegion()
{
	TOAD_XLIB_MTLOCK();
//cout << "destroying X11 region " << x11region << endl;
	XDestroyRegion(x11region);
	TOAD_XLIB_MTUNLOCK();
}


// move region by the specified amount
//---------------------------------------
void TRegion::Translate(int dx, int dy)
{
	TOAD_XLIB_MTLOCK();
	XOffsetRegion(x11region, dx, dy);
	TOAD_XLIB_MTUNLOCK();
}

// return the smallest rectangle containing the region
//-----------------------------------------------------
void TRegion::GetBoundary(TRectangle* r)
{
	XRectangle rect;
	TOAD_XLIB_MTLOCK();
	XClipBox(x11region, &rect);
	TOAD_XLIB_MTUNLOCK();
	
	r->x = rect.x;
	r->y = rect.y;
	r->w = rect.width;
	r->h = rect.height;
}

// return number of rectangles in the region
//-------------------------------------------
long TRegion::GetNumRects()
{
	return x11region->numRects;
}

// get rectangle 'n' from the region
//-----------------------------------
bool TRegion::GetRect(long n,TRectangle *r)
{
	if ( n<0 || n >= x11region->numRects)
		return false;
	r->x = x11region->rects[n].x1;
	r->y = x11region->rects[n].y1;
	r->w = x11region->rects[n].x2 - x11region->rects[n].x1;
	r->h = x11region->rects[n].y2 - x11region->rects[n].y1;
	return true;
}

void TRegion::operator&=(const TRegion &r)
{
	XIntersectRegion(x11region, r.x11region, x11region);
}

void TRegion::operator|=(const TRegion &r)
{
	XUnionRegion(x11region, r.x11region, x11region);
}

void TRegion::operator-=(const TRegion &r)
{
	XSubtractRegion(x11region, r.x11region, x11region);
}

void TRegion::operator^=(const TRegion &r)
{
	XXorRegion(x11region, r.x11region, x11region);
}

void TRegion::operator&=(const TRectangle &rect)
{
	TRegion r;
	r|=rect;
	XIntersectRegion(x11region, r.x11region, x11region);
}

void TRegion::operator|=(const TRectangle &r)
{
	XRectangle rect;
	rect.x=r.x;
	rect.y=r.y;
	rect.width=r.w;
	rect.height=r.h;
	XUnionRectWithRegion(&rect, x11region, x11region);
}

void TRegion::operator-=(const TRectangle &rect)
{
	TRegion r;
	r|=rect;
	XSubtractRegion(x11region, r.x11region, x11region);
}

void TRegion::operator^=(const TRectangle &rect)
{
	TRegion r;
	r|=rect;
	XXorRegion(x11region, r.x11region, x11region);
}

// add a rectangle to the region
//------------------------------------------------------
void TRegion::AddRect(const TRectangle &r)
{
	AddRect(r.x,r.y,r.w,r.h);
};

void TRegion::AddRect(int x,int y,int w,int h)
{
	if (w<0 || h<0)	{
		cerr << "TRegion::AddRect: warning illegal size\n";
		return;
	}
	XRectangle rect;
	rect.x=x;
	rect.y=y;
	rect.width=w;
	rect.height=h;
	XUnionRectWithRegion(&rect, x11region, x11region);
};
