/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

/*
 * This file looks really nasty because it's the oldest and has seen a
 * lot of modifications too. I will change this in the future...
 */

#include <errno.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>
#include <X11/Xmd.h>
#include <X11/Xatom.h>
#include <X11/Xlocale.h>
#include <cstdarg>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>

#define _TOAD_PRIVATE

#include <toad/toadbase.hh>
#include <toad/pen.hh>
#include <toad/window.hh>
#include <toad/font.hh>
#include <toad/region.hh>
#include <toad/dragndrop.hh>
#include <toad/gadget.hh>
#include <toad/dialogeditor.hh>

// TMessageBox
#include <toad/dialog.hh>
#include <toad/messagebox.hh>

#define SECURE


static string GetWindowProperty(Window source, Atom property, Atom type);
static string AtomName(Atom atom) {
	string result = "(None)";
	if (atom) {
		char *name = XGetAtomName(TOADBase::x11display, atom);
		result = name;
		XFree(name);
	}
	return result;
}
static string selection_kludge_data;
static bool selection_kludge_flag;

// define this for a periodic screen update every 1/12 seconds
// #define PERIODIC_PAINT

#include <deque>
#include <set>
#include <vector>

class TActionDeleteWindow:
	public TAction
{
		TWindow *_window;
	public:
		TActionDeleteWindow(TWindow *w) { _window = w; }
		void execute() { delete _window; }
};

extern XIC xic_current;

// X11 data
//---------------------------------------------------------------------------
Display*	TOADBase::x11display = NULL;
Visual*		TOADBase::x11visual = NULL;
int				TOADBase::x11depth = 0;
int				TOADBase::x11screen;
XEvent		TOADBase::x11event;
GC				TOADBase::x11gc = 0;
XContext	TOADBase::nClassContext;
Atom			TOADBase::xaWMSaveYourself;
Atom			TOADBase::xaWMDeleteWindow;
static Atom	xaWMProtocols;
Atom			TOADBase::xaWMMotifHints;

// TOAD data
//---------------------------------------------------------------------------
TWindow* 	TOADBase::twMainWindow = NULL;
TFont* 		TOADBase::default_font = NULL;
TFont*		TOADBase::bold_font = NULL;
int				nStatus;
bool			TOADBase::bAppIsRunning;
TWindow*	TOADBase::wndTopPopup;
bool			TOADBase::bSimulatedAutomaticGrab;
bool			TOADBase::lock_paint_queue = false;

int				TOADBase::argc;
char**		TOADBase::argv;
char**		TOADBase::envv;

struct TModalNode {
	TWindow *wnd;
	bool running;
	bool pushed;
};

typedef vector<TModalNode*> TModalStack;
static TModalStack modal_stack;

fd_set _io_set_rd, _io_set_wr, _io_set_ex;

static string executable_path;
static string executable_name;

const string& TOADBase::ExecutablePath()
{
	return executable_path;
}

const string& TOADBase::ExecutableName()
{
	return executable_name;
}

// called by 'TOADBase::Open' to set the 'executable_path' string
//----------------------------------------------------------------
static void get_executable_path(char *prgname)
{
	char buffer[PATH_MAX+1];
	string cwd;
	executable_path = prgname;
	int p = executable_path.rfind('/');
	if (p>0) {
		executable_name = executable_path.substr(p+1);
		executable_path = executable_path.substr(0,p)+"/"; 
	} else {
		executable_name = executable_path;
		executable_path = "";
	}    
	getcwd(buffer,PATH_MAX); cwd = buffer;
	chdir(executable_path.c_str());
	getcwd(buffer,PATH_MAX); executable_path = buffer;  
	executable_path+='/';
	chdir(cwd.c_str());
}

// Bell
//---------------------------------------------------------------------------
//. Rings the bell on the keyboard if possible. The specified volume is
//. relative to the base volume for the keyboard.
void TOADBase::Bell(int volume, int freq)
{
	if (volume<-100) volume=-100; else if (volume>100) volume=100;
	if (freq>=0) {
		XKeyboardControl c;
		c.bell_pitch = freq;
		XChangeKeyboardControl(x11display, KBBellPitch, &c);
	}
	XBell(x11display, volume);
}

// open/close toad
//---------------------------------------------------------------------------
TOADBase::~TOADBase()
{
}

bool TOADBase::Init(int argc,char** argv,char** envv)
{
	TOADBase::argc = argc;
	TOADBase::argv = argv;
	TOADBase::envv = envv;

	bool x11sync = false;
	string x11displayname;

	for(int i=1; i<argc; i++)	{
		if (strcmp(argv[i],"--sync")==0) {
			cout << "XSynchronize on\n";
			x11sync = true;
		} else
		if (strcmp(argv[i],"--dialog-editor")==0)	{
			cout << "dialog edit mode enabled\n";
			TDialogEditor::running = true;
		} else
		if (strcmp(argv[i],"--display")==0)	{
			i++;
			if (i>=argc) {
				cerr << "--display: missing display name" << endl;
			} else {
				x11displayname = argv[i];
			}
		}
	}

	bool i18n = true;

	if (setlocale(LC_ALL, "")==NULL) {
		if (setlocale(LC_ALL, "POSIX")==NULL) {
			if (setlocale(LC_ALL, "C")==NULL) {
				i18n = false;
			}
		}
	}

	if ((x11display = XOpenDisplay(x11displayname.c_str()))==NULL) {
		cerr << "Couldn't open display \"" << x11displayname << "\"\n";
		exit(1);
	}

	if (x11sync)
		XSynchronize(x11display, True);

	x11screen					= DefaultScreen(x11display);
	nClassContext			=	XUniqueContext();
	xaWMSaveYourself	= XInternAtom(x11display, "WM_SAVE_YOURSELF", False);
	xaWMDeleteWindow	= XInternAtom(x11display, "WM_DELETE_WINDOW", False);
	xaWMProtocols			= XInternAtom(x11display, "WM_PROTOCOLS", False);
	xaWMMotifHints		= XInternAtom(x11display, "_MOTIF_WM_HINTS", False);

	if (i18n)
		InitXInput();

	bSimulatedAutomaticGrab = false;
	wndTopPopup = NULL;

	InitColor();

	InitIO(ConnectionNumber(x11display));
	TGadget::InitStorage();
	InitThreads();
	InitDnD();

	// parse arguments
	get_executable_path(*argv);

	// set up default font
	//---------------------
	x11gc = DefaultGC(x11display, DefaultScreen(x11display));
	default_font=new TFont(TFont::SANS, TFont::PLAIN, 12);
	bold_font		=new TFont(TFont::SANS, TFont::BOLD, 12);

	TBitmap::Open();
	return true;
}

void TOADBase::Close()
{
	if (TDialogEditor::CtrlWindow()) {
		delete TDialogEditor::CtrlWindow();
		TDialogEditor::SetCtrlWindow(NULL);
	}

	TGadget::LoseStorage();
	TBitmap::Close();

	if (default_font)	{
		delete default_font;
		default_font = NULL;
	}
	if (bold_font) {
		delete bold_font;
	}
	CloseXInput();

	//	close connection to the X11 server
	XCloseDisplay(x11display);
	x11display = 0;
	RemoveAllIntMsg();
}

/*---------------------------------------------------------------------------*
 | without category    	                                                  	 |
 *---------------------------------------------------------------------------*/

// there seems to be a problem with nested exceptions
static bool show_exception_message = false;

//. <B>Please don't use this.</B><BR>
//. Use 'int TWindow.Run()' instead.<BR>
//. <I>Note: It's the old entry point into the message loop and might be
//. removed some day.</I>
int TOADBase::Run(TWindow *wnd, int, char**,char**)
{
	ENTRYEXIT("TOADBase::Run");
	twMainWindow = wnd;

	twMainWindow->Create();

	bAppIsRunning = true;
	string msg;
	while(bAppIsRunning) {
		try {
			HandleMessage();
		} catch(exception &e) {
			if (show_exception_message) {
				cerr << "caught another exception: " << e.what() << endl;
			} else {
				show_exception_message = true;
				MessageBox(NULL, "Encountered Exception", e.what(), MB_ICONEXCLAMATION | MB_OK);
				show_exception_message = false;
			}
		}
	}

	twMainWindow->Destroy();
//	delete twMainWindow;	// can't do this when using 'TWindow.Run'

	// flush paint event buffer
	//---------------------------------
	while(TWindow::_HavePaintEvents())
		TWindow::_DispatchPaintEvent();

	return nStatus;
}

//. Stop the toplevel event queue.<BR>
//. The toplevel event queue is the one you've called from ToadMain with
//. 'Run()'. The other event queues, invoked by TDialog.DoModal() will
//. terminate also.
void TOADBase::PostQuitMessage(int nExitCode)
{
	nStatus = nExitCode;
	EndAllModal();
	EndAllModeless();
	bAppIsRunning = false;
}

//. Send all stored event to the X Server.
void TOADBase::Flush()
{
	XFlush(x11display);
}

/*---------------------------------------------------------------------------*
 | message handling       	                                               	 |
 *---------------------------------------------------------------------------*/
typedef deque<PAction> TMessageQueue;
TMessageQueue _messages;

#ifdef __TOAD_THREADS
	static TThreadMutex mutexMessageQueue;
	#if 1
		#define THREAD_LOCK(A) A.Lock();
		#define THREAD_UNLOCK(A) A.Unlock();
	#else
		#define THREAD_LOCK(A) { cout << "lock " #A << " @ " << __LINE__ << endl; A.Lock(); }
		#define THREAD_UNLOCK(A) { A.Unlock(); cout << "unlock " #A << " @ " << __LINE__ << endl; }
	#endif
#else
	#define THREAD_LOCK(A)
	#define THREAD_UNLOCK(A)
#endif

//.  Add a new message/action to the message queue.<BR>
//.  This is one of the rare thread safe methods in TOAD.
//---------------------------------------------------------------------------
void TOADBase::SendMessage(TAction *action)
{
	THREAD_LOCK(mutexMessageQueue);
	#ifdef __TOAD_THREADS
		// wake thread running the message loop (main thread)
		if (TThread::WhoAmI()!=NULL)
			TThread::Kill(NULL, SIGUSR1);
	#endif
	_messages.push_back(action);
	THREAD_UNLOCK(mutexMessageQueue);
}

//.  Call `check(void*)' for all actions in the message queue being derived
//. from `TSignalNodeCheck'.<BR>
//.  This is used to disable pending messages from `TSignal.DelayedTrigger'.
//---------------------------------------------------------------------------
void TOADBase::RemoveMessage(void *obj)
{
	THREAD_LOCK(mutexMessageQueue);
	TMessageQueue::iterator p,e;
	p = _messages.begin();
	e = _messages.end();
	while(p!=e) {
		TAction *a = *p;
		TSignalBase::TSignalNodeCheck *snc 
			= dynamic_cast<TSignalBase::TSignalNodeCheck*>(a);
		if (snc)
			snc->check(obj);
		p++;
	}
	THREAD_UNLOCK(mutexMessageQueue);
}

void TOADBase::RemoveAllIntMsg()
{
	THREAD_LOCK(mutexMessageQueue);
	_messages.erase(_messages.begin(), _messages.end());
	THREAD_UNLOCK(mutexMessageQueue);
}

//. Instead of calling the assigned object:method pairs directly, put the
//. command onto TOADs' internal message queue and return immediately.
bool TSignalBase::DelayedTrigger()
{
	if (_list==NULL)
		return false;
	THREAD_LOCK(mutexMessageQueue);
	TSignalNode *p = _list;
	while(p) {
		TAction *a = p;
		_messages.push_back(a);
		p = p->_next;
	}
	THREAD_UNLOCK(mutexMessageQueue);
	return true;
}


//. Returns `true' when the message queue is not empty.
//---------------------------------------------------------------------------
bool TOADBase::PeekMessage()
{
	bool result;
	THREAD_LOCK(mutexMessageQueue);
	result = (XPending(x11display)!=0 || 
						TWindow::_HavePaintEvents() || 
						_messages.size()!=0 );
	THREAD_UNLOCK(mutexMessageQueue);
	return result;
}

//. Wait when the message queue is empty until a new message arrives
//. and handle one message.
//. Returns `false' when `PostQuitMessage(..)' was been called.
//---------------------------------------------------------------------------

#define CONSIDER_GRAB(etype) \
	x=x11event.xbutton.x; \
	y=x11event.xbutton.y; \
	if (wndTopPopup && wndTopPopup != window) { \
		if (!window->IsChildOf(wndTopPopup)) { \
			XTranslateCoordinates(x11display,window->x11window,wndTopPopup->x11window,x,y,&x,&y,&dummy_window); \
			window=wndTopPopup; \
		} \
	}

#define MODAL_BREAK \
	if( !modal_stack.empty() && \
			!window->IsChildOf(modal_stack.back()->wnd) ) { \
		if( modal_stack.back()->pushed ) \
			window=modal_stack.back()->wnd; \
		else \
			break; }

bool TOADBase::HandleMessage()
{
	ENTRYEXIT("TOADBase::HandleMessage()");

	bool dispatch_paint_event = false;
#ifdef PERIODIC_PAINT
	static struct timeval first_paint_event_age;
#endif

	// wait for an event
	//-------------------
	while(true) {
		XFlush(x11display);

		// test if any TOAD internal message exist
		//-----------------------------------------
		THREAD_LOCK(mutexMessageQueue);
		if (_messages.size()!=0 || !bAppIsRunning) {
			goto handle_event;								// => yes, handle event
		}
		THREAD_UNLOCK(mutexMessageQueue);

		// loop while event queue is empty
		//---------------------------------
		while (XPending(x11display)==0) {

			// don't wait when we have paint events
			//--------------------------------------
			if(TWindow::_HavePaintEvents()) {
				XSync(x11display, False);				// wait for other events
				if (XPending(x11display)!=0)
					break;
				dispatch_paint_event = true;
				THREAD_LOCK(mutexMessageQueue);
				goto handle_event;
			}
	    
			// wait for the events to come
			//--------------------------------------
			Select();	// won't return for `TIOObserver' and `TSimpleTimer' events

			THREAD_LOCK(mutexMessageQueue);
			if (_messages.size()!=0 || !bAppIsRunning) {
				goto handle_event;
			}
			THREAD_UNLOCK(mutexMessageQueue);
	  }

		// pop event from the event queue
		//--------------------------------
		XNextEvent(x11display, &x11event);
		
		if (XFilterEvent(&x11event, None))
			continue;

		// don't handle paint events, just collect them
		//-----------------------------------------------------------
		if(x11event.type == Expose || x11event.type == GraphicsExpose )	{
			TWindow *window;
			if(XFindContext(x11display, x11event.xany.window, nClassContext, (XPointer*)&window))
			{
				cerr << "toad: XFindContext failed for Expose event" << endl;
			} else {
				if (window && window->x11window) {
					window->handleX11Event();
#ifdef PERIODIC_PAINT					
					bool dispatch_all_paint_events = false;
					if (!TWindow::_HavePaintEvents()) {
						gettimeofday(&first_paint_event_age, NULL);
					} else {
						struct timeval crnt_time;
						gettimeofday(&crnt_time, NULL);
						if ( (ulong)(crnt_time.tv_sec -first_paint_event_age.tv_sec)*1000000UL
								+(ulong)(crnt_time.tv_usec-first_paint_event_age.tv_usec) 
								>= 1000000UL / 12UL ) 
						{
							dispatch_all_paint_events = true;
						}
					}
#endif
					window->Invalidate(
						x11event.xexpose.x, x11event.xexpose.y,
						x11event.xexpose.width, x11event.xexpose.height,
						false
					);
#ifdef PERIODIC_PAINT					
					if (dispatch_all_paint_events) {
						while(TWindow::_HavePaintEvents())
							TWindow::_DispatchPaintEvent();
					}
#endif
				}
			}
		}	else {
			THREAD_LOCK(mutexMessageQueue);
			goto handle_event;
		}
	}

handle_event:

#if 0
	if (!bAppIsRunning) {
		THREAD_UNLOCK(mutexMessageQueue);
		return false;
	}
#endif

	// dispatch local messages
	//--------------------------
	if (_messages.size()!=0)	{
		TMessageQueue::iterator p = _messages.begin();
		PAction action = *p;
		_messages.erase(p);
		THREAD_UNLOCK(mutexMessageQueue);
		action->execute();
		return bAppIsRunning;
	}
	THREAD_UNLOCK(mutexMessageQueue);

#ifdef PERIODIC_PAINT
	// flush all paint events 12 times per second
	//--------------------------------------------
	if (TWindow::_HavePaintEvents()) {
		struct timeval crnt_time;
		gettimeofday(&crnt_time, NULL);
		if ( (ulong)(crnt_time.tv_sec -first_paint_event_age.tv_sec)*1000000UL
				+(ulong)(crnt_time.tv_usec-first_paint_event_age.tv_usec) 
				>= 1000000UL / 12UL ) {
			while(TWindow::_HavePaintEvents())
				TWindow::_DispatchPaintEvent();
		}
	}
#endif

	// no other event, dispatch a single paint event
	//-----------------------------------------------
	if (dispatch_paint_event) {
		TWindow::_DispatchPaintEvent();
		return bAppIsRunning;
	}

	// first event dispatching switch for events not 
	// related to actual `TWindow' objects
	//-----------------------------------------------
	switch(x11event.type) {
		case ButtonRelease:
			if (DnDButtonRelease(x11event))
				return bAppIsRunning;
			break;
			
    case MotionNotify:
			if (DnDMotionNotify(x11event))
				return bAppIsRunning;
			break;
			
		case ClientMessage:
			if (DnDClientMessage(x11event))
				return bAppIsRunning;
			break;
			
		case SelectionClear:
			if (DnDSelectionClear(x11event));
				return bAppIsRunning;
			break;

		case SelectionNotify: {
			if (DnDSelectionNotify(x11event))
				return bAppIsRunning;
// start of hack
#if 0
			cout << "got SelectionNotify" << endl;
			cout << "  requestor: " << x11event.xselection.requestor << endl;
			cout << "  selection: " << AtomName(x11event.xselection.selection) << endl;
			cout << "  target   : " << AtomName(x11event.xselection.target) << endl;
			cout << "  property : " << AtomName(x11event.xselection.property) << endl;
			cout << "  time     : " << x11event.xselection.time << endl;
#endif
			if (x11event.xselection.selection==XA_PRIMARY &&
					x11event.xselection.target==XA_STRING)
			{
				if ( x11event.xselection.target==XA_STRING)
				{
					selection_kludge_data = GetWindowProperty(
						x11event.xselection.requestor,
						XA_PRIMARY,
						XA_STRING
					);
				}
				selection_kludge_flag = false;
			}
// end of hack
		} break;

		case SelectionRequest:
			if (DnDSelectionRequest(x11event))
				return bAppIsRunning;
			break;
	}

	// handle messages for window 0
	//------------------------------
	if (x11event.xany.window==0) {
		if (x11event.type == ClientMessage) {
			cerr<< "toad: unknown client message: " 
					<< AtomName(x11event.xclient.message_type) << endl;
		} else {
			cerr << "toad: event " << x11event.type << " for window 0" << endl;
		}		
		return bAppIsRunning;
	}

	// get TWindow object
	//---------------------
	TWindow *window;
	if(XFindContext(x11display, 
	x11event.xany.window, nClassContext, (XPointer*)&window))	{
		if (x11event.type==GraphicsExpose || x11event.type==NoExpose)	{
			// `x11event.xany.window' could be a bitmap => ignore the failure
		} else {
			static const char *eventname[34] = {
				"(reserved)",
				"(reserved)",
				"KeyPress",
				"KeyRelease",
				"ButtonPress",
				"ButtonRelease",
				"MotionNotify"
				"EnterNotify",
				"LeaveNotify",
				"FocusIn",
				"FocusOut",
				"KeymapNotify",
				"Expose",
				"GraphicsExpose",
				"NoExpose",
				"VisibilityNotify",
				"CreateNotify",
				"DestroyNotify",
				"UnmapNotify",
				"MapNotify",
				"MapRequest",
				"ReparentNotify",
				"ConfigureNotify",
				"ConfigureRequest",
				"GravityNotify",
				"ResizeRequest",
				"CirculateNotify",
				"CirculateRequest",
				"PropertyNotify",
				"SelectionClear",
				"SelectionRequest",
				"SelectionNotify",
				"ColormapNotify",
				"ClientMessage",
				"MappingNotify"
			};
			cerr<< "toad: fatal; XFindContext failed for window "
					<< x11event.xany.window << ". Eventtype was "
					<< eventname[x11event.type] << "." << endl;
		}
		return bAppIsRunning;	// ignore event
	}
	
	// window doesn't exist anymore
	if (!window || !window->x11window)
		return bAppIsRunning;

	// dispatch event
	//----------------

	window->handleX11Event();

	int x,y;
	Window dummy_window;
	switch(x11event.type)	{
		case Expose:
		case GraphicsExpose:
			cerr << "toad: internal error; unhandled expose event\n";
			break;

		case NoExpose:
			//	printf("NoExpose for '%s'\n", window->Title());
			break;
	
		// ButtonPress
		//-------------
		case ButtonPress:
			ToolTipClose();
			x=x11event.xbutton.x;
			y=x11event.xbutton.y;

			// special for popup windows
			//---------------------------
			if (wndTopPopup && (wndTopPopup!=window || 
													(x<0 || y<0 || 
													 x>=window->_w || y>=window->_h)) )
			{
//				printf("special for popup windows\n");
				if (!window->IsChildOf(wndTopPopup) 
					|| wndTopPopup==window)	
				{
					// printf("stop popup grab\n");
					// stop popup grab
					//-----------------

					// 1st: Destroy popup window
					wndTopPopup->closeRequest();
					wndTopPopup=NULL;

					// 2nd: Stop Grab
					window->UngrabMouse();

					// 3rd: Get window beneath the mouse pointer
					Window root,child,dest=None;
					int rx,ry,wx,wy;
					unsigned m;
					child=DefaultRootWindow(x11display);
					while(child) {
						dest = child;
						XQueryPointer(x11display, child, &root, &child, &rx,&ry, &wx,&wy, &m);
					}

					// 4th:	Get position of mouse pointer within this window
					XTranslateCoordinates(x11display, x11event.xbutton.root, dest,
						x11event.xbutton.x_root, x11event.xbutton.y_root,
						&x,&y,
						&dummy_window);

					// 5th: Redo event
					TWindow* wnd;
					if (XFindContext(x11display, dest, nClassContext, (XPointer*)&wnd) ) {
						// window is not part of the application
						XEvent se;
						se.xbutton.type				= ButtonPress;
						se.xbutton.window 		= dest;
						se.xbutton.root				= x11event.xbutton.root;
						se.xbutton.subwindow	= dest;
						se.xbutton.time				= CurrentTime;
						se.xbutton.x					= x;
						se.xbutton.y					= y;
						se.xbutton.x_root			= x11event.xbutton.x_root;
						se.xbutton.y_root			= x11event.xbutton.y_root;
						se.xbutton.state			= x11event.xbutton.state;
						se.xbutton.button			=	x11event.xbutton.button;
						se.xbutton.same_screen=	x11event.xbutton.same_screen;
						if (!XSendEvent(x11display, dest, False, ButtonPressMask, &se))
							printf(__FILE__": The conversion to the wire protocol format failed.\n");
						break;
					}	else {
						// window is part of the application
						wnd->mouseEnter(x, y, x11event.xbutton.state);
						window = wnd;
						x11event.xany.send_event = true;
					}
				}
			} // end of special for popup windows
			// do automatic grab when event came from XSendEvent
			if (x11event.xany.send_event) {
//				printf("simulating automatic grab\n");
				window->GrabMouse();
				bSimulatedAutomaticGrab = true;
			}

#if 0
//cout << "mouseXDown for window " << window->Title() << endl;
if (window->IsChildOf(dialog_editor_window)) {
	cout << "its the dialog editor" << endl;
}
#endif

			if (!window->IsChildOf(TDialogEditor::CtrlWindow()))
				MODAL_BREAK;

			{
				static Time last_click_time = 0;
				static Window last_click_window = 0;
				x11event.xbutton.state &= ~MK_DOUBLE;
				if (x11event.xbutton.time-last_click_time<250 &&
						last_click_window == x11event.xbutton.window )
				{
					x11event.xbutton.state |= MK_DOUBLE;
				}
				last_click_time = x11event.xbutton.time;
				last_click_window = x11event.xbutton.window;
			}
			switch(x11event.xbutton.button) {
				case Button1:

					// Button1 has a special meaning when TOADs integrated
					// dialog editor is used
					if (TDialogEditor::running && TDialogEditor::enabled) {
						// case 1: event occured in the editable window
						if (window->bDialogEditRequest) {
							TDialogEditor::DialogEditor()->SetEditWindow(window);
							TDialogEditor::DialogEditor()->mouseLDown(x, y, x11event.xbutton.state);
							window->GrabMouse(TWindow::TMMM_LBUTTON);
							bSimulatedAutomaticGrab = true;
							return bAppIsRunning;
						}
						// case 2: event occured in one of the editable windows children
						TWindow *p = window;
						int dx=0, dy=0;
						while(p->parent) {
							dx += p->_x;
							dy += p->_y;
							if (p->parent->bDialogEditRequest) {
								TDialogEditor::DialogEditor()->SetEditWindow(p->parent);
								TDialogEditor::DialogEditor()->mouseLDown(x+dx, y+dy, x11event.xbutton.state);
								p->parent->GrabMouse(TWindow::TMMM_LBUTTON);
								bSimulatedAutomaticGrab = true;
								return bAppIsRunning;
							}
							p = p->parent;
						}
					}
					window->mouseLDown(x-window->OriginX(), 
														 y-window->OriginY(), 
														 x11event.xbutton.state);
					break;
				case Button2:
					window->mouseMDown(x-window->OriginX(), 
														 y-window->OriginY(), 
														 x11event.xbutton.state);
					break;
				case Button3:
					window->mouseRDown(x-window->OriginX(),
														 y-window->OriginY(), 
														 x11event.xbutton.state);
					break;
			}
			break;

		// ButtonRelease
		//---------------
		case ButtonRelease:
			ToolTipClose();
			if (bSimulatedAutomaticGrab)
				window->UngrabMouse();
			CONSIDER_GRAB(xbutton);
			if (!window->IsChildOf(TDialogEditor::CtrlWindow()))
				MODAL_BREAK;
			// printf("subwindow: 0x%08lX\n",x11event.xbutton.subwindow);
			switch(x11event.xbutton.button)	{
				case Button1:
					// Button1 has a special meaning when TOADs integrated
					// dialog editor is used
					if (TDialogEditor::running && TDialogEditor::enabled) {
						// case 1: event occured in the editable window
						if (window->bDialogEditRequest) {
							TDialogEditor::DialogEditor()->SetEditWindow(window);
							TDialogEditor::DialogEditor()->mouseLUp(x, y, x11event.xbutton.state);
							return bAppIsRunning;
						}
						// case 2: event occured in one of the editable windows children
						TWindow *p = window;
						int dx=0, dy=0;
						while(p->parent) {
							dx += p->_x;
							dy += p->_y;
							if (p->parent->bDialogEditRequest) {
								TDialogEditor::DialogEditor()->SetEditWindow(p->parent);
								TDialogEditor::DialogEditor()->mouseLUp(x+dx, y+dy, x11event.xbutton.state);
								return bAppIsRunning;
							}
							p = p->parent;
						}
					}
					window->mouseLUp(x-window->OriginX(),
													 y-window->OriginY(), 
													 x11event.xbutton.state);
					break;
				case Button2:
					window->mouseMUp(x-window->OriginX(), 
													 y-window->OriginY(), 
													 x11event.xbutton.state);
					break;
				case Button3:
					window->mouseRUp(x-window->OriginX(), 
													 y-window->OriginY(), 
													 x11event.xbutton.state);
					break;
			}
			break;

		// EnterNotify
		//-------------
		case EnterNotify:
			if (window->_bToolTipAvailable)
				ToolTipOpen(window);
			CONSIDER_GRAB(xcrossing);
			window->mouseEnter(x-window->OriginX(), 
												 y-window->OriginY(), 
												 x11event.xcrossing.state);
			break;

		// LeaveNotify
		//-------------
		case LeaveNotify:
			ToolTipClose();
			CONSIDER_GRAB(xcrossing);
			window->mouseLeave(x-window->OriginX(), 
												 y-window->OriginY(), 
												 x11event.xcrossing.state);
			break;

		// KeyPress
		//----------
		case KeyPress:
			{
				ToolTipClose();

				if (!window->IsChildOf(TDialogEditor::CtrlWindow()))
					MODAL_BREAK;

				int count;
				char buffer[KB_BUFFER_SIZE+1];
				KeySym key;
				
				if (xic_current) {
					Status status;
					count = XmbLookupString(xic_current, 
																	&x11event.xkey, 
																	buffer, KB_BUFFER_SIZE,
																	&key,
																	&status );

					if (status==XLookupNone)
						break;
					if (status==XBufferOverflow) {
						cerr << "TOAD keyboard buffer overflow" << endl;
						XmbResetIC(xic_current);
						break;
					}
				} else {
					static XComposeStatus compose_status = {NULL, 0};
					count = XLookupString(&x11event.xkey,
																buffer, KB_BUFFER_SIZE, 
																&key, 
																&compose_status);
				}
				buffer[count]=0;				// add zero terminator to string

				if (TDialogEditor::running && 
						TDialogEditor::enabled && 
						TDialogEditor::EditWindow() &&
						!window->IsChildOf(TDialogEditor::CtrlWindow()) )
				{
					TDialogEditor::DialogEditor()->keyDown(key, buffer, x11event.xkey.state);
				} else {
					KeyDown(key, buffer, x11event.xkey.state);
				}
			}
			break;

		// KeyRelease
		//------------
		case KeyRelease:
			{
				ToolTipClose();
				MODAL_BREAK;

				char buffer[KB_BUFFER_SIZE+1];
				KeySym key;
				XComposeStatus dummy;		// not needed since X11R5
				int count = XLookupString(&x11event.xkey, buffer, KB_BUFFER_SIZE, &key, &dummy);
				buffer[count]=0;				// add zero terminator to string
				KeyUp(key, buffer, x11event.xkey.state);
			}
			break;

		// MotionNotify
		//--------------
		case MotionNotify:
			CONSIDER_GRAB(xmotion)
			if (!window->IsChildOf(TDialogEditor::CtrlWindow()))
				MODAL_BREAK;
			if (TDialogEditor::running && TDialogEditor::enabled) {
				TDialogEditor::DialogEditor()->mouseMove(x, y, x11event.xmotion.state);
			} else {
				window->mouseMove(x-window->OriginX(), 
													y-window->OriginY(), x11event.xmotion.state);
			}
			break;
			
		// ReparentNotify
		//----------------
		case ReparentNotify:
#if 0
				printf("ReparentNotify for '%s' to x=%4i, y=%4i, parent=%lx\n",
							 window->Title().c_str(),
							 x11event.xreparent.x, 
							 x11event.xreparent.y, 
							 (long) x11event.xreparent.parent);
#endif
			break;

		// ConfigureNotify
		//-----------------
		case ConfigureNotify:
			{
				/*
				printf("ConfigureNotify for '%s'\n"
						 "     to x=%4i, y=%4i, w=%4i, h=%4i\n"
						 "   from x=%4i, y=%4i, w=%4i, h=%4i\n"
				,window->Title(),
				x11event.xconfigure.x, x11event.xconfigure.y,
				x11event.xconfigure.width, x11event.xconfigure.height,
				window->_x,window->_y,
				(int)window->width, (int)window->height);
				*/
				// resize
				//--------
				if (	x11event.xconfigure.width  != window->_w
					 ||	x11event.xconfigure.height	!= window->_h )
				{
					// FVWM does not deliver the right position relative to the root
					// window after resizing the toplevel window. Instead it's (0,0).
					// The following code will solve the problem:
					// !!! ATTENTION: THE FOLLOWING CODE BADLY NEEDS TO BE OPTIMIZED !!!
					if (!window->Parent() && 
							!x11event.xconfigure.x && 
							!x11event.xconfigure.y )
					{
						Window root, wnd, *children;
						x=y=0;
						int xp,yp; unsigned n;
						wnd = x11event.xconfigure.window;
						do {
							XQueryTree(x11display, wnd, &root, &wnd, &children, &n);
							XFree(children);
							XGetGeometry(x11display, wnd, &root, &xp,&yp, &n,&n,&n,&n);
							x+=xp;
							y+=yp;
						}while(root!=wnd);
						x11event.xconfigure.x = x;
						x11event.xconfigure.y = y;
					}
					window->_w = x11event.xconfigure.width;
					window->_h = x11event.xconfigure.height;

//printf("resize: '%s' is set to (%i,%i)\n", window->Title(),(int)window->_w, (int)window->_h);
					window->resize();	// event structure maybe changed afterwards!
				}

				// REPOSITION EVENT
				if (	x11event.xconfigure.x != window->_x
					 ||	x11event.xconfigure.y != window->_y )
				{
/*					printf("Position of %s set to (%i,%i)\n",
						window->Title(),
						event.xconfigure.x,
						event.xconfigure.y); */
					window->_x = x11event.xconfigure.x;
					window->_y = x11event.xconfigure.y;
				}
			}
			break;

		//+---------------+
		//| ClientMessage |
		//+---------------+

		// here are parts of the drag'n drop stuff too 
		
		case ClientMessage:
			{
				if (x11event.xclient.message_type == xaWMProtocols) {
					Atom at=x11event.xclient.data.l[0];
					if (at==xaWMDeleteWindow)
						window->closeRequest();
					else if (at==xaWMSaveYourself)
						window->saveYourself();
				}
			}
			break;				

		case PropertyNotify:
			{
				#if 0
				// printf("property notify:\n");
				if (x11event.xproperty.atom) {
					const char *name;
					name = XGetAtomName(x11display, event.xproperty.atom);
					printf("property '%s' of window '%s' is changed\n",name,window->Title().c_str());
					XFree((void*)name);
				}
				#endif
			}
			break;

		case SelectionClear:
			break;

		case SelectionNotify:
			break;

		case SelectionRequest:
			break;

		//+---------------+
		//| MappingNotify |
		//+---------------+
		case MappingNotify:
			XRefreshKeyboardMapping(&x11event.xmapping);
			break;

		case FocusIn:
			DomainToWindow(window);
			break;
			
		case FocusOut:
			DomainToWindow(NULL);
			break;

		case DestroyNotify:
			//fprintf(stderr, "toad: DestroyNotify\n");
			break;
		case UnmapNotify:
			//fprintf(stderr, "toad: UnmapNotify\n");
			break;
		case MapNotify:
			//fprintf(stderr, "toad: MapNotify\n");
			break;
		// default:
			// fprintf(stderr, "toad: (TOADBase.ToadDispatchEvent) unhandled event occured\n");
			// fprintf(stderr, "      type was %i.\n",x11event.type);
	}
	return bAppIsRunning;
}

/*---------------------------------------------------------------------------*
 | modal dialog administration    	                                         |
 *---------------------------------------------------------------------------*/
//. Create window `wnd' as a modal window and return when `wnd' is being
//. closed.
void TOADBase::DoModal(TWindow *wnd)
{
	TModalNode node;
	node.wnd = wnd;
	node.running = true;
	node.pushed  = false;
	modal_stack.push_back(&node);

	wnd->Create();
	while(node.running) {
		try {
			HandleMessage();
		} catch(exception &e) {
			if (show_exception_message) {
				cerr << "caught another exception: " << e.what() << endl;
			} else {
				show_exception_message = true;
				MessageBox(NULL, "Encountered Exception", e.what(), MB_ICONEXCLAMATION | MB_OK);
				show_exception_message = false;
			}
		}
	}
	wnd->Destroy();
	modal_stack.pop_back();
}

//. obsolete
void TOADBase::PushModal(TWindow *wnd)
{
	TModalNode *node = new TModalNode();
	node->wnd     = wnd;
	node->running = false;
	node->pushed  = true;
	modal_stack.push_back(node);
}

//. obsolete
void TOADBase::PopModal(TWindow *wnd)
{
	if (modal_stack.empty()) {
		cerr << "toad: TOADBase::PopModal(); internal modal window stack is empty\n";
		return;
	}
	if (modal_stack.back()->wnd != wnd) {
		cerr << "toad: TOADBase::PopModal(); not called for current modal window\n";
		return;
	}
	if (!modal_stack.back()->pushed) {
		cerr << "toad: TOADBase::PopModal(); can't pop a modal window with event loop\n";
		return;
	}
	delete modal_stack.back();
	modal_stack.pop_back();
}

typedef set<TDialog*> TModelessList;
static TModelessList _modeless_list;

//. Destroy dialog.
//. <UL>
//.   <LI><B>Modal Dialog</B><BR>
//.       Stop the modal message loop to the window.
//.   <LI><B>Modeless Dialog</B><BR>
//.       Delete window or when the dialog is still referenced by a 
//.       smart pointer call <CODE>SetVisible(false)</CODE> called.
//. </UL>
void TOADBase::EndDialog(TWindow *wnd)
{
	if (wnd==twMainWindow) {
		PostQuitMessage(0);
		return;
	}

	// try 1st modal dialog, if there is one
	//---------------------------------------
	if (!modal_stack.empty() && modal_stack.back()->wnd==wnd) {
		modal_stack.back()->running = false;
		return;
	}

	// search modeless dialogs
	//-------------------------
	TDialog *dlg = dynamic_cast<TDialog*>(wnd);
	if (dlg) {
		TModelessList::iterator p = _modeless_list.find(dlg);
		if (p!=_modeless_list.end()) {
			if (dlg->_toad_ref_cntr==1) {
				_modeless_list.erase(p);
				dlg->_toad_ref_cntr=0;
				#if 0
					delete dlg;						// ouch!
				#else
					SendMessage(new TActionDeleteWindow(dlg));
				#endif
			} else {
				dlg->SetVisible(false);
			}
			return;
		}
	}

#if 1
	if (wnd->_toad_ref_cntr==0) {
		SendMessage(new TActionDeleteWindow(wnd));
	}
#endif
}

void TOADBase::EndAllModal()
{
	TModalStack::iterator p,e;
	p = modal_stack.begin();
	e = modal_stack.end();
	while(p!=e) {
		(*p)->running = false;
		p++;
	}
}

/*---------------------------------------------------------------------------*
 | modeless dialog administration																						 |
 *---------------------------------------------------------------------------*/
void TOADBase::DoModeless(TDialog *dlg)
{
	dlg->Create();
	dlg->_toad_ref_cntr++;
	_modeless_list.insert(dlg);
}

void TOADBase::EndAllModeless()
{
	_modeless_list.erase(_modeless_list.begin(), _modeless_list.end());
}

/*---------------------------------------------------------------------------*
 | messagebox																																 |
 *---------------------------------------------------------------------------*/
//. This function creates and displays a modal dialog that contains a text,
//. an icon and pushbuttons.<P>
//. The following values define <VAR>type</VAR> and can be joined by the
//. '|' operator:<P>
//. <TABLE BORDER=1 WIDTH=100%>
//. <TR><TH>
//. MB_ICONEXCLAMATION
//. </TH><TD>
//. <IMG SRC="img/icon_exclamation.gif">
//. </TD></TR><TR><TH>
//. MB_ICONHAND
//. </TH><TD>
//. <IMG SRC="img/icon_hand.gif">
//. </TD></TR><TR><TH>
//. MB_ICONSTOP
//. </TH><TD>
//. <IMG SRC="img/icon_stop.gif">
//. </TD></TR><TR><TH>
//. MB_ICONINFORMATION
//. </TH><TD>
//. <IMG SRC="img/icon_information.gif">
//. </TD></TR><TR><TH>
//. MB_ICONQUESTION
//. </TH><TD>
//. <IMG SRC="img/icon_question.gif">
//. </TD></TR><TR><TH>
//. MB_ABORTRETRYIGNORE
//. </TH><TD>
//. </TD></TR><TR><TH>
//. MB_OK
//. </TH><TD>
//. </TD></TR><TR><TH>
//. MB_OKCANCEL
//. </TH><TD>
//. </TD></TR><TR><TH>
//. MB_RETRYCANCEL
//. </TH><TD>
//. </TD></TR><TR><TH>
//. MB_YESNO
//. </TH><TD>
//. </TD></TR><TR><TH>
//. MB_YESNOCANCEL
//. </TH><TD>
//. </TD></TR><TR><TH>
//. MB_DEFBUTTON1
//. </TH><TD>
//. The 1st button is the default button, which is the default
//. setting.
//. </TD></TR><TR><TH>
//. MB_DEFBUTTON2
//. </TH><TD>
//. The 2nd button is the default button.
//. </TD></TR><TR><TH>
//. MB_DEFBUTTON3
//. </TH><TD>
//. The 3rd button is the default button.
//. </TD></TR>
//. </TABLE>
//. <P>
//. Possible return values are: <B>IDACCEPT, IDABORT, IDOK, IDRETRY, IDYES,
//. IDNO, IDCANCEL</B> and <B>IDIGNORE</B>.
unsigned TOADBase::MessageBox(TWindow* parent, 
															const string &title,
															const string &text,
															ulong type,
															TBitmap *bmp,
															EWindowPlacement placement)
{
	if (bAppIsRunning) {
		TMessageBox* msg = new TMessageBox(parent,
																			 title,
																			 text, 
																			 type,
																			 bmp,
																			 placement);
		DoModal(msg);
		int result = msg->GetResult();
		delete msg;
		return result;
	}	else {
		printf("MESSAGEBOX: %s\n"
					 "            %s\n",title.c_str(),text.c_str());
	}
	return 0;
}

//. Returns the width of the screen in pixels.
int TOADBase::ScreenWidth()
{
	return WidthOfScreen(DefaultScreenOfDisplay(x11display));
}

//. Returns the height of the screen in pixels.
int TOADBase::ScreenHeight()
{
	return HeightOfScreen(DefaultScreenOfDisplay(x11display));
}

//. Returns the position of the mouse pointer relative to the
//. root window. (Root window is the whole screen background.)
void TOADBase::GetMousePos(int *x,int *y)
{
	Window w1,w2;
	int c1,c2;
	unsigned m;
	XQueryPointer(
		x11display,
		RootWindow(x11display, x11screen),
		&w1,&w2,
		x,y,
		&c1,&c2,
		&m);
}

//. Sets the position of the mouse pointer relative to the root
//. window. (Root window is the whole screen background.)
void TOADBase::SetMousePos(int x,int y)
{
	XWarpPointer(
		x11display,
		None,
		RootWindow(x11display, x11screen),
		0,0,0,0,
		x,y
	);
}

//. PlaceWindow is intended to place top level windows, e.g. dialog windows.<BR>
//. There are several modes of operation:
//. <UL>
//.   <LI>PLACE_PARENT_CENTER<BR>
//.       Center window over `parent'.
//.   <LI>PLACE_PARENT_RANDOM<BR>
//.       Randomly place window over `parent'.
//.   <LI>PLACE_SCREEN_CENTER<BR>
//.       Place window in the middle of the screen.
//.   <LI>PLACE_SCREEN_RANDOM<BR>
//.       Place window randomly on the screen.
//.   <LI>PLACE_MOUSE_POINTER<BR>
//.       Place window under the mouse pointer.
//.   <LI>PLACE_PULLDOWN<BR>
//.       Place window under the parent.
//.   <LI>PLACE_TOOLTIP<BR>
//.       Place window near the parent.
//. </UL>
//. When you use the `PARENT_' types and `parent' is NULL, the function will
//. try the applications first window and when there is none, the whole screen.
//---------------------------------------------------------------------------
void TOADBase::PlaceWindow(TWindow *window, EWindowPlacement how, TWindow *parent=NULL)
{
	if (!window) {
		cerr << __FUNCTION__ << ": window is NULL\n" << endl;
		exit(1);
	}

	TRectangle who;
	window->GetShape(&who);

	int sw = ScreenWidth();
	int sh = ScreenHeight();

	if (parent && how!=PLACE_PULLDOWN && how!=PLACE_TOOLTIP ) {
		while(!parent->bShell && parent->Parent())
			parent=parent->Parent();
	}

	TRectangle where;
	switch(how) {
		case PLACE_PARENT_RANDOM:
		case PLACE_PARENT_CENTER:
			if (parent) {
				parent->GetShape(&where);
				break;
			}
			if (twMainWindow) {
				twMainWindow->GetShape(&where);
				break;
			}
		case PLACE_SCREEN_RANDOM:
		case PLACE_SCREEN_CENTER:
		case PLACE_MOUSE_POINTER:
			where.Set(0, 0, ScreenWidth(), ScreenHeight());
			break;
		case PLACE_PULLDOWN:
		case PLACE_TOOLTIP:
			if (parent==NULL) {
				cerr << __FUNCTION__ << ": parent is NULL, can't place window" << endl;
				return;
			}
			parent->GetShape(&where);
	}

	int x,y;

	switch(how) {
		case PLACE_SCREEN_RANDOM:
		case PLACE_PARENT_RANDOM: {
			x = where.x + (where.w>>1) - (who.w>>1);
			y = where.y + (where.h>>1) - (who.h>>1);
			int xr = (int) (0.5*where.w*rand()/(RAND_MAX+1.0));
			xr-=(where.w>>2);
			x+=xr;
			int yr = (int) (0.5*where.h*rand()/(RAND_MAX+1.0));
			yr-=(where.h>>2);
			y+=yr;
			} break;
		case PLACE_SCREEN_CENTER:
		case PLACE_PARENT_CENTER:
			x = where.x + (where.w>>1) - (who.w>>1);
			y = where.y + (where.h>>1) - (who.h>>1);
			break;
		case PLACE_MOUSE_POINTER:
			GetMousePos(&x,&y);
			x-=who.w>>1;
			y-=who.h>>1;
			break;
		case PLACE_PULLDOWN:
			parent->GetRootPos(&where.x, &where.y);
			x = where.x;
			y = where.y + where.h - 1;
			if (y+who.h>sh)
				y = where.y - who.h + 1;
			break;
		case PLACE_TOOLTIP:
			parent->GetRootPos(&where.x, &where.y);
			x = where.x + (where.w>>2);
			y = where.y + where.h + 5;
			if (y+who.h>sh)
				y = where.y - who.h - 5;
			break;
	}
	
	// keep window inside the screen boundarys
	// `dist' is an additional distance to hide the additional size most
	// window managers add with their frames
	static const int dist = 32;
	if (x+who.w>sw-dist)
		x = sw-who.w-dist;
	if (y+who.h>sh-dist)
		y = sh-who.h-dist;
	if (x<dist)
		x=dist;
	if (y<dist)
		y=dist;
	window->SetPosition(x,y);
}

//. <B>Experimental</B>
//. <P>
//. 
string TOADBase::GetSelection()
{
	selection_kludge_data.erase();
	selection_kludge_flag = true;

	// XA_PRIMARY
	XConvertSelection(
		x11display,
		XA_PRIMARY,								// the primary selection
		XA_STRING,								// type
		XA_PRIMARY,								// destination property
		twMainWindow->x11window,	// destination window
		CurrentTime);

	// timeout needed!
	while(selection_kludge_flag) {
		HandleMessage();
	}
	return selection_kludge_data;
}

// SetSelection

// ClearSelection

//---------------------------------------------------------------------------

static string resource_prefix = "file://";

//. Sets the prefix for the resource files, e.g. "file://resource/" or
//. "memory://". This feature is currently used by the dialog editor.
//. <P>
//. The default value is "file://".
void TOADBase::SetResourcePrefix(const string &str)
{
	resource_prefix = str;
}

//. Returns the current resource prefix, set with SetResourcePrefix.
const string& TOADBase::ResourcePrefix()
{
	return resource_prefix;
}

// duplicated from dragndrop.cc for experiments:
//-----------------------------------------------
string GetWindowProperty(Window source, Atom property, Atom type)
{
	assert(source!=None);
	assert(property!=None);

	Atom out_type;
	int format;
	unsigned long position, received, remaining;
	position = 0L;
	unsigned char *buffer;
	string data;
	do {
		if (XGetWindowProperty(TOADBase::x11display,
													 source,
													 property,
													 position, 1024L,
													 False,
													 type,
													 &out_type,
													 &format,
													 &received,
													 &remaining,
													 &buffer) == Success )
		{
			data.append((char*)buffer, received);
			position+=received/4L;
			XFree(buffer);
		} else {
			break;
		}
	}while(remaining!=0);
	return data;
}
