/******************************************************************************
 JPrefsFile.cc

						The Basic Preferences File Class

	Base class to manage a preferences file.  Derived classes must provide
	a location for the file.  Thus, creating a J*PrefsFile object should only
	require specifying a stem for the file name.

	Each item in the JPrefsFile has an id.  This id is specified when the
	item is created, and -must- be unique.  The id is the only guaranteed way
	to get at the data after it is created.

	For convenience, SetData() automatically creates a new item if the
	given id doesn't already exist.

	We ignore the issue of the file signature because preferences files
	are usually hidden from the user and should therefore be named to avoid
	conflicts between programs.  The names serve as a sufficient signature.

	There is still the problem of more than one program trying to open
	the same file, however.  Derived classes are therefore still required
	to write constructor functions.

	BASE CLASS = JFileArray

	Copyright  1995 John Lindal. All rights reserved.

 ******************************************************************************/

#include <JPrefsFile.h>
#include <JFileArrayIndex.h>
#include <JString.h>
#include <jAssert.h>

/******************************************************************************
 Constructor (protected)

	We take a JString& so that derived classes can pass us the result of
	a function that builds and returns a JString.

 ******************************************************************************/

JPrefsFile::JPrefsFile
	(
	const JString& fileName
	)
	:
	JFileArray(fileName, "")
{
}

/******************************************************************************
 Destructor

 ******************************************************************************/

JPrefsFile::~JPrefsFile()
{
}

/******************************************************************************
 SetData

 ******************************************************************************/

void
JPrefsFile::SetData
	(
	const JPrefID&	id,
	ostrstream&		data
	)
{
	if (IDValid(id))
		{
		SetElement(id, data);
		}
	else
		{
		AppendElement(data);
		(GetFileArrayIndex())->SetElementID(GetElementCount(), id);
		}
}

void
JPrefsFile::SetData
	(
	const JPrefID&		id,
	const JCharacter*	data
	)
{
	if (IDValid(id))
		{
		SetElement(id, data);
		}
	else
		{
		AppendElement(data);
		(GetFileArrayIndex())->SetElementID(GetElementCount(), id);
		}
}
