/******************************************************************************
 JXChooseFileDialog.cc

	Clients must call BuildWindow() after constructing the object.
	Derived classes must override BuildWindow() and call SetObjects()
	at the end of their implementation.

	BASE CLASS = JXCSFDialogBase

	Copyright  1996-98 by John Lindal. All rights reserved.

 ******************************************************************************/

#include <JXChooseFileDialog.h>
#include <JXUNIXDirTable.h>
#include <JUNIXDirInfo.h>
#include <JUNIXDirEntry.h>

#include <JXWindow.h>
#include <JXStaticText.h>
#include <JXTextButton.h>
#include <JXTextCheckbox.h>
#include <JXPathInput.h>
#include <JXPathHistoryMenu.h>
#include <JXScrollbarSet.h>
#include <jXGlobals.h>

#include <JTableSelection.h>
#include <JString.h>
#include <jAssert.h>

/******************************************************************************
 Constructor

 ******************************************************************************/

JXChooseFileDialog::JXChooseFileDialog
	(
	JXDirector*			supervisor,
	JUNIXDirInfo*		dirInfo,
	const JCharacter*	fileFilter,
	const JBoolean		allowSelectMultiple
	)
	:
	JXCSFDialogBase(supervisor, dirInfo, fileFilter),
	itsSelectMultipleFlag(allowSelectMultiple)
{
}

/******************************************************************************
 Destructor

 ******************************************************************************/

JXChooseFileDialog::~JXChooseFileDialog()
{
}

/******************************************************************************
 GetFullName

 ******************************************************************************/

JBoolean
JXChooseFileDialog::GetFullName
	(
	JString* fullName
	)
	const
{
	const JUNIXDirEntry* entry;
	if ((GetFileBrowser())->GetFirstSelection(&entry))
		{
		*fullName = entry->GetFullName();
		return kTrue;
		}
	else
		{
		return kFalse;
		}
}

/******************************************************************************
 GetFullNames

 ******************************************************************************/

JBoolean
JXChooseFileDialog::GetFullNames
	(
	JPtrArray<JString>* fullNameList
	)
	const
{
	fullNameList->DeleteAll();

	JPtrArray<JUNIXDirEntry> entryList;
	if ((GetFileBrowser())->GetSelection(&entryList))
		{
		const JSize count = entryList.GetElementCount();
		for (JIndex i=1; i<=count; i++)
			{
			JString* s = new JString((entryList.NthElement(i))->GetFullName());
			assert( s != NULL );
			fullNameList->Append(s);
			}
		return kTrue;
		}
	else
		{
		return kFalse;
		}
}

/******************************************************************************
 BuildWindow

 ******************************************************************************/

void
JXChooseFileDialog::BuildWindow
	(
	const JCharacter* origName,
	const JCharacter* message
	)
{
// begin JXLayout

    JXWindow* window = new JXWindow(this, 310,310, "");
    assert( window != NULL );
    SetWindow(window);

    itsOpenButton =
        new JXTextButton("Open", window,
                    JXWidget::kFixedRight, JXWidget::kFixedBottom, 219,249, 72,22);
    assert( itsOpenButton != NULL );
    itsOpenButton->SetShortcuts("^M");

    JXTextButton* cancelButton =
        new JXTextButton("Cancel", window,
                    JXWidget::kFixedRight, JXWidget::kFixedBottom, 220,280, 70,20);
    assert( cancelButton != NULL );
    cancelButton->SetShortcuts("^[");

    JXTextButton* homeButton =
        new JXTextButton("Home", window,
                    JXWidget::kFixedRight, JXWidget::kFixedBottom, 250,110, 40,20);
    assert( homeButton != NULL );
    homeButton->SetShortcuts("#H");

    JXStaticText* pathLabel =
        new JXStaticText("Path:", window,
                    JXWidget::kFixedLeft, JXWidget::kFixedBottom, 20,20, 40,20);
    assert( pathLabel != NULL );

    JXScrollbarSet* scrollbarSet =
        new JXScrollbarSet(window,
                    JXWidget::kHElastic, JXWidget::kFixedBottom, 20,110, 180,190);
    assert( scrollbarSet != NULL );

    JXStaticText* filterLabel =
        new JXStaticText("Filter:", window,
                    JXWidget::kFixedLeft, JXWidget::kFixedBottom, 20,50, 40,20);
    assert( filterLabel != NULL );

    JXTextCheckbox* showHiddenCB =
        new JXTextCheckbox("Show hidden files", window,
                    JXWidget::kFixedLeft, JXWidget::kFixedBottom, 60,80, 130,20);
    assert( showHiddenCB != NULL );
    showHiddenCB->SetShortcuts("#S");

    JXPathInput* pathInput =
        new JXPathInput(window,
                    JXWidget::kHElastic, JXWidget::kFixedBottom, 60,20, 200,20);
    assert( pathInput != NULL );

    JXInputField* filterInput =
        new JXInputField(window,
                    JXWidget::kHElastic, JXWidget::kFixedBottom, 60,50, 200,20);
    assert( filterInput != NULL );

    JXPathHistoryMenu* pathHistory =
        new JXPathHistoryMenu(1, "", window,
                    JXWidget::kFixedRight, JXWidget::kFixedBottom, 260,20, 30,20);
    assert( pathHistory != NULL );

    JXStringHistoryMenu* filterHistory =
        new JXStringHistoryMenu(1, "", window,
                    JXWidget::kFixedRight, JXWidget::kFixedBottom, 260,50, 30,20);
    assert( filterHistory != NULL );

    JXTextButton* upButton =
        new JXTextButton("Up", window,
                    JXWidget::kFixedRight, JXWidget::kFixedBottom, 220,110, 30,20);
    assert( upButton != NULL );
    upButton->SetShortcuts("#U");

    itsSelectAllButton =
        new JXTextButton("Select all", window,
                    JXWidget::kFixedRight, JXWidget::kFixedBottom, 220,140, 70,20);
    assert( itsSelectAllButton != NULL );

// end JXLayout

	SetObjects(scrollbarSet, pathLabel, pathInput, pathHistory,
			   filterLabel, filterInput, filterHistory,
			   itsOpenButton, cancelButton, upButton, homeButton,
			   itsSelectAllButton, showHiddenCB, origName, message);
}

/******************************************************************************
 SetObjects (protected)

 ******************************************************************************/

void
JXChooseFileDialog::SetObjects
	(
	JXScrollbarSet*			scrollbarSet,
	JXStaticText*			pathLabel,
	JXPathInput*			pathInput,
	JXPathHistoryMenu*		pathHistory,
	JXStaticText*			filterLabel,
	JXInputField*			filterInput,
	JXStringHistoryMenu*	filterHistory,
	JXTextButton*			openButton,
	JXTextButton*			cancelButton,
	JXTextButton*			upButton,
	JXTextButton*			homeButton,
	JXTextButton*			selectAllButton,
	JXTextCheckbox*			showHiddenCB,
	const JCharacter*		origName,
	const JCharacter*		message
	)
{
	if (itsSelectMultipleFlag)
		{
		(scrollbarSet->GetWindow())->SetTitle("Choose files");
		}
	else
		{
		(scrollbarSet->GetWindow())->SetTitle("Choose file");
		}

	itsOpenButton      = openButton;
	itsSelectAllButton = selectAllButton;

	SetButtons(openButton, cancelButton);
	JXCSFDialogBase::SetObjects(
		scrollbarSet, pathLabel, pathInput, pathHistory,
		filterLabel, filterInput, filterHistory,
		openButton, upButton, homeButton, NULL, showHiddenCB, message);

	JXUNIXDirTable* fileBrowser = GetFileBrowser();
	fileBrowser->AllowSelectFiles(kTrue, itsSelectMultipleFlag);
	ListenTo(fileBrowser);
	ListenTo(&(fileBrowser->GetTableSelection()));

	if (itsSelectMultipleFlag)
		{
		ListenTo(itsSelectAllButton);
		}
	else
		{
		itsSelectAllButton->Hide();
		}

	// select initial file

	JIndex index;
	if (!JStringEmpty(origName) &&
		fileBrowser->ClosestMatch(origName, &index))
		{
		const JUNIXDirEntry& entry = (GetDirInfo())->GetEntry(index);
		if (entry.GetName() == origName)
			{
			fileBrowser->SelectSingleEntry(index);
			}
		}
}

/******************************************************************************
 AdjustSizings (virtual protected)

	Adjust the sizing options for the widgets in the window after the
	message has been installed at the top.

 ******************************************************************************/

void
JXChooseFileDialog::AdjustSizings()
{
	JXCSFDialogBase::AdjustSizings();
	itsSelectAllButton->SetSizing(JXWidget::kFixedRight, JXWidget::kFixedTop);
}

/******************************************************************************
 Receive (virtual protected)

 ******************************************************************************/

void
JXChooseFileDialog::Receive
	(
	JBroadcaster*	sender,
	const Message&	message
	)
{
	JXUNIXDirTable* fileBrowser = GetFileBrowser();

	if (sender == fileBrowser && message.Is(JXUNIXDirTable::kFileDblClicked))
		{
		const JXUNIXDirTable::FileDblClicked* info =
			dynamic_cast(const JXUNIXDirTable::FileDblClicked*, &message);
		assert( info != NULL );
		if (info->IsActive())
			{
			EndDialog(kTrue);
			}
		}

	else if (sender == itsSelectAllButton && message.Is(JXButton::kPushed))
		{
		fileBrowser->SelectAll();
		}

	else if (sender == fileBrowser &&
			 (message.Is(JXWidget::kGotFocus) ||
			  message.Is(JXWidget::kLostFocus)))
		{
		UpdateDisplay();
		}

	else
		{
		if (sender == &(fileBrowser->GetTableSelection()))
			{
			UpdateDisplay();
			}

		JXCSFDialogBase::Receive(sender, message);
		}
}

/******************************************************************************
 UpdateDisplay (virtual protected)

	Disable the Open button if nothing is selected.

 ******************************************************************************/

void
JXChooseFileDialog::UpdateDisplay()
{
	JXCSFDialogBase::UpdateDisplay();

	JXUNIXDirTable* dirTable = GetFileBrowser();
	if (dirTable->HasFocus() && !dirTable->HasSelection())
		{
		itsOpenButton->Deactivate();
		}
	else
		{
		itsOpenButton->Activate();
		}
}

/******************************************************************************
 OKToDeactivate (virtual protected)

 ******************************************************************************/

JBoolean
JXChooseFileDialog::OKToDeactivate()
{
	if (!JXCSFDialogBase::OKToDeactivate())
		{
		return kFalse;
		}
	else if (Cancelled())
		{
		return kTrue;
		}

	JXPathInput* pathInput = GetPathInput();
	if (pathInput->HasFocus())
		{
		GoToItsPath();
		return kFalse;
		}

	JXInputField* filterInput = GetFilterInput();
	if (filterInput->HasFocus())
		{
		AdjustFilter();
		return kFalse;
		}

	JXUNIXDirTable* fileBrowser = GetFileBrowser();
	if (fileBrowser->GoToSelectedDirectory())
		{
		return kFalse;
		}

	JPtrArray<JUNIXDirEntry> entryList;
	if (fileBrowser->GetSelection(&entryList))
		{
		const JSize count = entryList.GetElementCount();
		for (JIndex i=1; i<=count; i++)
			{
			JUNIXDirEntry* entry = entryList.NthElement(i);
			entry->ForceUpdate();	// check that link hasn't been broken behind our back
			if (!entry->IsFile())
				{
				(GetDirInfo())->ForceUpdate();
				return kFalse;
				}
			}
		return kTrue;
		}
	else
		{
		return kFalse;
		}
}
