/******************************************************************************
 JXIntegerInput.cc

	Maintains a line of editable text for entering an integer value.

	BASE CLASS = JXInputField

	Copyright  1996 by John Lindal. All rights reserved.

 ******************************************************************************/

#include <JXIntegerInput.h>
#include <jXGlobals.h>
#include <JString.h>
#include <JMinMax.h>
#include <jMath.h>
#include <ctype.h>
#include <jAssert.h>

/******************************************************************************
 Constructor

 ******************************************************************************/

JXIntegerInput::JXIntegerInput
	(
	JXContainer*		enclosure,
	const HSizingOption	hSizing,
	const VSizingOption	vSizing,
	const JCoordinate	x,
	const JCoordinate	y,
	const JCoordinate	w,
	const JCoordinate	h
	)
	:
	JXInputField(enclosure, hSizing, vSizing, x,y, w,h)
{
	SetIsRequired();
	itsHasLowerLimitFlag = itsHasUpperLimitFlag = kFalse;
	SetValue(0);
}

/******************************************************************************
 Destructor

 ******************************************************************************/

JXIntegerInput::~JXIntegerInput()
{
}

/******************************************************************************
 GetValue

 ******************************************************************************/

JBoolean
JXIntegerInput::GetValue
	(
	JInteger* value
	)
	const
{
	return JConvertToBoolean(
			(GetText()).ConvertToInteger(value) && ValueValid(*value) );
}

/******************************************************************************
 SetValue

 ******************************************************************************/

void
JXIntegerInput::SetValue
	(
	const JInteger value
	)
{
	if (ValueValid(value))
		{
		const JString text(value, 0);
		if (text != GetText())
			{
			SetText(text);
			}
		}
}

/******************************************************************************
 Set limits

 ******************************************************************************/

void
JXIntegerInput::SetLimits
	(
	const JInteger minValue,
	const JInteger maxValue
	)
{
	assert( minValue < maxValue );
	SetLowerLimit(minValue);
	SetUpperLimit(maxValue);
}

void
JXIntegerInput::SetLowerLimit
	(
	const JInteger minValue
	)
{
	assert( !itsHasUpperLimitFlag || minValue < itsUpperLimit );

	itsHasLowerLimitFlag = kTrue;
	itsLowerLimit = minValue;

	JInteger v;
	if (!GetValue(&v))
		{
		SetValue(itsLowerLimit);
		}
}

void
JXIntegerInput::SetUpperLimit
	(
	const JInteger maxValue
	)
{
	assert( !itsHasLowerLimitFlag || itsLowerLimit < maxValue );

	itsHasUpperLimitFlag = kTrue;
	itsUpperLimit = maxValue;

	JInteger v;
	if (!GetValue(&v))
		{
		SetValue(itsUpperLimit);
		}
}

/******************************************************************************
 InputValid (virtual)

 ******************************************************************************/

JBoolean
JXIntegerInput::InputValid()
{
	if (!JXInputField::InputValid())
		{
		return kFalse;
		}
	else
		{
		const JString& text = GetText();

		if (!IsRequired() && text.IsEmpty())
			{
			return kTrue;
			}

		JInteger value;
		if (!text.ConvertToInteger(&value))
			{
			(JGetUserNotification())->ReportError("This field accepts only integers.");
			return kFalse;
			}

		const JBoolean valid = ValueValid(value);
		if (!valid && itsHasLowerLimitFlag && itsHasUpperLimitFlag)
			{
			JString errorStr  = "This field accepts only integers between ";
			errorStr += JString(itsLowerLimit, 0);
			errorStr += " and ";
			errorStr += JString(itsUpperLimit, 0);
			errorStr += " inclusive.";
			(JGetUserNotification())->ReportError(errorStr);
			}
		else if (!valid && itsHasLowerLimitFlag)
			{
			JString errorStr  = "This field accepts only integers greater than or equal to ";
			errorStr += JString(itsLowerLimit, 0);
			errorStr += ".";
			(JGetUserNotification())->ReportError(errorStr);
			}
		else if (!valid && itsHasUpperLimitFlag)
			{
			JString errorStr  = "This field accepts only integers less than or equal to ";
			errorStr += JString(itsUpperLimit, 0);
			errorStr += ".";
			(JGetUserNotification())->ReportError(errorStr);
			}

		return valid;
		}
}
