/*
 Copyright (C) 1999 Gerald L. Gay

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Library General Public License
 version 2 as published by the Free Software Foundation.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Library General Public License for more details.

 You should have received a copy of the GNU Library General Public
 License along with this library; see the file COPYING.  If not,
 write to the Free Software Foundation, Inc., 675 Mass Ave,
 Cambridge, MA 02139, USA.
*/

/* Include(s) */
#include "setup.h"

// The panels are all defined here...

WelcomePanel::WelcomePanel(Widget parent, XarmArg &formArgs, char *name)
  : SetupSubpanel(parent, formArgs, name)
{
    XarmArg args;

    setTitle(" Welcome");

    args.reset();
    args(XmNleftAttachment,  XmATTACH_FORM)
        (XmNtopAttachment,   XmATTACH_WIDGET)
        (XmNtopWidget,       titleLabel)
        (XmNrightAttachment, XmATTACH_FORM)
        (XmNrightOffset,     5)
        (XmNleftOffset,      5)
        (XmNtopOffset,       30);

    intro = new Label(*myForm, args, "introLabel");

    intro->labelString("Welcome to Xarm " XarmNumericVersion );

    args.reset();
    args(XmNleftAttachment,  XmATTACH_FORM)
        (XmNrightAttachment, XmATTACH_FORM)
        (XmNtopAttachment,   XmATTACH_WIDGET)
        (XmNtopWidget,       intro)
        (XmNleftOffset,      30)
        (XmNtopOffset,       20)
        (XmNrightOffset,     5)
        (XmNalignment,       XmALIGNMENT_BEGINNING);

    swLabel = new Label(*myForm, args, "swLabel");

    swLabel->labelString("Xarm provides a C++ interface to the Motif and\n"
			 "Xmext Libraries and also provides support for\n"
			 "some CDE features as well as the Xbae Matrix\n"
			 "and Caption Widgets.\n\n"
			 "You are welcome to use the included source to\n"
			 "this setup utility as a base for your own\n"
			 "installation GUI's.\n\n"
			 "This utility will extract and optionally\n"
			 "configure, compile and install Xarm "
             XarmNumericVersion
			 "\non your system.");

}

DirChooserPanel::DirChooserPanel(Widget parent, XarmArg &formArgs, char *name)
  : SetupSubpanel(parent, formArgs, name)
{
    XarmArg args;

    setTitle(" Choose Directory");
    sourceLocation = NULL;

    args.reset();
    args(XmNleftAttachment,  XmATTACH_FORM)
        (XmNtopAttachment,   XmATTACH_WIDGET)
        (XmNtopWidget,       titleLabel)
        (XmNrightAttachment, XmATTACH_FORM)
        (XmNrightOffset,     5)
        (XmNleftOffset,      20)
        (XmNtopOffset,       30)
        (XmNalignment,       XmALIGNMENT_BEGINNING);

    dirPrompt = new Label(*myForm, args, "dirPrompt");

    dirPrompt->labelString("Please enter the directory where you would\n"
			   "like to extract the Xarm source.\n\n"
			   "Files will be installed under:\n\n"
			   "[Directory Name]/Xarm-" XarmNumericVersion );

    args.reset();
    args(XmNleftAttachment,     XmATTACH_FORM)
        (XmNrightAttachment,    XmATTACH_FORM)
        (XmNtopAttachment,      XmATTACH_WIDGET)
        (XmNtopWidget,          dirPrompt)
        (XmNleftOffset,         30)
        (XmNtopOffset,          20)
        (XmNrightOffset,        5)
        (XmNhighlightThickness, 0);

    directory = new TextField(*myForm, args, "directory");

    // Here's something that really bugs me: I hate Text Fields
    // that leave the text cursor visible while you're not
    // entering data into it.  It's distracting and it can make
    // the text under the grey'd out cursor hard to read. So
    // I turn it off and only turn it on when the user is editing
    // the text field.
    directory->cursorPositionVisible(False);
    addCallback(this, &DirChooserPanel::tfGetFocus,  directory->widget(), XmNfocusCallback);
    addCallback(this, &DirChooserPanel::tfLoseFocus, directory->widget(), XmNlosingFocusCallback);

    char dirBuffer[1024];

    if (getcwd(dirBuffer, 1024) != NULL) {
        directory->value(dirBuffer);
    }
}

void DirChooserPanel::tfGetFocus(Widget, XtPointer, XtPointer)
{
    directory->cursorPositionVisible(True);
}

void DirChooserPanel::tfLoseFocus(Widget, XtPointer, XtPointer)
{
    directory->cursorPositionVisible(False);
}

bool DirChooserPanel::formOK()
{

    if (sourceLocation != NULL) {
        XtFree(sourceLocation);
    }

    sourceLocation = directory->value();

    if (sourceLocation == NULL) return false;

    int len = strlen(sourceLocation);

    if (len == 0) {
        theApp->warning("You must specify a parent directory\n"
			"under which to install Xarm " XarmNumericVersion );
	return false;
    }

    // See if there's a slash on the end already!
    // If there is, remove it.
    if (sourceLocation[len - 1] == '/') {
        sourceLocation[len - 1] = '\0';
	--len;
    }

    // Attempt to touch a file in the directory provided.

    char *fName = new char[len + 9];

    if (fName == NULL) {
        theApp->fatal("Memory allocation error\nin DirChooserPanel");
	return false;
    }

    sprintf(fName, "%s/XarmTmp", sourceLocation);

    ofstream fout(fName);

    if (!fout) {
        theApp->warning("Setup couldn't write to the directory\n"
			"provided. Make sure you have write\n"
			"permissions on the source directory.");
	delete [] fName;
	return false;
    }

    // That seemed to work. Close and remove the file.
    fout.close();
    remove(fName);

    delete [] fName;

    return true;
}

static char *xbaePrompt = "The Xarm configuration script only looks\n"
                          "in a few places for the Xbae files. You\n"
			  "can specify where to find Xbae if the\n"
			  "default search fails. The directory provided\n"
			  "here should contain include/Xbae and\n"
			  "lib/libXbae. Leaving this blank will cause\n"
			  "the Xarm configure script to search it's\n"
			  "built-in list of locations.";

XbaeDirPanel::XbaeDirPanel(Widget parent, XarmArg &formArgs, char *name)
  : StringPanel(parent, formArgs, name, " Specify Xbae Directory", xbaePrompt)
{
    // Nothing to do here!
}

bool XbaeDirPanel::validate()
{

    if (strValue == NULL) return true;

    int len = strlen(strValue);

    if (len == 0) return true;

    // See if there's a slash on the end already!
    // If there is, remove it.
    if (strValue[len - 1] == '/') {
        strValue[len - 1] = '\0';
	--len;
    }

    // Attempt to find the Xbae include directory

    char *fName = new char[len + 14];

    if (fName == NULL) {
        theApp->fatal("Memory allocation error\nin XbaeDirPanel");
	return false;
    }

    sprintf(fName, "%s/include/Xbae", strValue);

    struct stat statBuf;
    int         result;

    result = stat(fName, &statBuf);

    delete [] fName;

    // Here is one of those difficult choices. I can't think
    // of any reason you would want to configure Xarm to use
    // Xbae in a directory that doesn't have the right files
    // in it but you just never know!  If I return false from
    // here then I am forcing the user to enter a valid directory
    // based on what I know about Xbae. The user may have another
    // idea!  So for now, I'm not going to make this mandatory.

    if (result != 0) {
        theApp->warning("Setup couldn't find include/Xbae under\n"
			"the directory given. Please check your\n"
			"Xbae installation.");
    }

    return true;
}

static char *glPrompt = "The Xarm configuration script only looks\n"
                         "in a few places for the Mesa/OpenGL (GL) files.\n"
                         "You can specify where to find GL if the\n"
			 "default search fails. The directory provided\n"
			 "here should contain include/GL and\n"
			 "lib/libGLwM. Leaving this blank will cause\n"
			 "the Xarm configure script to search it's\n"
			 "built-in list of locations.";

GLDirPanel::GLDirPanel(Widget parent, XarmArg &formArgs, char *name)
  : StringPanel(parent, formArgs, name, " Specify GL Directory", glPrompt)
{
    // Nothing to do here!
}

bool GLDirPanel::validate()
{

    if (strValue == NULL) return true;

    int len = strlen(strValue);

    if (len == 0) return true;

    // See if there's a slash on the end already!
    // If there is, remove it.
    if (strValue[len - 1] == '/') {
        strValue[len - 1] = '\0';
	--len;
    }

    // Attempt to find the GL include directory

    char *fName = new char[len + 14];

    if (fName == NULL) {
        theApp->fatal("Memory allocation error\nin GLDirPanel");
	return false;
    }

    sprintf(fName, "%s/include/GL", strValue);

    struct stat statBuf;
    int         result;

    result = stat(fName, &statBuf);

    delete [] fName;

    // Here is one of those difficult choices. I can't think
    // of any reason you would want to configure Xarm to use
    // GL in a directory that doesn't have the right files
    // in it but you just never know!  If I return false from
    // here then I am forcing the user to enter a valid directory
    // based on what I know about Mesa. The user may have another
    // idea!  So for now, I'm not going to make this mandatory.

    if (result != 0) {
        theApp->warning("Setup couldn't find include/GL under\n"
			"the directory given. Please check your\n"
			"Mesa or OpenGL installation.");
    }

    return true;
}

static char *installDirPrompt = "The Xarm Library installs under /usr/local\n"
                                "by default. To install the library and header\n"
                                "files in an alternate location, enter the\n"
                                "full path name here:";

InstallDirPanel::InstallDirPanel(Widget parent, XarmArg &formArgs, char *name)
  : StringPanel(parent, formArgs, name, " Specify Installation Directory", installDirPrompt)
{
    // Nothing to do!!!
}

bool InstallDirPanel::validate()
{

    if (strValue == NULL) return true;

    int len = strlen(strValue);

    if (len == 0) return true;

    // See if there's a slash on the end already!
    // If there is, remove it.
    if (strValue[len - 1] == '/') {
        strValue[len - 1] = '\0';
	--len;
    }

    // Make sure that the path is absolute

    if (*strValue != '/') {
        theApp->warning("Please use an absolute path name\n"
			"for the installation directory.\n"
			"An absolute path name starts with\n"
			"a slash (/) character.");
	return false;
    }

    return true;
}

ReadyPanel::ReadyPanel(Widget parent, XarmArg &formArgs, char *name)
  : SetupSubpanel(parent, formArgs, name)
{
    XarmArg args;

    setTitle(" Ready to Install");

    args.reset();
    args(XmNleftAttachment,  XmATTACH_FORM)
        (XmNtopAttachment,   XmATTACH_WIDGET)
        (XmNtopWidget,       titleLabel)
        (XmNrightAttachment, XmATTACH_FORM)
        (XmNrightOffset,     5)
        (XmNleftOffset,      5)
        (XmNtopOffset,       30);

    readyPrompt = new Label(*myForm, args, "readyPrompt");

    readyPrompt->labelString("Setup is now ready to install the Xarm Library.");
}

ThankYouPanel::ThankYouPanel(Widget parent, XarmArg &formArgs, char *name)
  : SetupSubpanel(parent, formArgs, name)
{
    XarmArg args;

    setTitle(" Setup Completed");

    args.reset();
    args(XmNleftAttachment,  XmATTACH_FORM)
        (XmNtopAttachment,   XmATTACH_WIDGET)
        (XmNtopWidget,       titleLabel)
        (XmNrightAttachment, XmATTACH_FORM)
        (XmNrightOffset,     5)
        (XmNleftOffset,      5)
        (XmNtopOffset,       30);

    thanksPrompt = new Label(*myForm, args, "thanksPrompt");

    thanksPrompt->labelString("Setup has completed all requested operations.\n\n"
			      "I hope you enjoy using the Xarm library.\n\n"
			      "The latest version is always available at\n"
			      "http://www.xarm.org\n\n"
			      "If you have any comments, questions, complaints\n"
			      "or bug reports, send an email to\n"
			      "glgay@xarm.org");
}

			      

