/*
 Copyright (C) 1999 Gerald L. Gay

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Library General Public License
 version 2 as published by the Free Software Foundation.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Library General Public License for more details.

 You should have received a copy of the GNU Library General Public
 License along with this library; see the file COPYING.  If not,
 write to the Free Software Foundation, Inc., 675 Mass Ave,
 Cambridge, MA 02139, USA.
*/

#if !defined(setup_h)
#define setup_h

// I put all the includes here so that
// all the C files can just include "setup.h"
// without worrying.

#include <string>
#include <vector>
#include <iostream.h>
#include <fstream.h>
#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <Xarm/AppContext.h>
#include <Xarm/Form.h>
#include <Xarm/Frame.h>
#include <Xarm/Label.h>
#include <Xarm/ToggleB.h>
#include <Xarm/RowColumn.h>
#include <Xarm/Text.h>
#include <Xarm/TextF.h>
#include <Xarm/Gauge.h>
#include <Xarm/Separator.h>
#include <Xarm/PushB.h>
#include <Xarm/MessageB.h>
#include <Xarm/XarmXpm.h>
#include <Xarm/Callback.h>
#include <Xm/MwmUtil.h>

// Personally, I think this whole std namespace thing
// is a royal pain!
#if defined(XARM_HAS_NAMESPACES)
using namespace std;
#endif

//************************************************************
//* SetupSubpanel is the base class for all the panels that  *
//* appear.  It provides the base interface that all panels  *
//* are required to implement.                               *
//*                                                          *
//* manage()/unmanage() can be used to make decisions when   *
//*                     the panel is about to be displayed.  *
//*                                                          *
//* formOK() is called when the user clicks "Next" and       *
//*          before unmanage(). If the panel isn't happy     *
//*          with it's current settings, returning false     *
//*          will prevent the user from moving on.           *
//************************************************************

class SetupSubpanel
{

 protected:

    Form  *myForm;
    Label *titleLabel;

 public:

    SetupSubpanel(Widget parent, XarmArg &args, char *name);

    void setTitle(const char *str) { titleLabel->labelString(str); }

    virtual bool formOK()          { return true;                  }

    virtual void manage()   = 0;
    virtual void unmanage() = 0;
};

typedef vector<SetupSubpanel *> PanelVec;
typedef vector<int>             FormSequence;

//************************************************************
//* YesNoPanel gets a simple Yes or No answer from the user. *
//************************************************************

class YesNoPanel : public SetupSubpanel
{

 protected:

    bool          answer;

    Label        *prompt;
    RowColumn    *choiceBox;
    ToggleButton *yesButton;
    ToggleButton *noButton;

    // Track the state of the buttons
    void valueTracker(Widget, XtPointer, XtPointer);

 public:

    YesNoPanel(Widget parent, XarmArg &args, char *name,
	       const char *title, const char *text, bool defAnswer);

    virtual void manage()   { myForm->manage();   }
    virtual void unmanage() { myForm->unmanage(); }
    virtual bool formOK()   { return true;        }

    bool value()            { return answer;      }
};

//************************************************************
//* StringPanel gets a string value from the user.           *
//************************************************************

class StringPanel : public SetupSubpanel
{

    Label     *prompt;
    TextField *entry;

    void tfGetFocus(Widget, XtPointer, XtPointer);
    void tfLoseFocus(Widget, XtPointer, XtPointer);

protected:

    char      *strValue;

public:

    StringPanel(Widget parent, XarmArg &args, char *name,
	        const char *title, const char *promptText, const char *defValue = NULL);

    virtual void manage()   { myForm->manage();   }
    virtual void unmanage() { myForm->unmanage(); }

    char *value();

    virtual bool formOK();

    // validate() will be called from formOK() who will
    // return it's value. This way, to validate data
    // on a StringPanel, just override this one function.

    virtual bool validate() { return true; }
};

//************************************************************
//*               Now the custom panels...                   *
//************************************************************

class WelcomePanel : public SetupSubpanel
{

    Label *intro;
    Label *swLabel;

 public:

    WelcomePanel(Widget parent, XarmArg &args, char *name);

    virtual void manage()          { myForm->manage();   }
    virtual void unmanage()        { myForm->unmanage(); }
};

// DirChooserPanel isn't inherited from StringPanel
// because I changed his attachments just a little
// to make it look better. Beauty to the user always
// wins over beauty in the code!

class DirChooserPanel : public SetupSubpanel
{

    Label     *dirPrompt;
    TextField *directory;
    char      *sourceLocation;

    void tfGetFocus(Widget, XtPointer, XtPointer);
    void tfLoseFocus(Widget, XtPointer, XtPointer);

public:

    DirChooserPanel(Widget parent, XarmArg &args, char *name);

    virtual void manage()   { myForm->manage();      }
    virtual void unmanage() { myForm->unmanage();    }

    char *value()    { return sourceLocation; }

    virtual bool formOK();
};

class XbaeDirPanel : public StringPanel
{

public:

    XbaeDirPanel(Widget parent, XarmArg &args, char *name);

    virtual bool validate();
};

class GLDirPanel : public StringPanel
{

public:

    GLDirPanel(Widget parent, XarmArg &args, char *name);

    virtual bool validate();
};

class InstallDirPanel : public StringPanel
{

public:

    InstallDirPanel(Widget parent, XarmArg &args, char *name);

    virtual bool validate();
};

class ReadyPanel : public SetupSubpanel
{

    Label *readyPrompt;

public:

    ReadyPanel(Widget parent, XarmArg &args, char *name);

    virtual void manage()   { myForm->manage();   }
    virtual void unmanage() { myForm->unmanage(); }
    virtual bool formOK()   { return true;        }
};

// Define a structure to hold all the answers...

struct XarmOptions {
    string extractDir;
    bool   wantConfigure;
    bool   wantShared;
    bool   wantStatic;
    bool   wantCDE;
    bool   wantXbae;
    string xbaeDir;
    bool   wantGL;
    string glDir;
    bool   wantExamples;
    string installDir;
    string otherOptions;
    bool   wantCompile;
    bool   wantInstall;
};

class InstallingPanel : public SetupSubpanel
{
    Frame        *myFrame;
    ScrolledText *monitor;
    Label        *currAction;
    Gauge        *gauge;
    ToggleButton *waitHere;
    XarmOptions   myOptions;
    int           totalProgress;
    int           currProgress;
    int           state;
    bool          busy;

    FILE         *workFP;

    Boolean       installWork(XtPointer);

    void inputWatcher(XtPointer, int *, XtInputId *);

    void extractTarball();
    void configureXarm();
    void compile();
    void makeInstall();

public:

    InstallingPanel(Widget parent, XarmArg &args, char *name);

    void setOptions(const XarmOptions &);

    virtual void manage();
    virtual void unmanage() { myForm->unmanage(); }
    virtual bool formOK()   { return true;        }
};

class ThankYouPanel : public SetupSubpanel
{

    Label     *thanksPrompt;

public:

    ThankYouPanel(Widget parent, XarmArg &args, char *name);

    virtual void manage()   { myForm->manage();   }
    virtual void unmanage() { myForm->unmanage(); }
};

class Application : public AppContext
{

    PanelVec     panels;
    FormSequence sequence;

    Pixmap      logo;
    Form       *theForm;
    Label      *logoLabel;
    Separator  *sep;
    PushButton *backButton;
    PushButton *nextButton;
    PushButton *exitButton;
    RowColumn  *rc;
    Label      *nextLabel;
    Label      *backLabel;

    QuestionDialog *exitDialog;
    ErrorDialog    *fatalErr;
    WarningDialog  *warnDialog;

    bool allFinished;

    void nextCallback(Widget, XtPointer, XtPointer);
    void backCallback(Widget, XtPointer, XtPointer);
    void onExit(Widget, XtPointer, XtPointer);
    void reallyExit(Widget, XtPointer, XtPointer);

    void doInstall();

 public:

    Application(char *app_class,int &argc_in_out, char **argv_in_out);

    void fatal(char *message);
    void warning(char *message);
    void done();
    void almostDone();

};

// Some really useful values...

extern int   XarmTarballSize;
extern int   XarmFileCount;
extern int   XarmConfCount;
extern int   XarmMakeAllCount;
extern int   XarmMakeNoSharedCount;
extern int   XarmMakeNoStaticCount;
extern int   XarmMakeInstallCount;
extern int   XarmMakeInstallExamples;
extern int   XarmExamplesCount;
extern char *XarmTarballName;

extern string setupFile;

// It's useful to have a pointer to the application around
extern Application *theApp;

#endif
