/* ************************************************************************ 
 *         The Amulet User Interface Development Environment              *
 * ************************************************************************
 * This code was written as part of the Amulet project at                 *
 * Carnegie Mellon University, and has been placed in the public          *
 * domain.  If you are using this code or any part of Amulet,             *
 * please contact amulet@cs.cmu.edu to be put on the mailing list.        *
 * ************************************************************************/

extern "C" {
#include <stdlib.h>  // For abort ()
}

#if defined(GCC)
#include <string.h>
#else
extern "C" {
#ifdef _MSC_VER
#include <string.h>
#else
#include <strings.h>
#endif
}
#endif

#include <am_inc.h>

#include AM_IO__H

#include TYPES__H
#include REGISTRY__H
#include STDVALUE__H

void Am_Error (const char* error_string)
{
  cerr << "** Amulet Error: " << error_string << endl;
  cerr << "**  Program aborted." << endl;
#if defined(_WINDOWS) || defined(_MACINTOSH)
  cerr << "(press return to exit)" << endl;
  getchar();
#endif
  abort ();
}

void Am_Error ()
{
  cerr << "**  Program aborted." << endl;
#if defined(_WINDOWS) || defined(_MACINTOSH)
  cerr << "(press return to exit)" << endl;
  getchar();
#endif
  abort ();
}

Am_ID_Tag Am_Get_Unique_ID_Tag (const char* type_name, Am_ID_Tag in_class)
{
#define _FIRST_TAG 500
  static Am_ID_Tag current_tag = _FIRST_TAG;
  Am_ID_Tag tag = current_tag | in_class;
  if (Am_Type_Base (in_class))
    tag = in_class;
  else
    current_tag++;
  if (current_tag < _FIRST_TAG) // this should never happen.
    Am_Error ("*** Am_Get_Unique_ID_Tag: overflow!  Too many tags!");
#ifdef DEBUG
  if (type_name) Am_Register_Type_Name(tag, type_name);
#endif
  return tag;
}

Am_ID_Tag Am_Set_ID_Class (const char* type_name, Am_ID_Tag tag)
{
#ifdef DEBUG
  Am_Register_Type_Name (tag, type_name);
#endif
  return tag;
}

void Am_Wrapper::Print_Name(ostream& out) {
#ifdef DEBUG
  const char *name = Am_Get_Name_Of_Item(this);
  if (name) out << name;
  else {
    const char *type_name = Am_Get_Type_Name(ID());
    out << type_name << ":" << hex << (Am_Ptr)this << dec;
  }
#else
  out << hex << (Am_Ptr)this << dec;
#endif
}

static Am_Value_Type FONE    = Am_Set_ID_Class ("NO VALUE", 0x0000);
static Am_Value_Type WRAPPER = Am_Set_ID_Class ("wrapper",  0x1000);
static Am_Value_Type METHOD  = Am_Set_ID_Class ("method",   0x2000);

static Am_Value_Type NONE    = Am_Set_ID_Class ("NO VALUE", 0x0000);
static Am_Value_Type UNINIT  = Am_Set_ID_Class ("UNINITIALIZED", Am_UNINIT);
static Am_Value_Type INT     = Am_Set_ID_Class ("int",    Am_INT);
static Am_Value_Type LONG    = Am_Set_ID_Class ("long",   Am_LONG);
static Am_Value_Type BOOL    = Am_Set_ID_Class ("bool",   Am_BOOL);
static Am_Value_Type FLOAT   = Am_Set_ID_Class ("float",  Am_FLOAT);
static Am_Value_Type DOUB    = Am_Set_ID_Class ("double", Am_DOUBLE);
static Am_Value_Type STRIN   = Am_Set_ID_Class ("string", Am_STRING);
static Am_Value_Type CHAR    = Am_Set_ID_Class ("char",   Am_CHAR);
static Am_Value_Type VOID    = Am_Set_ID_Class ("void*",  Am_VOIDPTR);
static Am_Value_Type PROC  = Am_Set_ID_Class ("Am_Generic_Procedure*", Am_PROC);

void Am_Print_Type (ostream& os, Am_Value_Type type)
{
  os << Am_Get_Type_Name (type);
}

/////////////////////////////////////////////////////////////////
// The Am_String procedures
/////////////////////////////////////////////////////////////////

Am_WRAPPER_IMPL (Am_String);

Am_String Am_No_String;

Am_String::Am_String (const char* string, bool copy)
{
  data = new Am_String_Data (string, copy);
}

Am_String& Am_String::operator= (const char* string)
{
  if (data)
    data->Release ();
  data = new Am_String_Data (string);
  return *this;
}

Am_String::operator const char* () const
{
  if (data)
    return *data;
  else
    return NULL;
}

Am_String::operator char* ()
{
  if (data) {
    data = (Am_String_Data*)data->Make_Unique ();
    return *data;
  }
  else
    return NULL;
}

Am_String::operator== (const Am_String& test_string) const
{
  if (data && test_string.data) {
    const char* string = *data;
    return (string == (const char*)test_string) ||
           !strcmp (string, (const char*)test_string);
  }
  else
    return data == test_string.data;
}

Am_String::operator== (const char* test_string) const
{
  if (data) {
    const char* string = *data;
    return (string == test_string) ||
        !strcmp (string, test_string);
  }
  else
    return test_string == NULL;
}

ostream& operator<< (ostream& os, const Am_String& string)
{
  os << (const char*) string;
  return os;
}

/////////////////////////////////////////////////////////////////
// The Am_Value type procedures
/////////////////////////////////////////////////////////////////

Am_Value Am_No_Value;

inline bool wrapper_type (Am_Value_Type type)
{
  return Am_Type_Class (type) == Am_WRAPPER || type == Am_STRING ||
         type == Am_DOUBLE;
}
        
Am_Value::Am_Value (Am_Wrapper* initial)
{
  value.wrapper_value = initial;
  if (initial)
    type = (Am_Value_Type)initial->ID ();
  else
    type = Am_WRAPPER;
}

Am_Value::Am_Value (double initial)
{
  type = Am_DOUBLE;
  value.wrapper_value = new Am_Double_Data (initial);
}

Am_Value::Am_Value (const char* initial)
{
  type = Am_STRING;
  value.wrapper_value = new Am_String_Data (initial);
}

Am_Value::Am_Value (const Am_String& initial)
{
  type = Am_STRING;
  value.wrapper_value = initial;
}

Am_Value::Am_Value (Am_Method_Wrapper* initial)
{
  value.method_value = initial;
  if (initial)
    type = (Am_Value_Type)initial->ID ();
  else
    type = Am_METHOD;
}

Am_Value::Am_Value (const Am_Value& initial)
{
  type = initial.type;
  value = initial.value;
  if (wrapper_type (type) && value.wrapper_value)
    value.wrapper_value->Note_Reference ();
}

Am_Value::~Am_Value ()
{
  if (wrapper_type (type) && value.wrapper_value)
    value.wrapper_value->Release ();
}

static void type_error (const char* type_name, const Am_Value &value)
{
  cerr << "** Stored value of Am_Value is not of " << type_name
       << " type." << endl;
  cerr << "** It contains a value of type ";
  Am_Print_Type (cerr, value.type);
  cerr << "." << endl;
  Am_Error ();
}

Am_Value::operator Am_Wrapper* () const
{
  switch (Am_Type_Class (type)) {
  case Am_WRAPPER:
    if (value.wrapper_value)
      value.wrapper_value->Note_Reference ();
    return value.wrapper_value;
  case Am_NONE:
    switch (type) {
    case Am_STRING:
    case Am_DOUBLE:
      if (value.wrapper_value)
        value.wrapper_value->Note_Reference ();
      return value.wrapper_value;
    case Am_UNINIT:
      return NULL;
    case Am_INT:
    case Am_LONG:
    case Am_VOIDPTR:
    case Am_PROC:
      if (!value.voidptr_value)
        return NULL;
    }
  default:
    type_error ("Am_Wrapper*", *this);
    return NULL;
  }
}

Am_Value::operator Am_Ptr () const
{
  return (Am_Ptr)value.voidptr_value;
}

Am_Value::operator int () const
{
  switch (type) {
  case Am_INT:
  case Am_LONG:
    return (int)value.long_value; //use compiler's conversion
  case Am_BOOL:
    if (value.bool_value) return 1;
    else return 0;
  case Am_UNINIT:
    return 0;
  default:
    type_error ("int", *this);
    return 0;
  }
}

Am_Value::operator long () const
{
  switch (type) {
  case Am_LONG:
  case Am_INT:
    return value.long_value;
  case Am_FLOAT:
    return (long)value.float_value;
  case Am_DOUBLE:
    return (long)(double)*(Am_Double_Data*)value.wrapper_value;
  case Am_BOOL:
    if (value.bool_value) return 1L;
    else return 0L;
  case Am_UNINIT:
    return 0;
  default:
    type_error ("long", *this);
    return 0L;
  }
}

#ifndef NEED_BOOL
Am_Value::operator bool () const
{
  if (Am_Type_Class (type) != Am_NONE)
    return !!value.voidptr_value;
  switch (type) {
  case Am_INT:
  case Am_LONG:
    return !!value.long_value;
  case Am_BOOL:
    return value.bool_value;
  case Am_FLOAT:
    return !!value.float_value;
  case Am_DOUBLE:
    return !!(double)*(Am_Double_Data*)value.wrapper_value;
  case Am_CHAR:
    return !!value.char_value;
  case Am_STRING:
    return !!(const char*)*(Am_String_Data*)value.wrapper_value;
  case Am_VOIDPTR:
    return !!value.voidptr_value;
  case Am_PROC:
    return !!value.proc_value;
  case Am_METHOD:
    return !!value.method_value;
  case Am_UNINIT:
    return false;
  default:
    type_error ("bool", *this);
    return false;
  }
}
#endif

Am_Value::operator float () const
{
  switch (type) {
  case Am_FLOAT:
    return value.float_value;
  case Am_DOUBLE:
    return *(Am_Double_Data*)value.wrapper_value;
  case Am_INT:
  case Am_LONG:
    return (float)value.long_value;
  case Am_UNINIT:
    return 0.0f;
  default:
    type_error ("float", *this);
    return 0.0f;
  }
}

Am_Value::operator double () const
{
  switch (type) {
  case Am_FLOAT:
    return value.float_value;
  case Am_DOUBLE:
    return *(Am_Double_Data*)value.wrapper_value;
  case Am_INT:
  case Am_LONG:
    return value.long_value;
  case Am_UNINIT:
    return 0.0;
  default:
    type_error ("double", *this);
    return 0.0;
  }
}

Am_Value::operator char () const
{
  switch (type) {
  case Am_CHAR:
    return value.char_value;
  case Am_UNINIT:
    return '\0';
  default:
    type_error ("char", *this);
	  }
  return NULL; // should never get here, but need to return something

}

Am_Value::operator Am_Generic_Procedure* () const
{
  if (Am_Type_Class (type) == Am_METHOD) {
    if (value.method_value)
      type_error ("Am_Generic_Procedure*", *this);
    return NULL;
  }
  switch (type) {
  case Am_PROC:
    return value.proc_value;
  case Am_UNINIT:
    return NULL;
  case Am_INT:
  case Am_LONG:
  case Am_VOIDPTR:
    if (!value.voidptr_value)
      return NULL;
  default:
    type_error ("Am_Generic_Procedure*", *this);
    return NULL;
  }
}

Am_Value::operator Am_Method_Wrapper* () const
{
  if (Am_Type_Class (type) == Am_METHOD)
    return value.method_value;
  switch (type) {
  case Am_UNINIT:
    return NULL;
  case Am_PROC:
  case Am_INT:
  case Am_LONG:
  case Am_VOIDPTR:
    if (!value.voidptr_value)
      return NULL;
  default:
    type_error ("Am_Method_Wrapper*", *this);
    return NULL;
  }
}

Am_Value::operator== (Am_Wrapper* test_value) const
{
  if (Am_Type_Class (type) == Am_WRAPPER)
    return (value.wrapper_value == test_value) ||
      (test_value && value.wrapper_value &&
       (*value.wrapper_value == *test_value));
  switch (type) {
  case Am_STRING:
  case Am_DOUBLE:
    return (value.wrapper_value == test_value) ||
      (test_value && value.wrapper_value &&
       (*value.wrapper_value == *test_value));
  case Am_INT:
  case Am_LONG:
  case Am_VOIDPTR:
    if (!value.voidptr_value)
      return test_value == NULL;
  default:
    return false;
  }
}

Am_Value::operator== (Am_Ptr test_value) const
{
  return value.voidptr_value == test_value;
}

Am_Value::operator== (int test_value) const
{
  switch (type) {
  case Am_INT:
  case Am_LONG:
    return value.long_value == test_value;
  case Am_FLOAT:
    return value.float_value == test_value;
  case Am_DOUBLE:
    return (double)*(Am_Double_Data*)value.wrapper_value == test_value;
  default:
    return false;
  }
}

Am_Value::operator== (long test_value) const
{
  switch (type) {
  case Am_LONG:
  case Am_INT:
    return value.long_value == test_value;
  case Am_FLOAT:
    return value.float_value == test_value;
  case Am_DOUBLE:
    return (double)*(Am_Double_Data*)value.wrapper_value == test_value;
  default:
    return false;
  }
}

#ifndef NEED_BOOL
Am_Value::operator== (bool test_value) const
{
  if (Am_Type_Class (type) != Am_NONE)
    return !!value.voidptr_value == test_value;
  switch (type) {
  case Am_INT:
  case Am_LONG:
    return !!value.long_value == test_value;
  case Am_BOOL:
    return value.bool_value == test_value;
  case Am_FLOAT:
    return !!value.float_value == test_value;
  case Am_DOUBLE:
    return !!(double)*(Am_Double_Data*)value.wrapper_value == test_value;
  case Am_CHAR:
    return !!value.char_value == test_value;
  case Am_STRING:
    return !!(const char*)*(Am_String_Data*)value.wrapper_value == test_value;
  case Am_VOIDPTR:
    return !!value.voidptr_value == test_value;
  case Am_PROC:
    return !!value.proc_value == test_value;
  default:
    return false;
  }
}
#endif

Am_Value::operator== (float test_value) const
{
  switch (type) {
  case Am_FLOAT:
    return value.float_value == test_value;
  case Am_DOUBLE:
    return (double)*(Am_Double_Data*)value.wrapper_value == test_value;
  case Am_INT:
  case Am_LONG:
    return value.long_value == test_value;
  default:
    return false;
  }
}

Am_Value::operator== (double test_value) const
{
  switch (type) {
  case Am_FLOAT:
    return value.float_value == test_value;
  case Am_DOUBLE:
    return (double)*(Am_Double_Data*)value.wrapper_value == test_value;
  case Am_INT:
  case Am_LONG:
    return value.long_value == test_value;
  default:
    return false;
  }
}

Am_Value::operator== (char test_value) const
{
  if (type == Am_CHAR)
    return value.char_value == test_value;
  else
    return false;
}

Am_Value::operator== (const char* test_value) const
{
  switch (type) {
  case Am_STRING:
    return *(Am_String_Data*)value.wrapper_value == test_value;
  case Am_INT:
  case Am_LONG:
  case Am_VOIDPTR:
    return !value.voidptr_value && !test_value;
  default:
    return false;
  }
}

Am_Value::operator== (const Am_String& test_value) const
{
  switch (type) {
  case Am_STRING:
    return *(Am_String_Data*)value.wrapper_value == test_value;
  case Am_INT:
  case Am_LONG:
  case Am_VOIDPTR:
    return !value.voidptr_value && !(const char*)test_value;
  default:
    return false;
  }
}

Am_Value::operator== (Am_Generic_Procedure* test_value) const
{
  if (Am_Type_Class (type) == Am_METHOD)
    return !value.method_value && !test_value;
  switch (type) {
  case Am_PROC:
    return value.proc_value == test_value;
  case Am_INT:
  case Am_LONG:
  case Am_VOIDPTR:
    return !value.voidptr_value && !test_value;
  default:
    return false;
  }
}

Am_Value::operator== (Am_Method_Wrapper* test_value) const
{
  if (Am_Type_Class (type) == Am_METHOD)
    return value.method_value == test_value;
  switch (type) {
  case Am_PROC:
  case Am_INT:
  case Am_LONG:
  case Am_VOIDPTR:
    return !value.voidptr_value && !test_value;
  default:
    return false;
  }
}

Am_Value::operator== (const Am_Value& test_value) const
{
  switch (Am_Type_Class (type)) {
  case Am_NONE:
    switch (type) {
    case Am_STRING:
    case Am_DOUBLE:
      return test_value == value.wrapper_value;
    case Am_VOIDPTR:
      return test_value == value.voidptr_value;
    case Am_INT:
    case Am_LONG:
      return test_value == value.long_value;
    case Am_BOOL:
      return test_value == value.bool_value;
    case Am_FLOAT:
      return test_value == value.float_value;
    case Am_CHAR:
      return test_value == value.char_value;
   case Am_PROC:
     return test_value == value.proc_value;
    case Am_NONE:
      return test_value.type == Am_NONE;
    default:
      return false;
    }
  case Am_WRAPPER:
    return test_value == value.wrapper_value;
  case Am_METHOD:
    return test_value == value.method_value;
  }
  return NULL; // should never get here, but need to return something
}

Am_Value::operator!= (Am_Wrapper* test_value) const
{
  if (Am_Type_Class (type) == Am_WRAPPER)
    return (value.wrapper_value != test_value) &&
      (!test_value || !value.wrapper_value ||
       !(*value.wrapper_value == *test_value));
  switch (type) {
  case Am_STRING:
  case Am_DOUBLE:
    return (value.wrapper_value != test_value) &&
      (!test_value || !value.wrapper_value ||
       !(*value.wrapper_value == *test_value));
  case Am_INT:
  case Am_LONG:
  case Am_VOIDPTR:
    if (!value.voidptr_value)
      return test_value != NULL;
  default:
    return true;
  }
}

Am_Value::operator!= (Am_Ptr test_value) const
{
  return value.voidptr_value != test_value;
}

Am_Value::operator!= (int test_value) const
{
  switch (type) {
  case Am_INT:
  case Am_LONG:
    return value.long_value != test_value;
  case Am_FLOAT:
    return value.float_value != test_value;
  case Am_DOUBLE:
    return (double)*(Am_Double_Data*)value.wrapper_value != test_value;
  default:
    return true;
  }
}

Am_Value::operator!= (long test_value) const
{
  switch (type) {
  case Am_LONG:
  case Am_INT:
    return value.long_value != test_value;
  case Am_FLOAT:
    return value.float_value != test_value;
  case Am_DOUBLE:
    return (double)*(Am_Double_Data*)value.wrapper_value != test_value;
  default:
    return true;
  }
}

#ifndef NEED_BOOL
Am_Value::operator!= (bool test_value) const
{
  if (Am_Type_Class (type) != Am_NONE)
    return !!value.voidptr_value != test_value;
  switch (type) {
  case Am_INT:
  case Am_LONG:
    return !!value.long_value != test_value;
  case Am_BOOL:
    return value.bool_value != test_value;
  case Am_FLOAT:
    return !!value.float_value != test_value;
  case Am_DOUBLE:
    return !!(double)*(Am_Double_Data*)value.wrapper_value != test_value;
  case Am_CHAR:
    return !!value.char_value != test_value;
  case Am_STRING:
    return !!(const char*)*(Am_String_Data*)value.wrapper_value != test_value;
  case Am_VOIDPTR:
    return !!value.voidptr_value != test_value;
  case Am_PROC:
    return !!value.proc_value != test_value;
  default:
    return true;
  }
}
#endif

Am_Value::operator!= (float test_value) const
{
  switch (type) {
  case Am_FLOAT:
    return value.float_value != test_value;
  case Am_DOUBLE:
    return (double)*(Am_Double_Data*)value.wrapper_value != test_value;
  case Am_INT:
  case Am_LONG:
    return value.long_value != test_value;
  default:
    return true;
  }
}

Am_Value::operator!= (double test_value) const
{
  switch (type) {
  case Am_FLOAT:
    return value.float_value != test_value;
  case Am_DOUBLE:
    return (double)*(Am_Double_Data*)value.wrapper_value != test_value;
  case Am_INT:
  case Am_LONG:
    return value.long_value != test_value;
  default:
    return true;
  }
}

Am_Value::operator!= (char test_value) const
{
  if (type == Am_CHAR)
    return value.char_value != test_value;
  else
    return true;
}

Am_Value::operator!= (const char* test_value) const
{
  switch (type) {
  case Am_STRING:
    return !(*(Am_String_Data*)value.wrapper_value == test_value);
  case Am_INT:
  case Am_LONG:
  case Am_VOIDPTR:
    if (!value.voidptr_value)
      return test_value != NULL;
  default:
    return true;
  }
}

Am_Value::operator!= (const Am_String& test_value) const
{
  switch (type) {
  case Am_STRING:
    return !(*(Am_String_Data*)value.wrapper_value == test_value);
  case Am_INT:
  case Am_LONG:
  case Am_VOIDPTR:
    return value.voidptr_value || (const char*)test_value;
  default:
    return true;
  }
}

Am_Value::operator!= (Am_Generic_Procedure* test_value) const
{
  if (Am_Type_Class (type) == Am_METHOD)
    return value.voidptr_value || test_value;
  switch (type) {
  case Am_PROC:
    return value.proc_value != test_value;
  case Am_INT:
  case Am_LONG:
  case Am_VOIDPTR:
    return value.voidptr_value || test_value;
  default:
    return true;
  }
}

Am_Value::operator!= (Am_Method_Wrapper* test_value) const
{
  if (Am_Type_Class (type) == Am_METHOD)
    return value.method_value != test_value;
  switch (type) {
  case Am_PROC:
  case Am_INT:
  case Am_LONG:
  case Am_VOIDPTR:
    return value.voidptr_value || test_value;
  default:
    return true;
  }
}

Am_Value::operator!= (const Am_Value& test_value) const
{
  return (type != test_value.type) || (wrapper_type (type) &&
     value.wrapper_value != test_value.value.wrapper_value &&
     value.wrapper_value && test_value.value.wrapper_value &&
     !(*value.wrapper_value == *test_value.value.wrapper_value)) ||
     (value.voidptr_value != test_value.value.voidptr_value);
}

bool Am_Value::Valid () const
{
  if (Am_Type_Class (type) != Am_NONE)
    return value.voidptr_value != NULL;
  switch (type) {
  case Am_STRING:
  case Am_VOIDPTR:
    return value.voidptr_value != NULL;
  case Am_INT:
  case Am_LONG:
    return value.long_value != 0;
  case Am_BOOL:
    return value.bool_value != false;
  case Am_FLOAT:
    return value.float_value != 0.0;
  case Am_DOUBLE:
    return *(Am_Double_Data*)value.wrapper_value != 0.0;
  case Am_CHAR:
    return value.char_value != 0;
  case Am_PROC:
    return value.proc_value != NULL;
  default:
    return false;
  }
}

void Am_Value::Set_Empty ()
{
  if (wrapper_type (type) && value.wrapper_value)
    value.wrapper_value->Release ();
  value.voidptr_value = NULL;
  type = Am_NONE;
}

void Am_Value::Set_Uninitialized ()
{
  if (wrapper_type (type) && value.wrapper_value)
    value.wrapper_value->Release ();
  value.voidptr_value = NULL;
  type = Am_UNINIT;
}

Am_Value& Am_Value::operator= (Am_Wrapper* in_value)
{
  if (wrapper_type (type) && value.wrapper_value)
    value.wrapper_value->Release ();
  value.wrapper_value = in_value;
  if (in_value)
    type = in_value->ID ();
  else
    type = Am_WRAPPER;
  return *this;
}

Am_Value& Am_Value::operator= (Am_Ptr in_value)
{
  if (wrapper_type (type) && value.wrapper_value)
    value.wrapper_value->Release ();
  type = Am_VOIDPTR;
  value.voidptr_value = in_value;
  return *this;
}

Am_Value& Am_Value::operator= (int in_value)
{
  if (wrapper_type (type) && value.wrapper_value)
    value.wrapper_value->Release ();
  type = Am_INT;
  value.long_value = in_value;
  return *this;
}

Am_Value& Am_Value::operator= (long in_value)
{
  if (wrapper_type (type) && value.wrapper_value)
    value.wrapper_value->Release ();
  type = Am_LONG;
  value.long_value = in_value;
  return *this;
}

#ifndef NEED_BOOL
Am_Value& Am_Value::operator= (bool in_value)
{
  if (wrapper_type (type) && value.wrapper_value)
    value.wrapper_value->Release ();
  type = Am_BOOL;
  value.bool_value = in_value;
  return *this;
}
#endif

Am_Value& Am_Value::operator= (float in_value)
{
  if (wrapper_type (type) && value.wrapper_value)
    value.wrapper_value->Release ();
  type = Am_FLOAT;
  value.float_value = in_value;
  return *this;
}

Am_Value& Am_Value::operator= (double in_value)
{
  //if ((type == Am_DOUBLE) &&
  //    ((double)*(Am_Double_Data*)value.wrapper_value == in_value))
  //  return *this;
  if (wrapper_type (type) && value.wrapper_value)
    value.wrapper_value->Release ();
  type = Am_DOUBLE;
  value.wrapper_value = new Am_Double_Data (in_value);
  return *this;
}

Am_Value& Am_Value::operator= (char in_value)
{
  if (wrapper_type (type) && value.wrapper_value)
    value.wrapper_value->Release ();
  type = Am_CHAR;
  value.char_value = in_value;
  return *this;
}

Am_Value& Am_Value::operator= (const char* in_value)
{
  if (wrapper_type (type) && value.wrapper_value)
    value.wrapper_value->Release ();
  type = Am_STRING;
  value.wrapper_value = new Am_String_Data (in_value);
  return *this;
}

Am_Value& Am_Value::operator= (const Am_String& in_value)
{
  if (wrapper_type (type) && value.wrapper_value)
    value.wrapper_value->Release ();
  type = Am_STRING;
  value.wrapper_value = in_value;
  return *this;
}

Am_Value& Am_Value::operator= (Am_Generic_Procedure* in_value)
{
  if (wrapper_type (type) && value.wrapper_value)
    value.wrapper_value->Release ();
  type = Am_PROC;
  value.proc_value = in_value;
  return *this;
}

Am_Value& Am_Value::operator= (Am_Method_Wrapper* in_value)
{
  if (wrapper_type (type) && value.wrapper_value)
    value.wrapper_value->Release ();
  if (in_value)
    type = (Am_Value_Type)in_value->ID ();
  else
    type = Am_METHOD;
  value.method_value = in_value;
  return *this;
}

Am_Value& Am_Value::operator= (const Am_Value& in_value)
{
  if (wrapper_type (type) && value.wrapper_value)
    value.wrapper_value->Release ();
  type = in_value.type;
  value = in_value.value;
  if (wrapper_type (type) && value.wrapper_value)
    value.wrapper_value->Note_Reference ();
  return *this;
}

Am_Method_Wrapper::Am_Method_Wrapper(Am_ID_Tag* id_ptr,
				     Am_Generic_Procedure *p,
				     const char * name)
{
  ID_Ptr = id_ptr;
  Call = p;
#ifdef DEBUG
  Am_Register_Name(this, name);
#endif
}

void Am_Method_Wrapper::Print(ostream& os, Am_Method_Wrapper* wrapper) {
  if (wrapper) {
#ifdef DEBUG
    const char *name;
    if (name = Am_Get_Name_Of_Item(wrapper))
      os << name;
    else
#endif
    os << hex << (Am_Ptr)(wrapper->Call) << dec;
  }
  else os << "0x0";
}

ostream& operator<< (ostream& os, const Am_Value& value)
{
  switch (Am_Type_Class (value.type)) {
  case Am_NONE:
    switch (value.type) {
    case Am_INT:
    case Am_LONG:
      os << value.value.long_value;
      break;
    case Am_BOOL:
      os << value.value.bool_value;
      break;
    case Am_FLOAT:
      os << value.value.float_value;
      break;
    case Am_DOUBLE:
      os << *(Am_Double_Data*)value.value.wrapper_value;
      break;
    case Am_CHAR:
      os << value.value.char_value;
      break;
    case Am_STRING:
      os << (const char*)*(Am_String_Data*)value.value.wrapper_value;
      break;
    case Am_VOIDPTR:
      // explicitly cast, for the case when Am_Ptr == char*
      os << hex << (void*)value.value.voidptr_value << dec;
      break;
    case Am_PROC:
      os << hex << (Am_Ptr)value.value.proc_value << dec;
      break;
    case Am_NONE:
      os << "NONE (No value)";
      break;
    case Am_UNINIT:
      os << "Uninitialized (No value)";
      break;
    default:
      cerr << "** This value cannot be printed, type = ";
      Am_Print_Type (cerr, value.type);
      cerr << "." << endl;
      Am_Error ();
      break;
    }
    break;
  case Am_WRAPPER:
    if (value.value.wrapper_value) // complains about const with call to valid
      value.value.wrapper_value->Print_Name(os);
    else
      os << "(NULL)";
    break;
  case Am_METHOD:
    Am_Method_Wrapper::Print(os, value.value.method_value);
    break;
  }
  return os;
}
