/* ************************************************************************ 
 *         The Amulet User Interface Development Environment              *
 * ************************************************************************
 * This code was written as part of the Amulet project at                 *
 * Carnegie Mellon University, and has been placed in the public          *
 * domain.  If you are using this code or any part of Amulet,             *
 * please contact amulet@cs.cmu.edu to be put on the mailing list.        *
 * ************************************************************************/

/* This file contains the scroll bar widgets
   
   Designed and implemented by Brad Myers
*/

#include <am_inc.h>

#include AM_IO__H

#include WIDGETS_ADVANCED__H
#include STANDARD_SLOTS__H
#include VALUE_LIST__H
#include OPAL_ADVANCED__H  // for Am_DRAWONABLE, Am_Window_Coordinate
#include INTER_ADVANCED__H //needed for Am_Four_Ints

#include WIDGETS__H
#include GEM__H
#include INTER__H
#include OPAL__H
#include REGISTRY__H

///////////////////////////////////////////////////////////////////////////
// Arrows
///////////////////////////////////////////////////////////////////////////

void draw_motif_up_arrow (int left, int top, int width, int height,
			  bool depressed,
			  const Computed_Colors_Record& rec,
			  Am_Drawonable* draw)
{
  const Am_Style fill1 (depressed ? rec.data->background_style :
			rec.data->foreground_style);
  const Am_Style line1 (depressed ? rec.data->highlight_style :
			rec.data->shadow_style);
  const Am_Style line2 (depressed ? rec.data->shadow_style :
			rec.data->highlight_style);

  int center_x = left + width/2;
  int rightm1 = left + width - 1 - 1;
  int bottom = top + height - 1;
  
  //right line
  draw->Draw_2_Lines(line1, fill1,
		     center_x, top,
		     rightm1, bottom,
		     left, bottom);
  //left line
  draw->Draw_Line(line2, center_x, top, left, bottom);
}

void draw_motif_down_arrow (int left, int top, int width, int height,
			    bool depressed,
			    const Computed_Colors_Record& rec,
			    Am_Drawonable* draw) {
  const Am_Style fill1 (depressed ? rec.data->background_style :
			rec.data->foreground_style);
  const Am_Style line1 (depressed ? rec.data->shadow_style :
			rec.data->highlight_style);
  const Am_Style line2 (depressed ? rec.data->highlight_style :
			rec.data->shadow_style);

  int center_x = left + width/2;
  int right = left + width - 1;
  int bottom = top + height - 1;
  
  //left line
  draw->Draw_2_Lines(line1, fill1,
		     center_x, bottom,
		     left+1, top+1,
		     right, top+1);
  //right line
  draw->Draw_Line(line2, right, top, center_x, bottom);
}

void draw_motif_left_arrow (int left, int top, int width, int height,
			    bool depressed,
			    const Computed_Colors_Record& rec,
			    Am_Drawonable* draw) {

  const Am_Style fill1 (depressed ? rec.data->background_style :
			rec.data->foreground_style);
  const Am_Style line1 (depressed ? rec.data->highlight_style :
			rec.data->shadow_style);
  const Am_Style line2 (depressed ? rec.data->shadow_style :
			rec.data->highlight_style);

  int center_y = top + (height - 2)/2;
  int rightm1 = left + width - 2;
  int bottomm1 = top + height - 2;
  
  //bottom line
  draw->Draw_2_Lines(line1, fill1,
		     left, center_y,
		     rightm1, bottomm1,
		     rightm1, top);
  //top line
  draw->Draw_Line(line2, left, center_y, rightm1, top);
}


void draw_motif_right_arrow (int left, int top, int width, int height,
			     bool depressed,
			     const Computed_Colors_Record& rec,
			     Am_Drawonable* draw) {

  const Am_Style fill1 (depressed ? rec.data->background_style :
			rec.data->foreground_style);
  const Am_Style line1 (depressed ? rec.data->shadow_style :
			rec.data->highlight_style);
  const Am_Style line2 (depressed ? rec.data->highlight_style :
			rec.data->shadow_style);

  int center_y = top + height/2;
  int right = left + width - 1;
  int bottom = top + height - 1;
  
  //top line
  draw->Draw_2_Lines(line1, fill1,
		     left, bottom,
		     left, top + 1,
		     right, center_y);
  //bottom line
  draw->Draw_Line(line2, left + 1, bottom, right, center_y);
}

Am_Define_Method(Am_Draw_Method, void, scroll_arrow_draw,
		 (Am_Object self, Am_Drawonable* drawonable,
		  int x_offset, int y_offset)) {
  int left = (int)self.Get (Am_LEFT) + x_offset;
  int top = (int)self.Get (Am_TOP) + y_offset;
  int width = self.Get (Am_WIDTH);
  int height = self.Get (Am_HEIGHT);
  bool interim_selected = self.Get (Am_INTERIM_SELECTED);
  Computed_Colors_Record rec = self.Get (Am_STYLE_RECORD);
  Am_Widget_Look look = (Am_Widget_Look)(int)self.Get (Am_WIDGET_LOOK);
  Am_Scroll_Arrow_Direction dir =
      (Am_Scroll_Arrow_Direction)(int)self.Get(Am_SCROLL_ARROW_DIRECTION);

  // finally ready to draw it
  if (look == Am_MOTIF_LOOK) {
    switch (dir) {
    case Am_SCROLL_ARROW_UP:
      draw_motif_up_arrow(left, top, width, height,
			  interim_selected, rec, drawonable);
      break;
    case Am_SCROLL_ARROW_DOWN:
      draw_motif_down_arrow(left, top, width, height,
			    interim_selected, rec, drawonable);
      break;
    case Am_SCROLL_ARROW_LEFT:
      draw_motif_left_arrow(left, top, width, height,
			    interim_selected, rec, drawonable);
      break;
    case Am_SCROLL_ARROW_RIGHT:
      draw_motif_right_arrow(left, top, width, height,
			     interim_selected, rec, drawonable);
      break;
    default: Am_Error("Bad Am_Scroll_Arrow_Direction in arrow object");
    }// end switch
  }
  else Am_Error("Sorry, only the Motif Style implemented for now");
  // else no window so don't draw anything
}

///////////////////////////////////////////////////////////////////////////
// Vertical Scroll bar
///////////////////////////////////////////////////////////////////////////

//self is the indicator.  Interesting slots are in the scroll bar
Am_Define_Formula(int, scroll_indicator_pos) {
  int return_val;
  Am_Object scroll_bar = self.GV_Owner();
  if(!scroll_bar.Valid ()) return 10;
  int minpos = scroll_bar.GV(Am_SCROLL_AREA_MIN);
  int maxpos = scroll_bar.GV(Am_SCROLL_AREA_MAX);
  Am_Value value_v;
  Am_Value value_1;
  Am_Value value_2;
  scroll_bar.GVM(Am_VALUE, value_v);
  scroll_bar.GVM(Am_VALUE_1, value_1);
  scroll_bar.GVM(Am_VALUE_2, value_2);
  
  if (value_v.type == Am_INT && value_1.type == Am_INT
      && value_2.type == Am_INT)
    return_val = Am_Clip_And_Map(value_v, value_1, value_2, minpos, maxpos);
  else // calc in float and then convert result to integers
    return_val = (int) Am_Clip_And_Map((float)value_v, (float)value_1,
				 (float)value_2, (float)minpos, (float)maxpos);
  return return_val;
}

Am_Define_Formula(int, v_scroll_area_max) {
  Am_Object indicator = self.GV_Part(Am_SCROLL_INDICATOR);
  int indicator_size = indicator.GV(Am_HEIGHT);
  int minpos = self.GV(Am_SCROLL_AREA_MIN);
  int maxpos = (int)self.GV(Am_SCROLL_AREA_SIZE) - indicator_size + minpos;
  return maxpos;
}

Am_Define_Formula(int, h_scroll_area_max) {
  Am_Object indicator = self.GV_Part(Am_SCROLL_INDICATOR);
  int indicator_size = indicator.GV(Am_WIDTH);
  int minpos = self.GV(Am_SCROLL_AREA_MIN);
  int maxpos = (int)self.GV(Am_SCROLL_AREA_SIZE) - indicator_size + minpos;
  return maxpos;
}
 
Am_Define_Formula(int, scroll_indicator_size) {
  Am_Object scroll_bar = self.GV_Owner();
  if(!scroll_bar.Valid ()) return 10;
  int maxsize = scroll_bar.GV(Am_SCROLL_AREA_SIZE);
  float percent = scroll_bar.GV(Am_PERCENT_VISIBLE);
  if (percent > 1.0f) percent = 1.0f;
  int size = (int) (maxsize*percent);
  if (size < 6) size = 6;
  return size;
}

// set the positions of the arrows and indicator based on size of scroll bar.
// Only sets the left and width of the indicator, height and top are formulas 
Am_Define_Formula(int, v_scroll_layout_formula) {
  Am_Object arrow1 = self.GV_Part(Am_SCROLL_ARROW1);
  Am_Object arrow2 = self.GV_Part(Am_SCROLL_ARROW2);
  Am_Object indicator = self.GV_Part(Am_SCROLL_INDICATOR);
  int width = self.GV(Am_WIDTH);
  int height = self.GV(Am_HEIGHT);

  arrow1.Set(Am_LEFT, 5);
  arrow1.Set(Am_TOP, 5);
  arrow1.Set(Am_WIDTH, width-9);
  arrow1.Set(Am_HEIGHT, width-9); // square
  
  arrow2.Set(Am_LEFT, 4); //different from arrow1 ????
  arrow2.Set(Am_TOP, height - (width-9) - 6); //minus size of arrows
  arrow2.Set(Am_WIDTH, width-9);
  arrow2.Set(Am_HEIGHT, width-9); // square
  
  int arrow_size = 5 + (width-9) + 1;
  self.Set(Am_SCROLL_AREA_MIN, arrow_size); //bottom of top arrow
  self.Set(Am_SCROLL_AREA_SIZE, height - arrow_size - arrow_size - 2);

  indicator.Set(Am_WIDTH, width-8);
  return 0;  //return value not used
}

// set the positions of the arrows and indicator based on size of scroll bar
// Only sets the height and top of the indicator, left and width are formulas 
Am_Define_Formula(int, h_scroll_layout_formula) {
  Am_Object arrow1 = self.GV_Part(Am_SCROLL_ARROW1);
  Am_Object arrow2 = self.GV_Part(Am_SCROLL_ARROW2);
  Am_Object indicator = self.GV_Part(Am_SCROLL_INDICATOR);
  int width = self.GV(Am_WIDTH);
  int height = self.GV(Am_HEIGHT);

  arrow1.Set(Am_LEFT, 5);
  arrow1.Set(Am_TOP, 4);
  arrow1.Set(Am_WIDTH, height-9);
  arrow1.Set(Am_HEIGHT, height-9); // square
  
  arrow2.Set(Am_LEFT, width - (height-9) - 6); //minus size of arrows
  arrow2.Set(Am_TOP, 4);
  arrow2.Set(Am_WIDTH, height-9);
  arrow2.Set(Am_HEIGHT, height-9); // square
  
  int arrow_size = 5 + (height-9) + 1;
  self.Set(Am_SCROLL_AREA_MIN, arrow_size); //bottom of top arrow
  self.Set(Am_SCROLL_AREA_SIZE, width - arrow_size - arrow_size - 2);

  indicator.Set(Am_HEIGHT, height-8);

  return 0;  //return value not used
}

// draw the background, then draw the parts
Am_Define_Method(Am_Draw_Method, void, scroll_draw,
		 (Am_Object self, Am_Drawonable* drawonable,
		  int x_offset, int y_offset)) {
  int left = (int)self.Get (Am_LEFT) + x_offset;
  int top = (int)self.Get (Am_TOP) + y_offset;
  int width = self.Get (Am_WIDTH);
  int height = self.Get (Am_HEIGHT);
  bool key_selected = self.Get (Am_KEY_SELECTED);
  Computed_Colors_Record rec = self.Get (Am_STYLE_RECORD);
  Am_Widget_Look look = (Am_Widget_Look)(int)self.Get (Am_WIDGET_LOOK);
  // finally ready to draw it
  if (look == Am_MOTIF_LOOK) {
    if (key_selected) // draw box showing that the keyboard is over this item
      drawonable->Draw_Rectangle (Am_Key_Border_Line,
				  Am_No_Style, left, top, width, height);
      // draw the background box, leave 2 pixel border for the key-selected box
    Am_Draw_Motif_Box(left+2, top+2, width-4, height-4,
		      true, rec, drawonable);
  }
  else Am_Error("Sorry, only the Motif Style implemented for now");
    
  // now call the prototype's method to draw the parts

  Am_Draw_Method method;
  method = Am_Aggregate.Get(Am_DRAW_METHOD);
  method.Call (self, drawonable, x_offset, y_offset);
}

/////////////////////////////////////////////////////////////////
// For the interactors and commands
/////////////////////////////////////////////////////////////////

//////////////////  Indicator //////////////////////

//For the where test of the indicator

Am_Define_Method(Am_Where_Method, Am_Object, in_scroll_indicator,
		 (Am_Object /* inter */,
		  Am_Object object, Am_Object event_window,
		  Am_Input_Char /* ic */, int x, int y)) {
  Am_Object indicator = object.Get_Part(Am_SCROLL_INDICATOR);
  if (Am_Point_In_All_Owners(indicator, x, y, event_window)
      && (Am_Point_In_Obj (indicator, x, y, event_window).Valid()))
    return indicator;

  //if get here, then not indicator
  return Am_No_Object;
}

void set_up_for_undo(Am_Object inter_command, Am_Object scroll_command,
		     Am_Object scrollbar) {
  //set old value to current value
  Am_Value old_value;
  scrollbar.Get(Am_VALUE, old_value);
  scroll_command.Set(Am_OLD_VALUE, old_value);
  inter_command.Set(Am_OLD_VALUE, old_value);
}

Am_Define_Method(Am_Object_Method, void, save_pos_for_undo,
		 (Am_Object command_obj)) {
  Am_Object inter = command_obj.Get_Owner();
  Am_Object scrollbar = inter.Get_Owner();
  Am_Object scroll_command = scrollbar.Get_Part(Am_COMMAND);
  set_up_for_undo(command_obj, scroll_command, scrollbar);
  //find the top-level parent and mark it as not queued until the
  //final do action
  Am_Object obj;
  Am_Value value;
  obj = scroll_command;
  while (true) {
    obj.Get(Am_IMPLEMENTATION_PARENT, value);
    if (value.Valid()) {
      if (value.type != Am_OBJECT)
	break; //already marked as not queued
    }
    else { // not valid
      obj.Set(Am_IMPLEMENTATION_PARENT, Am_MARKER_FOR_SCROLL_INC);
      break;
    }
    obj = value;
  }
}
  
//clear the Am_IMPLEMENTATION_PARENT marker, and also restore the
//VALUE of the widget.
Am_Define_Method(Am_Object_Method, void, clear_save_pos_for_undo,
		 (Am_Object command_obj)) {
  Am_Object obj, parent;
  Am_Value value;
  obj = command_obj;
  while (true) {
    obj.Get(Am_IMPLEMENTATION_PARENT, value);
    if (value.Valid()) {
      if (value.type != Am_OBJECT) {
	if (value == Am_MARKER_FOR_SCROLL_INC)
	  obj.Set(Am_IMPLEMENTATION_PARENT, 0);
	break;
      }
    }
    else { // not valid
      break;
    }
    obj = value;
  }
}

//internal procedure used by horiz and vertical
void set_scrollbar_and_commands(Am_Value value, Am_Object scrollbar,
				Am_Object scroll_command,
				Am_Object inter_command) {
  scrollbar.Set(Am_VALUE, value);
  if (scroll_command.Valid ()) 
    scroll_command.Set(Am_VALUE, value);
  inter_command.Set(Am_VALUE, value);
}
  
void scroll_indicator_inter_interim_do(Am_Object inter, int cur_value) {
  Am_Object scrollbar = inter.Get_Owner();
  if (scrollbar.Valid ()) {
    Am_Object scroll_command;
    scroll_command = scrollbar.Get(Am_COMMAND);
    Am_Object indicator;
    indicator = inter.Get(Am_START_OBJECT);
    //map the x or y value to the current value
    Am_Value val1, val2, value;
    scrollbar.Get(Am_VALUE_1, val1);
    scrollbar.Get(Am_VALUE_2, val2);
    int minpos = scrollbar.Get(Am_SCROLL_AREA_MIN);
    int maxpos = scrollbar.Get(Am_SCROLL_AREA_MAX);
      
    //now map the value
    if (val1.type == Am_INT && val2.type == Am_INT) {
      value = Am_Clip_And_Map(cur_value, minpos, maxpos, val1, val2);
    }
    else { // calc in float
      value = Am_Clip_And_Map((float)cur_value, (float)minpos,
			      (float)maxpos, (float)val1, (float)val2);
    }
    set_scrollbar_and_commands(value, scrollbar, scroll_command,
			       inter.Get_Part(Am_COMMAND));
    //pretend this is a final function, so call all parent methods
    Am_Move_Grow_Register_For_Undo(inter);
  }
}
  
Am_Define_Method(Am_Current_Location_Method, void,
		 v_scroll_indicator_inter_interim_do,
		 (Am_Object inter, Am_Object /* object_modified */,
		  Am_Inter_Location data)) {
  int left, top, w, h;
  data.Get_Points(left, top, w, h);
  scroll_indicator_inter_interim_do(inter, top);
}

Am_Define_Method(Am_Current_Location_Method, void,
		 h_scroll_indicator_inter_interim_do,
		 (Am_Object inter, Am_Object /* object_modified */,
		  Am_Inter_Location data)) {
  int left, top, w, h;
  data.Get_Points(left, top, w, h);
  scroll_indicator_inter_interim_do(inter, left);
}

//////////////////  Arrows //////////////////////

// For the where test of arrow interactor: see if in an arrow
Am_Define_Method(Am_Where_Method, Am_Object, in_scroll_arrows,
		 (Am_Object /* inter */,
		  Am_Object object, Am_Object event_window,
		  Am_Input_Char /* ic */, int x, int y)) {
  Am_Object result = Am_No_Object;
  if (Am_Point_In_All_Owners(object, x, y, event_window))
    result = Am_Point_In_Part (object, x, y, event_window, false);
  if (result.Valid ()) {
    if (result == object.Get_Part(Am_SCROLL_ARROW1) ||
	result == object.Get_Part(Am_SCROLL_ARROW2))
      return result;
    else return Am_No_Object; // not an arrow
  }
  else return Am_No_Object;
}

void int_adjust_value(Am_Value &output_value, int cur_val,
		       bool towards_val1, int inc_amt, int val1, int val2) {
  int final_val;
  if (val1 < val2) {
    if (towards_val1) {
      if (cur_val > val2) cur_val = val2; //if starts off too big
      final_val = cur_val - inc_amt;
      if (final_val < val1) final_val = val1;
    }
    else {
      if (cur_val < val1) cur_val = val1; //if starts off too small
      final_val = cur_val + inc_amt;
      if (final_val > val2) final_val = val2;
    }
  }
  else {  // val1 >= val2
    if (towards_val1) {
      if (cur_val < val2) cur_val = val2; //if starts off too small
      final_val = cur_val + inc_amt;
      if (final_val > val1) final_val = val1;
    }
    else {
      if (cur_val > val1) cur_val = val1; //if starts off too big
      final_val = cur_val - inc_amt;
      if (final_val < val2) final_val = val2;
    }
  }
  output_value = final_val;
}
void float_adjust_value(Am_Value &output_value, float cur_val,
			bool towards_val1, float inc_amt,
			float val1, float val2) {
  float final_val;
  if (val1 < val2) {
    if (towards_val1) {
      if (cur_val > val2) cur_val = val2; //if starts off too big
      final_val = cur_val - inc_amt;
      if (final_val < val1) final_val = val1;
    }
    else {
      if (cur_val < val1) cur_val = val1; //if starts off too small
      final_val = cur_val + inc_amt;
      if (final_val > val2) final_val = val2;
    }
  }
  else {
    if (towards_val1) {
      if (cur_val < val2) cur_val = val2; //if starts off too small
      final_val = cur_val + inc_amt;
      if (final_val > val1) final_val = val1;
    }
    else {
      if (cur_val > val1) cur_val = val1; //if starts off too big
      final_val = cur_val - inc_amt;
      if (final_val < val2) final_val = val2;
    }
  }
  output_value = final_val;
}
      

void adjust_value(Am_Value &cur_value, bool towards_val1,
		      Am_Value inc_amt, Am_Value val1, Am_Value val2) {
  if (inc_amt.type == Am_INT && val1.type == Am_INT && val2.type == Am_INT)
    int_adjust_value(cur_value, (int)cur_value, towards_val1, inc_amt, val1,
		     val2);
  else float_adjust_value(cur_value, (float)cur_value, towards_val1,
			  (float) inc_amt, (float)val1, (float)val2 );
}
      
Am_Define_Method(Am_Object_Method, void, scroll_arrow_inter_command_do,
		 (Am_Object command)) {
  Am_Object arrow;
  Am_Object inter = command.Get_Owner();
  if(inter.Valid ()) {
    arrow = inter.Get(Am_INTERIM_VALUE);
    Am_Object scrollbar = inter.Get_Owner();
    if (scrollbar.Valid ()) {
      Am_Object scroll_command;
      scroll_command = scrollbar.Get(Am_COMMAND);
      if (scroll_command.Valid ())
	set_up_for_undo(command, scroll_command, scrollbar);

      Am_Value inc_amt, cur_value, val1, val2;
      scrollbar.Get(Am_SMALL_INCREMENT, inc_amt);
      scrollbar.Get(Am_VALUE_1, val1);
      scrollbar.Get(Am_VALUE_2, val2);
      scrollbar.Get(Am_VALUE, cur_value);
      bool towards_val1;
      if (arrow == scrollbar.Get_Part(Am_SCROLL_ARROW1))
	towards_val1 = true;
      else if (arrow == scrollbar.Get_Part(Am_SCROLL_ARROW2))
	towards_val1 = false;
      else Am_Error("command value not one of the arrows");
      adjust_value(cur_value, towards_val1, inc_amt, val1, val2);
      set_scrollbar_and_commands(cur_value, scrollbar, scroll_command,
				 command);
    }
  }
}
      
//////////////////  Background (page) //////////////////////

//internal procedure used by both horiz and vert
void scroll_page_inter_command_do (Am_Object command, bool vertical,
				   Am_Object ref_obj, int mousex, int mousey) {
  Am_Object inter = command.Get_Owner();
  if(inter.Valid ()) {
    Am_Object scrollbar = inter.Get_Owner();
    if (scrollbar.Valid ()) {
      Am_Object scroll_command;
      scroll_command = scrollbar.Get(Am_COMMAND);
      if (scroll_command.Valid ())
	set_up_for_undo(command, scroll_command, scrollbar);
      Am_Value inc_amt;
      Am_Value cur_value;
      Am_Value val1;
      Am_Value val2;
      scrollbar.Get(Am_LARGE_INCREMENT, inc_amt);
      scrollbar.Get(Am_VALUE_1, val1);
      scrollbar.Get(Am_VALUE_2, val2);
      scrollbar.Get(Am_VALUE, cur_value);
      bool towards_val1 = false;
      // see if click is above the indicator
      Am_Object indicator = scrollbar.Get_Part(Am_SCROLL_INDICATOR);
      int indicator_pos, x, y;
      if (Am_Translate_Coordinates (ref_obj, mousex, mousey,
				    scrollbar, x, y)) {
	if (vertical) {
	  indicator_pos = indicator.Get(Am_TOP); 
	  if (indicator_pos > y) towards_val1 = true;
	}
	else { //horizontal
	  indicator_pos = indicator.Get(Am_LEFT);
	  if (indicator_pos > x) towards_val1 = true;
	}

	adjust_value(cur_value, towards_val1, inc_amt, val1, val2);
	set_scrollbar_and_commands(cur_value, scrollbar, scroll_command,
				   command);
      }
    }
  }
}

Am_Define_Method(Am_Mouse_Event_Method, void,
		 vertical_scroll_page_inter_command_do,
		 (Am_Object command, int mouse_x, int mouse_y,
		  Am_Object ref_obj, Am_Input_Char /* ic */)) {
  scroll_page_inter_command_do(command, true, ref_obj, mouse_x, mouse_y);
}
Am_Define_Method(Am_Mouse_Event_Method, void,
		 horizontal_scroll_page_inter_command_do,
		 (Am_Object command, int mouse_x, int mouse_y,
		  Am_Object ref_obj, Am_Input_Char /* ic */)) {
  scroll_page_inter_command_do(command, false, ref_obj, mouse_x, mouse_y);
}


///////////////////////////////////////////////////////////////////////////
// Scrolling_Group
///////////////////////////////////////////////////////////////////////////

// scroll bar has SCROLL_BORDER pixels blank border all around.
// SCROLL_EXTRA_WIDTH is 2*SCROLL_BORDER
// SCROLL_MARGIN is # pixels between the scroll bars and the clip region
#define SCROLL_BORDER 2 
#define SCROLL_EXTRA_WIDTH 4
#define SCROLL_MARGIN 2 

// Am_Translate_Coordinates_Method:
//  - Stored in slot Am_TRANSLATE_COORDINATES_METHOD
//  - Given a point in the coordinate system of the group, converts
//    it to be in the coordinate system of scrolling group's owner
Am_Define_Method(Am_Translate_Coordinates_Method, void,
		 scroll_group_translate_coordinates,
		 (const Am_Object& self, const Am_Object& for_part,
		  int in_x, int in_y, int& out_x, int& out_y,
		  Am_Constraint_Context& cc)) {
  bool using_constraint = (cc.ID () != 0);
  //first, offset by the origin of the inside of the group
  if (using_constraint) {
    out_x = (int)self.GV (Am_LEFT) + in_x;
    out_y = (int)self.GV (Am_TOP) + in_y;
  }
  else {
    out_x = (int)self.Get (Am_LEFT) + in_x;
    out_y = (int)self.Get (Am_TOP) + in_y;
  }
  //cout << "scroll " << in_x << " " << in_y << " maps to " << out_x << " "
  //     << out_y << endl << flush;
  Am_Object h_scroller = self.Get_Part(Am_H_SCROLLER);
  Am_Object v_scroller = self.Get_Part(Am_V_SCROLLER);
  if (for_part != h_scroller && for_part != v_scroller) {
    // then is for something on the inside of the scrolling region
    if (using_constraint) {
      out_x += (int)self.GV (Am_CLIP_LEFT) - (int)self.GV(Am_X_OFFSET);
      out_y += (int)self.GV (Am_CLIP_TOP)  - (int)self.GV(Am_Y_OFFSET);
    }
    else {
      out_x += (int)self.Get (Am_CLIP_LEFT) - (int)self.Get(Am_X_OFFSET);
      out_y += (int)self.Get (Am_CLIP_TOP)  - (int)self.Get(Am_Y_OFFSET);
    }
    // cout << "after offset " << out_x << " " << out_y << endl << flush;
  }
}

//coords are in my owner's
bool in_inside_scroll_group(Am_Object in_obj, int x, int y) {
  if ((bool)in_obj.Get (Am_VISIBLE)) {
    int left, top, clipleft, cliptop, clipwidth, clipheight;
    left = in_obj.Get(Am_LEFT);
    top = in_obj.Get(Am_TOP);
    clipleft = in_obj.Get(Am_CLIP_LEFT);
    cliptop = in_obj.Get(Am_CLIP_TOP);
    if (x < left + clipleft || y < top + cliptop) return false;
    clipwidth = in_obj.Get(Am_CLIP_WIDTH);
    clipheight = in_obj.Get(Am_CLIP_HEIGHT);
    if ((x >= left + clipleft + clipwidth) ||
	(y >= top + cliptop + clipheight))
      return false;
    return true;
  }
  else return false;
}

//coords are w.r.t ref_obj, make them be w.r.t owner of in_obj
void translate_coords_to_owner(Am_Object in_obj, 
			       Am_Object ref_obj,
			       int x, int y, int &x1, int &y1) {
  Am_Object owner;
  owner = in_obj.Get_Owner();
  if (owner.Valid() && owner == ref_obj) {
    x1 = x;
    y1 = y;  //already OK
  }
  else { //not the owner, use expensive transformation
    Am_Translate_Coordinates (ref_obj, x, y, owner, x1, y1);
  }
}

Am_Define_Method(Am_Point_In_Method, Am_Object, scroll_group_point_in_obj,
		 (Am_Object in_obj, int x, int y, Am_Object ref_obj)) {
  if ((bool)in_obj.Get (Am_VISIBLE)) {
    Am_Object owner = in_obj.Get_Owner();
    if (owner.Valid()) {
      if (owner != ref_obj) // otherwise x,y ok
	Am_Translate_Coordinates (ref_obj, x, y, owner, x, y);
      int left = in_obj.Get(Am_LEFT);
      int top = in_obj.Get(Am_TOP);
      if ((x < left) || (y < top)) return Am_No_Object;
      if ((x >= left + (int)in_obj.Get (Am_WIDTH)) ||
	  (y >= top + (int)in_obj.Get (Am_HEIGHT)))
	return Am_No_Object;
      return in_obj;
    }
  }
  return Am_No_Object;
}


// x and y are in coord of ref_obj
Am_Define_Method(Am_Point_In_Or_Self_Method, Am_Object,
		 scroll_group_point_in_part,
		 (Am_Object in_obj, int x, int y,
		  Am_Object ref_obj, bool want_self, bool want_groups)) {
  //only works for objects inside my clip region, so can't "click through"
  //the scroll bars.
  int x1, y1;
  translate_coords_to_owner(in_obj, ref_obj, x, y, x1, y1);
  if (in_inside_scroll_group (in_obj, x1, y1)) {
    int offx, offy;
    scroll_group_translate_coordinates_proc(in_obj, Am_No_Object,
					    0, 0, offx, offy, 
					    *Am_Empty_Constraint_Context);
    x1 -= offx;  //translate coordinates to the inside 
    y1 -= offy;  //of the scrolling-group, which is the owner of the parts
    Am_Value_List comp;
    comp = in_obj.Get (Am_GRAPHICAL_PARTS);
    Am_Object object;
    for (comp.End (); !comp.First (); comp.Prev ()) {
      object = comp.Get ();
      if ((want_groups || !am_is_group_and_not_pretending(object)) &&
	  Am_Point_In_Obj (object, x1, y1, in_obj))
	return object;
    }
    //went through list, not in a part
    if (want_self && (want_groups || !am_is_group_and_not_pretending(in_obj)))
      return in_obj;
    else return Am_No_Object;
  }
  else return Am_No_Object;
}

Am_Define_Method(Am_Point_In_Or_Self_Method, Am_Object,
		 scroll_group_point_in_leaf,
		 (Am_Object in_obj, int x, int y, Am_Object ref_obj,
		  bool want_self, bool want_groups)) {
  //only works for objects inside my clip region, so can't "click through"
  //the scroll bars.
  int x1, y1;
  translate_coords_to_owner(in_obj, ref_obj, x, y, x1, y1);
  if (in_inside_scroll_group (in_obj, x, y)) {
    Am_Value val;
    in_obj.Get(Am_PRETEND_TO_BE_LEAF, val);
    if (val.Valid()) return in_obj; // true if slot exists and is non-null
    else {
      int offx, offy;
      scroll_group_translate_coordinates_proc(in_obj, Am_No_Object, 0, 0,
					      offx, offy, 
					      *Am_Empty_Constraint_Context);
      x1 -= offx;  //translate coordinates to the inside 
      y1 -= offy;  //of the scrolling-group, which is the owner of the parts
      Am_Value_List comp;
      comp = in_obj.Get (Am_GRAPHICAL_PARTS);
      Am_Object object, ret;
      for (comp.End (); !comp.First (); comp.Prev ()) {
	object = comp.Get ();
	ret = Am_Point_In_Leaf (object, x1, y1, in_obj, want_self,
				want_groups);
	if (ret.Valid()) return ret;
      }
      //went through list, not in a part
      if (want_self &&
	  (want_groups || !am_is_group_and_not_pretending(in_obj)))
	return in_obj;
      else return Am_No_Object;
    }
  }
  else // if not in me, return NULL
    return Am_No_Object;
}

int get_scroll_border_thickness(Am_Object &self, Am_Constraint_Context& cc) {
  Am_Value v;
  self.GVM(Am_LINE_STYLE, v);
  if (v.Valid()) {
    Am_Style border_style = v;
    short thickness;
    Am_Line_Cap_Style_Flag cap;
    border_style.Get_Line_Thickness_Values (thickness, cap);
    if (thickness == 0) thickness = 1;
    return thickness;
  }
  else return 0;
}

// The left of the area that is used for the view of the insides
Am_Define_Formula(int, scroll_clip_left) {
  int ret = get_scroll_border_thickness(self, cc);
  if ((bool)self.GV(Am_V_SCROLL_BAR) &&
      (bool)self.GV(Am_V_SCROLL_BAR_ON_LEFT)) {
    Am_Object v_scroller = self.GV_Part(Am_V_SCROLLER);
    int scroll_width = (int)v_scroller.GV(Am_WIDTH) -
      SCROLL_EXTRA_WIDTH + SCROLL_MARGIN;
    ret += scroll_width;
  }
  return ret;
}

// The top of the area that is used for the view of the insides
Am_Define_Formula(int, scroll_clip_top) {
  int ret = get_scroll_border_thickness(self, cc);
  if ((bool)self.GV(Am_H_SCROLL_BAR) &&
      (bool)self.GV(Am_H_SCROLL_BAR_ON_TOP)) {
    Am_Object h_scroller = self.GV_Part(Am_H_SCROLLER);
    int scroll_height = (int)h_scroller.GV(Am_HEIGHT)  -
      SCROLL_EXTRA_WIDTH + SCROLL_MARGIN; 
    ret += scroll_height;
  }
  return ret;
}


// The area of the group that is used for the view of the insides
Am_Define_Formula(int, scroll_clip_width) {
  int borderwidth = 2*get_scroll_border_thickness(self, cc);
  int group_width = (int)self.GV(Am_WIDTH) - borderwidth;
  if ((bool)self.GV(Am_V_SCROLL_BAR)) {
    Am_Object v_scroller = self.GV_Part(Am_V_SCROLLER);
    return group_width - (int) v_scroller.GV(Am_WIDTH) +
      SCROLL_EXTRA_WIDTH - SCROLL_MARGIN; 
  }
  else return group_width;
}

// The area of the group that is used for the view of the insides
Am_Define_Formula(int, scroll_clip_height) {
  int borderwidth = 2*get_scroll_border_thickness(self, cc);
  int group_height = (int)self.GV(Am_HEIGHT) - borderwidth;
  if ((bool)self.GV(Am_H_SCROLL_BAR)) {
    Am_Object h_scroller = self.GV_Part(Am_H_SCROLLER);
    return group_height - (int) h_scroller.GV(Am_HEIGHT) + 
      SCROLL_EXTRA_WIDTH - SCROLL_MARGIN;
  }
  else return group_height;
}

//Draw the scroll bars, then the background, then transform the
//coordinates and draw the graphical parts.
Am_Define_Method(Am_Draw_Method, void, scrolling_group_draw,
		 (Am_Object self, Am_Drawonable* drawonable,
		  int x_offset, int y_offset)) {
  int myleft = (int)self.Get (Am_LEFT) + x_offset;
  int mytop = (int)self.Get (Am_TOP) + y_offset;
  int mywidth = self.Get (Am_WIDTH);
  int myheight = self.Get (Am_HEIGHT);
  //Am_CLIP_LEFT and TOP are in my coord system, need window coords so add
  int insideleft = (int)self.Get(Am_CLIP_LEFT) + myleft;
  int insidetop = (int)self.Get(Am_CLIP_TOP) + mytop;
  int insideheight = self.Get(Am_CLIP_HEIGHT);
  int insidewidth = self.Get(Am_CLIP_WIDTH);
  Am_State_Store* state;
  if (mywidth && myheight) {
    // set a clip region here in case group is too small for the scroll bars
    drawonable->Push_Clip (myleft, mytop, mywidth, myheight);

    if ((bool)self.Get(Am_H_SCROLL_BAR)) {
      Am_Object h_scroller = self.Get_Part(Am_H_SCROLLER);
      state = Am_State_Store::Narrow (h_scroller.Get (Am_PREV_STATE));
      if (state->Visible (drawonable, myleft, mytop))
	Am_Draw (h_scroller, drawonable, myleft, mytop);
    }
    if ((bool)self.Get(Am_V_SCROLL_BAR)) {
      Am_Object v_scroller = self.Get_Part(Am_V_SCROLLER);
      state = Am_State_Store::Narrow (v_scroller.Get (Am_PREV_STATE));
      if (state->Visible (drawonable, myleft, mytop))
	Am_Draw (v_scroller, drawonable, myleft, mytop);
    }
    // draw the background
    Am_Style background_style, line_style;
    // use Am_INNER_FILL_STYLE if supplied, otherwise Am_FILL_STYLE
    Am_Value value;
    self.Get(Am_INNER_FILL_STYLE, value);
    if (Am_Type_Class (value.type) == Am_WRAPPER) background_style = value;
    else background_style = self.Get(Am_FILL_STYLE);
    //  filled with background_style
    self.Get(Am_LINE_STYLE, value);
    short thickness = 0;
    if (value.Valid()) {
      line_style = value;  //else leave line_style NULL
      Am_Line_Cap_Style_Flag cap;
      line_style.Get_Line_Thickness_Values (thickness, cap);
      if (thickness == 0) thickness = 1;
    }
    drawonable->Draw_Rectangle (line_style, background_style,
				insideleft-thickness, insidetop-thickness,
				insidewidth  + 2*thickness,
				insideheight + 2*thickness);

    drawonable->Pop_Clip (); // done scroll bar drawin.type
    //set clip of inside
    drawonable->Push_Clip (insideleft, insidetop, insidewidth, insideheight);
    //now offset the area by the x and y offsets
    int x_offset = self.Get(Am_X_OFFSET);
    int y_offset = self.Get(Am_Y_OFFSET);
    insideleft -= x_offset;
    insidetop -= y_offset;

    Am_Value_List components;
    components = self.Get (Am_GRAPHICAL_PARTS);
    Am_Object item;
    for (components.Start (); !components.Last (); components.Next ()) {
      item = components.Get ();
      state = Am_State_Store::Narrow (item.Get (Am_PREV_STATE));
      if (state->Visible (drawonable, insideleft, insidetop))
	Am_Draw (item, drawonable, insideleft, insidetop);
    }
    drawonable->Pop_Clip ();
  }
}

Am_Define_Method(Am_Invalid_Method, void, scrolling_group_invalid,
		 (Am_Object group, Am_Object which_part,
		  int left, int top, int width, int height)) {
  Am_Object owner = group.Get_Owner ();
  if (owner) {
    int my_left = group.Get (Am_LEFT);
    int my_top = group.Get (Am_TOP);
    int my_width = group.Get (Am_WIDTH);
    int my_height = group.Get (Am_HEIGHT);
    int final_left, final_top, final_width, final_height;
    Am_Object h_scroller = group.Get_Part(Am_H_SCROLLER);
    Am_Object v_scroller = group.Get_Part(Am_V_SCROLLER);
    // check if one of the scroll bars
    if (which_part != h_scroller && which_part != v_scroller) {
      // then transform based on offsets for the inside

      // adjust both the clip region and the object to be in my
      // parent's coordinate system
      int insideleft = group.Get(Am_CLIP_LEFT);
      int insidetop = group.Get(Am_CLIP_TOP);
      my_width = group.Get(Am_CLIP_WIDTH); //just re-set the value directly
      my_height = group.Get(Am_CLIP_HEIGHT);

      my_left += insideleft;
      my_top += insidetop;

      //now offset the area by the x and y offsets
      left -= (int)group.Get(Am_X_OFFSET);
      top -=  (int)group.Get(Am_Y_OFFSET);
    } // done with inside scrolling part

    // now clip incoming rectangle to my rectangle
    Am_Invalid_Rectangle_Intersect(left, top, width, height,
				   my_left, my_top, my_width, my_height,
				   final_left, final_top,
				   final_width, final_height);
    if ((final_width > 0) && (final_height > 0))
      Am_Invalidate (owner, group, final_left, final_top, final_width,
		     final_height);
  }
}

//in the Am_Scrolling_Group itself to define how big a page jump is
Am_Define_Formula(int, h_scroll_jump_page) {
  return (int)self.GV(Am_WIDTH) - 10;
}

//in the Am_Scrolling_Group itself to define how big a page jump is
Am_Define_Formula(int, v_scroll_jump_page) {
  return (int)self.GV(Am_HEIGHT) - 10;
}

Am_Define_Formula(int, v_scroll_left) {
  Am_Object group = self.GV_Owner();
  if ((bool)group.GV(Am_V_SCROLL_BAR_ON_LEFT)) return -SCROLL_BORDER;
  else // on right
    return (int)group.GV(Am_WIDTH) - (int)self.GV(Am_WIDTH) + SCROLL_BORDER;
}

Am_Define_Formula(int, v_scroll_top) {
  Am_Object group = self.GV_Owner();
  int ret = (int)group.GV(Am_CLIP_TOP) - SCROLL_BORDER - 1; 
  return ret;
}

Am_Define_Formula(int, v_scroll_height) {
  Am_Object group = self.GV_Owner();
  //SCROLL_EXTRA_WIDTH + 2 for 1-pixel border around scrollarea
  int ret = (int)group.GV(Am_CLIP_HEIGHT) + SCROLL_EXTRA_WIDTH + 2;
  return ret;
}

Am_Define_Formula(bool, v_scroll_visible) {
  Am_Object group = self.GV_Owner();
  if ((bool)group.GV(Am_VISIBLE) &&
      (bool)group.GV(Am_V_SCROLL_BAR)) return true;
  else return false;
}

Am_Define_Formula(bool, v_scroll_active) {
  Am_Object group = self.GV_Owner();
  if ((bool)group.GV(Am_ACTIVE) &&
      ((int)group.GV(Am_INNER_HEIGHT) >=
       (int)group.GV(Am_CLIP_HEIGHT)))
    return true;
  else {
    group.Set(Am_Y_OFFSET, 0);  //make sure at origin when go inactive
    group.Get_Part(Am_V_SCROLLER).Get_Part(Am_COMMAND).Set(Am_VALUE, 0);
    return false;
  }
}

Am_Define_Formula(int, v_scroll_value_2) {
  Am_Object group = self.GV_Owner();
  int val = (int)group.GV(Am_INNER_HEIGHT) - (int)group.GV(Am_CLIP_HEIGHT);
  if (val < 1) val = 1;
  return val;
}

Am_Define_Formula(float, v_scroll_percent_visible) {
  Am_Object group = self.GV_Owner();
  float innerh =  group.GV(Am_INNER_HEIGHT);
  if (innerh == 0.0f) innerh = 1.0f;
  float val = (float)group.GV(Am_CLIP_HEIGHT) / innerh;
  if (val > 1.0) val = 1.0f;
  return val;
}

//////// Horizontal scrollbar stuff

Am_Define_Formula(int, h_scroll_left) {
  Am_Object group = self.GV_Owner();
  return (int)group.GV(Am_CLIP_LEFT) - SCROLL_BORDER - 1;
}

Am_Define_Formula(int, h_scroll_top) {
  Am_Object group = self.GV_Owner();
  if ((bool)group.GV(Am_H_SCROLL_BAR_ON_TOP)) return -SCROLL_BORDER;
  else // on bottom
    return (int)group.GV(Am_HEIGHT) - (int)self.GV(Am_HEIGHT) + SCROLL_BORDER;
}

Am_Define_Formula(int, h_scroll_width) {
  Am_Object group = self.GV_Owner();
  //4 for key_sel border of scroll bar + 2 for 1-pixel border around scrollarea
  return (int)group.GV(Am_CLIP_WIDTH)+6;
}

Am_Define_Formula(bool, h_scroll_visible) {
  Am_Object group = self.GV_Owner();
  if ((bool)group.GV(Am_VISIBLE) &&
      (bool)group.GV(Am_H_SCROLL_BAR)) return true;
  else return false;
}

Am_Define_Formula(bool, h_scroll_active) {
  Am_Object group = self.GV_Owner();
  if ((bool)group.GV(Am_ACTIVE) &&
      ((int)group.GV(Am_INNER_WIDTH) >=
       (int)group.GV(Am_CLIP_WIDTH)))
    return true;
  else {
    group.Set(Am_X_OFFSET, 0);  //make sure at origin when go inactive
    group.Get_Part(Am_H_SCROLLER).Get_Part(Am_COMMAND).Set(Am_VALUE, 0);
    return false;
  }
}

Am_Define_Formula(int, h_scroll_value_2) {
  Am_Object group = self.GV_Owner();
  int val = (int)group.GV(Am_INNER_WIDTH) - (int)group.GV(Am_CLIP_WIDTH);
  if (val < 1) val = 1;
  return val;
}

Am_Define_Formula(float, h_scroll_percent_visible) {
  Am_Object group = self.GV_Owner();
  float innerw =  group.GV(Am_INNER_WIDTH);
  if (innerw == 0.0f) innerw = 1.0f;
  float val = (float)group.GV(Am_CLIP_WIDTH) / innerw;
  if (val > 1.0) val = 1.0f;
  return val;
}

//explicitly remove Am_V_SCROLLER and Am_H_SCROLLER from the
//graphical objects list so programmers iterating through the list
//won't be confused by them
void remove_scrollers(Am_Object scroll_group) {
  Am_Value_List components;
  components = scroll_group.Get (Am_GRAPHICAL_PARTS);
  // probably would be OK to just set the list to empty, but this is safer
  components.Start();
  if(!components.Member(scroll_group.Get_Part(Am_V_SCROLLER)))
    Am_Error("Scroll group doesn't contain V_scroller");
  components.Delete();

  components.Start();
  if(!components.Member(scroll_group.Get_Part(Am_H_SCROLLER)))
    Am_Error("Scroll group doesn't contain H_scroller");
  components.Delete();
  scroll_group.Set (Am_GRAPHICAL_PARTS, components);
}

////////////////////////////////////////////////////////////////////////
// Scroll group do and undo
////////////////////////////////////////////////////////////////////////

void scroll_group_inter_abort_do(Am_Object command_obj, Am_Slot_Key slot) {
  clear_save_pos_for_undo_proc(command_obj); //do the standard scrollbar stuff
  Am_Object inter = command_obj.Get_Owner();
  Am_Object scrollbar = inter.Get_Owner();
  Am_Object scroll_group = scrollbar.Get_Owner();
  Am_Object scroll_bar_command = scrollbar.Get_Part(Am_COMMAND);
  int val = scroll_bar_command.Get(Am_OLD_VALUE);
  scroll_group.Set(slot, val);
}

Am_Define_Method(Am_Object_Method, void, scroll_group_v_inter_abort_do,
		 (Am_Object command)) {
  scroll_group_inter_abort_do(command, Am_Y_OFFSET);
}

Am_Define_Method(Am_Object_Method, void, scroll_group_h_inter_abort_do,
		 (Am_Object command)) {
  scroll_group_inter_abort_do(command, Am_X_OFFSET);
}


//update the y_offset of the group
Am_Define_Method(Am_Object_Method, void, scroll_group_v_do,
		 (Am_Object command)) {
  Am_Object v_scroll = command.Get_Owner();
  Am_Object scroll_group = v_scroll.Get_Owner();
  int val = v_scroll.Get(Am_VALUE);
  scroll_group.Set(Am_Y_OFFSET, val);
}

//update the x_offset of the group
Am_Define_Method(Am_Object_Method, void, scroll_group_h_do,
		 (Am_Object command)) {
  Am_Object h_scroll = command.Get_Owner();
  Am_Object scroll_group = h_scroll.Get_Owner();
  int val = h_scroll.Get(Am_VALUE);
  scroll_group.Set(Am_X_OFFSET, val);
}

void scroll_group_general_undo_redo(Am_Object command_obj, bool undo,
				    bool selective, Am_Slot_Key slot) {
  Am_Value new_value, old_value;
  Am_Object scroll_bar, scroll_group;
  scroll_bar = command_obj.Get(Am_SAVED_OLD_OWNER);
  scroll_group = scroll_bar.Get_Owner();
  if (scroll_group.Valid()) {
    
    if (selective)  // then get current value from the scroll_bar
      scroll_bar.Get(Am_VALUE, new_value);
    else // get current value from the command_obj
      command_obj.Get(Am_VALUE, new_value);
  
    if (undo) command_obj.Get(Am_OLD_VALUE, old_value);
    else  // repeat
      command_obj.Get(Am_VALUE, old_value);

    command_obj.Set(Am_OLD_VALUE, new_value);
    command_obj.Set(Am_VALUE, old_value);
    //also set scroll group
    if (scroll_group.Valid()) {
      scroll_group.Set(slot, old_value);
    }
  }
}

Am_Define_Method(Am_Object_Method, void, scroll_group_v_undo,
		 (Am_Object command_obj)) {
  scroll_group_general_undo_redo(command_obj, true, false, Am_Y_OFFSET);
}
Am_Define_Method(Am_Object_Method, void, scroll_group_v_selective_undo,
		 (Am_Object command_obj)) {
  scroll_group_general_undo_redo(command_obj, true, true, Am_Y_OFFSET);
}
Am_Define_Method(Am_Object_Method, void, scroll_group_v_selective_repeat,
		 (Am_Object command_obj)) {
  scroll_group_general_undo_redo(command_obj, false, true, Am_Y_OFFSET);
}
Am_Define_Method(Am_Object_Method, void, scroll_group_h_undo,
		 (Am_Object command_obj)) {
  scroll_group_general_undo_redo(command_obj, true, false, Am_X_OFFSET);
}
Am_Define_Method(Am_Object_Method, void, scroll_group_h_selective_undo,
		 (Am_Object command_obj)) {
  scroll_group_general_undo_redo(command_obj, true, true, Am_X_OFFSET);
}
Am_Define_Method(Am_Object_Method, void, scroll_group_h_selective_repeat,
		 (Am_Object command_obj)) {
  scroll_group_general_undo_redo(command_obj, false, true, Am_X_OFFSET);
}


////////////////////////////////////////////////////////////////////////
  
//goes in scroll bar inter
Am_Define_Value_Formula(v_scroller_value_from_y_offset) {
  //value is an out parameter defined by the macro
  Am_Object scroll_group = self.GV_Owner();
  scroll_group.GVM(Am_Y_OFFSET, value);
}

//goes in scroll bar inter
Am_Define_Value_Formula(h_scroller_value_from_x_offset) {
  //value is an out parameter defined by the macro
  Am_Object scroll_group = self.GV_Owner();
  scroll_group.GVM(Am_X_OFFSET, value);
}

Am_Define_Value_Formula(get_impl_parent_from_group) {
  //self is cmd, owner is scroll_bar, owner is scroll_group
  Am_Object group = self.GV_Owner().GV_Owner();
  if (group.Valid())
    group.GVM(Am_COMMAND, value);
  else value = Am_NOT_USUALLY_UNDONE;
}

////////////////////////////////////////////////////////////////////////

//exported objects
Am_Object Am_Vertical_Scroll_Bar = 0;
Am_Object Am_Horizontal_Scroll_Bar = 0;
  
// internal objects
Am_Object Am_Scroll_Arrow;
Am_Object Am_Scrolling_Group;

void Am_Scroll_Widgets_Initialize () {
  Am_Object inter; // interactor in the widget
  Am_Object command_obj; 
  Am_Object_Advanced obj_adv; // to get at advanced features like
			       // local-only and demons.

  ///////////////////////////////////////////////////////////////////////////
  // Scroll Internal Command
  ///////////////////////////////////////////////////////////////////////////
  
  Am_Object Am_Scroll_Inter_Command = Am_Command.Create("Scroll_Inter_Command")
    .Set(Am_IMPLEMENTATION_PARENT, Am_Get_Owners_Command)
    .Set(Am_DO_METHOD, NULL)
    .Set(Am_UNDO_METHOD, Am_Widget_Inter_Command_Undo)
    .Set(Am_REDO_METHOD, Am_Widget_Inter_Command_Undo)
    .Set(Am_SELECTIVE_UNDO_METHOD, Am_Widget_Inter_Command_Selective_Undo)
    .Set(Am_SELECTIVE_REPEAT_SAME_METHOD,
	 Am_Widget_Inter_Command_Selective_Repeat)
    .Set(Am_SELECTIVE_REPEAT_ON_NEW_METHOD, NULL)
    ;

  ///////////////////////////////////////////////////////////////////////////
  // Scroll bars
  ///////////////////////////////////////////////////////////////////////////
  Am_Scroll_Arrow = Am_Graphical_Object.Create ("Am_Scroll_Arrow")
     .Set (Am_INTERIM_SELECTED, false)
     .Set (Am_SCROLL_ARROW_DIRECTION, (int)Am_SCROLL_ARROW_UP)
     .Set (Am_WIDGET_LOOK, (int)Am_MOTIF_LOOK)
     .Set (Am_FILL_STYLE, Am_Amulet_Purple)
     .Set (Am_STYLE_RECORD, Am_Get_Computed_Colors_Record_Form)
     .Set (Am_DRAW_METHOD, scroll_arrow_draw)
     ;
  obj_adv = (Am_Object_Advanced&)Am_Scroll_Arrow;
  obj_adv.Get_Slot (Am_SELECTED)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_INTERIM_SELECTED)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_FILL_STYLE)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_SCROLL_ARROW_DIRECTION)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);

  
  Am_Object inter1;
  Am_Object inter2;
  Am_Object inter3;
  
  Am_Object Scroll_Bar = Am_Group.Create("Scroll_Bar")
     .Set (Am_VALUE, 50)
     .Set (Am_WIDTH, 20)
     .Set (Am_HEIGHT, 200)
     .Set (Am_KEY_SELECTED, false)
     .Set (Am_ACTIVE, Am_Active_From_Command)
     .Set (Am_ACTIVE_2, true) // used by interactive tools
     .Set (Am_WIDGET_LOOK, (int)Am_MOTIF_LOOK)
     .Set (Am_FILL_STYLE, Am_Amulet_Purple)
     .Set (Am_STYLE_RECORD, Am_Get_Computed_Colors_Record_Form)
     .Set (Am_VALUE_1, 0) //default type is int, but can be float
     .Set (Am_VALUE_2, 100)
     .Set (Am_SMALL_INCREMENT, 1)
     .Set (Am_LARGE_INCREMENT, 10)
     .Set (Am_PERCENT_VISIBLE, 0.2)
     .Set (Am_DRAW_METHOD, scroll_draw)
     .Set (Am_SCROLL_AREA_SIZE, 166) //temp, set by layout
     .Set (Am_SCROLL_AREA_MIN, 17)   //temp, set by layout
     .Set (Am_WIDGET_START_METHOD, Am_Standard_Widget_Start_Method)
     .Set (Am_WIDGET_ABORT_METHOD, Am_Standard_Widget_Abort_Method)
     .Set (Am_WIDGET_STOP_METHOD, Am_Standard_Widget_Stop_Method)
     .Add_Part(Am_SCROLL_ARROW1, Am_Scroll_Arrow.Create("Scroll_Arrow1")
		.Set (Am_SCROLL_ARROW_DIRECTION, (int)Am_SCROLL_ARROW_UP)
		.Set (Am_FILL_STYLE, Am_From_Owner (Am_FILL_STYLE))
		.Set (Am_VISIBLE, Am_From_Owner (Am_VISIBLE))
		)
     .Add_Part(Am_SCROLL_ARROW2, Am_Scroll_Arrow.Create("Scroll_Arrow2")
		.Set (Am_SCROLL_ARROW_DIRECTION, (int)Am_SCROLL_ARROW_DOWN)
		.Set (Am_FILL_STYLE, Am_From_Owner (Am_FILL_STYLE))
		.Set (Am_VISIBLE, Am_From_Owner (Am_VISIBLE))
		)
     .Add_Part(Am_SCROLL_INDICATOR,
		Am_Border_Rectangle.Create("V_Scroll_Indicator")
		.Set (Am_FILL_STYLE, Am_From_Owner (Am_FILL_STYLE))
		.Set (Am_LEFT, 4) //some of these replaced with formulas below
	        .Set_Single_Constraint_Mode(Am_LEFT, false) //needed for abort
		.Set (Am_TOP, 4)
	        .Set_Single_Constraint_Mode(Am_TOP, false) //needed for abort
		.Set (Am_WIDTH, 12)
		.Set (Am_HEIGHT, 12)
		.Set (Am_VISIBLE, Am_From_Owner (Am_VISIBLE))
		)
     .Add_Part(Am_INTERACTOR, inter1 =
		   Am_Move_Grow_Interactor.Create("Scroll_Indicator_Inter")
	        .Set (Am_START_WHEN, Am_Default_Widget_Start_Char)
		.Set (Am_START_WHERE_TEST, in_scroll_indicator)
		.Set (Am_ACTIVE, Am_Active_And_Active2)
		.Set (Am_PRIORITY, 30.0) //so higher than the others
	        .Set (Am_DO_METHOD, NULL) //all work done by interim_do
		)
     .Add_Part(Am_ARROW_INTERACTOR,
		inter2 = Am_Choice_Interactor.Create("Scroll_Arrow_Inter")
	        .Set (Am_START_WHEN, Am_Default_Widget_Start_Char)
		.Set (Am_START_WHERE_TEST, in_scroll_arrows)
		.Set (Am_ACTIVE, Am_Active_And_Active2)
		.Set (Am_PRIORITY, 20.0) //so higher than the background inter
		)
     .Add_Part(Am_BACKGROUND_INTERACTOR,
		inter3 = Am_Choice_Interactor.Create("Scroll_Page_Inter")
	       .Set (Am_START_WHEN, Am_Default_Widget_Start_Char)
	       .Set (Am_ACTIVE, Am_Active_And_Active2)
	       .Set (Am_START_WHERE_TEST, Am_Inter_In)
	       .Set (Am_PRIORITY, 10.0) // higher than normal
		)
     .Add_Part (Am_COMMAND, Am_Command.Create("Scroll_Command")
		.Set (Am_VALUE, 50)
		.Set (Am_LABEL, "Scrollbar"))
     .Set (Am_SET_COMMAND_OLD_OWNER, Am_Set_Old_Owner_To_Me)
     ;

  // don't do any of the standard undo things
  inter1.Get_Part(Am_IMPLEMENTATION_COMMAND)
    .Set (Am_UNDO_METHOD, NULL)
    .Set (Am_REDO_METHOD, NULL)
    .Set (Am_SELECTIVE_UNDO_METHOD, NULL)
    .Set (Am_SELECTIVE_REPEAT_SAME_METHOD, NULL)
    .Set (Am_SELECTIVE_REPEAT_ON_NEW_METHOD, NULL);

  inter1.Add_Part(Am_COMMAND,
	  Am_Scroll_Inter_Command.Create("Command_In_Scroll_Indicator_Inter")
		  .Set(Am_START_DO_METHOD, save_pos_for_undo)
		  .Set(Am_DO_METHOD, clear_save_pos_for_undo)
		  .Set(Am_ABORT_DO_METHOD, clear_save_pos_for_undo)
		  );
  inter2.Add_Part(Am_COMMAND,
	  Am_Scroll_Inter_Command.Create("Command_In_Scroll_Arrow_Inter")
		  .Set(Am_DO_METHOD, scroll_arrow_inter_command_do)
		  );
  inter3.Add_Part(Am_COMMAND,
	  Am_Scroll_Inter_Command.Create("Command_In_Scroll_Page_Inter")
		  .Set(Am_DO_METHOD, vertical_scroll_page_inter_command_do)
		  );
  obj_adv = (Am_Object_Advanced&)Scroll_Bar;
  obj_adv.Get_Slot (Am_KEY_SELECTED)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_ACTIVE)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_WIDGET_LOOK)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_FINAL_FEEDBACK_WANTED)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_FONT)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_FILL_STYLE)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_VALUE_1)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_VALUE_2)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_PERCENT_VISIBLE)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);

  // all the next slots should not be inherited
  obj_adv.Get_Slot(Am_ACTIVE).Set_Inherit_Rule(Am_COPY);
  obj_adv.Get_Slot(Am_ACTIVE_2).Set_Inherit_Rule(Am_COPY);

  obj_adv.Get_Slot(Am_VALUE).Set_Single_Constraint_Mode(false);

  Am_Vertical_Scroll_Bar = Scroll_Bar.Create("Vertical_Scroll_Bar")
    .Set(Am_SCROLL_AREA_MAX, v_scroll_area_max)
    .Set(Am_LAYOUT, v_scroll_layout_formula)
    ;

  Am_Vertical_Scroll_Bar.Get_Part(Am_SCROLL_INDICATOR)
    .Set(Am_TOP, scroll_indicator_pos)
    .Set(Am_WIDTH, 12) //overridden by Am_V_Scroll_Layout_Formula
    .Set(Am_HEIGHT, scroll_indicator_size)
    ;

  Am_Vertical_Scroll_Bar.Get_Part(Am_INTERACTOR)
    .Set(Am_INTERIM_DO_METHOD, v_scroll_indicator_inter_interim_do);

  Am_Horizontal_Scroll_Bar = Scroll_Bar.Create("Horizontal_Scroll_Bar")
    .Set (Am_WIDTH, 200)
    .Set (Am_HEIGHT, 20)
    .Set (Am_SCROLL_AREA_MAX, h_scroll_area_max)
    .Set (Am_LAYOUT, h_scroll_layout_formula)
    ;

  Am_Horizontal_Scroll_Bar.Get_Part(Am_SCROLL_ARROW1)
    .Set(Am_SCROLL_ARROW_DIRECTION, (int)Am_SCROLL_ARROW_LEFT);
  Am_Horizontal_Scroll_Bar.Get_Part(Am_SCROLL_ARROW2)
    .Set(Am_SCROLL_ARROW_DIRECTION, (int)Am_SCROLL_ARROW_RIGHT);
  Am_Horizontal_Scroll_Bar.Get_Part(Am_SCROLL_INDICATOR)
    .Set_Name("H_Scroll_Indicator")
    .Set(Am_LEFT, scroll_indicator_pos)
    .Set(Am_WIDTH, scroll_indicator_size)
    .Set(Am_HEIGHT, 12) //overridden by Am_H_Scroll_Layout_Formula
    ;
  Am_Horizontal_Scroll_Bar.Get_Part(Am_BACKGROUND_INTERACTOR)
    .Get_Part(Am_COMMAND)
    .Set(Am_DO_METHOD, horizontal_scroll_page_inter_command_do);
  Am_Horizontal_Scroll_Bar.Get_Part(Am_INTERACTOR)
    .Set(Am_INTERIM_DO_METHOD, h_scroll_indicator_inter_interim_do);

  
  ///////////////////////////////////////////////////////////////////////////
  // Scrolling_Group
  ///////////////////////////////////////////////////////////////////////////
  Am_Object v_scroll, h_scroll;

  Am_Scrolling_Group = Am_Group.Create("Scrolling_Group")
     .Set (Am_X_OFFSET, 0) // x offset of the scrollable area
     .Set (Am_Y_OFFSET, 0) 
     .Set (Am_WIDTH, 150)
     .Set (Am_HEIGHT, 150)
     .Set (Am_ACTIVE, true)
     .Set (Am_ACTIVE_2, true) // used by interactive tools
     .Set (Am_WIDGET_LOOK, (int)Am_MOTIF_LOOK)
     .Set (Am_FILL_STYLE, Am_Amulet_Purple)
     .Set (Am_LINE_STYLE, Am_Black) //border around scrolling area
     .Set (Am_INNER_FILL_STYLE, 0) //if not 0, then inside of window color
     .Set (Am_H_SCROLL_BAR, true)
     .Set (Am_V_SCROLL_BAR, true)
     .Set (Am_H_SCROLL_BAR_ON_TOP, false)
     .Set (Am_V_SCROLL_BAR_ON_LEFT, false)
     .Set (Am_H_SMALL_INCREMENT, 10) // in pixels
     .Set (Am_H_LARGE_INCREMENT, h_scroll_jump_page)
     .Set (Am_V_SMALL_INCREMENT, 10) // in pixels
     .Set (Am_V_LARGE_INCREMENT, v_scroll_jump_page)
     .Set (Am_DRAW_METHOD, scrolling_group_draw)

     .Set (Am_INVALID_METHOD, scrolling_group_invalid)
     .Set (Am_TRANSLATE_COORDINATES_METHOD,
	   scroll_group_translate_coordinates)
     .Set (Am_POINT_IN_PART_METHOD, scroll_group_point_in_part)
     .Set (Am_POINT_IN_LEAF_METHOD, scroll_group_point_in_leaf)
     .Set (Am_POINT_IN_OBJ_METHOD,  scroll_group_point_in_obj)

     .Set (Am_INNER_WIDTH, 400)   // ** USE THE FORMULA FOR WIDTH FROM GROUPS
     .Set (Am_INNER_HEIGHT, 400)  // ** USE THE FORMULA FROM GROUPS
     .Set (Am_COMMAND, Am_NOT_USUALLY_UNDONE) // whether undoable
     .Set (Am_SET_COMMAND_OLD_OWNER, Am_Set_Old_Owner_To_Me)

     
     // internal slots
     .Set (Am_CLIP_LEFT, scroll_clip_left)
     .Set (Am_CLIP_TOP, scroll_clip_top)
     .Set (Am_CLIP_WIDTH, scroll_clip_width)
     .Set (Am_CLIP_HEIGHT, scroll_clip_height)
     .Add_Part(Am_V_SCROLLER, v_scroll = 
		Am_Vertical_Scroll_Bar.Create("V_Scroll_In_Group")
	        .Set(Am_VALUE, 0)
	        .Set(Am_VALUE, v_scroller_value_from_y_offset)
		.Set(Am_LEFT, v_scroll_left)
		.Set(Am_TOP, v_scroll_top)
		.Set(Am_HEIGHT, v_scroll_height)
		.Set(Am_VISIBLE, v_scroll_visible)
		.Set(Am_ACTIVE, v_scroll_active)
		.Set(Am_ACTIVE_2, Am_From_Owner (Am_ACTIVE_2))
		.Set(Am_WIDGET_LOOK, Am_From_Owner (Am_WIDGET_LOOK))
		.Set(Am_FILL_STYLE, Am_From_Owner (Am_FILL_STYLE))
		.Set(Am_VALUE_1, 0)
		.Set(Am_VALUE_2, v_scroll_value_2)
		.Set(Am_SMALL_INCREMENT, Am_From_Owner (Am_V_SMALL_INCREMENT))
		.Set(Am_LARGE_INCREMENT, Am_From_Owner (Am_V_LARGE_INCREMENT))
		.Set(Am_PERCENT_VISIBLE, v_scroll_percent_visible)
	       )
     .Add_Part(Am_H_SCROLLER, h_scroll = 
		Am_Horizontal_Scroll_Bar.Create("H_Scroll_In_Group")
	        .Set(Am_VALUE, 0)
	        .Set(Am_VALUE, h_scroller_value_from_x_offset)
		.Set(Am_LEFT, h_scroll_left)
		.Set(Am_TOP, h_scroll_top)
		.Set(Am_WIDTH, h_scroll_width)
		.Set(Am_VISIBLE, h_scroll_visible)
		.Set(Am_ACTIVE, h_scroll_active)
		.Set(Am_ACTIVE_2, Am_From_Owner (Am_ACTIVE_2))
		.Set(Am_WIDGET_LOOK, Am_From_Owner (Am_WIDGET_LOOK))
		.Set(Am_FILL_STYLE, Am_From_Owner (Am_FILL_STYLE))
		.Set(Am_VALUE_1, 0)
		.Set(Am_VALUE_2, h_scroll_value_2)
		.Set(Am_SMALL_INCREMENT, Am_From_Owner (Am_H_SMALL_INCREMENT))
		.Set(Am_LARGE_INCREMENT, Am_From_Owner (Am_H_LARGE_INCREMENT))
		.Set(Am_PERCENT_VISIBLE, h_scroll_percent_visible)

		)
     ;
  obj_adv = (Am_Object_Advanced&)Am_Scrolling_Group;
  obj_adv.Get_Slot (Am_LINE_STYLE)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_X_OFFSET)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_Y_OFFSET)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_H_SCROLL_BAR)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_V_SCROLL_BAR)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_H_SCROLL_BAR_ON_TOP)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_V_SCROLL_BAR_ON_LEFT)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);

  //explicitly remove Am_V_SCROLLER and Am_H_SCROLLER from the
  //graphical objects list so programmers iterating through the list
  //won't be confused by them
  remove_scrollers(Am_Scrolling_Group);

  v_scroll.Get_Part(Am_COMMAND)
    .Set(Am_LABEL, "Scroll Vertical")
    .Set(Am_DO_METHOD, scroll_group_v_do)
    .Set(Am_UNDO_METHOD, scroll_group_v_undo)
    .Set(Am_REDO_METHOD, scroll_group_v_undo)
    .Set(Am_SELECTIVE_UNDO_METHOD, scroll_group_v_selective_undo)
    .Set(Am_SELECTIVE_REPEAT_SAME_METHOD, scroll_group_v_selective_repeat)
    .Set(Am_SELECTIVE_REPEAT_ON_NEW_METHOD, NULL)
    //not undoable by default
    .Set(Am_IMPLEMENTATION_PARENT, get_impl_parent_from_group)
    .Set(Am_IMPLEMENTATION_CHILD, Am_DONT_UNDO_CHILDREN)
    ;
  v_scroll.Get_Part(Am_INTERACTOR).Get_Part(Am_COMMAND)
    .Set(Am_ABORT_DO_METHOD, scroll_group_v_inter_abort_do);

  h_scroll.Get_Part(Am_COMMAND)
    .Set(Am_LABEL, "Scroll Horizontal")
    .Set(Am_DO_METHOD, scroll_group_h_do)
    .Set(Am_IMPLEMENTATION_PARENT, get_impl_parent_from_group)
    .Set(Am_UNDO_METHOD, scroll_group_h_undo)
    .Set(Am_REDO_METHOD, scroll_group_h_undo)
    .Set(Am_SELECTIVE_UNDO_METHOD, scroll_group_h_selective_undo)
    .Set(Am_SELECTIVE_REPEAT_SAME_METHOD, scroll_group_h_selective_repeat)
    .Set(Am_SELECTIVE_REPEAT_ON_NEW_METHOD, NULL)
    .Set(Am_IMPLEMENTATION_CHILD, Am_DONT_UNDO_CHILDREN)
    ;
  h_scroll.Get_Part(Am_INTERACTOR).Get_Part(Am_COMMAND)
    .Set(Am_ABORT_DO_METHOD, scroll_group_h_inter_abort_do);
}
