#ifndef PixmapDraw_included
#define PixmapDraw_included

#include <qlist.h>
#include <qcolor.h>
#include <qframe.h>
#include <qpixmap.h>
#include <qscrbar.h>
#include <qfile.h>

#define MIN_CELLSIZE 1

class Previewer;

class PixmapDraw : public QFrame
{
  Q_OBJECT

public: // ctor, dtor
  PixmapDraw( QWidget *, const char * = NULL );

  ~PixmapDraw();

public: // methods
  void SetSize( const QSize& );
  // set a new size, all cells will be resized down to MIN_CELLSIZE
  // cells will maintain same width/height

  QSize GetSize();
  // return actual size

  const QColor &GetBackgroundColor();
  void SetBackgroundColor( const QColor & );

  void SetRightColor( const QColor & );
  void SetLeftColor( const QColor & );
  // Set the colors for the both used mousebuttons

  const QColor &GetRightColor();
  const QColor &GetLeftColor();
  // Get the colors for the both used mousebuttons

  enum DrawMode {
    DrawPoint = 1,
    DrawLine,
    DrawCircle,
    DrawRect,
    DrawFill,
    DrawRegion,
    DrawPaste,
    LastDrawMode
  };

  void SetMode( DrawMode );
  DrawMode GetMode();
  // Get and set various modes, see enum DrawMode above

  void ModeChange( DrawMode );
  // New drawing mode

  void SetupPreview( Previewer * );
  // Show up a preview window

  void Rotate( float );
  // Rotate pixmap

  enum FlipDirection {
    Vertical,
    Horizontal 
  };

  void Flip( FlipDirection );
  // Flip the pixmap

  bool LoadFile( const QString & );
  // Read pixmap from disk

  bool WriteFile( const QString &, const QString & );
  // Save pixmap to disk

  void SetPixelSize( int );
  // Set a new pixel size

  void SetLineStyle( int );
  // Set a new Line style

  void UseGrid( bool );
  // En/Disable the grid on zoomed views

  void Deselect();
  // Disable selection

  void GeneratePixmap( QString &, QString &, QFile * );
  // Generate pixmap data

signals:
  void Changed();
  void MouseTracking( int, int, int, int );
  void ZoomChanged( int );
  void CPAvailable();
  
public slots:
  void ZoomIn();
  void ZoomOut();

  bool Copy();
  bool Cut();
  bool Paste();
  // C&P 

  void ClipRegion();
  void TimerEvent();

private: // attributes
  int actCellSize;
  // actual width/height of a cell
  
  int numCols, numRows;
  // actual size

  QPixmap *itsPixmap;
  // the actual pixmap 

  QPixmap *itsClipboard;
  // clipboard buffer

  QRect clipboardRect;
  // Place where the clipboard content was originated from

  bool selectionVisible;
  // Set by TimerEvent when selection rect is drawn

  bool regionActive;
  // When true, all drawings are clipped by selected region

  QColor leftColor, rightColor;
  // The colors for the mouse press events

  QColor backGround;

  DrawMode mode;
  // actual drawing mode

  QPoint firstPoint;
  QPoint lastPoint;
  // Point where mouse press occured

  QScrollBar *itsHorScr;
  QScrollBar *itsVerScr;
  // The scrollbars

  int horOff, verOff;
  // Offset due to scroll

  int contRectWidth, contRectHeight;
  // corrected contentsRect()

  Previewer *previewWindow;
  // The preview Window

  int pixelWidth;
  // Actual pen size

  int lineStyle;
  // Actual line style

  bool useGrid;
  // Display a grid in zoomed views

  QTimer *itsTimer;
  // Timer used for showing selected areas

private: // methods
  virtual void drawContents( QPainter * );
  // The actual drawing method

  virtual void resizeEvent ( QResizeEvent * ); 

  virtual void mouseMoveEvent( QMouseEvent * );
  virtual void mousePressEvent( QMouseEvent * );
  virtual void mouseReleaseEvent( QMouseEvent * );
  virtual void enterEvent( QEvent * );
  virtual void leaveEvent( QEvent * );

  void Fill( int, int, const QColor & );
  void FillLine( QImage &, int, int, unsigned int, unsigned int );

  int CalcCellSize();
  void CalcContRect();

  int CoordTrans( int );

private slots:
  void HorScrBarChanged( int );
  void VerScrBarChanged( int );
};

inline void PixmapDraw::ModeChange( DrawMode m )
{
  mode = m; 
  setCursor( arrowCursor );
}
  
inline void PixmapDraw::SetLeftColor( const QColor &c )
{
  leftColor = c;
}

inline void PixmapDraw::SetRightColor( const QColor &c )
{
  rightColor = c;
}

inline const QColor &PixmapDraw::GetLeftColor()
{
  return leftColor;
}

inline const QColor &PixmapDraw::GetRightColor()
{
  return rightColor;
}

inline const QColor &PixmapDraw::GetBackgroundColor()
{
  return backGround;
}

inline void PixmapDraw::SetPixelSize( int s )
{
  pixelWidth = s;
}

inline void PixmapDraw::SetLineStyle( int s )
{
  lineStyle = s;
}

inline int PixmapDraw::CoordTrans( int mc )
{
  return mc / actCellSize;
}


#endif
