/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,   
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

#ifndef TSignal
#define TSignal TSignal

#include <toad/pointer.hh>

#ifdef TOAD_OLD_CONNECT
#  warning "using TOAD_OLD_CONNECT"
#  define CONNECT(A,B,C,D) { typedef typeof(*A) T; _toad_connect(A,&T::B,C,&D); }
#else
#  define OLD_CONNECT(A,B,C,D) { typedef typeof(*A) T; _toad_connect(A,&T::B,C,&D); }
#  define CONNECT(S, O, M, ARGS...) { typedef typeof(*O) T; S.Add(O, &T::M , ## ARGS ); }
#endif

class TAction:
	public TSmartObject
{
	public:
		TAction();
		virtual ~TAction();
		virtual void execute() = 0;
};

typedef GSmartPointer<TAction> PAction;

class TSignalBase
{
	protected:
		class TSignalNode;
		typedef GSmartPointer<TSignalNode> PSignalNode;

		class TSignalNode:
			public TAction
		{
				friend TSignalBase;
				PSignalNode _next;
			public:
				~TSignalNode();
		};

		// function, void
		template <class A>
		class GSignalNode1:
			public TSignalNode
		{
			protected:
				A(*_method)(void);
			public:
				GSignalNode1(A(*m)(void)) {
					_method = m;
				}
				void execute() {
					(*_method)();
				}
		};

		// function, one parameter
		template <class A, class U>
		class GSignalNode2:
			public TSignalNode
		{
			protected:
				A(*_method)(U);
				U _parameter;
			public:
				GSignalNode2(A(*m)(U), U p) {
					_method = m;
					_parameter = p;
				}
				void execute() {
					(*_method)(_parameter);
				}
		};

		// function, two parameter
		template <class A, class U1, class U2>
		class GSignalNode3:
			public TSignalNode
		{
			protected:
				A(*_method)(U1,U2);
				U1 _parameter1;
				U2 _parameter2;
			public:
				GSignalNode3(A(*m)(U1,U2), U1 p1, U2 p2) {
					_method = m;
					_parameter1 = p1;
					_parameter2 = p2;
				}
				void execute() {
					(*_method)(_parameter1, _parameter2);
				}
		};

	public:	
		// `check' is currently used by `TOADBase::RemoveMessage' to disable
		// some signal node when their object is being destroyed.
		class TSignalNodeCheck
		{
			public:
				virtual void check(void *object);
		};

	protected:
		// method, void
		template <class A, class T>
		class GSignalNode4:
			public TSignalNode, TSignalNodeCheck
		{
			protected:
				T *_object;
				A(T::*_method)(void);
			public:
				GSignalNode4(T *o, A(T::*m)(void)):
					_object(o), _method(m) {}
				void execute() {
					if (_object)
						(_object->*_method)();
				}
				void check(void *o) {
					if (o==(void*)_object)
						_object=0;
				}
		};

		// method, one parameter
		template <class A, class T, class U>
		class GSignalNode5:
			public TSignalNode, TSignalNodeCheck
		{
				T *_object;
				A(T::*_method)(U);
				U _parameter;
			public:
				GSignalNode5(T *o, A(T::*m)(U), U p):
					_object(o), _method(m), _parameter(p) {}
				void execute() {
					if (_object)
						(_object->*_method)(_parameter);
				}
				void check(void *o) {
					if (o==(void*)_object)
						_object=0;
				}
		};

		// method, two parameters
		template <class A, class T, class U1, class U2>
		class GSignalNode6:
			public TSignalNode, TSignalNodeCheck
		{
				T *_object;
				A(T::*_method)(U1,U2);
				U1 _parameter1;
				U2 _parameter2;
			public:
				GSignalNode6(T *o, A(T::*m)(U1,U2), U1 p1, U2 p2):
					_object(o), _method(m), _parameter1(p1), _parameter2(p2) {}
				void execute() {
					if (_object)
						(_object->*_method)(_parameter1, _parameter2);
				}
				void check(void *o) {
					if (o==(void*)_object)
						_object=0;
				}
		};

	private:
		PSignalNode _list;

	public:
		TSignalBase();
		virtual ~TSignalBase();
		
		void Add(TSignalNode *);

		void ClearAll();		
		bool Trigger();
		bool operator()(){return Trigger();}
		bool DelayedTrigger();
		bool IsConnected() { return _list; }
};

#define _TOAD_TSIGNAL_ADD \
	template<class R> \
	void Add( R(*m)(void) ) { \
		TSignalBase::Add(new GSignalNode1<R>(m)); \
	} \
	template <class R, class U, class V> \
	void Add( R(*m)(U), V p) { \
		TSignalBase::Add(new GSignalNode2<R,U>(m,p)); \
	} \
	template <class R, class U1, class U2, class V1, class V2> \
	void Add( R(*m)(U1,U2), V1 p1, V2 p2) { \
		TSignalBase::Add(new GSignalNode3<R,U1,U2>(m,p)); \
	} \
	template <class R, class T1, class T2> \
	void Add(T1 *o, R(T2::*m)(void)) { \
		TSignalBase::Add(new GSignalNode4<R,T1>(o,m)); \
	} \
	template <class R, class T1, class T2, class U, class V> \
	void Add(T1 *o, R(T2::*m)(U), V p) { \
		TSignalBase::Add(new GSignalNode5<R,T1,U>(o,m,p)); \
	} \
	template <class R, class T1, class T2, class U1, class U2, class V1, class V2> \
	void Add(T1 *o, R(T2::*m)(U1,U2), V1 p1, V2 p2) { \
		TSignalBase::Add(new GSignalNode6<R,T1,U1,U2>(o,m,p1,p2)); \
	}

class TSignal:
	public TSignalBase
{
	public:
		TSignal();
	
		_TOAD_TSIGNAL_ADD
};

template <class P1>
class GSignal1P:
	public TSignal
{
		template <class A, class T>
		class GNode:
			public TSignalNode, public TSignalNodeCheck
		{
			protected:
				typedef GSignal1P<P1> TOwner;
				TOwner *_owner;
				T *_object;
				A(T::*_method)(P1);

			public:
				GNode(TOwner *own, T *o, A(T::*m)(P1)): 
					_owner(own), _object(o), _method(m) {};
			
				void execute() {
					if (_object)
						(_object->*_method)(_owner->_p1);
				}
				void check(void *o) {
					if (o==(void*)_object)
						_object = 0;
				}
		};

		template <class A, class T>
		class GNodeCR:
			public TSignalNode, public TSignalNodeCheck
		{
			protected:
				typedef GSignal1P<P1> TOwner;
				TOwner *_owner;
				T *_object;
				A(T::*_method)(const P1&);

			public:
				GNodeCR(TOwner *own, T *o, A(T::*m)(const P1&)): 
					_owner(own), _object(o), _method(m) {};
			
				void execute() {
					if (_object)
						(_object->*_method)(_owner->_p1);
				}
				void check(void *o) {
					if (o==(void*)_object)
						_object = 0;
				}
		};

	public:
		P1 _p1;
		
		template <class A, class T1, class T2>
		void Add(T1 *o, A(T2::*m)(P1) ) {
			TSignalBase::Add(new GNode<A,T1>(this,o,m));
		}

		template <class A, class T1, class T2>
		void Add(T1 *o, A(T2::*m)(const P1&) ) {
			TSignalBase::Add(new GNodeCR<A,T1>(this,o,m));
		}
		
		_TOAD_TSIGNAL_ADD
		
		bool Trigger(const P1 &p1) {
			_p1 = p1;
			return TSignalBase::Trigger();
		}
		bool operator()(const P1 &p1) { return Trigger(p1); }

	private:
		bool Trigger() {}
		bool operator()(){return false;}
		bool DelayedTrigger() {}
};

template <class P1, class P2>
class GSignal2P:
	public TSignal
{
		template <class A, class T>
		class GNode:
			public TSignalNode, public TSignalNodeCheck
		{
			protected:
				typedef GSignal2P<P1,P2> TOwner;
				TOwner *_owner;
				T *_object;
				A(T::*_method)(P1,P2);

			public:
				GNode(TOwner *own, T *o, A(T::*m)(P1,P2)): 
					_owner(own), _object(o), _method(m) {};
			
				void execute() {
					if (_object)
						(_object->*_method)(_owner->_p1, _owner->_p2);
				}
				void check(void *o) {
					if (o==(void*)_object)
						_object = 0;
				}
		};

	public:
		P1 _p1;
		P2 _p2;

		template <class A, class T1, class T2>
		void Add(T1 *o, A(T2::*m)(P1,P2) ) {
			TSignalBase::Add(new GNode<A,T1>(this,o,m));
		}

		_TOAD_TSIGNAL_ADD

		bool Trigger(const P1 &p1, const P2 &p2) {
			_p1 = p1;
			_p2 = p2;
			return TSignalBase::Trigger();
		}
		
		bool operator()(const P1 &p1, const P2 &p2) { return Trigger(p1, p2); }

	private:
		bool Trigger() {}
		bool operator()(){return false;}
		bool DelayedTrigger() {}
};

template <class P1, class P2, class P3>
class GSignal3P:
	public TSignal
{
		template <class A, class T>
		class GNode:
			public TSignalNode, public TSignalNodeCheck
		{
			protected:
				typedef GSignal3P<P1,P2,P3> TOwner;
				TOwner *_owner;
				T *_object;
				A(T::*_method)(P1,P2,P3);

			public:
				GNode(TOwner *own, T *o, A(T::*m)(P1,P2,P3)): 
					_owner(own), _object(o), _method(m) {};
			
				void execute() {
					if (_object)
						(_object->*_method)(_owner->_p1, _owner->_p2, _owner->_p3);
				}
				void check(void *o) {
					if (o==(void*)_object)
						_object = 0;
				}
		};

	public:
		P1 _p1;
		P2 _p2;
		P3 _p3;

		template <class A, class T1, class T2>
		void Add(T1 *o, A(T2::*m)(P1,P2,P3) ) {
			TSignalBase::Add(new GNode<A,T1>(this,o,m));
		}

		_TOAD_TSIGNAL_ADD

		bool Trigger(const P1 &p1, const P2 &p2, const P3 &p3) {
			_p1 = p1;
			_p2 = p2;
			_p3 = p3;
			return TSignalBase::Trigger();
		}
		
		bool operator()(const P1 &p1, const P2 &p2, const P3 &p3) 
			{ return Trigger(p1, p2, p3); }

	private:
		bool Trigger() {}
		bool operator()(){return false;}
		bool DelayedTrigger() {}
};

#undef _TOAD_TSIGNAL_ADD

template <class T1, class T2, class R, class P1, class P2>
void _toad_connect(T1 *o, R(T2::*m)(P1), P2 p, TSignal* s)
{
	s->Add(o,m,p);
}

template <class T1, class T2, class R, class P2>
void _toad_connect(T1 *o, R(T2::*m)(void), P2, TSignal* s)
{
	s->Add(o,m);
}

#endif
