/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

#ifndef GDataManipulator
#define GDataManipulator GDataManipulator

#include <string>
#include <toad/datainterface.hh>
#include <toad/connect.hh>

class TDataManipulator {
	public:
		TDataManipulator() {};
		virtual ~TDataManipulator();	// force creation of virtual table in library (EGCS 1.0.3a)
		virtual void Apply() = 0;
		virtual void Reset() = 0;
		
	 static void Assimilate(const void*); 
	 static void Enable(const void*);
	 static void Disable(const void*);
		/*
			 to write something like
				 value = 17;
				 Assimilate(&value);
			 when there's no controller arround
		*/
		
	protected:
		static void _Register(TDataInterface*, TDataManipulator*);
		static void _Unregister(TDataInterface*, TDataManipulator*);
		static void _Assimilate(TDataInterface*, TDataManipulator*);
		virtual void _assimilate() = 0;
		TDataManipulator* _next;
		
		virtual TDataInterface* Interface() = 0;	// for `TDataManipulator::Assimilate'
};

// a data manipulator with special need how the data is stored
template <class T, class I> 
class GStrangeDataManipulator: public TDataManipulator
{
		I *_interface;
		TDataInterface* Interface() { return _interface; }
		bool _indirect;
	public:
		GStrangeDataManipulator() { _interface = NULL; _indirect = true; } 
		virtual ~GStrangeDataManipulator() {	
			if (_interface) {
				_Unregister(_interface, this);
				delete _interface; 
			}
		} 

		void SetInterface(I *interface) {
			if (_interface) {
				_Unregister(_interface, this);
				delete _interface;
			}
			_interface = interface;
			_Register(_interface, this);
			Reset();
		}
		I *Interface() const { return _interface; } 

		// `true' : manipulate data after each call to `ValueChanged()'
		// `false': manipulate data after each call to `Apply()'
		void SetDirect(bool b) { _indirect = !b; }
		bool Direct() const { return !_indirect; }

		void Apply() { 
			if (_interface) {
				T data = getData();
				_interface->SetValue(data);
				_interface->SetX(data);
				_Assimilate(_interface, this);
			}
		} 
		void Reset() { if (_interface) putData(_interface->Value()); }

		// set the manipulators data
		void SetValue(T data)
		{
			if (_interface) {
				_interface->SetX(data);
				_Assimilate(_interface, this);
			}
			putData(data);
			if (!_indirect && _interface)
				_interface->SetValue(data);
		}
		
		// return the manipulators data
		const T& Value()
		{
			return getData();
		}

		GSignal1P<T> sigValueChanged;

	protected:
		void _assimilate() {
			if (_interface)
				 putData(_interface->GetX());
		}

		// a subclassed manipulator must call this when it's data was modified
		void ValueChanged()
		{
			const T& data = getData();
			if (_interface) {
				if (!_indirect)
					_interface->SetValue(data);
				_interface->SetX(data);
				_Assimilate(_interface, this);
			}
			sigValueChanged(data);
		}

		// subclasses must implement these to receiver & deliver their current
		// value
		virtual void putData(const T& data) = 0;
		virtual const T& getData() = 0;
};

template <class T, class I>
class GDataManipulator: public GStrangeDataManipulator<T,I>
{
	protected:
		T _data;

		void putData(const T& data) {	_data = data;	valueChanged(); }
		const T& getData() { return _data; }
		
		virtual void valueChanged() { };
};

class TStringInterface;

class TStringManipulator: public GDataManipulator<string, TStringInterface>
{
	public:
		~TStringManipulator();	// force creation of virtual table

		template <class T>
		void SetData(T *data)
		{
			SetInterface(NewStringInterface(data));
		}
};

class TIntInterface;

class TIntManipulator: public GDataManipulator<int, TIntInterface>
{
	public:
		~TIntManipulator();			// force creation of virtual table
		template <class T>
		void SetData(T *data)
		{
			SetInterface(NewIntInterface(data));
		}
};

class TBoolInterface;

class TBoolManipulator: public GDataManipulator<bool, TBoolInterface>
{
	public:
		~TBoolManipulator();

		template <class T>
		void SetData(T *data)
		{
			SetInterface(NewBoolInterface(data));
		}
};

#endif
