/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef TBinaryFile
#define	TBinaryFile TBinStream

typedef unsigned long ulong;
typedef unsigned char byte;

#include <iostream>
#include <string>

class TBinStreamBase
{
	public:
		TBinStreamBase() {
			_endian = LITTLE;
		}
	
		enum EEndian { LITTLE, BIG };
		void SetEndian(EEndian e) {
			_endian = e;
		}

	protected:
		EEndian _endian;
};

class TInBinStream:
	public virtual TBinStreamBase
{
	protected:
		istream *in;

	public:
		TInBinStream(istream*);

		ulong TellRead()  { return in->tellg(); }	// get
		void SeekRead(ulong p, ios::seek_dir d=ios::beg) { in->seekg(p,d); }
		ulong ReadCount() { return in->gcount(); }
		bool Eof() { return in->eof();}

		void Read(signed char* buffer, size_t count);
		void Read(unsigned char* buffer, size_t count);
		void ReadString(char*,int len);
		void ReadString(unsigned char *buf,int len)
			{return ReadString((char*)buf,len);}
		void ReadString(string*,int len);
		string ReadString(int len);
		bool CompareString(const char*,int len);
		void Unget(){in->unget();}
		
		signed int ReadSByte();							// 8 bit signed integer
		unsigned int ReadByte();						// 8 bit unsigned integer
		signed int ReadSWord();							// 16 bit signed integer
		unsigned int ReadWord();						// 16 bit unsigned integer
		signed long ReadSDWord();						// 32 bit signed integer
		unsigned long ReadDWord();					// 32 bit unsigned integer
#if 0
		double ReadDouble();								// 64bit IEEE 754 coded floating-point
		float ReadFloat();									// 32bit IEEE 754 coded floating-point
#endif

		int ReadInt() { return ReadSDWord(); }
};

class TOutBinStream:
	public virtual TBinStreamBase
{
	protected:
		ostream *out;

	public:
		TOutBinStream(ostream*);

		void Flush() { out->flush(); }
		ulong TellWrite() { return out->tellp(); }	// put
		void SeekWrite(ulong p, ios::seek_dir d=ios::beg) { out->seekp(p,d); }

		void Write(const signed char* buffer, size_t count);
		void Write(const unsigned char* buffer, size_t count);
		void Write(signed char* buffer, size_t count);
		void Write(unsigned char* buffer, size_t count);

		void WriteSByte(signed int);				// 8bit signed integer
		void WriteByte(unsigned int);				// 8bit unsigned integer
		void WriteSWord(signed int);				// 16bit signed integer
		void WriteWord(unsigned int);				// 16bit unsigned integer
		void WriteSDWord(signed long);			// 32bit signed integer
		void WriteDWord(unsigned long);			// 32bit unsigned integer
#if 0
		void WriteDouble(double);						// 64bit IEEE 754 coded floating-point
		void WriteFloat(float);							// 32bit IEEE 754 coded floating-point
#endif

		void WriteString(const char*, unsigned len);
		void WriteString(const char*);
		void WriteString(const string&);

		void WriteInt(int d) { WriteDWord(d); };
};

class TBinStream:
	public TInBinStream, public TOutBinStream
{
	public:
		TBinStream(iostream *io):TInBinStream(io), TOutBinStream(io) {};
};

#endif
