/* vmanpg 1.1 - svgalib pager for man pages.
 * Copyright (C) 1998,2000 Russell Marks.
 *
 * vmanpg.c - the pager itself.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <vga.h>
#include "font.h"
#include "readnbkey.h"
#include "config.h"


/* doesn't start at zero as they're stored in a string;
 * also, these match the values used in font.c.
 */
#define ATTR_NORMAL	1
#define ATTR_BOLD	2
#define ATTR_ITALIC	3



/* deal with backspaces, figuring out for each char whether it's
 * normal, bold or italic.
 */
void parse_attrs(unsigned char *input,
	unsigned char *plaintext,unsigned char *attrs)
{
unsigned char *iptr=input,*tptr=plaintext,*aptr=attrs;

while(*iptr)
  {
  if(*iptr!=8)	/* backspace */
    {
    *tptr++=*iptr++;
    *aptr++=ATTR_NORMAL;
    }
  else
    {
    iptr++;		/* move on to the overstrike char */
    if(*iptr=='_' || tptr[-1]=='_')
      {
      aptr[-1]=ATTR_ITALIC;
      if(tptr[-1]=='_') tptr[-1]=*iptr;
      }
    else
      if(*iptr==tptr[-1])	/* same as prev. char */
        aptr[-1]=ATTR_BOLD;
    iptr++;		/* skip to next char */
    }
  }

/* end the output strings */

*tptr=*aptr=0;
}


/* read in all of input file from stdin, parsing attrs */
int grab_input(unsigned char **lines_ptr,int **lineofs_ptr)
{
static unsigned char buf[1024];	/* XXX should allow arbitrary line length */
static unsigned char plaintext[1024],attrs[1024];
int numlines=0;
unsigned char *lines=NULL;
int lines_size=0x8000,lines_incr=0x4000;
int *lineofs=NULL;
int lineofs_idx=0;
int lineofs_size=0x2000,lineofs_incr=0x1000;
int current_offset=0;
int len;

if((lines=malloc(lines_size))==NULL ||
   (lineofs=malloc(lineofs_size))==NULL)
  fprintf(stderr,"vman: not enough memory\n"),exit(1);

while(fgets(buf,sizeof(buf),stdin)!=NULL)
  {
  len=strlen(buf);
  
  /* chop LF */
  if(len>0 && buf[len-1]=='\n') buf[--len]=0;
  
  parse_attrs(buf,plaintext,attrs);
  len=strlen(plaintext);	/* attrs will also be this length */
  
  /* make lines storage bigger if needed, then copy line */
  if(current_offset+(len+1)*2>=lines_size)
    {
    lines_size+=lines_incr;
    if((lines=realloc(lines,lines_size))==NULL)
      fprintf(stderr,"vman: not enough memory\n"),exit(1);
    }
  strcpy(lines+current_offset,plaintext);
  strcpy(lines+current_offset+len+1,attrs);
  
  /* make offset-of-line array bigger if needed, then store offset */
  if((lineofs_idx+1)*sizeof(int)>=lineofs_size)
    {
    lineofs_size+=lineofs_incr;
    if((lineofs=realloc(lineofs,lineofs_size))==NULL)
      fprintf(stderr,"vman: not enough memory\n"),exit(1);
    }
  lineofs[lineofs_idx]=current_offset;
  lineofs_idx++;
  current_offset+=(len+1)*2;
  numlines++;
  }

freopen("/dev/tty","r",stdin);

*lines_ptr=lines;
*lineofs_ptr=lineofs;
return(numlines);
}


/* mark_opt is 1 for `mark' prompt, 2 for `goto' prompt, else 0 */
void drawpage(int top,unsigned char *lines,int *lineofs,int numlines,
	int mark_opt)
{
static unsigned char twocharbuf[2]={0,0};
static unsigned char buf[256];
char *manpg=getenv("MAN_PN");
int tx,ty;
int h=get_font_height();
unsigned char *ptr;
int len,f,curline=top;
int bot,pcnt;

vmode_clear();

ty=Y_MARGIN;
while(ty+h<=STATUS_LINE_YPOS-STATUS_LINE_MARGIN)
  {
  if(curline>=numlines) break;
  
  tx=X_MARGIN;
  ptr=lines+lineofs[curline];
  len=strlen(ptr);
  for(f=0;f<len;f++)
    {
    /* XXX could do with a draw-one-char routine instead of this crock! */
    *twocharbuf=ptr[f];
    tx+=vmode_drawtext(tx,ty,ptr[len+1+f],twocharbuf);
    if(tx>=640-X_MARGIN) break;
    }
  
  curline++;
  ty+=h;
  }

/* do status line */
switch(mark_opt)
  {
  case 0:
    *buf=0;
    if(manpg) sprintf(buf,"Manual page %s ",manpg);
    sprintf(buf+strlen(buf),"line %d/%d (",top+1,numlines);
    bot=curline;
    pcnt=(bot*100)/numlines;
    if(pcnt>=100)
      strcat(buf,"END)");
    else
      sprintf(buf+strlen(buf),"%d%%)",pcnt);
    break;
  
  case 1:
    strcpy(buf,"mark:");
    break;
    
  case 2:
    strcpy(buf,"goto mark:");
    break;
    
  default:
    fprintf(stderr,"can't happen\n"),exit(1);
  }

vmode_drawtext(STATUS_LINE_XPOS,STATUS_LINE_YPOS,0,buf);

vmode_refresh();
}


int getkey()
{
int key=RK_NO_KEY;

/* svgalib uses select in vga_getkey, so doing a usleep is
 * a bit paranoid. But I s'pose the implementation could change,
 * and better safe than sorry.
 */
while(key==RK_NO_KEY)
  {
  usleep(20000);
  key=readnbkey();
  }

return(key);
}


/* this is pretty BFI, and assumes a short needle */
unsigned char *my_strcasestr(unsigned char *haystack,unsigned char *needle)
{
static unsigned char needle_lcase[256];
int f,nlen=strlen(needle);
unsigned char *hptr,*hchk,*nptr;

for(f=0;f<nlen;f++)
  needle_lcase[f]=tolower(needle[f]);
needle_lcase[f]=0;

hptr=haystack;
while(*hptr)
  {
  hchk=hptr; nptr=needle_lcase;
  while(tolower(*hchk)==*nptr)
    {
    nptr++;
    if(*nptr==0) return(hptr);	/* match found */
    hchk++;
    }
  hptr++;
  }

return(NULL);
}


void do_search(int *top_ptr,unsigned char *searchtext,int dir,
	unsigned char *lines,int *lineofs,int numlines)
{
static unsigned char buf[256],ibuf[40],*iptr,*bptr;
/* the input buffer is small to restrict input to one line onscreen */
int top=*top_ptr;
int done=0,key,f;
int ignore_case,found;
char *ret;

strcpy(buf,(dir==1)?"/":"?");

/* now need to grab a line of input */
*ibuf=0;
iptr=ibuf;
while(!done)
  {
  /* make copy of input line to display */
  bptr=buf+1;
  for(f=0;f<strlen(ibuf);f++)
    {
    if(ibuf[f]==9)	/* tab */
      {
      *bptr++='^';
      *bptr++='I';
      }
    else
      *bptr++=ibuf[f];
    }
  *bptr++='_';
  *bptr=0;
  
  vmode_clearlines(STATUS_LINE_YPOS-1,479);
  vmode_drawtext(STATUS_LINE_XPOS,STATUS_LINE_YPOS,0,buf);
  vmode_refresh();
  
  key=getkey();
  if((key>=32 && key<=126) || key==9)	/* allow tab too */
    {
    if(iptr<ibuf+sizeof(ibuf)-1)
      {
      *iptr++=key;
      *iptr=0;
      }
    }
  else
    {
    switch(key)
      {
      case 8: case 127:
        if(iptr>ibuf)
          *--iptr=0;
        break;
      case RK_ENTER:
        done=1; break;
      case RK_ESC:
        done=1; break;
      }
    }
  }

if(key==RK_ESC) return;

if(*ibuf)
  strcpy(searchtext,ibuf);	/* else copy ibuf as next old text */
else
  {
  strcpy(ibuf,searchtext);	/* use old text if none given */
  
  if(*ibuf==0)
    {
    vmode_clearlines(STATUS_LINE_YPOS-1,479);
    vmode_drawtext(STATUS_LINE_XPOS,STATUS_LINE_YPOS,0,
    	"No previous search text  (press RETURN)");
    vmode_refresh();
    getkey();
    return;
    }
  }

/* if there's any uppercase letters in it, *don't* ignore case. */
ignore_case=1;
for(f=0;f<strlen(ibuf);f++)
  if(isupper(ibuf[f])) { ignore_case=0; break; }

found=0;
top+=dir;	/* don't search current line */
while(top>=0 && top<numlines)
  {
  if(ignore_case)
    ret=my_strcasestr(lines+lineofs[top],ibuf);
  else
    ret=strstr(lines+lineofs[top],ibuf);
  
  if(ret!=NULL)
    {
    *top_ptr=top;
    found=1;
    break;
    }
  
  top+=dir;
  }

if(!found)
  {
  vmode_clearlines(STATUS_LINE_YPOS-1,479);
  vmode_drawtext(STATUS_LINE_XPOS,STATUS_LINE_YPOS,0,
  	"Pattern not found  (press RETURN)");
  vmode_refresh();
  getkey();
  }
}


int calc_lpp()
{
return((480-2*Y_MARGIN-STATUS_LINE_HEIGHT)/get_font_height());
}



void display_file(unsigned char *lines,int *lineofs,int numlines)
{
static int marks[1024];
static unsigned char searchtext[40];	/* small, but see do_search() */
int lines_per_page=0;
int top=0;
int quit=0;
int key,key2,tmp,f;
int prevtop=0;		/* for the '' command */

for(f=0;f<1024;f++) marks[f]=0;
marks['$']=numlines-1;
*searchtext=0;

vmode_start();

lines_per_page=calc_lpp();

while(!quit)
  {
  /* the page is always redrawn, since it's pretty quick
   * and the screen update itself is near-instant.
   */
  drawpage(top,lines,lineofs,numlines,0);
  
  key=getkey();
  
  switch(key)
    {
    case RK_CURSOR_UP: case 'k':
      top--; break;
    case RK_CURSOR_DOWN: case 'j': case RK_ENTER:
      top++; break;
    case RK_PAGE_UP: case 'b': case 'w': case 127:
      top-=lines_per_page; break;
    case RK_PAGE_DOWN: case ' ': case 'z':
      top+=lines_per_page; break;
    case RK_HOME: case '<': case 'g':
      prevtop=top;
      top=0; break;
    case RK_END: case '>': case 'G':
      prevtop=top;
      top=numlines-1; break;
    case 'm':
      drawpage(top,lines,lineofs,numlines,1);
      marks[getkey()]=top;
      break;
    case '\'':
      tmp=top;	/* save for next prevtop */
      drawpage(top,lines,lineofs,numlines,2);
      key2=getkey();
      if(key2=='\'')
        top=prevtop;
      else
        top=marks[key2];
      prevtop=tmp;
      break;
    case '/':
      prevtop=top;
      do_search(&top,searchtext, 1,lines,lineofs,numlines);
      break;
    case '?':
      prevtop=top;
      do_search(&top,searchtext,-1,lines,lineofs,numlines);
      break;
    case 'i':
      invert=!invert; break;
    case RK_F1:
      set_point_size(18); break;
    case RK_F2:
      set_point_size(14); break;
    case RK_F3:
      set_point_size(12); break;
    case RK_ESC: case 'q': case 'Q':
      quit=1; break;
    }
  
  lines_per_page=calc_lpp();
  
  if(top<0) top=0;
  
  if(top+lines_per_page>=numlines)
    {
    if(numlines<lines_per_page)
      top=0;
    else
      top=numlines-lines_per_page;
    }
  }

vga_setmode(TEXT);
}


/* check we're running on a console */
void check_vc()
{
struct stat sbuf;
int major,minor;
int fd;

/* see if terminal is a console */
fd=dup(2);
fstat(fd,&sbuf);
major=sbuf.st_rdev>>8;
minor=sbuf.st_rdev&0xff;
close(fd);
if(major==4 && minor<64) return;	/* if on a console, ok */

/* otherwise give up. I can't see any way to stop svgalib fscking with
 * stdin in this case, and at least this is better than waiting
 * for the man page input from a VC...!
 */

/* (Well, ok, I *could* read the input as root before doing vga_init.
 * But you could measure the size of the security hole that'd open
 * up in light-years. ;-) And messing about with seteuid or whatever
 * is just too easy to screw up. It's not worth the grief, all in all.)
 */

fprintf(stderr,"Sorry, you *must* run vman/vmanpg from a console.\n");
exit(1);
}


void parse_env()
{
char *var=getenv("VMANPG"),*ptr;

if(var==NULL) return;

for(ptr=var;*ptr;ptr++)
  {
  switch(*ptr)
    {
    case 'p':
      if(!set_point_size(atoi(ptr+1)))
        {
        fprintf(stderr,
        	"vman: unsupported point size in VMANPG, using default\n");
        sleep(1);
        }
      /* any digits will be skipped over, so no need to skip them here :-) */
      break;
    
    case 'i':
      invert=1;
      break;
    }
  }
}


int main(int argc,char *argv[])
{
unsigned char *lines;
int *lineofs;
int numlines;

check_vc();

vga_disabledriverreport();
vga_init();

parse_env();

numlines=grab_input(&lines,&lineofs);	/* allocates lines/lineofs */
fprintf(stderr,"\x1b[H\x1b[J");		/* clear text screen */
display_file(lines,lineofs,numlines);

free(lines);
free(lineofs);
exit(0);
}
