/* [jntex.c wk 07.09.94] A LaTeX preprocessor
 *	Copyright (c) 1994-95 by Werner Koch (dd9jn)
 */

#include <wk/tailor.h>
RCSID("$Id: jntex.c,v 1.2 1996/01/25 20:21:47 wernerk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <wk/file.h>
#include <wk/direc.h>
#include <wk/string.h>

/****** constants *********/
/* ISO 8859-1 */
#define c8Ae 0xc4
#define c8ae 0xe4
#define c8Oe 0xd6
#define c8oe 0xf6
#define c8Ue 0xdc
#define c8ue 0xfc
#define c8SZ 0xdf

#define c8agrave 0xe0
#define c8aacute 0xe1
#define c8acirc  0xe2
#define c8Agrave 0xc0
#define c8Aacute 0xc1
#define c8Acirc  0xc2
#define c8ograve 0xf2
#define c8oacute 0xf3
#define c8ocirc  0xf4
#define c8Ograve 0xd2
#define c8Oacute 0xd3
#define c8Ocirc  0xd4
#define c8ugrave 0xf9
#define c8uacute 0xfa
#define c8ucirc  0xfb
#define c8Ugrave 0xd9
#define c8Uacute 0xda
#define c8Ucirc  0xdb
#define c8egrave 0xe8
#define c8eacute 0xe9
#define c8ecirc  0xea
#define c8Egrave 0xc8
#define c8Eacute 0xc9
#define c8Ecirc  0xca
#define c8igrave 0xec
#define c8iacute 0xed
#define c8icirc  0xee
#define c8Igrave 0xcc
#define c8Iacute 0xcd
#define c8Icirc  0xce

#define c8para	 0xa7


/* Codepage 437/850 */
#define c4Ae 0x8e
#define c4ae 0x84
#define c4Oe 0x99
#define c4oe 0x94
#define c4Ue 0x9a
#define c4ue 0x81
#define c4SZ 0xe1

#if 0 /* babel german mode */
#define s_Ae   "\"A"
#define s_ae   "\"a"
#define s_Oe   "\"O"
#define s_oe   "\"o"
#define s_Ue   "\"U"
#define s_ue   "\"u"
#define s_SZ   "\"s"
#else /* plain TeX */
#define s_Ae   "\\\"{A}"
#define s_ae   "\\\"{a}"
#define s_Oe   "\\\"{O}"
#define s_oe   "\\\"{o}"
#define s_Ue   "\\\"{U}"
#define s_ue   "\\\"{u}"
#define s_SZ   "\\ss{}"
#endif


#define c_agrave    "\\`{a}"
#define c_aacute    "\\'{a}"
#define c_acirc     "\\^{a}"
#define c_Agrave    "\\`{A}"
#define c_Aacute    "\\'{A}"
#define c_Acirc     "\\^{A}"
#define c_ograve    "\\`{o}"
#define c_oacute    "\\'{o}"
#define c_ocirc     "\\^{o}"
#define c_Ograve    "\\`{O}"
#define c_Oacute    "\\'{O}"
#define c_Ocirc     "\\^{O}"
#define c_ugrave    "\\`{u}"
#define c_uacute    "\\'{u}"
#define c_ucirc     "\\^{u}"
#define c_Ugrave    "\\`{U}"
#define c_Uacute    "\\'{U}"
#define c_Ucirc     "\\^{U}"
#define c_egrave    "\\`{e}"
#define c_eacute    "\\'{e}"
#define c_ecirc     "\\^{e}"
#define c_Egrave    "\\`{E}"
#define c_Eacute    "\\'{E}"
#define c_Ecirc     "\\^{E}"
#define c_igrave    "\\`{i}"
#define c_iacute    "\\'{i}"
#define c_icirc     "\\^{i}"
#define c_Igrave    "\\`{I}"
#define c_Iacute    "\\'{I}"
#define c_Icirc     "\\^{I}"

#define c_para	   "\\S{}"

/******** typedefs ********/

/******* globals **********/
static struct {
	int tty;
	int quiet;
	int verbose;
	int noexec;
	int interactive;
	int all;
	const char *cd;
    } opt ;

static char *outname, *cwd;
static FILE *fpout;

/****** prototypes ********/
static void Process( const char *inname );
static int  LookIntoLog( const char *texfile, const char *patter );
/******* Functions ********/

const char *CopyRight( int level )
{
    const char *p;
    switch( level ) {
      case 10:
      case 0:	p = "jntex - v1.00b; "
		    "Copyright 1994-95 by Werner Koch (dd9jn)" ; break;
      case 13:	p = "jntex"; break;
      case 14:	p = "1.00b"; break;
      case 1:
      case 11:	p = "Usage: jntex [options] [inputfiles](-h for help)";
		break;
      case 2:
      case 12:	p =
    "\nSyntax: jntex [options] [inputfiles]\n"
    "LaTeX preprocessor (for Babel language germanb)\n"
    "Options summary:\n"
    " -C dir = use this diretory for all output\n"
    " -a = run as often as needed\n"
    " -i = interactive (no \\batchmode)\n"
    " -n = do not execute LaTeX\n"
    " -q = quiet mode\n"
    " -v = verbose output\n"
    " -h = help\n";
	break;
      default:	p = WklibCopyRight(level);
    }
    ShowCopyRight(level);
    return p;
}


int main( int argc, char **argv )
{
    char *s, *inname;
    int rc, filter, i;

    rc = 0;
    outname = cwd = NULL;
    if( ArgExpand( &argc, &argv, 4 | 1 ) )
	Error(4,GetStr(12));

    opt.cd = NULL;
    for( s=""; --argc && **++argv == '-' && *s != '-'; )
	for( s = *argv + 1 ; *s && *s != '-' ; s++ )
	    switch( *s ) {
	      case 'n': opt.noexec++; break;
	      case 'i': opt.interactive++; break;
	      case 'v': opt.verbose++; break;
	      case 'q': opt.quiet++; break;
	      case 'a': opt.all++; break;
	      case 'C':
		if( s[1] ) {
		    opt.cd = s+1; while( *s ) s++; s--;
		}
		else if( argc > 1 ) {
		    --argc; ++argv;
		    opt.cd = *argv;
		    s = "C"; /* faked */
		}
		break;
	      case 'h' :
	      case '?' : CopyRight(0) ; CopyRight(2); break;
	      default  : Error(3,GetStr(15),s );
	    }

    if( !opt.quiet ) {
	if( (opt.tty = IsTerminal(0)) )
	    setvbuf(stdout,NULL,_IONBF,0);
    }
    else
	opt.verbose = 0 ;

    filter = argc == 0;
    if( opt.cd )
	cwd = cwdname();
    if( filter )
	argc = 1;
    for(rc=0; !rc && argc; argc--, argv++ ) {
	inname = NULL;
	if( !filter ) {
	    s = extname(*argv);
	    if( !FileCmpName(s,".tex") )
		Error(2,"can't use an inputfile named '*.tex'");
	    else if( !*s ) {
		inname = xmalloc(strlen(*argv) + 5 );
		strcpy(inname, *argv);
		strcat(inname, ".doc");
	    }
	    else
		inname = xstrdup(*argv);
	    free(s);
	    /* outputfile ffnen */
	    if( opt.cd ) {
		if( opt.verbose )
		    Info("switching to '%s'", opt.cd);
		if( ChangeDirectory( opt.cd ) )
		    Error(1002,"chdir(\"%s\") failed", opt.cd );
	    }
	    s = xmalloc( strlen(inname) + 5 );
	    strcpy(s, inname);
	    FileExtAppend(s, "tex");
	    fpout= xfopen( s, "w" );
	    outname = s;
	}
	else {
	    if( opt.cd ) {
		if( opt.verbose )
		    Info("switching to '%s'", opt.cd);
		if( ChangeDirectory( opt.cd ) )
		    Error(1002,"chdir(\"%s\") failed", opt.cd );
	    }
	    fpout= opt.noexec? stdout : xfopen( outname=xstrdup("a.tex"),"w");
	}


	fprintf(fpout,"%%-- created by %s v%s(%s)%s\n",
		CopyRight(13), CopyRight(14), CopyRight(24), CopyRight(32) );
	if( !opt.interactive )
	    fputs("\\batchmode\n", fpout);

	/* wieder zum Source diretory */
	if( opt.cd && ChangeDirectory( cwd ) )
	    Error(1002,"chdir(\"%s\") failed", cwd );
	if( !opt.quiet )
	    Info("processing '%s'", inname? inname : "[stdin]");
	Process(inname);
	FREE(inname);
	if( fpout != stdout )
	    fclose(fpout);

	/* und wieder zum working directory */
	if( opt.cd && ChangeDirectory( opt.cd ) )
	    Error(1002,"chdir(\"%s\") failed", opt.cd );

	if( !opt.noexec ) {
	    s = xmalloc( 20 + strlen(outname) );
	    sprintf(s,"latex %s", outname);
	    if( opt.verbose )
		Info("Exec: %s", s );
	    rc = system(s);
	    if( !rc && opt.all ) {
		if( LookIntoLog(outname,
				"Rerun to get cross-references right") )
		    rc = system(s);
		if( !rc && LookIntoLog(outname,
				"Rerun to get cross-references right") ) {
		    Info("%s: must rerun to cross-reference", outname);
		    rc = 16;
		}
	    }
	    if( !rc && (i=LookIntoLog(outname,"\nOverfull \\")) )
		Info("%s: %d \"Overfull\" warning%s detected", outname,
					    i, i==1?"":"s");
	    if( opt.verbose && !rc )
		Info("Latex completed" );
	    if( rc )
		Info("Latex failed: rc=%d", rc );
	    free(s);
	}
	if( opt.cd ) {
	    if( opt.verbose )
		Info("switching back to '%s'", cwd);
	    if( ChangeDirectory( cwd ) )
		Error(1002,"chdir(\"%s\") failed", cwd );
	}
    }

    FREE(cwd);
    return rc? 1:0;
}



/****************
 * Einen File verarbeiten, wir befinden uns hier im Source Directory
 * falls inname == NULL, benutzen wir stdin
 */

static void
Process( const char *inname )
{
    static int sentinel;
    int c, cp437, verbatim, include, i, i2;
    int pos; /* used in verbatim sections */
    char *s;
    FILE *fpin;

    if( ++sentinel > 20 )
	Error(2,"\\input{} are nested too deep");
    fpin = inname? xfopen( inname, "r" ) : stdin;
    /* read the first 2 bytes */
    cp437 = 0;
    if( (c = getc(fpin)) == EOF )
	goto endoffile;
    putc( c, fpout );
    if( c == '%' ) {
	if( (c = getc(fpin)) == EOF )
	    goto endoffile;
	putc( c, fpout );
	if( c == '+' )
	    cp437++;
    }
    if( opt.verbose )
	Info("using charset %s", cp437?"ibm-cp437":"iso-8859-1");
    /* Wir versuchen hier \begin{verbatim} und
     * \end{verbatim} zu erkennen um dort die Umlaute
     * anders zu uebersetzen ( -> ae, etc), da wir noch keinen
     * 8 bit zeichesatz haben und listings sonst etwas seltsam aussehen
     * Erkennung funktioniert erst ab der 2. Zeile und nur wenn
     * die Strings am Anfang der Zeile stehen.
     * Deweiteren wird ein \input{filename} hier bereits durchgefhrt,
     * falls dieses mit .doc oder .inc endet und keinen absoluten Path angibt;
     * dies ist notwendig, damit dessen Inhalt auch hier geparsed wird.
     * Das \input{} darf muss nicht unbedingt am Zeilenanfang stehen
     */
    verbatim = i = include = i2 = pos = 0;
    s = NULL;
    while( (c= getc(fpin)) != EOF ) {
	if( verbatim ) {
	    if( c == "\n\\end{verbatim}"[i] ) {
		i++;
		if( i == 15 ) {
		    verbatim = 0;
		    i = 0;
		}
	    }
	    else
		i = c == '\n'? 1 : 0;

	    if( c == '\t' ) {
		int j, r = 8 - ( (pos<8)? pos:(pos-8) % 8 ) ;
		putc( ' ', fpout ); pos++ ;
		for( j = 1 ; j < r ; j++, pos++ )
		    putc( ' ', fpout );
		s = ""; /* inhibit other output */
	    }
	    else if( cp437 ) {
		switch(c & 0xff) {
		  case c4Ae: s = "Ae" ; break;
		  case c4ae: s = "ae" ; break;
		  case c4Oe: s = "Oe" ; break;
		  case c4oe: s = "oe" ; break;
		  case c4Ue: s = "Ue" ; break;
		  case c4ue: s = "ue" ; break;
		  case c4SZ: s = "sz" ; break;
		}
	    }
	    else {
		switch(c & 0xff) {
		  case c8Ae: s = "Ae" ; break;
		  case c8ae: s = "ae" ; break;
		  case c8Oe: s = "Oe" ; break;
		  case c8oe: s = "oe" ; break;
		  case c8Ue: s = "Ue" ; break;
		  case c8ue: s = "ue" ; break;
		  case c8SZ: s = "sz" ; break;
		  /* andere haben keinen Zweck */
		}
	    }
	    if( c == '\n' )
		pos = 0;
	    else
		pos += s ? strlen(s) : 1;
	}
	else if( include ) {
	    char buf[200]; /* wir indizieren hier wieder mit i2 */

	    if( c == /*{*/'}' ) {
		/* der filename ist in buf */
		buf[i2] = 0;
		include = i2 = 0;
		StripWSpaces(buf);
		if( *buf != '/' ) {
		    s = extname(buf);
		    if( !FileCmpName(s,".doc") || !FileCmpName(s,".inc") ) {
			FREE(s);
			fputs( "null}\n", fpout ); /* include "null.tex" */
			if( !opt.quiet )
			    Info("%*sincluding '%s'", sentinel*2,"",buf);
			Process(buf); /* source this file */
			continue;
		    }
		    FREE(s);
		}
		/* wollen wir doch nicht selbst machen */
		strcat(buf,/*{*/"}");
		s = buf;
	    }
	    else {
		if( i2 < DIM(buf)-2 ) /* wir brauchen noch ein Zeichen Reserve*/
		    buf[i2++] = c;
		continue; /* wir drfen noch nichts ausgeben */
	    }
	}
	else {
	    if( c == "\n\\begin{verbatim}"[i] ) {
		i++;
		if( i == 17 ) {
		    verbatim++;
		    i = i2 = pos = 0;
		}
	    }
	    else
		i = c == '\n'? 1 : 0;
	    if( c == "\\input{"[i2] ) {
		i2++;
		if( i2 == 7 ) {
		    include++;
		    i = i2 = 0;
		}
	    }
	    else
		i2 = c == '\\'? 1:0;

	    if( cp437 ) {
		switch(c & 0xff) {
		  case c4Ae: s = s_Ae ; break;
		  case c4ae: s = s_ae ; break;
		  case c4Oe: s = s_Oe ; break;
		  case c4oe: s = s_oe ; break;
		  case c4Ue: s = s_Ue ; break;
		  case c4ue: s = s_ue ; break;
		  case c4SZ: s = s_SZ ; break;
		}
	    }
	    else {
		switch(c & 0xff) {
		  case c8Ae: s = s_Ae ; break;
		  case c8ae: s = s_ae ; break;
		  case c8Oe: s = s_Oe ; break;
		  case c8oe: s = s_oe ; break;
		  case c8Ue: s = s_Ue ; break;
		  case c8ue: s = s_ue ; break;
		  case c8SZ: s = s_SZ ; break;

		  case c8agrave  : s = c_agrave  ; break;
		  case c8aacute  : s = c_aacute  ; break;
		  case c8acirc	 : s = c_acirc	 ; break;
		  case c8Agrave  : s = c_Agrave  ; break;
		  case c8Aacute  : s = c_Aacute  ; break;
		  case c8Acirc	 : s = c_Acirc	 ; break;
		  case c8ograve  : s = c_ograve  ; break;
		  case c8oacute  : s = c_oacute  ; break;
		  case c8ocirc	 : s = c_ocirc	 ; break;
		  case c8Ograve  : s = c_Ograve  ; break;
		  case c8Oacute  : s = c_Oacute  ; break;
		  case c8Ocirc	 : s = c_Ocirc	 ; break;
		  case c8ugrave  : s = c_ugrave  ; break;
		  case c8uacute  : s = c_uacute  ; break;
		  case c8ucirc	 : s = c_ucirc	 ; break;
		  case c8Ugrave  : s = c_Ugrave  ; break;
		  case c8Uacute  : s = c_Uacute  ; break;
		  case c8Ucirc	 : s = c_Ucirc	 ; break;
		  case c8egrave  : s = c_egrave  ; break;
		  case c8eacute  : s = c_eacute  ; break;
		  case c8ecirc	 : s = c_ecirc	 ; break;
		  case c8Egrave  : s = c_Egrave  ; break;
		  case c8Eacute  : s = c_Eacute  ; break;
		  case c8Ecirc	 : s = c_Ecirc	 ; break;
		  case c8igrave  : s = c_igrave  ; break;
		  case c8iacute  : s = c_iacute  ; break;
		  case c8icirc	 : s = c_icirc	 ; break;
		  case c8Igrave  : s = c_Igrave  ; break;
		  case c8Iacute  : s = c_Iacute  ; break;
		  case c8Icirc	 : s = c_Icirc	 ; break;

		  case c8para	 : s = c_para	 ; break;
		}
	    }
	}
	if( s ) {
	    fputs( s, fpout );
	    s = NULL;
	}
	else
	    putc( c, fpout );
    }
  endoffile:
    if( ferror(fpin) )
	Error(2,"read error");
    if( ferror(fpout) )
	Error(2,"write error");
    if( fpin != stdin )
	fclose(fpin);
    sentinel--;
}

static int
LookIntoLog( const char *texfile, const char *pattern )
{
    int c, rc, i, patternlen;
    char *s;
    FILE *fp;

    s = xmalloc( strlen(texfile) + 5 );
    strcpy(s, texfile);
    FileExtAppend(s, "log");
    fp = xfopen( s, "r" );

    patternlen = strlen(pattern);
    i = rc = 0;
    while( (c= getc(fp)) != EOF ) {
	if( c == pattern[i] ) {
	    i++;
	    if( i == patternlen ) {
		i = 0;
		rc++;
	    }
	}
	else
	    i = c == *pattern? 1 : 0;
    }
    if( ferror(fp) )
	Error(2,"read error");
    fclose(fp);
    free(s);
    return rc;
}


/*** bottom of file ***/
