/* [wcmd.c wk 2.7.91] W-Editor Commands
 *	Copyright (c) 1991 by Werner Koch (dd9jn)
 * This file is part of the W-Editor.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * History:
 * 03.01.93 wk	removed unused vars and other cleanups
 * 11.06.93 wk	added option 'l' to locate command syntax checking
 * 19.08.94 wk	comments may now also start with a hash
 * 25.08.94 wk	sigint now trapped in loops
 */

#include "wtailor.h"

#define DBG_PARSING 0
#define DBG_EXEC    0
#define DBG_RECORDER 0

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <wk/lib.h>
#include <wk/file.h>
#include <wk/string.h>
#include <wk/mouse.h>

#if __ZTC__ && 0
  #include <page.h>
  #define USE_ZORTECH_PAGE 1
#endif

#if __ZTC__ >= 0x0300 && ( __COMPACT__ || __LARGE__ || __VCM__ )
  /* there is a serious bug in the assembly implementation  */
  /* of strtoul() - only the offset of p will be set; so we */
  /* have to correct this problem ourself. */
  #include <dos.h>
  #define PATCH_STRTOUL_BUG 1
#endif

#define MODULE_WCMD 1

#include "w.h"
#include "wscreen.h"
#include "wcmd.h"
#include "wfile.h"

#include "wcmdtbl.h"
#include "wkbddrv.h"

/****** constants *********/
#define MAX_NESTING 10
#define MAX_MACROLEN 5000  /* max. laenge eine Macros in einem profile */
#define MAX_MACOUTLINE 70  /* max- laenge einer Ausgabezeile bei .keydefs */
			   /* dies ist nur eine ca. Angabe, da ein Kommando */
			   /* incl. Argumente laenger sein kann */

#define MAX_RECORDS  500


#define CMDTBLHASH_SIZE 0x80 /* 128 entries */
#define CMDTBLHASH_MASK 0x7f


static const char symTerminators[] = " \t\'\"];01234567890/!:^~" ;
static const char symTerminatorsNoDig[] = " \t\'\"];/!:^~" ;
static const char whiteSpaces[] = " \t\n" ;

#if USE_ZORTECH_PAGE
    #define SIZEOF_PAGEBLOCK4CMDS 10000
#endif


/****************
 * This is a table containing the symbolic names of the keys.
 * The Table is ordered in the sequence of the namecodes
 */

struct {
	const char *name;
	const int   keyCode;
    } keyNameTbl[] = {
	{   "space"     , ' '           },
	{   "zero"      , '0'           },
	{   "one"       , '1'           },
	{   "two"       , '2'           },
	{   "three"     , '3'           },
	{   "four"      , '4'           },
	{   "five"      , '5'           },
	{   "six"       , '6'           },
	{   "seven"     , '7'           },
	{   "eight"     , '8'           },
	{   "nine"      , '9'           },
	{   "a"         , 'A'           },
	{   "b"         , 'B'           },
	{   "c"         , 'C'           },
	{   "d"         , 'D'           },
	{   "e"         , 'E'           },
	{   "f"         , 'F'           },
	{   "g"         , 'G'           },
	{   "h"         , 'H'           },
	{   "i"         , 'I'           },
	{   "j"         , 'J'           },
	{   "k"         , 'K'           },
	{   "l"         , 'L'           },
	{   "m"         , 'M'           },
	{   "n"         , 'N'           },
	{   "o"         , 'O'           },
	{   "p"         , 'P'           },
	{   "q"         , 'Q'           },
	{   "r"         , 'R'           },
	{   "s"         , 'S'           },
	{   "t"         , 'T'           },
	{   "u"         , 'U'           },
	{   "v"         , 'V'           },
	{   "w"         , 'W'           },
	{   "x"         , 'X'           },
	{   "y"         , 'Y'           },
	{   "z"         , 'Z'           },
	{   "lclick"    , K_BTN_LEFT    },
	{   "ldclick"   , K_BTN_LEFTDBL },
	{   "rclick"    , K_BTN_RIGHT   },
	{   "rdclick"   , K_BTN_RIGHTDBL},
	{   "del"       , K_DEL         },
	{   "down"      , K_DOWN        },
	{   "end"       , K_END         },
	{   "f1"        , K_F1          },
	{   "f10"       , K_F10         },
	{   "f11"       , K_F11         },
	{   "f12"       , K_F12         },
	{   "f2"        , K_F2          },
	{   "f3"        , K_F3          },
	{   "f4"        , K_F4          },
	{   "f5"        , K_F5          },
	{   "f6"        , K_F6          },
	{   "f7"        , K_F7          },
	{   "f8"        , K_F8          },
	{   "f9"        , K_F9          },
	{   "home"      , K_HOME        },
	{   "ins"       , K_INS         },
	{   "left"      , K_LEFT        },
	{   "pad1"      , K_PAD_1        },
	{   "pad2"      , K_PAD_2        },
	{   "pad3"      , K_PAD_3        },
	{   "pad4"      , K_PAD_4        },
	{   "pad5"      , K_PAD_5        },
	{   "pad6"      , K_PAD_6        },
	{   "pad7"      , K_PAD_7        },
	{   "pad8"      , K_PAD_8        },
	{   "pad9"      , K_PAD_9        },
	{   "paddel"    , K_PAD_DEL     },
	{   "paddiv"    , K_PAD_DIV     },
	{   "padins"    , K_PAD_INS      },
	{   "padminus"  , K_PAD_MINUS   },
	{   "padmul"    , K_PAD_MUL     },
	{   "padplus"   , K_PAD_PLUS    },
	{   "padenter"  , K_PAD_RETURN  },
	{   "pgdn"      , K_PGDN        },
	{   "pgup"      , K_PGUP        },
	{   "right"     , K_RIGHT       },
	{   "up"        , K_UP          },
	{   "mf1l"      , K_USER_FNC0 }, /* special mouse function 1 for left click */
	{   "mf1r"      , K_USER_FNC1 }, /* special mouse function 1 for right click */
	{   "mf2l"      , K_USER_FNC2 }, /* special mouse function 2 for left click */
	{   "mf2r"      , K_USER_FNC3 }, /* special mouse function 2 for right click */
	{   "mf3l"      , K_USER_FNC4 }, /* special mouse function 3 for left click */
	{   "mf3r"      , K_USER_FNC5 }, /* special mouse function 3 for right click */
	{   "mf4l"      , K_USER_FNC6 }, /* special mouse function 4 for left click */
	{   "mf4r"      , K_USER_FNC7 }, /* special mouse function 4 for right click */
	{   "esc"       , K_VK_ESCAPE   },
	{   "enter"     , K_VK_RETURN      },
	{   "backspace" , K_VK_RUBOUT      },
	{   "tab"       , K_VK_TAB         },
	{ NULL }
    };

/******* typedefs ********/
typedef struct {
	int   keyId;		/* 0 = unused */
	cmd_t *cmd;		/* pointer to bounded commands */
    } cmdTbl_t;

typedef struct s_cmdTblLink {	/* used as hash table entry */
	struct s_cmdTblLink *next;
	int   keyId;
	cmd_t *cmd;
    } *t_cmdTblLink;


typedef struct {
	char *name;		/* malloced name of function, NULL = unused */
	cmd_t *cmd;		/* pointer to bounded commands */
    } fncTbl_t;

/******* globals **********/
static int isInitialized;
static byte cmdLookUp['Z'-'A'+1];

#if USE_ZORTECH_PAGE
static void *pageBlock4Cmds;
#endif
static cmdTbl_t *cmdTbl;
static size_t  cmdTblSize;
static cmd_t *asciiCmdTbl; /* default commands */

static fncTbl_t *fncTbl;    /* Malloced Array of NULLs or Ptrs to malloced */
			    /* cmd.cmd is: index of Array + CMDFNC_BASE */
static size_t  fncTblSize;  /* strings, fncTblSize gives the Arraysize */


static t_cmdTblLink unusedCmdTblLinks;
static t_cmdTblLink cmdTblHashArray[CMDTBLHASH_SIZE];
static int cmdTblHashValid;



static int    *recordArray; /* records the keyIds */
static size_t  recordArrayIdx;
static int     recorderActive;
static cmd_t  *recordDefCmd;

static int lastCmdError;
static int *adrOfCurNoErrorVar;

static char *leapTarget;

/******* prototypes *******/
static void CleanUp( void *dummy );
static int String2KeyId( const char *string, size_t *scnlen );
static int String2CmdFnc( const char *string, size_t *scnlen );
static int IsValidFncName( const char *s, size_t len );
static char *KeyId2String( int keyId, char *buffer, size_t buflen );
static int ParseCmd( const char *line, cmd_t **retcmd, const char **endp );
static void DeleteBinding( int keyId );
static void DeleteFunction( int cmdId );
static void DeleteAllBindings(void);
static cmd_t *KeyId2Cmd( int keyId, int keyValue );
static void ClearCommandHashTable(void);
static void PutCmd2CmdTbl( int keyId, cmd_t *cmd );
static cmd_t *PrintCmdLine( cmd_t * cmd, char *buffer, size_t n );
static int DoExecCmd( cmd_t *, int *, int );
static cmd_t *AllocCmd( size_t stringLen );
static void FreeCmd( cmd_t *cmd );
static cmd_t *CopyCmd( cmd_t *cmd );
/******* functions ********/

/****************
 * Initialisieren des SubSystems
 * mode = 1 : translate codes to ISO 8859-1
 */

void InitCmdTables( int mode )
{
    int i, last;
    const char *p;

    if( isInitialized )
	return;
    KybrdDrvOpen( mode );
#if USE_ZORTECH_PAGE
    pageBlock4Cmds = xmalloc( SIZEOF_PAGEBLOCK4CMDS );
    page_initialize( pageBlock4Cmds, SIZEOF_PAGEBLOCK4CMDS );
#endif
    /* build the look up table */
    for( i=0; i < DIM(cmdLookUp) ; i++ )
	cmdLookUp[i] = 0;  /* default to start of command */
    for(last=' ', i=0; p = command[i].strg ; i++ ) {
	command[i].strgLen = (byte)strlen(p);
	if( *p != last && *p >= 'a' && *p <= 'z' ) {
	    cmdLookUp[ *p - 'a' ] = (byte)i;
	    last = *p;
	}
    }

    asciiCmdTbl = xcalloc( 256, sizeof *asciiCmdTbl );
    cmdTblSize = 100;
    cmdTbl = xcalloc( cmdTblSize , sizeof *cmdTbl );
    fncTblSize = 10;
    fncTbl = xcalloc( fncTblSize , sizeof *fncTbl );
    cmdTblHashValid = 0;
    AddCleanUp( CleanUp, NULL );
    isInitialized++;
    for(i=0; i < 256; i++ ) {
	asciiCmdTbl[i].cmd = CMD_PUT;
	asciiCmdTbl[i].type= ARGTYPE_ASCII;
	asciiCmdTbl[i].arg.keyId = i;
    }
}


static void
CleanUp( void *dummy )
{
    size_t n;
    t_cmdTblLink r, r2;

    if( !isInitialized )
	return;
    for(n=0; n < fncTblSize ; n++ ) {
      #if DBG_EXEC | DBG_PARSING
	printf( "%3u. Function '%s'  ", n, fncTbl[n].name );
	if( fncTbl[n].cmd ) {
	    cmd_t *x;

	    x = fncTbl[n].cmd;
	    printf("code %d", x->cmd );
	    for(x = x->nxtCmd; x ; x = x->nxtCmd )
		printf(", %d", x->cmd );
	    putchar('\n');
	}
	else
	    puts("undefined");
      #endif
	if( fncTbl[n].name ) {
	    DeleteFunction(CMDFNC_BASE + n);
	    free( fncTbl[n].name );
	}
    }
    free( fncTbl );
    DeleteAllBindings();

    ClearCommandHashTable();
    for(r=unusedCmdTblLinks; r; r = r2 ) {
	r2 = r->next;
	free(r);
    }
    unusedCmdTblLinks = NULL;

    free(cmdTbl);
    free(asciiCmdTbl);
    FreeCmd( recordDefCmd );
    free( recordArray );
#if USE_ZORTECH_PAGE
    free( pageBlock4Cmds );
#endif
    isInitialized = 0;
}



/****************
 * Die Funktion gibt zu einem Cmd-Code den String zurueck
 * mode: 0 = den langen Namen zurueckgeben
 *	 1 = den abgekuerzten Namen zurueckgeben
 * Returns: String oder NULL wenn nicht bekannt.
 */

const char *Cmd2String( int cmd, int mode )
{
    const char *p;
    byte last_len;
    int i, last_i;

    if( cmd >= CMDFNC_BASE ) {
	cmd -= CMDFNC_BASE;
	if( cmd < fncTblSize )
	    if( p = fncTbl[cmd].name )
		return p;
	return NULL ; /* can't be */
    }


    last_len = mode ? 100:0;
    last_i = -1;
    for(i=0; command[i].strg ; i++ )
	if( cmd == command[i].cmd ) {
	    if( !mode ) { /* search for long name */
		if( command[i].strgLen > last_len )
		    last_len = command[last_i=i].strgLen;
	    }
	    else {  /* search for short name */
		if( command[i].strgLen < last_len )
		    last_len = command[last_i=i].strgLen;
	    }
	}
    return last_i >= 0 ? command[last_i].strg : NULL;
}



/****************
 * Sucht in der Tabelle der Komamndos nach dem angegebenen Kommando
 * Der String darf nur das Kommando ohne blanks etc enthalten.
 * Returns:  0 nicht gefunden
 *	     commandcode
 */

int String2Cmd( const char *string )
{
    const char *p;
    int i;

    for(i=0; p = command[i].strg ; i++ )
	if( !strcmpl( p, string ) )
	    return command[i].cmd;
    return 0;
}


/****************
 * Einen String der eine Keynamen beschreibt in eine entsprechende keyId
 * umwandeln. Der String wird beendet, durch stringende Zeichen, blank, ']'
 * oder '='.  Ist dieser Keyname nicht bekannt, so wird 0 zurueckgegeben.
 * Returns: KeyId oder 0 wenn unkueltige Key
 */

static int String2KeyId( const char *string, size_t *scnlen )
{
    size_t len;
    int keyId, keyMod, i;
    const char *p;

    keyId = keyMod = 0;
    if( *scnlen = len = strcspn( string, " \t]=;'\"" ) ) {
	if( string[1] == '-' ) {
	    keyMod = string[0];
	    string += 2;
	    len -= 2; /* len was >= 2 */
	}
	if( len ) {
	    for(i=0; p = keyNameTbl[i].name; i++ )
		if( len == strlen(p) )
		    if( !memcmp( string, p, len ) ) {
			keyId = keyNameTbl[i].keyCode;
			break;
		    }
	    keyId &= KEYCOD_MASK;
	    if( keyId ) {
		switch( keyMod ) {
		  case 0: break;
		  case 's': case 'S': keyId |= KEYMOD_SHIFT ; break;
		  case 'c': case 'C': keyId |= KEYMOD_CTRL  ; break;
		  case 'a': case 'A': keyId |= KEYMOD_ALT   ; break;
		  case 'u': case 'U': keyId |= KEYMOD_USR   ; break;
		  case 'x': case 'X': keyId |= KEYMOD_EXT   ; break;
		  case 'd': case 'D': keyId |= KEYMOD_SHIFT|KEYMOD_CTRL ;break;
		  case 'b': case 'B': keyId |= KEYMOD_SHIFT|KEYMOD_ALT  ;break;
		  default: keyId = 0; break; /* invalid modifier */
		}
	    }
	}
    }

    return keyId;
}


/****************
 * Eine String in als Functionname definieren, falls eine Function
 * dieses Namens noch nicht existier, so wird sie neu angelegt.
 * Returns: 0 = Invalid Functionname
 *	    CMDFNC_BASE + n = id of cmd
 */

static int String2CmdFnc( const char *string, size_t *scnlen )
{
    size_t len;
    int i, cmd;
    char *q;
    fncTbl_t *tbl;


    cmd = 0;
    if( *scnlen = len = strcspn( string, " \t]=;'\"" ) ) {
	for(i=0; i < fncTblSize; i++ )
	    if( q = fncTbl[i].name )
		if( len == strlen(q) )
		    if( !memicmp( string, q, len ) ) {
			cmd = CMDFNC_BASE + i;
			break;
		    }
	if( !cmd ) {
	    /* not yet found, find a empty slot and */
	    /* allocate a new Name */
	    if( IsValidFncName( string, len ) )
		for(i=0; i < fncTblSize; i++ )
		    if( !fncTbl[i].name ) {
			fncTbl[i].name = xcalloc(1,len+1);
			memcpy(fncTbl[i].name,string,len);
			fncTbl[i].cmd = NULL ; /* not yet defined */
			cmd = CMDFNC_BASE + i;
			break;
		    }
	}
	if( !cmd && fncTblSize <= KEYCOD_MASK ) {
	    if( IsValidFncName( string, len ) ) {
		/* no free slot - extend the Table */
		if( !(fncTbl = realloc( fncTbl, (fncTblSize + 50)
						* sizeof *fncTbl )) )
		    Error(4,"Not enough memory"
			    " to extend function table");
		tbl = fncTbl+fncTblSize;
		tbl->name = xcalloc(1,len+1);
		memcpy(tbl->name,string,len);
		tbl->cmd = NULL;
		cmd = CMDFNC_BASE + fncTblSize;
		fncTblSize += 50;
		/* realloc does only a malloc, */
		/* so we have to reset the memory */
		for(i=1,tbl++; i < 50; i++,tbl++ )
		    tbl->name = NULL;
	    }
	}
    }

    return cmd;
}



static int IsValidFncName( const char *s, size_t len )
{
    if( *s && len ) {
	if( *s == '_' || isalpha(*s) ) {
	    while( s++, --len )
		if( !(isalnum( *s ) || *s == '_') )
		    return 0;
	    return 1;
	}
    }
    return 0;
}


/****************
 * Eine KeyId in eine String umwandeln
 * ist der key nicht bekannt, so wird [n-]? zurueckgegeben.
 * Buffer sollte mindestens 3 Zeichen lang sein.
 */

static char *
KeyId2String( int keyId, char *buffer, size_t buflen )
{
    size_t len;
    const char *p;
    int i;

    xassert( buflen >= 3 );

    if( keyId & KEYMOD_SHIFT )
	*buffer = keyId & KEYMOD_CTRL ? 'd':
		  keyId & KEYMOD_ALT  ? 'b':'s';
    else if( keyId & KEYMOD_CTRL  )
	*buffer = 'c';
    else if( keyId & KEYMOD_ALT   )
	*buffer = 'a';
    else if( keyId & KEYMOD_USR   )
	*buffer = 'u';
    else if( keyId & KEYMOD_EXT   )
	*buffer = 'x';
    else
	*buffer = 0;
    if( *buffer ) {
	buffer[1] = '-';
	len = 2;
    }
    else
	len = 0;


    keyId &= KEYCOD_MASK;

    for(i=0; p = keyNameTbl[i].name; i++ )
	if( keyNameTbl[i].keyCode == keyId )
	    break;
    if( p )
	mem2str( buffer+len, p, buflen-len );
    else {
	buffer[len] = (byte)'?' ;
	buffer[len+1] = 0;
    }

    return buffer;
}



/****************
 * Diese Funktion analysiert eine Kommando, dabei werden die Argumente
 * mit analysiert.
 * Das Kommando wird beendet, entweder durch das Stringende, ein ';' ,
 * ']' oder implizit.
 * Bei einem Fehler enthaelt cmd nicht alle Werte.
 * Returns: ErrorCode
 */

static int ParseCmd( const char *line, cmd_t **retcmd, const char **endp )
{
    int err, i, args, c;
    size_t len, len1, len2, usedLen;
    const char *p, *parg, *orgline;
    char *sarg;
    cmd_t cmd;

    err = 0;
    *retcmd = NULL;
    cmd.cmd = 0;
    orgline = line;
    line += strspn( line, whiteSpaces ); /* skip Blanks */
    if( isdigit( *line ) )
	err = ERR_INVCMD;
    else {
	/* now where is a digit or a string */
	for(p=line; args = *p && *p != ']' && *p != ';'; p++ )
	    if( isdigit( *p ) || *p == '\"' || *p == '\'' || *p == '%' ||
				 *p == ':' || *p == '^' || *p == '~' )
		break;
	parg = p;
	for( p--; p >= line; p-- )  /* remove blanks at end */
	    if( !isspace( *p ) )
		break;

	c = tolower( *line );
	if( c == '\'' || c == '\"' ) { /* issue pseudoCmd */
	    cmd.cmd = CMD_PUT;
	    cmd.type = ARGTYPE_STRING;
	}
	else if( c == '%' ) { /* issue pseudoCmd */
	    cmd.cmd = CMD_PUT;
	    cmd.type = ARGTYPE_VAR;
	}
	else if( c == ':' ) { /* issue pseudoCmd */
	    cmd.cmd = CMD_ENTER;
	    line++;
	    cmd.type = *line == '%'? ARGTYPE_VAR : ARGTYPE_STRING;
	}
	else if( c == '^' ) { /* issue pseudoCmd */
	    cmd.cmd = CMD_CALC;
	    line++;
	    cmd.type = ARGTYPE_OP;
	}
	else if( c == '~' ) { /* issue pseudoCmd */
	    cmd.cmd = CMD_ASSIGN;
	    line++;
	    cmd.type = ARGTYPE_VAR;
	    cmd.type = *line == '%'? ARGTYPE_VAR : ARGTYPE_NO;
	}
	else if( !(len = p - line+1 ) )
	    err = ERR_INVCMD;
	else { /* so look for the Command code */
	    /* laenge der ersten Woerter bestimmen */
	    len1 = strcspn( line, symTerminators );
	    len2 = len1+strspn( line+len1, whiteSpaces );
	    len2 += strcspn( line+len2, symTerminators );
	    i = (c >= 'a' && c <= 'z') ? cmdLookUp[c-'a'] : 0;
	    for(; p = command[i].strg ; i++ ) {
		switch( command[i].flag ) {
		  case 1 : usedLen = len1; break;
		  case 2 : usedLen = len2; break;
		  default: usedLen = len;  break;
		}
		if( command[i].strgLen == usedLen ) {
		    if( !memicmp( p, line, usedLen ) ) {
			cmd.cmd = command[i].cmd;
			cmd.type = command[i].argtype;
			line += usedLen;
			line += strspn( line, whiteSpaces );
			break;
		    }
		}
	    }

	    if( !p ) {
		if( *line == '/' ) {
		    cmd.cmd = CMD_LOCATE;
		    cmd.type = ARGTYPE_LOC;
		}
		else {	/* is it a function ? */
		    len = strcspn( line, symTerminatorsNoDig );
		    for(i=0; i < fncTblSize; i++ )
			if( p = fncTbl[i].name )
			    if( len == strlen(p) )
				if( !memicmp( line, p, len ) ) {
				    cmd.cmd = CMDFNC_BASE + i;
				    cmd.type = ARGTYPE_NO;
				    line += len;
				    line += strspn( line, whiteSpaces );
				    break;
				}
		    if( !p ) {
			/* noch ein paar test fuer falls : */
			/* "c/", "change/", "l/" etc angegeben wurde */
			if( !memcmp( line, "change", 6 ) ) {
			    cmd.cmd = CMD_CHANGE;
			    cmd.type = ARGTYPE_CHG;
			    line+=6;
			}
			else if( !memcmp( line, "locate", 6 ) ) {
			    cmd.cmd = CMD_LOCATE;
			    cmd.type = ARGTYPE_LOC;
			    line+=6;
			}
			else if( *line == 'c' )  {
			    cmd.cmd = CMD_CHANGE;
			    cmd.type = ARGTYPE_CHG;
			    line++;
			}
			else if( *line == 'l' ) {
			    cmd.cmd = CMD_LOCATE;
			    cmd.type = ARGTYPE_LOC;
			    line+=6;
			}
			else
			    err = ERR_INVCMD;
		    }
		}
	    }
	}

	if( !err ) {
	    size_t xlen;
	    int flg;

	    if( cmd.type == ARGTYPE_FILEORVAR )
		cmd.type = *line == '%' ? ARGTYPE_VAR : ARGTYPE_FILE;
	    else if( cmd.type == ARGTYPE_STRORVAR )
		cmd.type = *line == '%' ? ARGTYPE_VAR : ARGTYPE_STRING;

	    switch( cmd.type ) {
	      case ARGTYPE_NO :
		break;

	      case ARGTYPE_CHG:
	      case ARGTYPE_LOC:
		line += strspn( line, whiteSpaces );
		for(len=0, p=line+1; *p != *line && *p ; p++ )
		    len++;
		if( *p && cmd.type == ARGTYPE_CHG)
		    for( p++, len++; *p != *line && *p ; p++ )
			len++;
		if( !*p && !(cmd.type == ARGTYPE_LOC && *line == '/') )
		    err = ERR_LOCCMD;
		else {
		    len+=2; /* we need the delimiters */
		    if( *p ) {
			flg = 0;
			p++;
			if( xlen = strcspn( p, symTerminators ) ) {
			    for(flg=0; xlen; p++, xlen--, len++) {
				if( !(flg&1) && (*p == 'm' || *p == 'M') )
				    flg |=1;
				else if( !(flg&16) && (*p == 'l' || *p == 'L'))
				    flg |=16;
				else if( !(flg&32) && (*p == 'r' || *p == 'R'))
				    flg |=32;
				else if( !(flg&2) && *p == '-' )
				    flg |=2;
				else if( !(flg&4) &&
					 (*p == '*' && cmd.type==ARGTYPE_CHG))
				    flg |=4;
				else if( !(flg&8) && (*p == 'e' || *p == 'E'))
				    flg |=8;
				else if( !(flg&8) && (*p == 'a' || *p == 'A'))
				    flg |=8;  /* any and exact are exclusiv, */
				else	      /* so we can use the same bit */
				    break;
			    }
			}
			flg = 0;
		    }
		    else
			flg=1;

		    if( !(*retcmd = AllocCmd( len )) )
			err = ERR_NOMEM;
		    else {
			memcpy( *retcmd, &cmd, sizeof( cmd_t ) );
			/* add the string */
			sarg = (*retcmd)->arg.string;
			if( flg )
			    len--;
			for( ;len ; len-- )
			    *sarg++ = *line++;
			if( flg ) /* shortform, we have to add the slash */
			    *sarg++ = '/';
			*sarg = '\0';
		    }
		}
		break;

	      case ARGTYPE_VAR:
		/* look for the length of the var */
		for(len=0, p=++line; *p && (isalnum(*p) || *p == '_'); p++ )
		    len++;
		if( *p == '%' ) {
		    cmd.type = ARGTYPE_VARENV;
		    xlen = 1 ;
		}
		else
		    xlen = 0;
		if( !(*retcmd = AllocCmd( len )) )
		    err = ERR_NOMEM;
		else {
		    memcpy( *retcmd, &cmd, sizeof( cmd_t ) );
		    /* add the string */
		    sarg = (*retcmd)->arg.string;
		    for( ;len ; len-- )
			*sarg++ = *line++;
		    *sarg = '\0';
		    line += xlen;
		}
		break;

	      case ARGTYPE_FILE:
	      case ARGTYPE_STRING:  /* defaults to an empty string */
		/* look for the length of the string */
		if( *line == '\"' || *line == '\'' ) {
		    for(len=0, p=line+1; *p != *line && *p ; p++ )
			len++;
		    line++;
		    xlen=1;
		}
		else if( cmd.type == ARGTYPE_FILE ) {
		    for(len=0, p=line; *p && *p != ']' && *p != ';'; p++ )
			len++;
		    xlen=0;
		}
		else {
		    len = strcspn( line, symTerminatorsNoDig );
		    xlen=0;
		}
		if( !(*retcmd = AllocCmd( len )) )
		    err = ERR_NOMEM;
		else {
		    memcpy( *retcmd, &cmd, sizeof( cmd_t ) );
		    /* add the string */
		    sarg = (*retcmd)->arg.string;
		    for( ;len ; len-- )
			*sarg++ = *line++;
		    *sarg = '\0';
		    line += xlen;
		}
		break;

	      case ARGTYPE_NUMBER:
		cmd.arg.number = strtoul(line, (char**)&p, 0 );
	      #if PATCH_STRTOUL_BUG
		p = MK_FP( FP_SEG(line), FP_OFF(p) );
	      #endif
		line = p;
		break;

	      case ARGTYPE_ONOFF:
		len = strcspn( line, symTerminators );
		if( len == 3 && !memcmp( line, "off", 3 ) )
		    cmd.arg.number = 0;
		else if( len == 2 && !memcmp( line, "on", 2 ) )
		    cmd.arg.number = 1;
		else
		    err = ERR_SETCMD; /* no defaults allowed */
		line += len;
		break;

	      case ARGTYPE_ANYEXA:
		len = strcspn( line, symTerminators );
		if( len == 3 && !memcmp( line, "any", 3 ) )
		    cmd.arg.number = 0;
		else if( len == 5 && !memcmp( line, "exact", 5 ) )
		    cmd.arg.number = 1;
		else
		    err = ERR_SETCMD; /* no defaults allowed */
		line += len;
		break;


	      case ARGTYPE_OP:
		len = strcspn( line, symTerminators );
		if( !len )
		    cmd.arg.number = OPERATOR_NOP;
		else {
		    for(i=0; p = operatorTbl[i].name ; i++ )
			if( len == operatorTbl[i].len )
			    if( !memcmp( line, p, len ) ) {
				cmd.arg.number = operatorTbl[i].code;
				break;
			    }
		    if( !p )
			err = ERR_INVOP; /* no defaults allowed */
		}
		line += len;
		break;

	      case ARGTYPE_KEY:
		cmd.arg.keyId = String2KeyId( line, &xlen );
		line += xlen;
		if( !cmd.arg.keyId )
		    err = ERR_INVKEY;
		break;
	      case ARGTYPE_KEYDEF:
		if( !(cmd.arg.keyId = String2KeyId( line, &xlen )) )
		    if( cmd.arg.keyId = String2CmdFnc( line, &xlen ) )
			cmd.type = ARGTYPE_FNCDEF;
		line += xlen;
		if( !cmd.arg.keyId )
		    err = ERR_INVNAME;
		break;

	      default: BUG();
	    }
	}
    }

    if( !err ) {
	if( !*retcmd ) {
	    if( !(*retcmd = AllocCmd( 0 )) )
		err = ERR_NOMEM;
	    else
		memcpy( *retcmd, &cmd, sizeof( cmd_t ) );
	}
    }

    *endp = line;
    return err;
}



/****************
 * Eine CmdFile lesen ( pro-file ) und die alten cmds durch diese dort ersetzen
 * wird ein leerer string angegeben,so werden alle Cmds geloescht und die
 * interne (default) Table aktiviert. wird NULL uebergeben, so werden nur alle
 * cmds geloescht aber keine aktiviert - fuer CleanUp.
 * Ist modeFlag =1, so wird im FehlerFall die fehlerhafte Zeile dort angezeigt.
 * Returns: LineNumber
 */

int ReadCmdFile( const char *filename, int modeFlag )
{
    int err,i, nodefault, more, rewound;
    cmd_t *cmd;
    FILE *st;
    char *buffer;
    size_t n, nbytes, nread, helppos;
    long lineNr;
    char helpBuf[20];

    nread=0; /* controlled by <more> (only to avoid compiler warning)*/
    err=0;
    if( !filename ) {
	nodefault = 1;
	filename = "";
    }
    else
	nodefault = 0;
    if( !*filename ) { /* delete all and set to internal defaults */
	DeleteAllBindings();
	if( !nodefault )
	    for(i=0; defaultMacros[i]; i++ ) {
		if( ParseCmdLine( defaultMacros[i], &cmd ) )
		    Bug("Invalid internal command: %s", defaultMacros[i] );
		if( cmd )
		    ExecTempCmd( cmd );
	    }
    }
    else {  /* read from Profile */
      #if W_FULL
	lineNr = 0;
	FREE(leapTarget);
	rewound = 0;
	if( st = fsopen( filename, "r" ) ) {
	    helppos = 0;
	    buffer = xmalloc( MAX_MACROLEN+1 );
	    more = 0;
	    while( !err ) {
		if( !more )
		    nread = 0;
		else if( nread + MAX_LINELEN >= MAX_MACROLEN ) {
		    err = ERR_MAC2LNG;
		    continue;
		}

		if( !(nbytes = FExtRead( buffer+nread, MAX_LINELEN, st,
			      F_NFILE_LF | F_NFILE_TAB, 0, &helppos ) ) ) {
		    if( leapTarget && !rewound ) {
			rewind(st);
			lineNr = 0;
			rewound++;
			continue;
		    }
		    else
			break; /* end of file */
		}
		lineNr++;
		if( buffer[nread] == '*' || buffer[nread] == '#' )
		    continue; /* comment line */
		/* convert line to C-String */
		for( n=nread; n < nread+nbytes-1; n++ )
		    if( !buffer[n] )
			buffer[n] = ' ';
		buffer[n] = '\0';
		StripWSpaces(buffer);
		n = strlen(buffer);
		if( leapTarget ) {
		    if( n && buffer[n-1]==':' ) {
			buffer[n-1] = 0;
			if( !stricmp(buffer,leapTarget) ) {
			    FREE(leapTarget);
			    rewound = 0;
			}
		    }
		    continue;
		}
		more = 0;
		if( n && buffer[n-1] == '\\' ) {
			buffer[n-1] = ';';
			more = 1;
			nread = n;
		    }
		if( !more ) {
		    /* process line */
		    if( err = ParseCmdLine( buffer, &cmd ) ) {
			if( n && buffer[n-1]==':'
			      && !strpbrk(buffer," ;[]\t\n") ) {
			    err = 0; /* looks like a leap target */
			}
			else {
			    if( modeFlag ) {
				sprintf( helpBuf, "line %ld : ", lineNr );
				StrNShift( buffer, strlen(helpBuf),
						    MAX_MACROLEN-1);
				memcpy(buffer, helpBuf, strlen(helpBuf) );
				Write2CurCmdLine( buffer );
			    }
			    break;
			}
		    }
		    else if( cmd ) {
			if( err = ExecTempCmd( cmd ) )
			    break;
		    }
		}
		if( SigIntPoll() )
		    err = ERR_CMDINT;
	    }
	    free( buffer );
	    fclose( st );
	    if( !err && leapTarget )
		err = ERR_NOLABEL;
	    FREE(leapTarget);
	}
	else
      #endif
	    err = ERR_NOPRO;	/* profile not found */
    }
    return err;
}



/****************
 * Loeschen der Commands zu einer bestimmten KeyId
 */

static void DeleteBinding( int keyId )
{
    cmd_t *cmd, *ncmd;
    size_t n;

    if( !keyId )
	return;
    for(n=0; n < cmdTblSize; n++ )
	if( cmdTbl[n].keyId == keyId ) {
	    for( cmd = cmdTbl[n].cmd; cmd ; cmd = ncmd ) {
		ncmd = cmd->nxtCmd;
		FreeCmd( cmd );
	    }
	    cmdTbl[n].keyId = 0;
	    cmdTbl[n].cmd = NULL;
	    cmdTblHashValid = 0;
	    return;
	}
}


/****************
 * Dei Function loeschen, der Name bleibt allerdings
 * definiert.
 */

static void DeleteFunction( int id )
{
    cmd_t *cmd, *ncmd;


    xassert( id >= CMDFNC_BASE && id <= CMDFNC_BASE + fncTblSize );
    id -= CMDFNC_BASE;
    xassert( fncTbl[id].name ) ;

    for( cmd = fncTbl[id].cmd; cmd ; cmd = ncmd ) {
	ncmd = cmd->nxtCmd;
	FreeCmd( cmd );
    }
    fncTbl[id].cmd = NULL;
}



/****************
 *  Loeschen aller Bindings
 */

static void DeleteAllBindings()
{
    cmd_t *cmd, *ncmd;
    size_t n;

    for(n=0; n < cmdTblSize; n++ )
	if( cmdTbl[n].keyId ) {
	    for( cmd = cmdTbl[n].cmd; cmd ; cmd = ncmd ) {
		ncmd = cmd->nxtCmd;
		FreeCmd( cmd );
	    }
	    cmdTbl[n].keyId = 0;
	    cmdTbl[n].cmd = NULL;
	}
    cmdTblHashValid = 0;
}


/****************
 * Eine Zeile von Kommandos untersuchen.
 * Faengt die Zeile mit einem CMD_DEFINE an, so wird falls kein Fehler
 * aufgetreten, diese Commandsequence in der KeyTable gespeichert.
 * Ist retcmd != NULL, so zeigt es auf ein direkt ausfuehrbars Kommando.
 * andernfalls handlet es sich um ein keydef kommando.
 * Returns: ErrorCode.
 */

int ParseCmdLine( const char *cmdline, cmd_t **retcmd )
{
    cmd_t *cmd, *nxtcmd, *firstcmd;
    int keyId , type;
    int err;
    const char *p, *endp;

    err = 0;
    firstcmd = cmd = nxtcmd = NULL;
    p = cmdline;
  #if DBG_PARSING
    printf("Parsing: (%s) ", cmdline );
  #endif
    while( p += strspn( p, whiteSpaces ), !err && *p ) {
	if( *p == '[' )
	    p++;
	else if( *p == ';' )
	    p++;
	if( !(err = ParseCmd( p, &nxtcmd, &endp )) ) {
	    p = endp;
	    p += strspn( p, whiteSpaces );
	    if( nxtcmd->cmd == CMD_DEFINE && *p == '=' )
		p++;
	    else if( *p == ']' || *p == ';' )
		p++;
	    nxtcmd->nxtCmd = NULL;
	    if( !firstcmd )
		firstcmd = cmd = nxtcmd;
	    else {
		cmd->nxtCmd = nxtcmd;
		cmd = nxtcmd;
		nxtcmd = NULL;
	    }
	}
    }

    if( !err ) {
	if( !firstcmd )
	    *retcmd = NULL;
	else if( firstcmd->cmd == CMD_DEFINE ) {
	    cmd = firstcmd->nxtCmd ;
	    keyId = firstcmd->arg.keyId;
	    type  = firstcmd->type;
	    FreeCmd( firstcmd ); /* we dont need the definition any more */
	    firstcmd = cmd;   /* this is the new one */
	    if( type == ARGTYPE_FNCDEF ) {
	      #if DBG_PARSING
		printf("fncdef ");
	      #endif
		/* keyid is here the cmdId !*/
		xassert( keyId >= CMDFNC_BASE &&
			 keyId <  CMDFNC_BASE + fncTblSize);
		DeleteFunction( keyId );
		if( firstcmd )
		    fncTbl[keyId-CMDFNC_BASE].cmd = firstcmd;
	    }
	    else {
	      #if DBG_PARSING
		printf("keydef ");
	      #endif
		DeleteBinding( keyId );        /* delete old one */
		if( firstcmd )	/* if not: it will work as an undefine */
		    PutCmd2CmdTbl( keyId, firstcmd );
	    }
	    *retcmd = NULL;
	}
	else
	    *retcmd = firstcmd;

      #if DBG_PARSING
	if( firstcmd ) {
	    cmd_t *x;
	    printf("--> %d", firstcmd->cmd );
	    for(x = firstcmd->nxtCmd; x ; x = x->nxtCmd )
		printf(", %d", x->cmd );
	    putchar('\n');
	}
	else
	    printf("--> null\n");
      #endif
    }
    else {
      #if DBG_PARSING
	printf("--> failed\n");
      #endif
	/* cleanup here */
	FreeCmd( nxtcmd );
	for( cmd = firstcmd ; cmd ; cmd = nxtcmd ) {
	    nxtcmd = cmd->nxtCmd;
	    FreeCmd( cmd );
	}
    }
    return err;

}



/****************
 * Fuer einen KeyId einen Pointer auf das erste Command zurueckliefern
 * oder NULL falls kein Command vorhanden
 * Returns: Pointer to cmd
 */

static cmd_t *
KeyId2Cmd( int keyId, int keyValue )
{
    size_t n;

    if( !keyId )
	return NULL;

    if( cmdTblHashValid ) {
	t_cmdTblLink r;

	for( r = cmdTblHashArray[keyId & CMDTBLHASH_MASK]; r ; r = r->next )
	    if( r->keyId == keyId )
		return r->cmd;
    }
    else {
	for(n=0; n < cmdTblSize; n++ )
	    if( cmdTbl[n].keyId == keyId )
		return cmdTbl[n].cmd;
    }

    /* use the keyValue */
    if( keyValue > 0 && keyValue < 256 )
	return &asciiCmdTbl[keyValue];

    /* no valid key value, so try to break down the keyid */
    if( (keyId & KEYCOD_MASK) < 256 )
	return &asciiCmdTbl[keyId&0xff];
    return NULL;
}


/****************
 * Einen Pointer fuer ein Command in die CmdTable eintragen.
 */

static void
PutCmd2CmdTbl( int keyId, cmd_t *cmd )
{
    size_t n;
    cmdTbl_t *tbl;

    if( !keyId )
	return ; /* ist ja kennung fuer freien Eintrag */
    for(n=0; n < cmdTblSize; n++ )
	if( cmdTbl[n].keyId == keyId ) { /* bereits in Table - ersetzen */
	    cmdTbl[n].cmd = cmd;
	    return;
	}

    cmdTblHashValid = 0; /* invalidate hash table */
    for(n=0; n < cmdTblSize; n++ )
	if( !cmdTbl[n].keyId ) {  /* neuer Eintrag */
	    cmdTbl[n].keyId = keyId;
	    cmdTbl[n].cmd = cmd;
	    return;
	}
    /* command table is full */
    if( !(cmdTbl = realloc( cmdTbl, (cmdTblSize + 50) * sizeof *cmdTbl )) )
	Error(4,"Not enough memory to extend keydef table");
    tbl = cmdTbl+cmdTblSize;
    cmdTblSize += 50;
    tbl->keyId = keyId;
    tbl->cmd = cmd;
    /* realloc does only a malloc, so we have to reset the memory */
    for(n=1,tbl++; n < 50; n++,tbl++ ) {
	tbl->keyId = 0;
	tbl->cmd = NULL;
    }

}



static void
ClearCommandHashTable()
{
    t_cmdTblLink r, r2;
    int i;

    for(i=0; i < CMDTBLHASH_SIZE; i++ ) {
	for(r=cmdTblHashArray[i]; r; r = r2 ) {
	    r2 = r->next;
	    r->next = unusedCmdTblLinks;
	    unusedCmdTblLinks = r;
	}
	cmdTblHashArray[i] = NULL;
    }
    cmdTblHashValid = 0;
}


void
RehashCommandTable()
{
    t_cmdTblLink r;
    int i;
    size_t n;

    if( cmdTblHashValid )
	return ; /* still valid */

    ClearCommandHashTable();
    for(n=0; n < cmdTblSize; n++ )
	if( i = cmdTbl[n].keyId ) {
	    if( r = unusedCmdTblLinks )
		unusedCmdTblLinks = r->next;
	    else
		r = xmalloc( sizeof *r );
	    r->cmd = cmdTbl[n].cmd;
	    r->keyId = i;
	    i &= CMDTBLHASH_MASK;
	    r->next = cmdTblHashArray[i];
	    cmdTblHashArray[i] = r;
	    return;
	}
    cmdTblHashValid = 1;
}

/****************
 * Diese Funktion holt die naechsten verfuegbaren KeyId.
 * Ist keiner da, so wird 0 zurueckgegeben.
 */

int
GetKeyId(int *ret_value)
{
    int keyid;
    keyid = KybrdDrvGetKeyId(ret_value);
    if( keyid && recorderActive ) {
	if( recordArrayIdx < MAX_RECORDS )
	    recordArray[recordArrayIdx++] = keyid;
    }
    return keyid;
}


/****************
 * Returns a value suitable for Yes/No querie etc.
 */

int
GetKeyValue()
{
    int v;

    if( GetKeyId( &v ) )
	return v & 255;
    else
	return 0;
}


/****************
 * level 0 := get name of key
 *	 1 := check for definition and return string with pointer value
 *	      or empty string (uses another buffer );
 */

const char *
GetKeyName( int level, int keyid )
{
    static char buffer[40], buffer2[13];
    size_t n;

    if( level == 1 )  {
	*buffer2 = 0;
	/* fixme: we could you the hash table */
	if( keyid )
	    for(n=0; n < cmdTblSize; n++ )
		if( cmdTbl[n].keyId == keyid )
		    sprintf(buffer2, "#%u", n );
	return buffer2;
    }
    else
	KeyId2String( keyid, buffer, DIM(buffer)-1 );
    return buffer;
}



/****************
 * Die keydefs in den entsprechenden file schreiben
 */

#if W_FULL
int Cmd_Keydefs( cmd_t *dummycmd )
{
    int err=0, fhd,i;
    size_t n, alen;
    char *buffer;
    cmd_t *cmd;
    const char *s;

    if( !(err=Cmd_Edit(".keydefs")) ) {
	fhd = QryScreenFile();
	DeleteAllLines( fhd );
	buffer = xmalloc( MAX_LINELEN*2 );
	for(i=0; !err && (s=KybrdDrvEnumKeyDef(1,i)); i++ )  {
	    strcpy( buffer, "defkey " );
	    strcat(buffer, s);
	    err = InsertLine( fhd, buffer, strlen(buffer) );
	}
	for(n=0; n < cmdTblSize && !err ; n++ )
	    if( cmdTbl[n].keyId ) {
		cmd = cmdTbl[n].cmd;
		strcpy( buffer, "def " );
		KeyId2String( cmdTbl[n].keyId, buffer + 4, 40 );
		strcat(buffer, " = ");
		alen = strlen(buffer);
		if( cmd ) {
		    while( cmd = PrintCmdLine( cmd, buffer,strlen(buffer))) {
			strcat(buffer, " \\");
			if( err = InsertLine( fhd, buffer, strlen(buffer)) )
			    break;
			sprintf( buffer, "%*s", (int)alen, "" );
		    }
		}
		if( !err )
		    err = InsertLine( fhd, buffer, strlen(buffer) );
	    }
	for(n=0; n < fncTblSize && !err ; n++ )
	    if( fncTbl[n].name	) {
		cmd = fncTbl[n].cmd;
		strcpy( buffer, "def " );
		strcpy( buffer + 4, fncTbl[n].name );
		strcat(buffer, " = ");
		alen = strlen(buffer);
		if( cmd ) {
		    while( cmd = PrintCmdLine( cmd, buffer,strlen(buffer))) {
			strcat(buffer, " \\");
			if( err = InsertLine( fhd, buffer, strlen(buffer)) )
			    break;
			sprintf( buffer, "%*s", (int)alen, "" );
		    }
		}
		if( !err )
		    err = InsertLine( fhd, buffer, strlen(buffer) );
	    }
	free(buffer);
	SeekLine( fhd, 0 );
	if( GetFileTotLines( fhd) > 1 )
	    DeleteLine( fhd );
	Move2Pos( fhd, 0, 0 );
	ResetFileFlag( fhd, WFILE_CHG );
    }
    return err;
}
#endif

/****************
 * This function prints a keydefinition to a buffer and returns
 * a pointer to this new allocated buffer, caller must free the buffer !
 */
#if W_FULL
char *GetKeyDefString( int keyId )
{
    char * buffer;

    buffer = xmalloc(1000);
    strcpy( buffer, "def " );
    KeyId2String( keyId, buffer+4, 20 );
    strcat( buffer, " = " );

    if( PrintCmdLine( KeyId2Cmd(keyId,0), buffer, strlen(buffer) ) ) {
	memcpy( buffer, "???", 3 ); /* force error if used as def command */
	ShowMessageAsInfo("Truncated; use KEYDEF command");
    }
    return buffer;
}

/****************
 * Returns: NULL Zeile complett ausgegeben
 *	    cmd  cmd welches nicht mehr ausgegeben wurde
 */

static cmd_t *PrintCmdLine( cmd_t * cmd, char *buffer, size_t n )
{
    int abbrev, i, flag;
    char buf[40];


    abbrev = GetSetOption( SETOPT_ABBREV ) ? 1 : 0;
    for( ; cmd ; cmd = cmd->nxtCmd ) {
	flag = 0;
	if( cmd->cmd == CMD_PUT ) {
	    if( cmd->type == ARGTYPE_VAR )
		n += sprintf(buffer+n, "%%%s ", cmd->arg.string );
	    else if( cmd->type == ARGTYPE_VARENV )
		n += sprintf(buffer+n, "%%%s%% ", cmd->arg.string );
	    else
		n += sprintf(buffer+n, "'%s'", cmd->arg.string );
	    if( !abbrev ) {
		strcpy( buffer+n, " ");
		n++;
	    }
	}
	else {
	    if( cmd->cmd == CMD_CALC )
		n += sprintf( buffer+n, "^" );
	    else if( cmd->cmd == CMD_ASSIGN )
		n += sprintf( buffer+n, "~" );
	    else if( cmd->cmd == CMD_ENTER )
		n += sprintf( buffer+n, ":" );
	    else {
		n += sprintf( buffer+n, " %s", Cmd2String( cmd->cmd,abbrev ));
		flag++;
	    }
	    switch( cmd->type ) {
	      case ARGTYPE_NO:
		  if( flag ) {
		      strcpy( buffer+n, ";"); n++;
		  }
		  break;
	      case ARGTYPE_STRING:
	      case ARGTYPE_CHG:
	      case ARGTYPE_LOC:
		if( !flag ) {
		    if( strpbrk( cmd->arg.string, whiteSpaces ) )
			n += sprintf(buffer+n, "'%s'", cmd->arg.string );
		    else
			n += sprintf(buffer+n, "%s", cmd->arg.string );
		}
		else if( *cmd->arg.string ) {
		    n += sprintf(buffer+n, " '%s';", cmd->arg.string );
		}
		else {
		    strcpy( buffer+n, ";"); n++;
		}
		break;

	      case ARGTYPE_VAR:
		n += sprintf(buffer+n, "%%%s ", cmd->arg.string );
		break;

	      case ARGTYPE_VARENV:
		n += sprintf(buffer+n, "%%%s%% ", cmd->arg.string );
		break;

	      case ARGTYPE_FILE:
		if( *cmd->arg.string )
		    n += sprintf(buffer+n, " %s;", cmd->arg.string );
		else {
		    strcpy( buffer+n, ";"); n++;
		}
		break;
	      case ARGTYPE_NUMBER:
		if( cmd->arg.number )
		    n += sprintf(buffer+n, " %lu;", cmd->arg.number );
		else {
		    strcpy( buffer+n, ";"); n++;
		}
		break;
	      case ARGTYPE_ONOFF:
		n += sprintf( buffer+n, cmd->arg.number ? " on;" : " off;");
		break;
	      case ARGTYPE_ANYEXA:
		n += sprintf( buffer+n, cmd->arg.number ? " exact;" : " any;");
		break;
	      case ARGTYPE_KEYDEF:
		n += sprintf(buffer+n, " %s =",
			     KeyId2String(cmd->arg.keyId,buf,DIM(buf)));
		break;
	      case ARGTYPE_KEY:
		n += sprintf(buffer+n, " %s;",
			     KeyId2String(cmd->arg.keyId,buf,DIM(buf)));
		break;

	      case ARGTYPE_OP:
		for(i=0; operatorTbl[i].name; i++ )
		    if( operatorTbl[i].code == cmd->arg.number ) {
			n += sprintf(buffer+n, flag ? " %s;" : "%s ",
					       operatorTbl[i].name );
			break;
		    }
		break;
	    }
	    if( !abbrev && flag ) {
		strcpy( buffer+n, " ");
		n++;
	    }
	}

	if( n > MAX_MACOUTLINE ||
	    cmd->cmd == CMD_BEGIN ||
	    cmd->cmd == CMD_END   )
	    return cmd->nxtCmd;

    }
    return NULL;
}
#endif

/****************
 * Den letzten errorcode zurueckgeben, beim starten eines Macros wird dieser
 * immer zurueckgesetzt.
 */

int GetLastCmdError(void)
{
    return lastCmdError;
}


/****************
 * Set the target for the "leap to" command,
 * this is only used when processing a macro
 */

int
SetLeapTarget( const char *s )
{
    free(leapTarget);
    leapTarget = xstrdup(s);
    StripWSpaces(leapTarget);
    if( !*leapTarget ) {
	FREE(leapTarget);
	return ERR_INVLAB;
    }
    return 0;
}


/****************
 * Eine Kommandofolge verarbeiten ( macro im Profile )
 * cmd == NULL ist gueltig. und dann loeschen
 */

int
ExecTempCmd( cmd_t *cmd )
{
    int err;
    cmd_t *ncmd;

    err = ExecCmd( cmd );

    /* and free the Command */
    for(; cmd ; cmd = ncmd ) {
	ncmd = cmd->nxtCmd;
	FreeCmd( cmd );
    }
    return err;
}


/****************
 * Ein Kommandofolge zu einer KeyId ausfuehren.
 */

int ExecKeyCmd( int keyId, int keyValue )
{
    return ExecCmd( KeyId2Cmd( keyId, keyValue ) );
}


int ExecFunction( cmd_t *cmd )
{
    int id = cmd->cmd;

    if( id >= CMDFNC_BASE ) {
	id -= CMDFNC_BASE;
	if( id < fncTblSize )
	    if( fncTbl[id].cmd )
		return ExecCmd( fncTbl[id].cmd );
	    else
		return ERR_UDFFNC;  /* undefined function */
    }

    return ERR_NOTFNC; /* not a function */
}


/****************
 * Verarbeitungsschleife
 */

int ExecCmd( cmd_t *cmd )
{
    int err, noerr;

    lastCmdError = noerr = 0;
    adrOfCurNoErrorVar = &noerr; /* oh what a hack */

    err = DoExecCmd( cmd, &noerr, 0 );

    if( err == ERR_PSEUDO_END )
	err = 0;

    return err;
}


/****************
 * Nur genau ein Komando bzw. einen Block
 * ausfuehren. Dabei wird der Kontext des Aufrufers benutzt
 * Darf nicht direkt aufgerufen werden, sondern wird von Call,Jmp,While,Until
 * benutzt.
 */

int ExecOneCmdBlock( cmd_t *cmd )
{
    int err;
    err = DoExecCmd( cmd, adrOfCurNoErrorVar, 1 );
    if( err == ERR_PSEUDO_END )
	err = 0;
    return err;
}


/****************
 * Die Verarbeitungsschleife
 * mode =1 : Nur den naechsten Block/Kommando ausfuehren.
 */

static int DoExecCmd( cmd_t *cmd, int *noerr, int mode )
{
    static int instance;
    int err,i;
  #if DBG_EXEC
    cmd_t *x;
  #endif

    err = 0;
    instance++;
  #if DBG_EXEC
    printf("%*sDoExecCmd%s\n", (instance-1)*4,"", mode ? " for one block":"");
  #endif
    for(; cmd ; cmd = err||mode ? NULL : cmd->nxtCmd ) {
      #if DBG_EXEC
	printf("%*scmd code %d", (instance-1)*4,"", cmd->cmd );
	for(x = cmd->nxtCmd; x ; x = x->nxtCmd )
	    printf(", %d", x->cmd );
	putchar('\n');
      #endif
	if( SigIntPoll() )
	    err = ERR_CMDINT;
	else if( cmd->cmd == CMD_BEGIN ) {
	  #if DBG_EXEC
	    printf("%*sBEGIN encountered\n", (instance-1)*4,"");
	  #endif
	    if( instance < MAX_NESTING ) {
		err = DoExecCmd( cmd->nxtCmd, noerr, 0 );
		if( err == ERR_PSEUDO_END )
		    err = 0;
		if( !err ) {
		    /* find the corresponding end and set one ahead */
		  #if DBG_EXEC
		    printf("%*sLooking for End  ", (instance-1)*4,"" );
		    printf("code %d", cmd->cmd );
		    for(x = cmd->nxtCmd; x ; x = x->nxtCmd )
			printf(", %d", x->cmd );
		    putchar('\n');
		  #endif
		    for(err = ERR_MISEND,i=0; err && cmd; cmd = cmd->nxtCmd )
			if( cmd->cmd == CMD_BEGIN )
			    i++;
			else if( cmd->cmd == CMD_END )
			    if( !--i ) {
				err = 0;
				break;
			    }
		  #if DBG_EXEC
		    if( err )
			printf("%*s..... NOT FOUND  ", (instance-1)*4,"" );
		    else {
			printf("%*s..... found End  ", (instance-1)*4,"" );
			printf("code %d", cmd->cmd );
			for(x = cmd->nxtCmd; x ; x = x->nxtCmd )
			    printf(", %d", x->cmd );
		    }
		    putchar('\n');
		  #endif
		}
	    }
	    else
		err = ERR_BLKNEST;
	}
	else if( cmd->cmd == CMD_END ) {
	  #if DBG_EXEC
	    printf("%*sEND encountered\n", (instance-1)*4,"");
	  #endif
	    break;  /* for loop */
	}
	else if( err = ProcessCmd( cmd ) ) {
	  #if DBG_EXEC
	    printf("%*sProcessCmd returns %d\n", (instance-1)*4,"", err);
	  #endif
	    if( err == ERR_PSEUDO_SKIP ) {
		err = 0;
		if( cmd->nxtCmd ) {
		    cmd = cmd->nxtCmd;	/* skip at least the next command */
		    if( cmd->cmd == CMD_END ) /* remove possible pending end */
			cmd = cmd->nxtCmd;
		    if( cmd->nxtCmd ) {
			if( cmd->cmd == CMD_BEGIN ) { /* or a complete block */
			  #if DBG_EXEC
			    printf("%*sSkipping Block\n", (instance-1)*4,"" );
			    printf("code %d", cmd->cmd );
			    for(x = cmd->nxtCmd; x ; x = x->nxtCmd )
				printf(", %d", x->cmd );
			    putchar('\n');
			  #endif
			    for(err = ERR_MISEND,i=0; err && cmd;
							    cmd=cmd->nxtCmd)
				if( cmd->cmd == CMD_BEGIN )
				    i++;
				else if( cmd->cmd == CMD_END )
				    if( !--i ) {
					err = 0;
					break;
				    }
			  #if DBG_EXEC
			    if( err )
				printf("%*s.... END MISSING ",
							 (instance-1)*4,"" );
			    else {
				printf("%*s....... skipped  ",
							 (instance-1)*4,"" );
				printf("code %d", cmd->cmd );
				for(x = cmd->nxtCmd; x ; x = x->nxtCmd )
				    printf(", %d", x->cmd );
			    }
			    putchar('\n');
			  #endif
			}
		    }
		}
	    }
	    else if( err == ERR_PSEUDO_END )
		;
	    else if( err == ERR_PSEUDO_CE )
		err = lastCmdError = 0;
	    else if( err == ERR_PSEUDO_NE ) {
		err = 0;
		*noerr = 1;
	    }
	    else {
		lastCmdError = err;
		if( *noerr && err != ERR_CMDINT) /*error trapping is disabled*/
		    err = 0;
	    }
	}
      #if W_FULL
	else if( demoModeFlag )
	    DemoProcess();
      #endif

    }


  #if DBG_EXEC
    if( err )
	printf("%*serr= %d\n", (instance-1)*4,"", err);
    printf("%*sLeaving DoExecCmd\n", (instance-1)*4,"");
  #endif
    instance--;
    return err;
}


void PrefixNextGetKeyId( char pref )
{
    KybrdDrvSetPrefix(pref);
}


int Cmd_RecordStart( cmd_t *cmd )
{
    if( !recordArray ) {
	recordArray = malloc( MAX_RECORDS+1 * sizeof *recordArray );
	recordDefCmd = NULL;
	if( !recordArray )
	    return ERR_NOMEM;
    }
    Cmd_RecordCancel();
    recordArrayIdx = 0;
    if( !(recordDefCmd = CopyCmd( cmd )) )
	return ERR_NOMEM;
    recorderActive = 1;
    return 0;
}

int Cmd_RecordStop(void)
{
    cmd_t *cmd, *acmd, *firstcmd, *lastcmd;
    size_t n;
    static cmd_t nopCmd = { CMD_NOP, NULL, ARGTYPE_NO };
    int err=0;

    recorderActive = 0;
    firstcmd = NULL;
    lastcmd=0; /*avoid warning, controlled by <firstcmd> */
  #if DBG_RECORDER
    puts("Stopping recorder");
  #endif
    for(n=0; n < recordArrayIdx; n++ ) {
      #if DBG_RECORDER
	printf( "\tprocessing %d. entry\n", n);
      #endif
	if( !(cmd = KeyId2Cmd( recordArray[n], 0 )) )
	    cmd = &nopCmd;
	if( cmd->cmd == CMD_RECORD_STOP )
	    break;
	for( ; cmd ; cmd = cmd->nxtCmd ) {
	    if( !(acmd = CopyCmd( cmd )) ) {
		err = ERR_NOMEM;
		goto retLabel;
	    }
	    acmd->nxtCmd = NULL;
	    if( !firstcmd )
		firstcmd = acmd;
	    else
		lastcmd->nxtCmd = acmd;
	    lastcmd = acmd;
	}
    }

    if( recordDefCmd->type == ARGTYPE_FNCDEF ) {
      #if DBG_PARSING || DBG_RECORDER
	printf("recorded fncdef ");
      #endif
	/* keyid is here the cmdId !*/
	xassert( recordDefCmd->arg.keyId >= CMDFNC_BASE &&
		 recordDefCmd->arg.keyId <  CMDFNC_BASE + fncTblSize);
	DeleteFunction( recordDefCmd->arg.keyId );
	if( firstcmd )
	    fncTbl[recordDefCmd->arg.keyId-CMDFNC_BASE].cmd = firstcmd;
    }
    else {
      #if DBG_PARSING || DBG_RECORDER
	printf("recorded keydef ");
      #endif
	DeleteBinding( recordDefCmd->arg.keyId );   /* delete old one */
	if( firstcmd )	/* if not: it will work as an undefine */
	    PutCmd2CmdTbl( recordDefCmd->arg.keyId, firstcmd );
    }


  retLabel:
    if( err ) {
      #if DBG_RECORDER
	puts("error - cleaning up");
      #endif
	for( cmd = firstcmd; cmd ; cmd = acmd ) {
	    acmd = cmd->nxtCmd;
	    FreeCmd( cmd );
	}
    }
    FreeCmd(recordDefCmd);
    recordDefCmd = NULL;
  #if DBG_RECORDER
    puts("Recorder stopped");
  #endif
    return err;
}


int Cmd_RecordCancel(void)
{
    recorderActive = 0;
    FreeCmd(recordDefCmd);
    recordDefCmd = NULL;
    return 0;
}


/****************
 * private Cmd Allocation function
 * stringLen is the length of the String ( excluding the '\0' )
 * (works together with FreeCmd() )
 * Returns: NULL if out of memory
 */

static cmd_t *AllocCmd( size_t stringLen )
{
#if USE_ZORTECH_PAGE
    unsigned off;

    off = page_calloc( pageBlock4Cmds, sizeof( cmd_t )+stringLen );
    return off ? (page_toptr(pageBlock4Cmds, off)) : NULL;
#else
    return calloc( 1, sizeof( cmd_t )+stringLen );
#endif
}


/****************
 * private Cmd Free function
 * (works together with AllocCmd() )
 */

static void FreeCmd( cmd_t *cmd )
{
#if USE_ZORTECH_PAGE
    if( cmd )
	page_free( pageBlock4Cmds, (char*)cmd - (char*)pageBlock4Cmds );
#else
    free( cmd );
#endif
}

static cmd_t *CopyCmd( cmd_t *cmd )
{
    cmd_t *n;
    size_t stringLen;

    if( cmd->type == ARGTYPE_FILE ||
	cmd->type == ARGTYPE_STRING ||
	cmd->type == ARGTYPE_VAR    ||
	cmd->type == ARGTYPE_CHG    ||
	cmd->type == ARGTYPE_LOC    )
	stringLen = strlen(cmd->arg.string)+1;
    else
	stringLen = 0;
    if( n = AllocCmd( stringLen ) )
	memcpy( n, cmd, sizeof( cmd_t )+stringLen );
    return n;
}


#if TEST

void main( int argc, char **argv )
{
    cmd_t *cmd;
    const char *endp;
    int err;

    argv++;
    if( err = ReadCmdFile( "w.pro", -1 ) )
	printf( "Errorcode: %d\n", err );
    else {
	PrintBindings();
	puts("Switching to default:");
	if( err = ReadCmdFile( NULL, -1 ) )
	    printf( "Errorcode: %d\n", err );
	else
	    puts("Ready");
	PrintBindings();
    }
    exit(0);
}

#endif


/*** bottom of file ***/
