/* [wmark.c wk 6.9.91] W-Editor Mark Operations
 *	Copyright (c) 1991 by Werner Koch (dd9jn)
 * This file is part of the W-Editor.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * History:
 *  6.12.92 wk	new Command Cmd_StripBlanks()
 */

#include "wtailor.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <wk/lib.h>
#include <wk/string.h>
#if __ZTC__
  #include <sound.h>
#elif __WATCOMC__
  #include <i86.h>
#endif

#include "w.h"
#include "wcmd.h"
#include "wscreen.h"
#include "wfile.h"

/****** constants *********/
/******* typedefs ********/
/******* globals **********/
/******* prototypes *******/
static int GetALabelFromUser(void);
/******* functions ********/


/****************
 * Goto to mark
 * mode 0 = to start of mark
 *	1 = to end of mark
 */

int Cmd_GotoMark( int mode )
{
    int fileHd, srcfhd, err, type;
    ushort pos;
    ulong lineNr;
    const char *p;
    unsigned flags;

    err = 0;
    fileHd = QryScreenFile();
    if( type = MarkInfo( &srcfhd ) ) {
	p = GetFileInfo( srcfhd, &flags );
	if( srcfhd != fileHd )	/* switch to file */
	    err = Cmd_Edit( p );
	if( !err ) {
	    fileHd = srcfhd ;
	    xassert( fileHd == QryScreenFile() );
	    if( flags & WFILE_INCMD )
		ResetFileFlag( fileHd, WFILE_INCMD );
	    lineNr = mode ? MarkEnd( &pos ) :
			    MarkStart( &pos );
	    if( type == MARKTYPE_LINE )
		Move2Line( fileHd, lineNr );
	    else
		Move2Pos( fileHd, lineNr, pos );
	}
    }
    else
	err = ERR_NOMRK;

    return err;
}


/****************
 * Change case of mark
 * mode 0 = to lowercase
 *	1 = to uppercase
 */

int Cmd_ChangeCase( int mode )
{
    int fileHd, err, i, mtype;
    ushort pos1, pos2, pos, nbytes;
    ulong lineNr, line1, line2, lnr;
    unsigned flags;
    char *p;

    err = 0;
    mtype = MarkInfo( &fileHd );
    GetFileInfo( fileHd, &flags );
    if( mtype && (flags & WFILE_RO) ){
	err = ERR_EDTRO;
	goto retLabel;
    }
    switch( mtype ) {
      case MARKTYPE_LINE:
	lineNr = GetFilePos( fileHd, NULL );
	line1 = MarkStart( NULL );
	line2 = MarkEnd( NULL );
	for( lnr = line1; lnr <= line2 && !err; lnr++ ) {
	    SeekLine( fileHd, lnr );
	    p = GetEditBuf( fileHd, &nbytes );
	    for(i=0; i < nbytes; i++,p++ )
		*p = mode ? toupper(*p) : tolower(*p);
	    if( SigIntPoll() )
		err = ERR_CMDINT;
	}
	SeekLine( fileHd, lineNr );
	break;

      case MARKTYPE_BLOCK:
	lineNr = GetFilePos( fileHd, NULL );
	line1 = MarkStart( &pos1);
	line2 = MarkEnd( &pos2);
	xassert( pos1 < MAX_LINELEN && pos2 < MAX_LINELEN );
	pos = pos2 - pos1 + 1; /* used as length */
	for( lnr = line1; lnr <= line2 && !err; lnr++ ) {
	    SeekLine( fileHd, lnr );
	    p = GetEditBuf( fileHd, &nbytes );
	    for(i=0, p += pos1; i < pos; i++,p++ )
		*p = mode ? toupper(*p) : tolower(*p);
	    if( SigIntPoll() )
		err = ERR_CMDINT;
	}
	SeekLine( fileHd, lineNr );
	break;

       default: err = ERR_NOMRK; break;
    }

  retLabel:
    return err;
}



/****************
 * Innerhalb einer Linemark blanks am Ende entfernen
 */

int Cmd_StripBlanks(void)
{
    int fileHd, err;
    ushort nbytes;
    ulong lineNr, line1, line2, lnr;
    unsigned flags;
    char *p;

    err = 0;
    if( MarkInfo( &fileHd ) == MARKTYPE_LINE ) {
	GetFileInfo( fileHd, &flags );
	lineNr = GetFilePos( fileHd, NULL );
	line1 = MarkStart( NULL );
	line2 = MarkEnd( NULL );
	if( flags & WFILE_RO )
	    err = ERR_EDTRO;
	for( lnr = line1; lnr <= line2 && !err; lnr++ ) {
	    SeekLine( fileHd, lnr );
	    p = GetEditBuf( fileHd, &nbytes );
	    for( ; nbytes; nbytes-- )
		if( p[nbytes-1] != ' ' )
		    break;
	    SetEditBuf( fileHd, nbytes );
	    if( SigIntPoll() )
		err = ERR_CMDINT;
	}
	SeekLine( fileHd, lineNr );
    }
    else
	err = ERR_NOLMRK; /* we need a line mark */

    return err;
}




/****************
 * Copy the mark to the actual position
 */

int Cmd_CopyMark(void)
{
    int srcfhd, dstfhd, err, same, mtype, insNewLine, adjust;
    const char *txt;
    char *p, *buffer;
    ulong line1, line2, lnr, lnr2, lineNr, lines;
    ushort pos1, pos2, xlen, dstpos, nsrc, ndst, n;
    ulong srcLineNr;
    ushort srcPos;

    err = 0;
    buffer=NULL; /* avoid warning, is controlled by <same> */
    srcLineNr=0; /* avoid warning, is controlled by <mtype,...> */
    dstfhd = QryScreenFile();
    FlushEditBuf( dstfhd );
    lineNr = GetFilePos( dstfhd, &dstpos );
    mtype = MarkInfo( &srcfhd );
    if( mtype && srcfhd != dstfhd )
	srcLineNr = GetFilePos( srcfhd, &srcPos );
    switch( mtype ) {
      case MARKTYPE_LINE:
	line1 = MarkStart( NULL );
	line2 = MarkEnd( NULL );
	if( same = (srcfhd == dstfhd) )
	    if( !(  lineNr >= line2 || lineNr < line1 ) )
		err = ERR_STCNFL; /* source and target conflict */
	lines = 0;
	for( lnr = line1, lnr2 = lineNr; lnr <= line2 && !err; lnr++, lnr2++ ){
	    SeekLine( srcfhd, lnr+lines );
	    txt = GetPtr2Line( srcfhd, &nsrc );
	    SeekLine( dstfhd, lnr2);
	    err = InsertLine( dstfhd, txt, nsrc );
	    if( same && lineNr < line1 )
		lines++;
	    if( SigIntPoll() )
		err = ERR_CMDINT;
	}
	break;

      case MARKTYPE_BLOCK:
	line1 = MarkStart( &pos1);
	line2 = MarkEnd( &pos2);
	xlen = pos2 - pos1 + 1;
	same = line1 == lineNr && dstfhd == srcfhd;
	adjust = 0;
	if( srcfhd == dstfhd )
	    if( !(  lineNr > line2 ||
		    lineNr + (line2-line1) < line1 ||
		    dstpos > pos2 ||
		    (adjust = (same && dstpos <= pos1))
		 ) )
		err = ERR_STCNFL; /* source and target conflict */
	if( same )
	    buffer = xmalloc( MAX_LINELEN+1 );
	insNewLine = 0;
	for( lnr = line1, lnr2 = lineNr; lnr <= line2 && !err; lnr++, lnr2++) {
	    SeekLine( srcfhd, lnr );
	    txt = GetPtr2Line( srcfhd, &nsrc );
	    if( same ) {
		memcpy( buffer, txt, nsrc );
		txt = buffer;
	    }
	    if( insNewLine ) {
		FlushEditBuf( dstfhd );  /* insert a new line  when at end */
		SeekLine( dstfhd, lnr2 );
		if( err = InsertLine( dstfhd, "", 0 ) )
		    break;
	    }
	    else if( SeekLine( dstfhd, lnr2 ) )
		insNewLine = 1;

	    p = GetEditBuf( dstfhd, &ndst );
	    /* Fillup with blanks */
	    for(n=ndst; n < dstpos && n < MAX_LINELEN-1; n++ )
		p[n] = ' ';
	    if( dstpos < ndst) { /* shift to the right */
		if( dstpos+xlen < MAX_LINELEN )
		    if( n = MAX_LINELEN-dstpos-xlen ) {
			memmove( p+dstpos+xlen, p+dstpos, n );
			if( ndst+xlen <= MAX_LINELEN )
			    ndst += xlen;
			else
			    ndst = MAX_LINELEN;
			SetEditBuf( dstfhd, ndst );
		    }
	    }

	    /* insert the text */
	    for(n = dstpos;
		 n < dstpos+xlen && pos1 < nsrc && n < MAX_LINELEN-1;
							   n++, nsrc-- )
		p[n] = txt[pos1+n-dstpos];
	    for(; n < dstpos+xlen && n < MAX_LINELEN-1; n++ )
		p[n] = ' ';
	    if( n > ndst )
		SetEditBuf(dstfhd, n );
	    if( SigIntPoll() )
		err = ERR_CMDINT;
	}
	if( !err && adjust )
	    AdjustBlockMark( srcfhd, xlen );
	if( same )
	    free( buffer );
	break;

      default:
	err = ERR_NOMRK;
	break;
    }

    if( mtype && srcfhd != dstfhd )
	Move2Pos( srcfhd, srcLineNr, srcPos );
    Move2Pos( dstfhd, lineNr, dstpos );

    return err;
}



/****************
 * Shift the mark : Direction: 0 = to left, 1 = to right
 */

int Cmd_ShiftMark( int direction )
{
    int fhd, err, mtype, trunc;
    char *p;
    saveFilePos_t savePos;
    ulong lnr, line1, line2;
    ushort pos1, len, nbytes;

    err = 0;
    if( !(mtype = MarkInfo( &fhd )))
	err = ERR_NOMRK;
    else {
	FlushEditBuf( fhd );
	SaveFilePos( fhd, &savePos );
	line1 = MarkStart( &pos1);
	line2 = MarkEnd( NULL);
	if( mtype == MARKTYPE_LINE )
	    pos1= 0;
	trunc=0;
	for( lnr = line1; lnr <= line2 && !err; lnr++) {
	    SeekLine( fhd, lnr );
	    p = GetEditBuf( fhd, &nbytes );
	    if( pos1 >= nbytes ) {
		if( direction ) { /* if shift right, fill up with blanks */
		    while( nbytes <= pos1 && nbytes < MAX_LINELEN )
			p[nbytes++] = ' ';
		    SetEditBuf( fhd, nbytes );
		}
	    }
	    else {
		len = nbytes - pos1;
		if( direction ) { /* shift to the right */
		    if( nbytes < MAX_LINELEN-2 ) {
			memmove( p+pos1+1, p+pos1, len );
			p[pos1] = ' ';
			SetEditBuf( fhd, ++nbytes );
			if( nbytes == MAX_LINELEN-2 )
			    trunc=1;
		    }
		}
		else {	/* shift to left */
		    if( pos1+1 < nbytes )
			memmove( p+pos1, p+pos1+1, len );
		    SetEditBuf( fhd, nbytes-1 );
		}
	    }
	    if( SigIntPoll() )
		err = ERR_CMDINT;
	}
	RestFilePos( &savePos );
	if( !err && trunc )
	    ShowMessage( GetErrorString(ERR_LTRUNC) );
    }

    return err;
}


/****************
 * Replace a textarea with a block mark
 */
#if W_FULL
int Cmd_OverlayBlock( cmd_t *cmd )
{
    int srcfhd, dstfhd, err, eof;
    size_t dm1, dm2;
    ushort dm3, dm4;
    const char *text[1];
    ushort pos1, pos2, xlen, dstpos, nbytes, n, srcnbytes;
    static ushort lenTbl[1];
    ulong line1, line2, lnr, lnr2, lineNr;
    char *p;
    saveFilePos_t savePos;

    err = 0;
    if( MarkInfo( &srcfhd ) == MARKTYPE_BLOCK ) {
	dstfhd = QryScreenFile();
	SaveFilePos( dstfhd, &savePos );
	lineNr = GetFilePos( dstfhd, &dstpos );
	line1 = MarkStart( &pos1);
	line2 = MarkEnd( &pos2);
	xassert( pos1 < MAX_LINELEN && pos2 < MAX_LINELEN );
	xlen = pos2 - pos1 + 1;
	if( srcfhd == dstfhd )
	    if( !(dstpos > pos2 ||
		  dstpos + xlen -1 < pos1 ||
		  lineNr > line2 ||
		  lineNr + (line2 - line1) < line1) )
		err = ERR_STCNFL; /* source and target conflict */
	eof = 0;
	for( lnr = line1, lnr2 = lineNr; lnr <= line2 && !err; lnr++, lnr2++) {
	    GetLineBlock( srcfhd, lnr, text, lenTbl, 1,
			  &dm1, &dm3, &dm2, &dm4);
	    if( eof ) {
		FlushEditBuf( dstfhd );
		if( err = InsertLine( dstfhd, "", 0 ) )
		    break;
	    }
	    else
		eof = SeekLine( dstfhd, lnr2 );
	    p = GetEditBuf( dstfhd, &nbytes );
	    srcnbytes = *lenTbl;
	    if( pos1 < srcnbytes )
		srcnbytes -= pos1;
	    else
		srcnbytes = 0;
	    for(n=nbytes; n < dstpos && n < MAX_LINELEN-1; n++ )
		p[n] = ' ';
	    for(n=dstpos; n < dstpos+xlen && srcnbytes && n < MAX_LINELEN-1;
							n++, srcnbytes-- )
		p[n] = (*text)[pos1+n-dstpos];
	    for(; n < dstpos+xlen && n < MAX_LINELEN-1; n++ )
		p[n] = ' ';
	    if( n > nbytes )
		SetEditBuf(dstfhd, n );
	    if( SigIntPoll() )
		err = ERR_CMDINT;
	}
	RestFilePos( &savePos );
    }
    else
	err = ERR_NOBMRK;

    return err;
}
#endif

/****************
 * Delete the mark
 */

int Cmd_DeleteMark(void)
{
    int err, fhd, orgfhd;
    ushort nbytes, len, pos1, pos2, savePos, savePos2;
    char *p;
    ulong line1, line2, lnr, saveLine, saveLine2;

    err = 0;
    saveLine2 = 0; /* avoid warning, controled by fhd,orgfhd */
    orgfhd = fhd = QryScreenFile();
    saveLine = GetFilePos( fhd, &savePos );
    if( !(err=Cmd_GotoMark(0)) ) {
	fhd = QryScreenFile();
	saveLine2 = GetFilePos( fhd, &savePos2 );
	FlushEditBuf( fhd );
	switch( MarkInfo( NULL ) ) {
	  case MARKTYPE_LINE:
	    line1 = MarkStart( NULL );
	    line2 = MarkEnd( NULL );
	    SetMark( fhd, 0 ); /* clear mark */
	    SeekLine( fhd, line1 );
	    for( lnr = line1; lnr <= line2 && !err; lnr++ ) {
		err = DeleteLine( fhd );
		if( SigIntPoll() )
		    err = ERR_CMDINT;
	    }
	    if( !GetFileTotLines( fhd ) )
		InsertLine( fhd, "", 0 );
	    break;

	  case MARKTYPE_BLOCK:
	    line1 = MarkStart( &pos1);
	    line2 = MarkEnd( &pos2);
	    for( lnr = line1; lnr <= line2 && !err; lnr++) {
		SeekLine( fhd, lnr );
		p = GetEditBuf( fhd, &nbytes );
		if( pos1 >= nbytes )
		    continue;
		if( nbytes > pos2 ) {
		    len = pos2-pos1+1;
		    memmove( p+pos1, p+pos1+len, nbytes - pos2 - 1 );
		}
		else
		    len = nbytes - pos1;
		SetEditBuf( fhd, nbytes-len );
		if( SigIntPoll() )
		    err = ERR_CMDINT;
	    }
	    if( !err )	/* delete the mark */
		SetMark( fhd, 0 );

	    break;

	  default:
	    err = ERR_NOMRK;
	    break;
	}
    }

    if( orgfhd != fhd )
	Move2Pos( fhd, saveLine2, savePos2 );
    Move2Pos( orgfhd, saveLine, savePos );
    return err;
}



/****************
 * Fill the mark with character c
 */

int Cmd_FillMark( cmd_t * cmd, int c )
{
    int fhd, err;
    ushort pos1, pos2, pos, nbytes;
    ulong line1, line2, lnr;
    char *p;
    saveFilePos_t savePos;

    err = 0;
    fhd = QryScreenFile();
    SaveFilePos( fhd, &savePos );
    switch( MarkInfo( &fhd ) ) {
      case MARKTYPE_LINE:
	line1 = MarkStart( NULL );
	line2 = MarkEnd( NULL );
	for( lnr = line1; lnr <= line2 && !err; lnr++ ) {
	    SeekLine( fhd, lnr );
	    p = GetEditBuf( fhd, &nbytes );
	    memset( p, c, MAX_LINELEN-1 );
	    SetEditBuf(fhd, MAX_LINELEN-1);
	    if( SigIntPoll() )
		err = ERR_CMDINT;
	}
	break;

      case MARKTYPE_BLOCK:
	line1 = MarkStart( &pos1);
	line2 = MarkEnd( &pos2);
	xassert( pos1 < MAX_LINELEN && pos2 < MAX_LINELEN );
	pos = pos2 - pos1 + 1; /* used as length */
	for( lnr = line1; lnr <= line2 && !err; lnr++ ) {
	    SeekLine( fhd, lnr );
	    p = GetEditBuf( fhd, &nbytes );
	    if( pos1 > nbytes ) { /* Fillup with blanks */
		while( nbytes < pos1 )
		    p[nbytes++] = ' ';
		SetEditBuf( fhd, nbytes );
	    }
	    memset( p+pos1, c, pos );
	    if( pos2 >= nbytes )
		SetEditBuf(fhd, pos2+1 );
	    if( SigIntPoll() )
		err = ERR_CMDINT;
	}
	break;

      default:
	err = ERR_NOMRK;
	break;
    }
    RestFilePos( &savePos );
    return err;
}


/****************
 * An der aktuellen Position en Label setzen
 */

#if W_FULL
int Cmd_SetLabel( cmd_t *cmd )
{
    int err=0, label;

    if( !(label = *cmd->arg.string) )
	label = GetALabelFromUser();
    label = isalpha( label ) ? toupper( label ) : 0;
    if( !label )
	err = ERR_INVLAB;
    else
	err = FileSetLabel( label, QryScreenFile() );
    return err;
}




/****************
 * Zu dem entsprechenden Label gehen, oder falls nicht gesetzt
 * eine Fehlermeldung ausgeben.
 */

int Cmd_GotoLabel( cmd_t *cmd )
{
    int err=0;
    int fhd, label;
    unsigned flags;
    ulong lnr;
    ushort pos;

    if( !(label = *cmd->arg.string) )
	label = GetALabelFromUser();
    label = isalpha( label ) ? toupper( label ) : 0;
    if( !label )
	err = ERR_INVLAB;
    else if( !(err = FileGetLabel( label, &fhd, &lnr, &pos )) ) {
	if( fhd != QryScreenFile() )  /* switch to file */
	    err = Cmd_Edit( GetFileInfo( fhd, &flags ) );
	if( !err ) {
	    xassert( fhd == QryScreenFile() );
	    ResetFileFlag( fhd, WFILE_INCMD );
	    Move2Pos( fhd, lnr, pos );
	}
    }
    return err;
}




static int GetALabelFromUser()
{
    int c;

    ShowMessage( "Which label?" );
    while( !(c=GetKeyValue()) )
	;
    ShowMessage(NULL);
    return ( c<256 && isalpha(c) ) ? c : 0;
}


/****************
 * Play the mark
 */

int Cmd_PlayMark(void)
{
  #if __ZTC__  || (__WATCOMC__ && (MSDOS || DOS386))
    int fhd, err=0, mtype, freq, durat, n, i;
    ushort pos, posx, pos1, pos2, nbytes;
    ulong line1, line2, lnr;
    const char *str;
    saveFilePos_t savePos;
    char buf[10];

    fhd = QryScreenFile();
    if( mtype = MarkInfo( &fhd ) ) {
	SaveFilePos( fhd, &savePos );
	line1 = MarkStart( &pos1);
	line2 = MarkEnd( &pos2);
	if( mtype == MARKTYPE_LINE ) {
	    pos1 = 0;
	    pos2 = MAX_LINELEN-1;
	}
	for( lnr = line1; lnr <= line2 && !err; lnr++ ) {
	    SeekLine( fhd, lnr );
	    str = GetPtr2Line( fhd, &nbytes );
	    str += pos = pos1;
	    posx = pos2;
	    if( posx >= nbytes )
		posx = nbytes ? nbytes-1:0;
	    while( pos < posx && !err ) {
		for( n=0; n < 2; n++ ) {
		    for(; pos <= posx; pos++, str++ )
			if( isdigit( *str ) )
			    break;
		    for(i=0; pos <= posx &&
			     i < DIM(buf) && isdigit(*str); str++, pos++ )
			buf[i++] = *str;
		    buf[i] = '\0';
		    durat = atoi( buf );
		    if( !n )
			freq = durat;
		}
		if( freq < 0 | freq > 20000 )
		    freq = 440;
		if( durat > 5000 || durat < 0 )
		    durat = 500; /* default to 500ms on error */
		/* play that note */
	      #if __ZTC__
	       #if OS2
		sound_note( freq, durat );
	       #else
		freq = (int)((freq * 1331L)/1000L);
		sound_beep( freq );
		if( durat > 200 )
		    Sleep( durat - 200 );
	       #endif
	      #elif (__WATCOMC__ && (MSDOS || DOS386))
		sound( freq );
		for(; durat > 200; durat -= 200 )
		    delay( 200 );
		delay(durat);
		nosound();
	      #endif
		if( SigIntPoll() )
		    err = ERR_CMDINT;
	    }
	}
	RestFilePos( &savePos );
    }
    else
	err = ERR_NOMRK;
    return err;
  #else /* not ZTC */
    return ERR_USPCMD;
  #endif
}
#endif


/*** bottom of file ***/
