/* [wmove.c wk 16.10.91] W-Editor Move Cursor
 *	Copyright (c) 1991 by Werner Koch (dd9jn)
 * This file is part of the W-Editor.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Header: /usr/src/master/apps/w/src/wmove.c,v 1.4 1996/09/10 12:20:05 wk Exp $
 */

#include "wtailor.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <limits.h>
#include <wk/lib.h>

#include "w.h"
#include "wscreen.h"
#include "wfile.h"

/****** constants *********/
/******* typedefs ********/
/******* globals **********/
static int lastWasEof;

/******* prototypes *******/
static ulong GetSyncedFilePos( int fhd, ushort *filX, ulong *filY,
					ushort *winX, ulong *winY,
					ushort *crsX, ushort *crsY,
					   int *sx, int *sy );
/******* functions ********/

int GetLastEofState()
{
    return lastWasEof;
}


/****************
 * Local function: nearly the same as GetAllFilePos, but we will first
 * see that the actual line is visible and the Screen Size is returned,
 * where sy is coorected according to winY
 */

static ulong GetSyncedFilePos( int fhd, ushort *filX, ulong *filY,
					ushort *winX, ulong *winY,
					ushort *crsX, ushort *crsY,
					int *sx, int *sy	   )
{
    ulong nlines;

    lastWasEof = 0;
    GetScreenSize( sx, sy );
    nlines = GetAllFilePos( fhd, filX, filY, winX, winY, crsX, crsY );
    if( !*winY )
	--*sy;	 /* topline is visible */
    if( *filY < *winY || *filY >= *winY+*sy ) {
	Move2Line( fhd, *filY );
	GetScreenSize( sx, sy );
	nlines = GetAllFilePos( fhd, filX, filY, winX, winY, crsX, crsY );
	if( !*winY )
	    --*sy;   /* topline is visible */
	xassert( !(*filY < *winY || *filY >= *winY+*sy) );
    }
    return nlines;
}



void MoveDown( int fhd, ulong count )
{
    ulong  nlines, filY, winY;
    ushort filX, winX, crsY, crsX;
    int    sx, sy;

    nlines = GetSyncedFilePos( fhd, &filX, &filY, &winX, &winY,
				    &crsX, &crsY, &sx, &sy );
    for(;count; count-- ) {
	if( filY+1 >= nlines ) {  /* end of file */
	     lastWasEof = 1;
	     if( crsY == sy-1 ) { /* scroll one more down */
		winY++;
		crsY--;
	     }
	}
	else if( crsY < sy-1 ) {
	    crsY++;
	    filY++;
	}
	else {
	    winY++; /* scroll */
	    filY++;
	}
    }

    SeekAndPosLine( fhd, filX, filY );
    SetTextAreaPos( fhd, winX, winY, crsX, crsY );
}




void MoveUp( int fhd, ulong count )
{
    ulong  nlines, filY, winY;
    ushort filX, winX, crsY, crsX;
    int    sx, sy;

    nlines = GetSyncedFilePos( fhd, &filX, &filY, &winX, &winY,
				    &crsX, &crsY, &sx, &sy );

    for( ;count ;count-- ) {
	if( crsY ) {
	    crsY--;
	    filY--;
	}
	else if( winY ) {
	    winY--;
	    filY--;
	}
	else
	    break;
    }

    SeekAndPosLine( fhd, filX, filY );
    SetTextAreaPos( fhd, winX, winY, crsX, crsY );
}



void MoveDownScrl( int fhd, ulong count )
{
    ulong  nlines, filY, winY;
    ushort filX, winX, crsY, crsX;
    int    sx, sy;

    nlines = GetSyncedFilePos( fhd, &filX, &filY, &winX, &winY,
				    &crsX, &crsY, &sx, &sy );

    for(;count; count-- ) {
	if( filY+1 >= nlines ) {   /* end of file */
	    lastWasEof = 1;
	    break;
	}
	else if( winY+sy > nlines ) { /* bottominfoline is visible */
	    if( crsY < sy-1 ) {
		crsY++;
		filY++;
	    }
	}
	else { /* scroll */
	    filY++;
	    winY++;
	}
    }

    SeekAndPosLine( fhd, filX, filY );
    SetTextAreaPos( fhd, winX, winY, crsX, crsY );
}



void MoveUpScrl( int fhd, ulong count )
{
    ulong  nlines, filY, winY;
    ushort filX, winX, crsY, crsX;
    int    sx, sy;

    nlines = GetSyncedFilePos( fhd, &filX, &filY, &winX, &winY,
				    &crsX, &crsY, &sx, &sy );

    for( ;count ;count-- ) {
	if( !filY )
	    break;
	else if( winY ) {
	    winY--;
	    filY--;
	}
	else if( crsY ) {
	    crsY--;
	    filY--;
	}
	else
	    break;
    }

    SeekAndPosLine( fhd, filX, filY );
    SetTextAreaPos( fhd, winX, winY, crsX, crsY );
}



void MoveLeft( int fhd, ushort count )
{
    ulong  nlines, filY, winY;
    ushort filX, winX, crsY, crsX;
    int    sx, sy;

    nlines = GetSyncedFilePos( fhd, &filX, &filY, &winX, &winY,
				    &crsX, &crsY, &sx, &sy );

    for( ;count ;count-- ) {
	if( crsX  ) {
	    crsX--;
	    filX--;
	}
	else if( winX ) {
	    winX--;
	    filX--;
	}
	else
	    break;
    }

    SeekAndPosLine( fhd, filX, filY );
    SetTextAreaPos( fhd, winX, winY, crsX, crsY );
}



void MoveRight( int fhd, ushort count )
{
    ulong  nlines, filY, winY;
    ushort filX, winX, crsY, crsX;
    int    sx, sy;

    nlines = GetSyncedFilePos( fhd, &filX, &filY, &winX, &winY,
				    &crsX, &crsY, &sx, &sy );

    for( ;count ;count-- ) {
	if( filX < MAX_LINELEN-1 ) {
	    if( crsX+1 < sx )
		crsX++;
	    else
		winX++;
	    filX++;
	}
	else
	    break;
    }

    SeekAndPosLine( fhd, filX, filY );
    SetTextAreaPos( fhd, winX, winY, crsX, crsY );
}



void MoveLeftScrl( int fhd, ushort count )
{
    ulong  nlines, filY, winY;
    ushort filX, winX, crsY, crsX;
    int    sx, sy;

    nlines = GetSyncedFilePos( fhd, &filX, &filY, &winX, &winY,
				    &crsX, &crsY, &sx, &sy );

    for( ;count ;count-- ) {
	if( winX  ) {
	    winX--;
	    filX--;
	}
	else if( crsX ) {
	    crsX--;
	    filX--;
	}
	else
	    break;
    }

    SeekAndPosLine( fhd, filX, filY );
    SetTextAreaPos( fhd, winX, winY, crsX, crsY );
}



void MoveRightScrl( int fhd, ushort count )
{
    ulong  nlines, filY, winY;
    ushort filX, winX, crsY, crsX;
    int    sx, sy;

    nlines = GetSyncedFilePos( fhd, &filX, &filY, &winX, &winY,
				    &crsX, &crsY, &sx, &sy );

    for( ;count ;count-- ) {
	if( filX+1 < MAX_LINELEN-1 ) {
	    if( winX+sx < MAX_LINELEN-1 )
		winX++;
	    else
		crsX++;
	    filX++;
	}
	else
	    break;
    }

    SeekAndPosLine( fhd, filX, filY );
    SetTextAreaPos( fhd, winX, winY, crsX, crsY );
}



void Move2Line( int fhd, ulong lnr )
{
    ulong  nlines, filY, winY;
    ushort filX, winX, crsY, crsX;
    int    sx, sy;


    lastWasEof = 0;
    GetScreenSize( &sx, &sy );
    nlines = GetAllFilePos( fhd, &filX, &filY, &winX, &winY, &crsX, &crsY );
    if( !winY )
	sy--;	/* topline is visible */

    if( lnr >= nlines ) {
	lastWasEof = 1;
	lnr = nlines-1;
    }
    filY = lnr;
    if( !(filY >= winY && filY < winY+sy) )
	winY = filY > sy/2 ? filY - sy/2 : 0;
    crsY = filY - winY;

    SeekAndPosLine( fhd, filX, filY );
    SetTextAreaPos( fhd, winX, winY, crsX, crsY );
}



void Move2Column( int fhd, ushort nr )
{
    ulong  nlines, filY, winY;
    ushort filX, winX, crsY, crsX;
    int    sx, sy;

    GetScreenSize( &sx, &sy );
    nlines = GetAllFilePos( fhd, &filX, &filY, &winX, &winY, &crsX, &crsY );
    if( !winY )
	sy--;	/* topline is visible */

    if( nr >= MAX_LINELEN )
	nr = MAX_LINELEN-1;
    filX = nr;
    if( !(filX >= winX && filX < winX+sx) )
	winX = filX > sx/2 ? filX - sx/2 : 0;
    crsX = filX - winX;

    SeekAndPosLine( fhd, filX, filY );
    SetTextAreaPos( fhd, winX, winY, crsX, crsY );
}


/****************
 * Move to a edge of the window
 * mode: 0 = top, 1=right, 2 = bottom, 3 = left
 *	 4 = center line
 *	 5 = top line
 */

void Move2Edge( int fhd, int mode )
{
    ulong  nlines, filY, winY;
    ushort filX, winX, crsY, crsX;
    int    sx, sy;

    nlines = GetSyncedFilePos( fhd, &filX, &filY, &winX, &winY,
				    &crsX, &crsY, &sx, &sy );

    switch( mode ) {
      case 0: /* top edge */
	filY = winY;
	crsY = 0;
	break;

      case 1: /* right edge */
	crsX = sx-1;
	filX = winX+crsX;
	break;

      case 2: /* bottom edge */
	if( winY+sy > nlines ) { /* bottominfoline is visible */
	    filY = nlines-1;
	    crsY = filY - winY;
	}
	else {
	    crsY = sy-1;
	    filY = winY + crsY;
	}
	break;

      case 3: /* left edge */
	filX = winX;
	crsX = 0;
	break;

      case 4: /* center line */
	if( filY > (sy/2) ) {
	    winY = filY - (sy/2);
	    crsY = sy/2;
	}
	break;

      case 5: /* top line ( make the current line the top one ) */
	winY = filY;
	crsY = 0;
	break;
    }


    SeekAndPosLine( fhd, filX, filY );
    SetTextAreaPos( fhd, winX, winY, crsX, crsY );
}



void Move2Pos( int fhd, ulong lnr, ushort cnr )
{
    SeekAndPosLine( fhd, cnr, lnr );
    Move2Line( fhd, lnr );
    Move2Column( fhd, cnr );
}




void MoveCmdLeft( int fhd, ushort count )
{
    ushort nbytes, pos, win, crs;
    int    sx;

    sx = GetCmdLineLen();
    GetAllCmdLineInfo( fhd, &nbytes, &pos, &win, &crs );

    for( ;count ;count-- ) {
	if( crs ) {
	    crs--;
	    pos--;
	}
	else if( win ) {
	    win--;
	    pos--;
	}
	else
	    break;
    }

    SetAllCmdLineInfo( fhd, pos, win, crs );
}


void MoveCmdLeftScrl( int fhd, ushort count )
{
    ushort nbytes, pos, win, crs;
    int    sx;

    sx = GetCmdLineLen();
    GetAllCmdLineInfo( fhd, &nbytes, &pos, &win, &crs );

    for( ;count ;count-- ) {
	if( win ) {
	    win--;
	    pos--;
	}
	else if( crs ) {
	    crs--;
	    pos--;
	}
	else
	    break;
    }

    SetAllCmdLineInfo( fhd, pos, win, crs );
}


void MoveCmdRight( int fhd, ushort count )
{
    ushort nbytes, pos, win, crs;
    int    sx;

    sx = GetCmdLineLen();
    GetAllCmdLineInfo( fhd, &nbytes, &pos, &win, &crs );

    for( ;count ;count-- ) {
	if( pos >= MAX_LINELEN-1 )
	    break;
	else if( crs < sx-1 ) {
	    crs++;
	    pos++;
	}
	else {
	    win++;
	    pos++;
	}
    }

    SetAllCmdLineInfo( fhd, pos, win, crs );
}


void MoveCmdRightScrl( int fhd, ushort count )
{
    ushort nbytes, pos, win, crs;
    int    sx;

    sx = GetCmdLineLen();
    GetAllCmdLineInfo( fhd, &nbytes, &pos, &win, &crs );

    for( ;count ;count-- ) {
	if( pos >= MAX_LINELEN-1 )
	    break;
	else if( win+sx < MAX_LINELEN-1 ) {
	    win++;
	    pos++;
	}
	else {
	    crs++;
	    pos++;
	}
    }

    SetAllCmdLineInfo( fhd, pos, win, crs );
}


void MoveCmd2Pos( int fhd, ushort new )
{
    ushort nbytes, pos, win, crs;
    int    sx;

    sx = GetCmdLineLen();
    GetAllCmdLineInfo( fhd, &nbytes, &pos, &win, &crs );

    pos = new;
    if( !(pos >= win && pos < win+sx) )
	win = pos > sx/2 ? (pos - sx/2) : 0;
    crs = pos - win;

    SetAllCmdLineInfo( fhd, pos, win, crs );
}


void MoveCmd2Edge( int fhd, int where )
{
    ushort nbytes, pos, win, crs;
    int    sx;

    sx = GetCmdLineLen();
    GetAllCmdLineInfo( fhd, &nbytes, &pos, &win, &crs );

    if( where == 3 ) /* left edge */
	pos = win;
    else
	pos = win +sx -1;

    SetAllCmdLineInfo( fhd, pos, win, crs );
}

/*** bottom of file ***/
