/* [wproc.c wk 29.7.91] W-Editor Dispatcher
 *	Copyright (c) 1991 by Werner Koch (dd9jn)
 * This file is part of the W-Editor.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * History:
 *  9.05.92 wk	Cmd_Quit benutzt jetzt die neue Funktion: GetPrevFileHandle().
 *		An ZEditProc_ wird die extension lowercased angehangen,
 *		notwendig, da durch HPFS der Case nicht geaendert wird und
 *		die Variablen casesensitiv sind.
 * 11.05.92 wk	Bug in protected mode environments: Write2CmdLine() could
 *		get an NULL from CopyFromHistory(), now it replaces NULL by ""
 * 20.12.92 wk	Neue Load option -s und -m
 * 03.01.93 wk	Some cleanups, removed some systemdep. functions
 *
 *
 * TODOs:
 * Remove:  dos_getdrive() dos_getdiskfreespace() _chkstack()
 *
 */

#include "wtailor.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <wk/lib.h>
#include <wk/string.h>
#include <wk/file.h>
#include <wk/hwacc.h>
#include <wk/environ.h>
#if __ZTC__
  #include <dos.h>
  #include <sound.h>
#elif __WATCOMC__
  #include <dos.h>
#endif

#include "w.h"
#include "wcmd.h"
#include "wscreen.h"
#include "wfile.h"
#include "wkbddrv.h"
#include "hilite.h"

/****** constants *********/
/******* typedefs ********/
/******* globals **********/
static int confirmChangePending;
static char *lastChangePattern; /* malloced string */
static int write2CmdLinePending;
static struct {
	int abbrev;	/* off / on */
	int autolock;	/* off / on */
	int lockcheck;	/* off / on */
	int searchcase; /* any / exact */
	int pureascii;	/* off / on */
	int tabexpand;	/* off / on */
	int crtsave;	/* 0= off, n = seconds */
	int blankstrip; /* off / on */
	int backup;	/* # of lines */
	int appctrlz;	/* off / on */
	int memswap;   /* off / on */
	int loadupd;   /* off / on */
	int history;   /* 0 = off, > 0 = # of stroed lines */
	int readbinary; /* off / on */
	int savemode;	/* 0 = standard, 1 = rename, 2 = backup, */
			/* 3 = rename and backup */
			/* 4 = store backupfile in mirror dir "/tmp/w.bak" */
	int tabmode;	/* 0 = standard */
			/* 1 = insert tabs, but not in lines containing string*/
	int unixsave;	/* use UNIX savemode as default (off/on) */
	int refresh;	/* 0 = no refresh or refresh interval in seconds */
	int intclip;	/* true = force using of internal clipboard */
	int latin;
	int editcmd;	/* on/off */
    } setOptions;

static char **history;	 /* each line ist malloced, */
static int historySize;  /* number of lines to be stored in the history */
			 /* 0 = disabled */
static int historyIdx;	 /* current index into history array */
static int historyDisabled;
static int toggleHd1 = -1, toggleHd2 = -1;
static int toggleInd;

/******* prototypes *******/
static int SimpleCmds( int cmdCode );
static void ShowSetOption( int opt );
static int GetACharFromUser(void);
static void Write2CmdLine( int fileHd, const char * string );
static int DoCommit( void );
static int DoEdit( const char *string, int mode  );
static int Cmd_Execute(void);
static void Add2History( const char *str );
static void InitHistory(void);
static void CopyFromHistory( int mode );
static int CheckSpecialHistory( int value );

/******* functions ********/

/****************
 * Ein einzelnes Command ausfuehren.
 */

int ProcessCmd( cmd_t *cmd )
{
    static int sentinel;
    int err, fileHd, inCmdLine, i,j;
    unsigned flags;
    ushort pos, x, y;
    ulong lineNr;
    char *p;
    const char *txt;
    ushort tabArray[MAX_TABS];
    ushort len;

    err = 0;
    sentinel++;
    if( sentinel > 20 ) {
	err = ERR_2DEEP;
	goto endLabel;
    }
    if( sigIntPending ) {
	err = ERR_CMDINT;
	goto endLabel;
    }
    fileHd = QryScreenFile();
    GetFileInfo( fileHd, &flags );
    inCmdLine = flags & WFILE_INCMD;
    if( confirmChangePending ) {
	if( cmd->cmd != CMD_CONFIRM_CHANGE ) {
	    confirmChangePending = 0;
	    free( lastChangePattern );
	    ShowMessage(NULL);
	}
    }

  #if 0
    printf("CMD No: %d\n", cmd->cmd);
  #endif
    switch( cmd->cmd ) {
    #if W_FULL
      case CMD_ASSIGN:
	err = Cmd_Assign( cmd );
	break;

      case CMD_BEGIN: /* may be called by Cmd_Call */
	err = ExecCmd( cmd );
	break;
    #endif

      case CMD_BACKTAB:
	if( inCmdLine ) {  /* use the default settings in the command line */
	    GetPtr2CmdLine( fileHd, &pos );
	    if( pos ) {
		pos = ((pos-1)/8)*8;
		MoveCmd2Pos( fileHd, pos);
	    }
	}
	else {
	    GetFilePos( fileHd, &pos );
	    y = GetFilesTabSettings( fileHd, tabArray, DIM(tabArray) );
	    for(i=0; i < y; i++ )
		if( tabArray[i] >= pos ) {
		    Move2Column( fileHd, i? tabArray[i-1]:0 );
		    break;
		}
	    if( i == y )
		if( pos ) {
		    pos = ((pos-1)/8)*8;
		    MoveCmd2Pos( fileHd, pos);
		}
	}
	break;

      case CMD_BEGIN_LINE:
	if( inCmdLine )
	    MoveCmd2Pos( fileHd, 0);
	else
	    Move2Column( fileHd, 0 );
	break;


      case CMD_BOTTOM:
	FlushEditBuf( fileHd );
	Move2Line( fileHd, ULONG_MAX );
	break;

      case CMD_BOTTOM_EDGE:
	FlushEditBuf( fileHd );
	Move2Edge( fileHd, 2 );
	break;

    #if W_FULL
      case CMD_CENTER_IN_MARGINS:
	err = Cmd_CenterInMargins();
	break;

      case CMD_CENTER_LINE:
	Move2Edge( fileHd, 4 );
	break;
    #endif

      case CMD_CHANGE:
	xassert( cmd->type == ARGTYPE_CHG );
	err = Cmd_Change( cmd->arg.string, 0 );
	if( err == ERR_PSEUDO_CFN ) {
	    lastChangePattern = xstrdup( cmd->arg.string );
	    confirmChangePending = 1;
	    ShowMessageAsInfo("Please confirm change !");
	    err = 0;
	}
	break;

    #if W_FULL
      case CMD_Q_DIR:
      case CMD_CHDIR:
	if( *cmd->arg.string && cmd->cmd == CMD_CHDIR )  /* change */
	    err = Cmd_ChDir( cmd );
	else   /* show current directory */
	    err = Cmd_ShowDir( cmd );
	break;
    #endif

      case CMD_CLEAR_MARKS:
	ResetMarkStack();
	SetMark( fileHd, 0 );
	break;


      case CMD_COLUMN:
	pos = cmd->arg.number < MAX_LINELEN? cmd->arg.number:MAX_LINELEN;
	if( pos )
	    pos--;
      #if W_FULL
	else {
	    if( err = CalcStackPopULong( &lineNr ) )
		break;
	    pos = lineNr < MAX_LINELEN ? lineNr : MAX_LINELEN;
	    if( pos )
		pos--;
	}
      #endif
	if( inCmdLine )
	    MoveCmd2Pos( fileHd, pos);
	else
	    Move2Column( fileHd, pos );
	break;

    #if W_FULL
      case CMD_CSTR:
	err = Cmd_CStr(cmd);
	break;

      case CMD_POSATMOUSE:
	i = ScreenMousePos( &i, &lineNr, &pos );    /* i is dummy for Buttons*/
	if( i == 2 ) {
	    ResetFileFlag( fileHd, WFILE_INCMD );
	    Move2Line( fileHd, lineNr );
	    Move2Column( fileHd, pos );
	}
	else if( i == 1 ) {
	    SetFileFlag( fileHd, WFILE_INCMD );
	    MoveCmd2Pos( fileHd, pos);
	}
	/* kein Fehler, wenn nicht im window */
	break;
    #endif

      case CMD_COMMAND_TOGGLE:
	FlushEditBuf( fileHd );
	GetFileInfo( fileHd, &flags );
	FlushEditBuf( fileHd );
	if( flags & WFILE_INCMD )
	    ResetFileFlag( fileHd, WFILE_INCMD );
	else
	    SetFileFlag( fileHd, WFILE_INCMD );
	break;

      case CMD_CONFIRM_CHANGE:
	if( confirmChangePending ) {
	    ShowMessage(NULL);
	    err = Cmd_Change( lastChangePattern, 1 );
	    confirmChangePending=0;
	    free( lastChangePattern );
	}
	else
	    err = ERR_NOCHGP;
	break;

      case CMD_COPY_FROM_COMMAND:
	FlushEditBuf( fileHd );
	p = GetPtr2CmdLine( fileHd, &pos );
	lineNr = GetFilePos( fileHd, NULL );
	err = InsertLine( fileHd, p, strlen(p) );
	SeekLine( fileHd, lineNr );
	break;

      case CMD_COPY_MARK:
	err = Cmd_CopyMark();
	break;

      case CMD_MOVE_MARK:
	if( !(err = Cmd_CopyMark()) )
	    if( !(err = Cmd_DeleteMark() ) )
		err = Cmd_Switch2NamedFile( GetFileInfo( fileHd, &flags ) );
	break;

      case CMD_DELETE_MARK:
	err = Cmd_DeleteMark();
	break;

      case CMD_COPY_TO_HISTORY:
	if( cmd->type == ARGTYPE_VAR || cmd->type == ARGTYPE_VARENV ) {
	    Add2History( p = VarValue( cmd ) );
	    FreeVarValue( p );
	}
	else
	    Add2History( cmd->arg.string );
	break;

      case CMD_COPY_TO_COMMAND: {
	    txt = GetPtr2Line( fileHd, &len );
	    p = xmalloc( len + 1 );
	    memcpy(p, txt, len );
	    for(i=0; i < len; i++ )
		if( !p[i] )
		    p[i] = ' ';
	    p[i] = '\0';
	    Write2CmdLine( fileHd, p );
	    free(p);
	}
	break;

      case CMD_DEFKEY:
	err = KybrdDrvDefKey( (p = VarValue( cmd )));
	FreeVarValue( p );
	break;

      case CMD_LEAP_TO:
	err = SetLeapTarget( p = VarValue( cmd ));
	FreeVarValue( p );
	break;

      case CMD_CURSOR_COMMAND:
	FlushEditBuf( fileHd );
	SetFileFlag( fileHd, WFILE_INCMD);
	break;

      case CMD_CURSOR_DATA:
	FlushEditBuf( fileHd );
	ResetFileFlag( fileHd, WFILE_INCMD);
	break;

      case CMD_DELETE_CHAR:
	if( !(x = (ushort)cmd->arg.number) )
	    x++;
	for( ;x && !err ; x-- )
	    err = Cmd_DeleteChar();
	break;

      case CMD_DELETE_LINE:
	if( !inCmdLine )
	    err = Cmd_DeleteLine();
	break;

    #if W_FULL
      case CMD_DIR:
	err = Cmd_Dir( cmd );
	break;

      case CMD_EDITLIST:
	err = Cmd_EditList( cmd );
	break;

      case CMD_VARLIST:
	err = Cmd_VarList( cmd );
	break;


      case CMD_KEYDEFS:
	err = Cmd_Keydefs( cmd );
	break;

      case CMD_DETACH:
	if( cmd->type == ARGTYPE_VAR || cmd->type == ARGTYPE_VARENV )
	    p = VarValue( cmd );
	else
	    p = NULL;
	err = Cmd_Detach( p ? p : cmd->arg.string );
	if( p )
	    FreeVarValue( p );
	break;

      case CMD_KILL_PROCESS:
	err = Cmd_KillProcess( cmd->arg.number );
	break;

      case CMD_CANCEL_PROCESS:
	err = Cmd_CancelProcess( cmd->arg.number );
	break;

      case CMD_SHOW_PROCESS:
	err = Cmd_ShowProcess( cmd->arg.number );
	break;

      case CMD_DOS:
	err = Cmd_Dos( cmd );
	break;
    #endif

      case CMD_DOWN:
	FlushEditBuf( fileHd );
	if( inCmdLine ) {
	    ResetFileFlag( fileHd, WFILE_INCMD);
	    Move2Edge( fileHd, 2 );
	}
	else {
	    if( !(x = (ushort)cmd->arg.number) )
		x++;
	    MoveDown( fileHd, x );
	}
	break;

      case CMD_VIEW:
      case CMD_REEDIT:
      case CMD_EDIT:
      case CMD_LOAD:
      case CMD_NEW:
       #if W_FULL
	if( cmd->type == ARGTYPE_VAR || cmd->type == ARGTYPE_VARENV )
	    p = VarValue( cmd );
	else
       #endif
	    p = NULL;
	if( p ? *p:*cmd->arg.string ) {
	    if( GetFileHandle( p? p: cmd->arg.string ) == -1 )
		if( IsInternalFileName( p? p:cmd->arg.string ) )
		    err = ERR_INTNAM;
	}
	if( !err ) {
	    if( cmd->cmd == CMD_REEDIT )
		err = Cmd_ReEdit( p ? p : cmd->arg.string );
	    else if( cmd->cmd == CMD_LOAD )
		err = Cmd_Load( p ? p : cmd->arg.string );
	    else if( cmd->cmd == CMD_NEW )
		err = Cmd_New( p ? p : cmd->arg.string );
	    else if( cmd->cmd == CMD_VIEW )
		err = Cmd_View( p ? p : cmd->arg.string );
	    else if( !setOptions.editcmd && (p? *p : *cmd->arg.string) )
		err = Cmd_Load( p ? p : cmd->arg.string );
	    else
		err = Cmd_Edit( p ? p : cmd->arg.string );
	}
      #if W_FULL
	if( p )
	    FreeVarValue( p );
      #endif
	break;


      case CMD_END_LINE:
	if( inCmdLine ) {
	    p = GetPtr2CmdLine( fileHd, &pos );
	    pos = strlen(p);
	    MoveCmd2Pos( fileHd, pos );
	}
	else {
	    GetPtr2Line( fileHd, &pos );
	    Move2Column( fileHd, pos );
	}
	break;

    #if W_FULL
      case CMD_ERASE:
	err = Cmd_Erase( cmd );
	break;
    #endif

      case CMD_ERASE_BEGIN_LINE:
	err = Cmd_EraseBeginLine();
	break;

      case CMD_ERASE_END_LINE:
	err = Cmd_EraseEndLine();
	break;

      case CMD_ERROR:
	FlushEditBuf( fileHd );
	err = (int)cmd->arg.number;
	if( !err )
	    err = ERR_PSEUDO_CE;
	break;

      case CMD_ESCAPE: {
	    char buf[2];

	    buf[0] = GetACharFromUser();
	    buf[1] = '\0';
	    err = Cmd_Put( buf );
	}
	break;


      case CMD_EXECUTE:
	if( cmd->type == ARGTYPE_STRING && !*cmd->arg.string )
	    err = Cmd_Execute();
	else
	    err = Cmd_ExecuteString( cmd->arg.string, cmd->type );
	break;

      case CMD_EXECUTECLEAR:
	err = ExecuteCmdLine();
	break;


      case CMD_FILE:
	if( !(err = Cmd_Save( cmd )) ) {
	    /* reset, cause save will not reset it when a file is specified*/
	    ResetFileFlag( fileHd, WFILE_CHG );
	    err = Cmd_Quit();
	}
	break;

      case CMD_FILL_MARK:
	if( !MarkInfo( NULL ) )
	    err = ERR_NOMRK;
	else
	    err = Cmd_FillMark( cmd, GetACharFromUser() );
	break;

      case CMD_FIND_BLANK_LINE:
	err = Cmd_FindBlankLine();
	break;

      case CMD_FIRST_NONBLANK:
	err = Cmd_FirstNonBlank();
	break;

      case CMD_NEXT_NONBLANK:
	err = Cmd_NextNonBlank();
	break;

      case CMD_NEXT_BLANK:
	err = Cmd_NextBlank();
	break;

    #if W_FULL
      case CMD_GOTO_LABEL:
	err = Cmd_GotoLabel( cmd );
	break;
    #endif


      case CMD_INSERT_LINE:
	err = Cmd_InsertLine( NULL, 0 );
	break;

      case CMD_INSERT_MODE:
	SetFileFlag( fileHd, WFILE_INSERT );
	break;

      case CMD_INSERT_TOGGLE:
	GetFileInfo( fileHd, &flags );
	if( flags & WFILE_INSERT)
	    ResetFileFlag( fileHd, WFILE_INSERT);
	else
	    SetFileFlag( fileHd, WFILE_INSERT);
	break;

      case CMD_JOIN:
	err = Cmd_Join();
	break;

      case CMD_KEY:
	err = ExecKeyCmd( cmd->arg.keyId, 0 );
	break;

      case CMD_LEFT:
	if( !(x = (ushort)cmd->arg.number) )
	    x++;
	if( inCmdLine )
	    MoveCmdLeft( fileHd, x );
	else
	    MoveLeft( fileHd, x );
	break;

      case CMD_LEFT_EDGE:
	if( inCmdLine )
	    MoveCmd2Edge( fileHd, 3 );
	else
	    Move2Edge( fileHd, 3 );
	break;

    #if W_FULL
      case CMD_LEFT_MARGIN:
	err = Cmd_GotoMargin( 0 );
	break;
    #endif

      case CMD_LINE:
	if( cmd->arg.number )
	    lineNr = cmd->arg.number-1;
	else {
	  #if W_FULL
	    if( err = CalcStackPopULong( &lineNr ) )
		break;
	    if( lineNr )
		lineNr--;
	  #else
	    lineNr = 0;
	  #endif
	}
	FlushEditBuf( fileHd );
	Move2Line( fileHd, lineNr );
	ResetFileFlag( fileHd, WFILE_INCMD);
	break;

      case CMD_GOTO_LINEID:
	if( cmd->arg.number )
	    lineNr = cmd->arg.number;
	else {
	    if( err = CalcStackPopULong( &lineNr ) )
		break;
	}
	err = GetLineNrFromLineId( fileHd, &lineNr );
	if( !err ) {
	    FlushEditBuf( fileHd );
	    Move2Line( fileHd, lineNr );
	    ResetFileFlag( fileHd, WFILE_INCMD);
	}
	break;


      case CMD_LOCATE:
	ShowMessageAsInfo("Search in progress ...");
	err = Cmd_Locate( cmd );
	ShowMessage(NULL);
	break;

      case CMD_LOWERCASE:
	err = Cmd_ChangeCase(0);
	break;

    #if W_FULL
      case CMD_MACRO:
	if( cmd->type == ARGTYPE_FILE )
	    err = ReadCmdFile( cmd->arg.string, 1 );
	else {
	    p = VarValue( cmd );
	    err = ReadCmdFile( p, 1 );
	    FreeVarValue(p);
	}
	break;

      case CMD_RECORD_START:
	err = Cmd_RecordStart(cmd);
	break;
      case CMD_RECORD_STOP:
	err = Cmd_RecordStop();
	break;
      case CMD_RECORD_CANCEL:
	err = Cmd_RecordCancel();
	break;
      #endif

      case CMD_MARK_BLOCK:
	err = SetMark( fileHd, MARKTYPE_BLOCK);
	break;

      case CMD_SORT:
	if( cmd->type == ARGTYPE_VAR || cmd->type == ARGTYPE_VARENV )
	    p = VarValue( cmd );
	else
	    p = NULL;
	ShowMessageAsInfo("Sort in progress ...");
	err = SortLines( fileHd, p ? p : cmd->arg.string );
	ShowMessage(NULL);
	if( p )
	    FreeVarValue( p );
	break;

    #if W_FULL
      case CMD_MARK_CHAR:
	err = SetMark( fileHd, MARKTYPE_CHAR );
	break;
    #endif

      case CMD_MARK_LINE:
	err = SetMark( fileHd, MARKTYPE_LINE );
	break;

      case CMD_NAME:
	FlushEditBuf( fileHd );
	err = RenameFile( fileHd, cmd->arg.string );
	break;

    #if W_FULL
      case CMD_OVERLAY_BLOCK:
	err = Cmd_OverlayBlock( cmd );
	break;
    #endif

      case CMD_PAGE_DOWN:
	GetScreenSize( &i, &j );
	MoveDownScrl( fileHd, j-2 );
	break;

      case CMD_PAGE_UP:
	GetScreenSize( &i, &j );
	MoveUpScrl( fileHd, j-2 );
	break;

    #if W_FULL
      case CMD_PARAGRAPH_MARGIN:
	err = Cmd_GotoMargin( 2 );
	break;
    #endif

      case CMD_POP_MARK:
	err = PopMark();
	break;

    #if W_FULL
      case CMD_PRINT:
	err = Cmd_Print( cmd , 0 );
	break;

      case CMD_PRINT_MARK:
	err = Cmd_Print( cmd , 1 );
	break;
    #endif

      case CMD_PUSH_MARK:
	if( !(err = PushMark()) )
	    err = SetMark( fileHd, 0 );
	break;

      case CMD_PUT:
	if( cmd->type == ARGTYPE_ASCII ) {
	    char buf[2];

	    buf[0] = cmd->arg.keyId & 0xff;
	    buf[1] = '\0';
	    err = Cmd_Put( buf);
	}
	else if( cmd->type == ARGTYPE_STRING ) {
	    err = Cmd_Put( cmd->arg.string );
	}
	else {
	  #if W_FULL
	    if( *(p = VarValue( cmd )) )
		err = Cmd_Put( p );
	    FreeVarValue( p );
	  #endif
	}
	break;

    #if W_FULL
      case CMD_PUTENV:
	if( putenv( cmd->arg.string ) )
	    err = ERR_PUTENV;
	break;
     #endif

      case CMD_REPLACE_MODE:
	ResetFileFlag( fileHd, WFILE_INSERT );
	break;

      case CMD_RIGHT:
	if( !(x = (ushort)cmd->arg.number) )
	    x++;
	if( inCmdLine )
	    MoveCmdRight( fileHd, x );
	else
	    MoveRight( fileHd, x );
	break;

      case CMD_RIGHT_EDGE:
	if( inCmdLine )
	    MoveCmd2Edge( fileHd, 1 );
	else
	    Move2Edge( fileHd, 1 );
	break;

      case CMD_RUBOUT:
	if( inCmdLine )
	    MoveCmdLeft( fileHd, 1 );
	else
	    MoveLeft( fileHd, 1 );
	err = Cmd_DeleteChar();
	break;

      case CMD_SAVE:
	err = Cmd_Save( cmd );
	break;

      case CMD_SCROLLDOWN:
	if( !(x = (ushort)cmd->arg.number) )
	    x++;
	MoveDownScrl( fileHd, x );
	break;

      case CMD_SCROLLLEFT:
	if( !(x = (ushort)cmd->arg.number) )
	    x++;
	if( x > MAX_LINELEN )
	    x = MAX_LINELEN;
	if( inCmdLine )
	    MoveCmdLeftScrl( fileHd, x );
	else
	    MoveLeftScrl( fileHd, x );
	break;

      case CMD_SCROLLRIGHT:
	if( !(x = (ushort)cmd->arg.number) )
	    x++;
	if( x > MAX_LINELEN )
	    x = MAX_LINELEN;
	if( inCmdLine )
	    MoveCmdRightScrl( fileHd, x );
	else
	    MoveRightScrl( fileHd, x );

	break;

      case CMD_SCROLLUP:
	FlushEditBuf( fileHd );
	if( !(x = (ushort)cmd->arg.number) )
	    x++;
	MoveUpScrl( fileHd, x );
	break;

      case CMD_SET_ABBREV:
	PutSetOption( SETOPT_ABBREV, cmd->arg.number );
	break;

      case CMD_SET_APPCTRLZ:
	PutSetOption( SETOPT_APPCTRLZ, cmd->arg.number );
	break;

      case CMD_SET_MEMSWAP:
	PutSetOption( SETOPT_MEMSWAP, cmd->arg.number );
	break;

      case CMD_SET_HISTORY:
	if( CheckSpecialHistory( cmd->arg.number ) ) {
	    PutSetOption( SETOPT_HISTORY, cmd->arg.number );
	    InitHistory();
	}
	break;

      case CMD_SET_BACKUP:
	PutSetOption( SETOPT_BACKUP, cmd->arg.number );
	break;

      case CMD_SET_BLANKSTRIP:
	PutSetOption( SETOPT_BLANKSTRIP, cmd->arg.number );
	break;

      case CMD_SET_CRTSAVE:
	PutSetOption( SETOPT_CRTSAVE, cmd->arg.number );
	break;

      case CMD_SET_REFRESH:
	PutSetOption( SETOPT_REFRESH, cmd->arg.number );
	break;

      case CMD_SET_DISPLAY:
	err = ERR_USPCMD;
	break;

      case CMD_SET_HSCROLL:
	err = ERR_USPCMD;
	break;

    #if W_FULL
      case CMD_SET_LABEL:
	err = Cmd_SetLabel( cmd );
	break;

      case CMD_SET_MARGINS:
	err = Cmd_SetMargins( cmd );
	break;

      case CMD_SET_SAVEMODE:
	PutSetOption( SETOPT_SAVEMODE, cmd->arg.number );
	break;
      case CMD_SET_TABMODE:
	PutSetOption( SETOPT_TABMODE, cmd->arg.number );
	break;
      case CMD_SET_UNIXSAVE:
	PutSetOption( SETOPT_UNIXSAVE, cmd->arg.number );
	break;
    #endif

      case CMD_SET_NOTABS:
	if( cmd->arg.number )
	    SetFileFlag( fileHd, WFILE_NOTAB );
	else
	    ResetFileFlag( fileHd, WFILE_NOTAB );
	break;

      case CMD_SET_HILITE:
	if( !stricmp(cmd->arg.string,"on") ) {
	    SetFileFlag( fileHd, WFILE_HILITE );
	    HiliteReset(fileHd);
	}
	else if( !stricmp(cmd->arg.string,"off") )
	    ResetFileFlag( fileHd, WFILE_HILITE );
	else {
	    SetFilesHiliteType( fileHd, cmd->arg.string );
	    HiliteReset(fileHd);
	}
	RedrawScreen();
	break;

      case CMD_SET_PUREASCII:
	PutSetOption( SETOPT_PUREASCII, cmd->arg.number );
	break;

      case CMD_SET_READBINARY:
	PutSetOption( SETOPT_READBINARY, cmd->arg.number );
	break;


      case CMD_SET_RO:
	SetFileFlag( fileHd, WFILE_RO );
	break;

      case CMD_SET_RW:
	GetFileInfo( fileHd, &flags );
	if( flags & WFILE_INT )
	    err = ERR_SETRW;
	else {
	    ResetFileFlag( fileHd, WFILE_RO );
	    GetFileInfo( fileHd, &flags );
	    if( flags & WFILE_RO )
		err = ERR_SETRW;
	}
	break;

      case CMD_SET_SEARCHCASE:
	PutSetOption( SETOPT_SEARCHCASE, cmd->arg.number );
	break;

      case CMD_SET_TABEXPAND:
	PutSetOption( SETOPT_TABEXPAND, cmd->arg.number );
	break;

    #if W_FULL

      case CMD_SET_SUPSCREEN:
	BindSupplementaryScreen( cmd->arg.number ? fileHd : -1 );
	break;


      case CMD_SET_TABS:
	err = Cmd_SetTabs( cmd );
	break;
    #endif

      case CMD_TAB:
	if( inCmdLine ) {  /* use the default settings in the command line */
	    GetPtr2CmdLine( fileHd, &pos );
	    if( pos < MAX_LINELEN-9) {
		pos = (pos/8+1)*8;
		MoveCmd2Pos( fileHd, pos);
	    }
	}
	else {
	    GetFilePos( fileHd, &pos );
	    y = GetFilesTabSettings( fileHd, tabArray, DIM(tabArray) );
	    for(i=0; i < y; i++ )
		if( tabArray[i] > pos ) {
		    Move2Column( fileHd, tabArray[i] );
		    break;
		}
	    if( i == y )
		if( pos < MAX_LINELEN-9) {
		    pos = (pos/8+1)*8;
		    MoveCmd2Pos( fileHd, pos);
		}
	}
	break;

      case CMD_TOP:
	FlushEditBuf( fileHd );
	Move2Line( fileHd, 0 );
	break;

      case CMD_TOP_EDGE:
	FlushEditBuf( fileHd );
	Move2Edge( fileHd, 0 );
	break;

      case CMD_UNDO:
	if( inCmdLine ) { /* clear it, got no backup */
	    *GetPtr2CmdLine( fileHd, &pos ) = '\0';
	    MoveCmd2Pos( fileHd, 0 );
	}
	else {
	    ClearEditBuf( fileHd );
	}
	break;

      case CMD_UNMARK:
	SetMark( fileHd, 0 );
	break;

      case CMD_HISTORY_UP:
	CopyFromHistory(0);
	break;

      case CMD_HISTORY_DOWN:
	CopyFromHistory(1);
	break;

      case CMD_UP:
	FlushEditBuf( fileHd );
	if( inCmdLine ) {
	    ResetFileFlag( fileHd, WFILE_INCMD);
	    Move2Edge( fileHd, 2 );
	}
	else {
	    if( !(x = (ushort)cmd->arg.number) )
		x++;
	    MoveUp( fileHd, x );
	}
	break;

    #if W_FULL
      case CMD_ENTER:
	err = Cmd_Enter( cmd );
	break;

      case CMD_JMP:
	err = Cmd_Jmp( cmd );
	break;

      case CMD_CALL:
	err = Cmd_Call( cmd );
	break;

      case CMD_UNTIL:
	err = Cmd_Until( cmd );
	break;

      case CMD_WHILE:
	err = Cmd_While( cmd );
	break;

      case CMD_CALC:
	err = Cmd_Calc( cmd );
	break;

      case CMD_Q_KEY:
	p = GetKeyDefString( cmd->arg.keyId );
	Write2CmdLine( fileHd, p );
	free(p);
	break;


      case CMD_RING:
	if( cmd->arg.number )
	    SetFileFlag( fileHd, WFILE_RING );
	else
	    ResetFileFlag( fileHd, WFILE_RING );
	break;
    #endif

      case CMD_SET_KEYPREFIX:
	PrefixNextGetKeyId( *cmd->arg.string );
	break;

    #if W_FULL
      case CMD_LOCK:
	SetFileFlag( fileHd, WFILE_LOCK );
	txt = GetFileInfo( fileHd, &flags );
	if( (flags & WFILE_LOCK) && (flags & WFILE_RO) )
	    ShowMessage( "Locked by: %s", GetFileLockOwner( txt ) );
	else if( !(flags & WFILE_LOCK) && !(flags & WFILE_INT) )
	    err = ERR_FINUSE;
	break;

      case CMD_UNLOCK:
	UnlockFile( fileHd );
	break;

      case CMD_SET_AUTOLOCK:
	PutSetOption( SETOPT_AUTOLOCK, cmd->arg.number );
	break;

      case CMD_SET_LOCKCHECK:
	PutSetOption( SETOPT_LOCKCHECK, cmd->arg.number );
	break;

      case CMD_SET_INTCLIP:
	PutSetOption( SETOPT_INTCLIP, cmd->arg.number );
	break;

      case CMD_PAUSE:
	lineNr = cmd->arg.number ;
	if( !lineNr )
	    lineNr = 1 ;
	UpdateScreen();
	ShowMessageAsInfo("Pausing ...");
	for( ;lineNr && !err; lineNr-- ) {
	    Sleep( 1000 );
	    err = SigIntPoll();
	}
	ShowMessageAsInfo(NULL);
	break;

      case CMD_SET_DLAYOUT:
	err = ScreenChangeLayout( cmd->arg.number );
	RedrawScreen();
	break;

      case CMD_SET_DCOLORS:
	err = ScreenChangeColors( cmd->arg.string );
	RedrawScreen();
	break;

      case CMD_SET_LATIN:
	if( !(err = SwitchCPForMode( cmd->arg.number? 1 : 0  )) ) {
	    ScreenSetTrans( cmd->arg.number? 1 : 0 );
	    KybrdDrvSetTrans( cmd->arg.number? 1: 0 );
	    PutSetOption( SETOPT_LATIN, cmd->arg.number );
	    RedrawScreen();
	}
	break;
     #endif

      case CMD_HELP:
	err = Cmd_Help(cmd->arg.string);
	break;

      case CMD_BACKTAB_WORD:
	err = Cmd_GotoWord2(1,cmd);
	break;

      case CMD_TAB_WORD:
	err = Cmd_GotoWord2(3,cmd);
	break;

      case CMD_BEGIN_WORD:
	err = Cmd_GotoWord2( 0, cmd );
	break;

      case CMD_END_WORD:
	err = Cmd_GotoWord2(2,cmd);
	break;



      case CMD_TOGGLE:
	if( toggleHd1 == -1 && toggleHd2 == -1 )
	    toggleHd1 = fileHd; /* noch keiner gesetzt */
	else if( toggleHd1 == -1 && toggleHd2 != fileHd ) {
	    toggleHd1 = fileHd;
	    toggleInd = 0;
	    if( toggleHd2 != -1 )
		err = Cmd_Switch2FileHd( toggleHd2 );
	}
	else if( toggleHd2 == -1 && toggleHd1 != fileHd ) {
	    toggleHd2 = fileHd;
	    toggleInd = 1;
	    if( toggleHd1 != -1 )
		err = Cmd_Switch2FileHd( toggleHd1 );
	}
	else if( toggleHd1 == fileHd ) {
	    if( toggleHd2 != -1 )
		err = Cmd_Switch2FileHd( toggleHd2 );
	}
	else if( toggleHd2 == fileHd ) {
	    if( toggleHd1 != -1 )
		err = Cmd_Switch2FileHd( toggleHd1 );
	}
	else if( toggleInd ) {
	    toggleInd = 0;
	    toggleHd1 = fileHd;
	    err = Cmd_Switch2FileHd( toggleHd2 );
	}
	else {
	    toggleInd = 1;
	    toggleHd2 = fileHd;
	    err = Cmd_Switch2FileHd( toggleHd1 );
	}
	break;

      case CMD_SWITCH_CONSOLE:
      #if USE_VT_LINUX
	err = VTLinuxSwitchConsole( cmd->arg.number );
      #else
	err = ERR_USPCMD;
      #endif
	break;

      case CMD_SET_EDITCMD:
	PutSetOption( SETOPT_EDITCMD, cmd->arg.number );
	break;

      default:
	if( (err = SimpleCmds( cmd->cmd )) == -1 ) {
	    err = ExecFunction( cmd );
	    if( err == ERR_NOTFNC ) {
	      #if W_FULL
		ShowMessage( "Invalid Command - Code = %d , Type = %d",
						    cmd->cmd, cmd->type );
		err = 0;
	      #else
		err = ERR_USPCMD;
	      #endif
	    }
	}
	break;
    }

    Try2AllocReservedMemory();

  endLabel:
    sentinel--;
    return err;
}


static int
SimpleCmds( int cmdCode )
{
    int err=0;


    switch( cmdCode ) {
    #if W_FULL
      case CMD_STRIPBLANKS:
	err = Cmd_StripBlanks();
	break;

      case CMD_END:
	err = ERR_UXPEND;
	break;

      case CMD_NOERROR:
	err = ERR_PSEUDO_NE;
	break;

      case CMD_COMMIT:
	err = DoCommit();
	break;

      case CMD_AUTOMARK:
	ScreenAutoMark();
	break;

      case CMD_ASCII:
	Cmd_Ascii();
	break;
     #endif


     #if W_FULL
      case CMD_BEEP:
      #if __ZTC__
	sound_beep(700);
      #endif
	break;

      case CMD_HISTORY:
	err = Cmd_History();
	break;

      case CMD_COPY:
	err = Cmd_Copy();
	break;

      case CMD_CUT:
	err = Cmd_Cut();
	break;

      case CMD_PASTE:
	err = Cmd_Paste();
	break;
     #endif

      case CMD_BEGIN_MARK:
	err = Cmd_GotoMark( 0 );
	break;

      case CMD_CLEAR_MESSAGE:
	ShowMessage(NULL);
	break;

    #if W_FULL
      case CMD_DEMO_END:
	demoModeFlag = 0;
	break;

      case CMD_DEMO_FAST:
	demoModeFlag = 1;
	break;

      case CMD_DEMO_SLOW:
	demoModeFlag = 2;
	break;
    #endif

      case CMD_END_MARK:
	err = Cmd_GotoMark( 1 );
	break;

    #if W_FULL
      case CMD_INDENT:
	err = Cmd_GotoMargin( 3 );
	break;

      case CMD_MATCHBLOCK:
	err = Cmd_MatchBlock();
	break;

      case CMD_NEXT_VIEW:
	err = ERR_USPCMD;
	break;

      case CMD_NEXT_WINDOW:
	err = ERR_USPCMD;
	break;
    #endif

      case CMD_NOP:
	break;

    #if W_FULL
      case CMD_PLAY_MARK:
	err = Cmd_PlayMark();
	break;
    #endif

      case CMD_QUIT:
	err = Cmd_Quit();
	break;

    #if W_FULL
      case CMD_EXIT:
	err = Cmd_Exit();
	break;
    #endif

      case CMD_REDRAW:
	RedrawScreen();
	break;

    #if W_FULL
      case CMD_REFLOW:
	err = Cmd_Reflow();
	break;

      case CMD_RIGHT_MARGIN:
	err = Cmd_GotoMargin( 1 );
	break;
    #endif

      case CMD_SHIFT_LEFT:
	err = Cmd_ShiftMark( 0 );
	break;

      case CMD_SHIFT_RIGHT:
	err = Cmd_ShiftMark( 1 );
	break;

      case CMD_SPLIT:
	err = Cmd_Split();
	break;

      case CMD_SPLIT_SCREEN:
	err = ERR_USPCMD;
	break;


    #if W_FULL
      case CMD_TREE:
	err = Cmd_Tree();
	break;
    #endif

      case CMD_UPPERCASE:
	err = Cmd_ChangeCase(1);
	break;

      case CMD_ZOOM_WINDOW:
	err = ERR_USPCMD;
	break;

      case CMD_SHOW_COPYING:
	ShowGNUInfo(0);
	break;

      case CMD_SHOW_WARRANTY:
	ShowGNUInfo(1);
	break;

    #if W_FULL
      case CMD_Q_ABBREV:
	ShowSetOption( SETOPT_ABBREV );
	break;

      case CMD_Q_APPCTRLZ:
	ShowSetOption( SETOPT_APPCTRLZ );
	break;

      case CMD_Q_MEMSWAP:
	ShowSetOption( SETOPT_MEMSWAP );
	break;

      case CMD_Q_BACKUP:
	ShowSetOption( SETOPT_BACKUP );
	break;

      case CMD_Q_HISTORY:
	ShowSetOption( SETOPT_HISTORY );
	break;

      case CMD_Q_BLANKSTRIP:
	ShowSetOption( SETOPT_BLANKSTRIP );
	break;

      case CMD_Q_CRTSAVE:
	ShowSetOption( SETOPT_CRTSAVE );
	break;

      case CMD_Q_DISKSPACE:
	ShowMessageAsInfo( "%luK diskspace available"
			  #if MSDOSFILESYSTEM
			    " on current drive"
			  #endif
			    , QryAvailableDiskSpace() );
	break;

      case CMD_Q_HEAP:
      #if __ZTC__
	  #if MEM_DEBUG
	    ShowMessageAsInfo( "Heap: %ld in use, %ld max. used. Stack: %u",
				memdbg_info(0), memdbg_info(1), _chkstack() );
	  #else
	    ShowMessageAsInfo( "Stack: %u", _chkstack() );
	  #endif
      #else
	err = ERR_USPCMD;
      #endif
	break;

      case CMD_Q_HSCROLL:
	err = ERR_USPCMD;
	break;

      case CMD_Q_MARGINS:
	Cmd_QueryMargins();
	break;

      case CMD_Q_MEMORY:
	ShowMessageAsInfo( "%luK memory available",
					   HW_QueryFreeMemory()/1024 );
	break;

      case CMD_Q_PUREASCII:
	ShowSetOption( SETOPT_PUREASCII );
	break;

      case CMD_Q_SEARCHCASE:
	ShowSetOption( SETOPT_SEARCHCASE );
	break;

      case CMD_Q_TABEXPAND:
	ShowSetOption( SETOPT_TABEXPAND );
	break;

      case CMD_Q_TABS:
	Cmd_QueryTabs();
	break;

      case CMD_Q_AUTOLOCK:
	ShowSetOption( SETOPT_AUTOLOCK );
	break;

      case CMD_Q_LOCKCHECK:
	ShowSetOption( SETOPT_LOCKCHECK );
	break;

      case CMD_DEMO_KEYS:
	while( !SigIntPoll() ) {
	    int i, v;

	    if( i = GetKeyId(&v) )
		ShowMessageAsInfo( "%04x %02x (%s) %s", i, v,
				   GetKeyName(0,i), GetKeyName(1,i));
	}
	ShowMessageAsInfo(NULL);
	break;

    #endif
      case CMD_REHASH:
	RehashCommandTable();
	break;

      default: err = -1; break; /* let caller process this error */
    }
    return err;
}



/****************
 * Get a "Set"-option, op ist SETOPT_xxxxx
 */

int GetSetOption( int opt )
{
    int r;

    switch( opt ) {
      case SETOPT_ABBREV:	r = setOptions.abbrev;	    break;
      case SETOPT_AUTOLOCK:	r = setOptions.autolock;    break;
      case SETOPT_LOCKCHECK:	r = setOptions.lockcheck;   break;
      case SETOPT_SEARCHCASE:	r = setOptions.searchcase;  break;
      case SETOPT_PUREASCII:	r = setOptions.pureascii;   break;
      case SETOPT_TABEXPAND:	r = setOptions.tabexpand;   break;
      case SETOPT_CRTSAVE:	r = setOptions.crtsave;     break;
      case SETOPT_BACKUP:	r = setOptions.backup;	    break;
      case SETOPT_BLANKSTRIP:	r = setOptions.blankstrip;  break;
      case SETOPT_APPCTRLZ:	r = setOptions.appctrlz;    break;
      case SETOPT_MEMSWAP:	r = setOptions.memswap;     break;
      case SETOPT_LOADUPD:	r = setOptions.loadupd;     break;
      case SETOPT_HISTORY:	r = setOptions.history;     break;
      case SETOPT_READBINARY:	r = setOptions.readbinary;  break;
      case SETOPT_SAVEMODE:	r = setOptions.savemode;    break;
      case SETOPT_TABMODE:	r = setOptions.tabmode;     break;
      case SETOPT_UNIXSAVE:	r = setOptions.unixsave;    break;
      case SETOPT_REFRESH:	r = setOptions.refresh;     break;
      case SETOPT_INTCLIP:	r = setOptions.intclip;     break;
      case SETOPT_LATIN:	r = setOptions.latin;	    break;
      case SETOPT_EDITCMD:	r = setOptions.editcmd;     break;
      default: Bug( "get: set option %d unkown", opt );
    }
    return r;
}


/****************
 * Set a "Set"-option, op ist SETOPT_xxxxx
 */

void PutSetOption( int opt, int val )
{
    switch( opt ) {
      case SETOPT_ABBREV:	setOptions.abbrev      = val; break;
      case SETOPT_AUTOLOCK:	setOptions.autolock    = val; break;
      case SETOPT_LOCKCHECK:	setOptions.lockcheck   = val; break;
      case SETOPT_SEARCHCASE:	setOptions.searchcase  = val; break;
      case SETOPT_PUREASCII:	setOptions.pureascii   = val; break;
      case SETOPT_TABEXPAND:	setOptions.tabexpand   = val; break;
      case SETOPT_CRTSAVE:	setOptions.crtsave     = val; break;
      case SETOPT_BACKUP:	setOptions.backup      = val; break;
      case SETOPT_BLANKSTRIP:	setOptions.blankstrip  = val; break;
      case SETOPT_APPCTRLZ:	setOptions.appctrlz    = val; break;
      case SETOPT_MEMSWAP:	setOptions.memswap     = val; break;
      case SETOPT_LOADUPD:	setOptions.loadupd     = val; break;
      case SETOPT_HISTORY:	setOptions.history     = val; break;
      case SETOPT_READBINARY:	setOptions.readbinary  = val; break;
      case SETOPT_SAVEMODE:	setOptions.savemode    = val & 7 ; break;
      case SETOPT_TABMODE:	setOptions.tabmode     = val & 1 ; break;
      case SETOPT_UNIXSAVE:	setOptions.unixsave    = val; break;
      case SETOPT_REFRESH:	setOptions.refresh     = val; break;
      case SETOPT_INTCLIP:	setOptions.intclip     = val; break;
      case SETOPT_LATIN:	setOptions.latin       = val; break;
      case SETOPT_EDITCMD:	setOptions.editcmd     = val; break;
      default: Bug( "put: set option %d unkown", opt );
    }
}

/****************
 * Returns a ptr to a statically allocated buffer, contaioning a verbal
 * descrition of the corresponding option setting, if a unkown option
 * is given, NULL will be returned.
 */

const char *GetSetOptionString( int opt )
{
  #if W_FULL
    static char buf[50];
    char nbuf[10], *u;
    static char t_on[] = "on", t_off[] = "off";
    const char *t;

    switch( opt ) {
      case SETOPT_ABBREV:
	t = "abbrev"; u = setOptions.abbrev ? t_on:t_off;
	break;
      case SETOPT_AUTOLOCK:
	t = "autolock"; u = setOptions.autolock ? t_on:t_off;
	break;
      case SETOPT_LOCKCHECK:
	t = "lockcheck"; u = setOptions.lockcheck ? t_on:t_off;
	break;
      case SETOPT_SEARCHCASE:
	t = "searchcase"; u = setOptions.searchcase ? "exact":"any";
	break;
      case SETOPT_PUREASCII:
	t = "pureascii"; u = setOptions.pureascii ? t_on:t_off;
	break;
      case SETOPT_TABEXPAND:
	t = "tabexpand"; u = setOptions.tabexpand ? t_on:t_off;
	break;
      case SETOPT_CRTSAVE:
	t = "crtsave"; sprintf( u = nbuf, "%d", setOptions.crtsave );
	break;
      case SETOPT_BACKUP:
	t = "backup"; sprintf( u = nbuf, "%d", setOptions.backup );
	break;
      case SETOPT_BLANKSTRIP:
	t = "blankstrip"; u = setOptions.blankstrip ? t_on:t_off;
	break;
      case SETOPT_APPCTRLZ:
	t = "appctrlz"; u = setOptions.appctrlz ? t_on:t_off;
	break;
      case SETOPT_MEMSWAP:
	t = "memswap"; u = setOptions.memswap ? t_on:t_off;
	break;
      case SETOPT_LOADUPD:
	t = "loadupd"; u = setOptions.loadupd ? t_on:t_off;
	break;
      case SETOPT_HISTORY:
	t = "history"; sprintf( u = nbuf, "%d", setOptions.history );
	break;
      case SETOPT_READBINARY:
	t = "readbinary"; u = setOptions.readbinary ? t_on:t_off;
	break;
      case SETOPT_SAVEMODE:
	t = "savemode"; sprintf( u = nbuf, "%d", setOptions.savemode );
	break;
      case SETOPT_TABMODE:
	t = "tabmode"; sprintf( u = nbuf, "%d", setOptions.tabmode );
	break;
      case SETOPT_UNIXSAVE:
	t = "unixsave"; u = setOptions.unixsave ? t_on:t_off;
	break;
      case SETOPT_REFRESH:
	t = "refresh"; sprintf( u = nbuf, "%d", setOptions.refresh );
	break;
      case SETOPT_INTCLIP:
	t = "intclip"; u = setOptions.intclip ? t_on:t_off;
	break;
      case SETOPT_LATIN:
	t = "latin"; u = setOptions.latin ? t_on:t_off;
	break;
      case SETOPT_EDITCMD:
	t = "editcmd"; u = setOptions.editcmd ? t_on:t_off;
	break;
      default: return NULL;
    }

    sprintf( buf, "set %s %s" , t, u );
    return buf;
  #else
    return NULL;
  #endif
}


static void ShowSetOption( int opt )
{
  #if W_FULL
    const char *p;
    if( p = GetSetOptionString(opt) )
	Write2CurCmdLine( p );
    else
	Bug( "show: set option %d unknown", opt );
  #endif
}



static int GetACharFromUser()
{
    int i;

    ShowMessageAsInfo( "Please type a character !" );
    while( !(i=GetKeyValue()) )
	;

    if( i<256)
	ShowMessage(NULL);
    else {
	i = ' ';
	ShowMessage("Not a character; blank used");
    }
    return i;
}



/****************
 * public Function to write a string to the current cmdLine
 */

void Write2CurCmdLine( const char * string )
{
    Write2CmdLine( QryScreenFile(), string );
}

/****************
 * Write a string to the cmdline of fileHd and set the Cursor of
 * the cmdline to the first position.
 */

static void Write2CmdLine( int fileHd, const char * string )
{
    char *p;
    ushort pos;

    p = GetPtr2CmdLine( fileHd, &pos );
    mem2str( p, string? string: "", MAX_LINELEN );
    MoveCmd2Pos( fileHd, 0 );
    write2CmdLinePending = 1;
}



#if W_FULL
int Cmd_Exit()
{
    int fhd, err, i, firstfhd, changed;
    unsigned flags;
    const char *p;

    err = changed = 0;
    firstfhd = fhd = QryScreenFile();
    do {
	FlushEditBuf( fhd );
	p = GetFileInfo( fhd, &flags );
	if( (flags & WFILE_INT) && !IsScratchFile( p ) )
	    ;
	else if( !(flags & WFILE_CHG) )
	    ;
	else
	    changed++;
	fhd = GetNextFileHandle( fhd );
    } while( firstfhd != fhd );
    if( changed ) {
	ShowMessage("Do you really want to exit and loose all changes ?"
						" Type y or n");
	for(;;) {
	    while( !(i=GetKeyValue()) )
		;
	    if( i > 255 )
		break;
	    else if( (i=toupper(i)) == 'Y' ) {
		changed = 0;
		break;
	    }
	    else if( i == 'N' )
		break;
	}
	ShowMessage(NULL);
    }

    if( !changed )
	ReleaseAllFiles();

    return err;
}
#endif


int Cmd_Quit()
{
    int  fHd, okay, err, i, nextfhd;
    unsigned flags;
    const char *p;

    err = okay = 0;
    fHd = QryScreenFile();
    FlushEditBuf( fHd );
    p = GetFileInfo( fHd, &flags );
    if( (flags & WFILE_INT) && !IsScratchFile(p) )
	okay++;
    else if( !(flags & WFILE_CHG) )
	okay++;
    else {
	ShowMessage("Do you really want to quit? Type y or n");
	for(;;) {
	    while( !(i=GetKeyValue()) )
		;
	    if( i > 255 )
		break;
	    else if( (i=toupper(i)) == 'Y' ) {
		okay++;
		break;
	    }
	    else if( i == 'N' )
		break;
	}
	ShowMessage(NULL);
    }

    if( okay ) {
      #if W_FULL
	if( QrySupplementaryScreenFile() == fHd )
	    BindSupplementaryScreen(-1); /* remove supplementary Window */
	if( toggleHd1 == fHd )
	    toggleHd1 = -1;
	if( toggleHd2 == fHd )
	    toggleHd2 = -1;
      #endif
	nextfhd = GetPrevFileHandle( fHd );
	ReleaseFile( fHd );
	if( nextfhd != fHd && GetFileCount() ) {  /*switch to next file */
	    GetFileInfo( nextfhd, &flags );
	    if( GetFileCount() > 1  && !(flags & WFILE_RING) )
		nextfhd = GetPrevFileHandle( nextfhd );
	    BindScreen( nextfhd );
	}
    }

    Try2AllocReservedMemory();

    return err;
}


static int DoCommit( void )
{
    int err=0, i, starthd;
    unsigned flags;
    cmd_t cmd;

    FlushEditBuf(-1); /* flush all Buffers */
    i = starthd = QryScreenFile();
    do {
	GetFileInfo(i, &flags );
	if( !(flags & WFILE_INT) &&
	    !(flags & WFILE_RO ) &&
	     (flags & WFILE_CHG) ) {

	    if( !(err = Cmd_Switch2FileHd( i )) ) {
		memset( &cmd, 0, sizeof cmd );
		cmd.cmd = CMD_SAVE;
		cmd.type = ARGTYPE_STRING;
		err = Cmd_Save( &cmd );
	    }
	}
	i = GetNextFileHandle( i );
    } while( !err && i != starthd );
    if( !err )
	err = Cmd_Switch2FileHd( starthd );

    return err;
}


/****************
 * This will be also be used from main() to load the files
 * from the commandline
 */

int Cmd_Edit( const char *string )
{
    return DoEdit( string, 0 );
}

int Cmd_ReEdit( const char *string )
{
    return DoEdit( string, 1 );
}

int Cmd_Load( const char *string )
{
    return DoEdit( string, 2 );
}

int Cmd_New( const char *string )
{
    return DoEdit( string, 3 );
}

int Cmd_View( const char *string )
{
    return DoEdit( string, 20 );
}


/* String may contain options, which are only applicable if a real read
 * occurs
 * -z	 = unused option
 * -fu	 = force reading standard format
 * -fd	 = same as fu
 * -fl	 = format of file is: length prefixed file
 * (-f<n> = with n is a digit: read with fixed record length)
 * -nt	 = no tab expanding
 * -nb	 = unused option
 * -s<n> = start with line/record n
 * -m<n> = maximum lines/records to load
 * -j0	 = ZEditPrcs werden nicht ausgefuehrt
 * -ji	 = tag lines with lineIds (always done for UNIX)
 * -jR	 = force reload
 * --	 = end of options
 */

static int
DoEdit( const char *filename, int mode	)
{
    int fHd, lastFHd, i, j, err, new, doload, doview;
    unsigned flags;
    const char *p, *s, *s2;
    char *string;
    char *buf;
    ushort tabArray[MAX_TABS], lmarg, rmarg, dmarg, reclen;
    ulong startline, maxlines;
    long hl;
    struct {
	int ctrlz:1;
	int upr:1;
	int fix:1;
	int bin:1;
	int notabs:1;
	int nobstrip:1;
	int noeditprc:1;
	int taglines:1;
	int reload:1;

	int trydef:1;
    } o;

    reclen=0; /*(only to avoid compiler warning) controlled by o.bin,o.fix */
    /* optionprocessing */
    startline = maxlines = 0; /* all */
    s = filename;
    s2 = NULL;
    string = NULL;
    memset( &o, 0, sizeof o );
    o.trydef = 1;
  #ifdef UNIX /* there is enough memory */
    o.taglines = 1;
  #endif
    buf = NULL;
    for(;;) {
	while( *s ) {
	    if( isspace( *s ) )
		s++;
	    else if( *s == '-' ) {
		o.trydef = 0;
		if( *++s == '-' ) {
		    s++;
		    while( isspace( *s ) )
			s++;
		    break; /* while loop */
		}
		for( ; *s && !isspace(*s); s++ ) {
		    switch( *s ) {
		      case 'z' : o.ctrlz++; break;
		      case 'F' :
		      case 'f' :
			switch( s[1] ) {
			  case 'd' : s++; break; /* dummy option */
			  case 'u' : s++; break; /* dummy option */
			  case 'l' : o.upr++ ; s++;break;
			  case 'b' : o.bin++ ; s++;
			    GetScreenSize( &i, &j ); /* get def reclen*/
			    reclen = i;
			    /* fall thru */
			  default:
			    if( isdigit( s[1] ) ) {
				if( !o.bin )
				    o.fix++;
				s++;
				i = atoi( s ) ;
				if( i < 1 || i >= MAX_LINELEN ) {
				    if( buf )
					FreeVarValue( buf );
				    err = ERR_INVRLEN;
				    goto failure;
				}
				reclen = i;
				while( isdigit(*s) )
				    s++ ;
				s--;
			    }
			    break ;
			}
			break;
		      case 'n' :
			switch( s[1] ) {
			  case 't' : o.notabs=1; s++; break;
			  case 'b' : o.nobstrip++; s++; break;
			}
			break;
		      case 's': /* startline */
			s++;
			hl = atol( s );
			startline =  hl < 1 ? 0 : hl;
			while( isdigit(*s) )
			    s++ ;
			s--;
			break;
		      case 'm': /* maxlines */
			s++;
			hl = atol( s );
			maxlines =  hl < 1 ? 0 : hl;
			while( isdigit(*s) )
			    s++ ;
			s--;
			break;
		      case 'j':
			s++;
			switch( *s ) {
			  case '0': o.noeditprc = 1; break;
			  case 'i': o.taglines = 1; break;
			  case 'R': o.reload = 1; break;
			  default:
			    if( buf )
				FreeVarValue( buf );
			    err = ERR_INVOPT;
			    goto failure;
			}
			break;
		      default:
			if( buf )
			    FreeVarValue( buf );
			err =  ERR_INVOPT;
			goto failure;
		    }
		}
	    }
	    else
		break;	/* while loop */
	}
	if( !s2 )
	    s2 = s;
	if( o.trydef ) {  /* try default options */
	    o.trydef = 0; /* but not next time */
	    s = buf = VarValueStr( "ZDefEditOpt", ARGTYPE_VAR );
	}
	else
	    break;
    }
    if( buf )
	FreeVarValue( buf );
    /* wir brauchen eventuell einen Stringbuffer von min. 10 bytes */
    string = xmalloc(strlen(s2)+10);
    strcpy( string, s2 );
    /* end optionprocessing */

    /* we have to convert backslashes to slashes at this point */
  #if MSDOSFILESYSTEM
    {	char *q;
	for(q=string; *q; q++ )
	    if( *q == '\\' )
		*q = '/';
    }
  #endif

    new = err = doload= doview = 0;
    if( mode == 20 ) {
	mode = 2;
	doview = 1;
    }
    lastFHd = fHd = QryScreenFile();
    FlushEditBuf( fHd );
    if( !*string ) { /* get next file in ring */
	if( mode == 3 ) {/* new */
	    /* try to find an unused number */
	    j = 0;
	    do {
		p = GetFileInfo( fHd, &flags );
		if( flags & WFILE_INT ) {
		    if( i = IsScratchFile( p )) {
			if( j && i > (j+1) )
			    break; /* do while loop */
			if( i > j )
			    j = i;
		    }
		}
		fHd = GetNextFileHandle( fHd );
	    } while( fHd != lastFHd );
	    fHd = lastFHd;
	    sprintf(string, ".%d", j + 1 );
	    doload++;
	}
	else if( mode )
	    err = ERR_MISNAM;
	else
	    do {
		fHd = GetNextFileHandle( fHd );
		GetFileInfo( fHd, &flags );
	    } while( fHd != lastFHd && !(flags & WFILE_RING) );
    }
    else if( (fHd=GetFileHandle( string )) != -1 ) {
	if( mode == 3 )
	    err = ERR_FINUSE;
	else if( !o.reload && (!mode || mode == 2) ) {
	    /* file is in editor, switch to this file */
	    if( !FileCmpName( string, ".unnamed" ) )
		SetFileFlag( fHd, WFILE_RING );
	}
	else {
	  #if W_FULL
	    if( QrySupplementaryScreenFile() == fHd )
		BindSupplementaryScreen(-1); /* remove supplementary Window */
	    if( toggleHd1 == fHd )
		toggleHd1 = -1;
	    if( toggleHd2 == fHd )
		toggleHd2 = -1;
	  #endif
	    ReleaseFile( fHd );
	    doload++;
	}
    }
    else
	doload++;
    if( doload ) {
	err = AllocFile( &fHd, string,
			    (mode == 3 ? 3 : mode == 2 ? 2 :
			     (GetSetOption( SETOPT_LOADUPD )? 1:0)) |
			    (o.notabs ? (1<<4):0) |
			    (o.taglines? (1<<5):0) |
			    (doview    ? (1<<6):0) ,
		o.upr ? WFILE_FMT_2PR :
		o.fix ? WFILE_FMT_FIX :
		o.bin ? WFILE_FMT_BIN :
			WFILE_FMT_STD,	(o.fix||o.bin) ? reclen : 0 ,
			startline, maxlines );
	if( !err ) {
	    /* set tabstops */
	    if( GetFileCount() == 1 ) { /* this is the first file */
		for(i=0; i < MAX_TABS; i++ ) /* so set to default */
		    tabArray[i] = i*8+8;
		    dmarg = lmarg = 0;
		    rmarg = MAX_LINELEN-1;
	    }
	    else { /* get from active file */
		i = GetFilesTabSettings( lastFHd, tabArray, DIM(tabArray) );
		GetFilesMargins( lastFHd, &lmarg, &rmarg, &dmarg );
	    }
	    SetFilesTabSettings( fHd, tabArray, i );
	    SetFilesMargins( fHd, lmarg, rmarg, dmarg );
	    new++;
	}
    }

    if( !err ) {
	BindScreen( fHd );
	if( new && !o.noeditprc ) {
	  #if W_FULL
	    err = Cmd_ExecuteString( "ZEditPrc", ARGTYPE_VAR );
	    if( !err && *(p=GetFilesExt( fHd )) ) {
		buf = xmalloc( 9 + strlen(p) + 1 );
		strcpy( buf, "ZEditPrc_" );
		strcat( buf, p );
		strlwr( buf+9); /* extension soll immer lowercase sein */
		err = Cmd_ExecuteString( buf, ARGTYPE_VAR );
		free(buf);
	    }
	  #endif
	}
    }

  failure:
    free(string);
    return err;
}



/****************
 * This function switchs to the named file.
 */

int Cmd_Switch2NamedFile( const char *name )
{
    int err=0;

    if( GetFileHandle( name ) == -1 )
	err = ERR_NOFILE;
    else
	err = Cmd_Edit( name );
    return err;
}


int Cmd_Switch2FileHd( int fhd )
{
    unsigned flags;
    return Cmd_Load( GetFileInfo( fhd, &flags ) );
}


int ExecuteCmdLine()
{
    int err, fileHd;
    ushort pos;
    char *p;

    err = 0;
    fileHd = QryScreenFile();
    write2CmdLinePending = 0;
    err = Cmd_Execute();
    p = GetPtr2CmdLine( fileHd, &pos );
    Add2History(p);
    if( !write2CmdLinePending ) {
	if( p ) {  /* at this point the file may be released */
	    *p = '\0';
	    MoveCmd2Pos( fileHd, 0 );
	}
    }
    write2CmdLinePending = 0;
    return err;
}



/****************
 * Den Inhalt des CmdlineBuffer ausfuehren, diesen buffer aber nicht
 * loeschen.
 */

static int Cmd_Execute()
{
    cmd_t *cmd;
    int fHd, err;
    char *p;
    ushort pos;

    err = 0;
    fHd = QryScreenFile();
    p = GetPtr2CmdLine( fHd, &pos );
    if( strlen(p) ) {
	if( !(err=ParseCmdLine( p, &cmd ) ) ) {
	    if( cmd ) {
		if( cmd->cmd == CMD_LOCATE || cmd->cmd == CMD_CHANGE )
		    write2CmdLinePending=1; /* do not delete line */
		err = ExecTempCmd( cmd );
	    }
	}
    }
    Try2AllocReservedMemory();
    return err;
}



/****************
 * Den Inhalt eines strings oder var ausfhren
 */

int Cmd_ExecuteString( const char *string, int argType )
{
    cmd_t *cmd;
    int err;
    char *varValue;

    err = 0;
    if( strlen(string) ) {
	if( argType == ARGTYPE_STRING )
	    err=ParseCmdLine( string, &cmd );
	else {
	    varValue = VarValueStr( string, argType );
	    if( *varValue )
		err=ParseCmdLine( varValue, &cmd );
	    else
		cmd = NULL;
	    FreeVarValue( varValue );
	}
	if( !err && cmd )
	    err = ExecTempCmd( cmd );
    }
    Try2AllocReservedMemory();
    return err;
}




/****************
 *  den string str in die Commandline History anhaengen
 */

static void Add2History( const char *str )
{
  #if W_FULL
    int i;

    if( historyDisabled )
	return;
    if( !historySize || !str )
	return;
    if( !strlen(str) )
	return;
    xassert( history );
    for(i=0; i < historySize; i++ )
	if( !history[i] )
	    break;
    if( i < historySize ) { /* enough room, add to list */
	history[i] = xstrdup(str);
	historyIdx = i;
    }
    else {  /* table full, free the first, and shift the rest */
	free( history[0] );
	for( i=1; i < historySize; i++ )
	    history[i-1] = history[i];
	/* and insert the new one */
	history[i-1] = xstrdup(str);
	historyIdx = i-1; /* make this the new position */
    }
  #endif
}



/****************
 * Initialize the commandline history system
 * number of elements will be taken from setOptions.history
 * This must be called, every time after setOptions.history is changed.
 * it also clear the history buffer;
 */

static void InitHistory()
{
  #if W_FULL
    int i;

    if( historySize ) { /* clear all */
	xassert(history);
	for(i=0; i < historySize; i++ )
	    free( history[i] );
	FREE(history);
    }
    xassert( !history );
    if( setOptions.history > 50 )
	setOptions.history = 50;
    historySize = setOptions.history;
    historyIdx = 0;
    if( historySize )
	history = xcalloc( historySize, sizeof *history );
  #endif
}


/****************
 * Die CommandLine durch die history ersetzen
 * mode 0 = durch den letzten eintrag und dann zurueck
 *	!0 = durch den naechsten Eintrag  und dann vor
 */

static void CopyFromHistory( int mode )
{
  #if W_FULL
    if( !history || !historySize )
	return;
    xassert( historyIdx < historySize );
    if( !mode ) {
	Write2CurCmdLine( history[historyIdx] );
	if( historyIdx )
	    historyIdx--;
    }
    else {
	if( historyIdx+1 < historySize ) {
	    if( history[historyIdx+1] ) {
		historyIdx++;
		Write2CurCmdLine( history[historyIdx] );
	    }
	    else
		Write2CurCmdLine( "" );
	}
	else
	    Write2CurCmdLine( "" );
    }
  #endif
}


/****************
 * ffne den file: .history und schreibe die History dorthinein.
 */

int Cmd_History(void)
{
    int err=0, i, fhd;

    if( !(err=Cmd_Edit(".history")) ) {
	fhd = QryScreenFile();
	ResetFileFlag( fhd, WFILE_RO );
	DeleteAllLines( fhd );
	for(i=0; i < historySize && i <= historyIdx; i++ )
	    if( history[i] )
		InsertLine( fhd, history[i], strlen(history[i]) );
	SeekLine( fhd, 0 );
	if( GetFileTotLines( fhd ) > 1 )
	    DeleteLine( fhd );
	Move2Pos( fhd, 0, 0 );
	ResetFileFlag( fhd, WFILE_CHG );
	SetFileFlag( fhd, WFILE_RO );
    }
    return err;
}


/****************
 * Returns: 0 if value is a special number
 */

static int CheckSpecialHistory( int value )
{
    switch( value ) {
      case 111: CheckFileBuffers(); return 0;
      default: return -1;
    }
}


void EnableHistoryUpdate(int onoff)
{
    historyDisabled = !onoff;
}


/*** bottom of file ***/
