/* [databuf.c wk 1.5.92] Scrollable Data Buffers
 *	Copyright (c) 1988-93 by Werner Koch (dd9jn)
 *  This file is part of WkLib.
 *
 *  WkLib is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WkLib is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 *
 * Funktionen um Daten in scrollbarer Form zu speichern; diese koenne mit
 * einem oder mehreren Ausgabemedien gekoppelt werden. Solle der Implementation
 * von Tracebuffer, Terminals u.a. dienen
 * Namen der Funktion fangen mit 'DaBf' an.
 *
 * History:
 * 03.05.93 wk	OS/2 support und jetzt in wk1-lib
 */

#include <wk/tailor.h>
RCSID("$Id: databuf.c,v 1.7 1995/03/08 16:54:19 wk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>
#include <string.h>
#include <wk/lib.h>
#include <wk/string.h>
#if OS2
    #define INCL_DOSPROCESS 1
    #include <os2.h>
#endif

#include <wk/dabf.h>

/***** constants *******/
#define MAX_BUFFERS 10
#define MAX_NOTIFIES 5
#define DEFAULT_LINES 50

static const dabfAttr_t defaultAttr = { 7 };

/**** types ************/
typedef struct {
	ushort	size;		/* number of array elements (0 = unused slot)*/
	char	**lines;	/* Array of ptrs to strings */
	dabfAttr_t  **attr;	/* kann auf NULL sein */
	int    type;
	unsigned flags;
	const void *appData;	/* pointer to private data of Application */
	union {
	  struct {
	    ushort  next;	/* index for next write */
	    ushort  last;	/* index of  last write */
	    char    incomplete; /* line not yet complete */
	  } std;	    /* used for buffer type STD */
	  struct {
	    ushort rows, cols;
	    ushort x,y;       /* current cursor position */
	    dabfAttr_t a;     /* current attribute */
	    ushort val1,val2;
	    ushort savex,savey;
	    int state;
	  } ansi;	    /* used for buffer type ANSI */
	} t;
    } buffer_t;

typedef struct {
	void *p;		/* pointer (NULL = unused slot)*/
	byte count;		/* no of locks on this pointer */
	byte freeFlag;		/* true, free this pointer */
    } ptrLock_t;

/**** global vars ******/
static void (*notifyTbl[MAX_NOTIFIES])(int,int,...);
static buffer_t bufferTbl[MAX_BUFFERS];
static ptrLock_t *ptrLockTbl;
static size_t	 ptrLockTblSize;
static char *printfBuffer;

/****** protos *********/
static int PutString( int hd, const char *string, int mode );
static void PutLines( int hd, char *string );
static void Notify( int hd, int reason );
static void Notify2( int hd, int reason,
		      ushort x, ushort y, dabfAttr_t a, byte c );
static void *LockPtr( void *p );
static void UnlockPtr( void *p );
static void FreePtr( void *p );
static void PutAnsiChar( int hd, buffer_t *buf, int c );
static void SetAnsiAttr( buffer_t *buf, ushort val );
static void ScrollAnsi( buffer_t *buf );
/***** functions *******/

#ifdef __IBMC__     /* ist immer OS/2 */
  #ifdef  __MULTI__	/* brauchen wir aber nur bei multi threaded */
    #define ENTERCRIT() DosEnterCritSec()
    #define LEAVECRIT() DosExitCritSec()
  #else
    #define ENTERCRIT()
    #define LEAVECRIT()
  #endif
#else
  #if OS2
    #define ENTERCRIT() DosEnterCritSec()
    #define LEAVECRIT() DosExitCritSec()
  #else
    #define ENTERCRIT()
    #define LEAVECRIT()
  #endif
#endif


int DaBfGetInfo( int hd, int *type, unsigned *flags, long *size )
{
    buffer_t *buffer;

    if( hd < 0 || hd >= MAX_BUFFERS )
	return E_DABF_INVHD;
    buffer = bufferTbl + hd;
    if( !buffer->size )
	return E_DABF_NOHD;
    *type = buffer->type;
    *flags= buffer->flags;
    if( buffer->type == DABF_T_ANSI )
	*size= ((buffer->t.ansi.rows<<8) | (buffer->t.ansi.cols&0xff)) & 0xfff;
    else
	*size=buffer->size;
    return 0;
}


/****************
 * Einen Ptr auf beliebige Daten speichern , wenn NULL angegeben, so werden
 * die Daten nicht gespeichert, sondern der alte Ptr zurueckggegeben
 * Returns: NULL wenn keine Daten gespeichert oder Handle nicht gueltig ist.
 *	    gespeicherter Ptr
 */

void *DaBfAppData( int hd, const void *appData )
{
    buffer_t *buffer;
    const void *old;

    if( hd < 0 || hd >= MAX_BUFFERS )
	return NULL;
    buffer = bufferTbl + hd;
    if( !buffer->size )
	return NULL;
    old = buffer->appData;
    if( appData )
	buffer->appData = appData;
    return (void*)old;
}



/****************
 * Eine Notify Funktion registrieren, diese wird bei verschieden
 * Ereignissen aufgerufen mit den Paramtern: Handle des Events und Reason
 * Returns: -1 = Error: Too many Notify Functions
^*	    sonst : Identifier fuer deregister
 */

int DaBfRegisterNotify( void (*fnc)(int,int,...) )
{
    int i;

    for(i=0; i < MAX_NOTIFIES; i++ )
	if( !notifyTbl[i] ) {
	    notifyTbl[i] = fnc ;
	    return i;
	}
    return -1;	/* too many notify functions */
}

void DaBfDeRegisterNotify( int hd )
{
    xassert( hd >=0 && hd < MAX_NOTIFIES );
    notifyTbl[hd] = NULL;
}



/****************
 * Erzeugen eine Databuffers
 * Gibt einen Handle fuer die weiteren Funktion zurueck oder eine Fehlercode
 * Parameter sind:  - Art des Databuffers
 *		      DABF_T_STD:
 *			einfache Zeichen in Zeilenform ohne Attribute
 *			in Scrollform; d.h. die ersten Zeilen werden durch
 *			die folgenden ersetzt, sobald die max. Zeilenzahl
 *			erreicht ist.
 *		      DABF_T_ANSI:
 *			ANSI Terminal like Buffer
 *		    - Flags je nach Databuffer
 *		      DABF_F_ATTR : Attribute werden unterstuetzt
 *			Dies ist z.Z. nur fuer DABF_T_ANSI vollstaendig
 *			implementiert
 *		    - einen Sizeparameter  (oder 0 fuer default Werte)
 *		      Bei DABF_T_STD: max. Anzahl von Zeilen  (min. 3, def. 50)
 *		      Bei DABF_T_ANSI: Byte 0: Anzahl Spalten (min. 5, def. 80)
 *				       Byte 1: Anzahl Zeilen  (min. 3, def. 25)
 * Returns: 0 = Okay oder Errorcode
 */

int DaBfCreate( int *retHd, int type, unsigned flags, long size )
{
    int i, j;
    ushort cols=0, rows;  /* avoid compiler warning about uninitialized use */
    buffer_t *buffer;
    char *p;
    dabfAttr_t *ap;

    if( !ptrLockTbl ) { /* allocate the table at the first call */
	ptrLockTblSize = 100;
	ptrLockTbl = xcalloc( ptrLockTblSize, sizeof *ptrLockTbl );
	printfBuffer = xmalloc(500);
    }

    if( type == DABF_T_STD ) {
	if( !size )
	    size = DEFAULT_LINES;
	if( size < 3 || size >= 0xffff )  /* must be less than ffff ! */
	    return E_DABF_INVARG;
	rows = size;
    }
    else if( type == DABF_T_ANSI ) {
	if( !size )
	    size = 25 << 8 | 80;
	cols = size & 0xff;
	rows = (size >> 8)&0xff;
	if( cols < 5 || rows < 3 )
	    return E_DABF_INVARG;
    }
    else
	return E_DABF_INVARG;

    ENTERCRIT();
    for(i=0; i < MAX_BUFFERS; i++ )
	if( !bufferTbl[i].size ) {
	    (buffer = bufferTbl+i)->size = rows;
	    LEAVECRIT();
	    *retHd = i; /* return handle */
	    buffer->type = type;
	    buffer->flags= flags;
	    buffer->appData = NULL;
	    if( !(buffer->lines = calloc( rows, sizeof *buffer->lines)) ) {
		buffer->size = 0;
		return E_DABF_NOMEM;
	    }
	    if( flags & DABF_F_ATTR ) {
		if( !(buffer->attr = calloc( rows, sizeof *buffer->attr)) ) {
		    free( buffer->lines );
		    buffer->size = 0;
		    return E_DABF_NOMEM;
		}
	    }
	    else
		buffer->attr = NULL;
	    if( type == DABF_T_ANSI ) {
		buffer->t.ansi.state = 0;
		buffer->t.ansi.rows = rows;
		buffer->t.ansi.cols = cols;
		buffer->t.ansi.x = buffer->t.ansi.y = 0;
		buffer->t.ansi.savex = buffer->t.ansi.savey = 0;
		buffer->t.ansi.a = defaultAttr;
		for(i=0; i < rows; i++ ) {
		    buffer->lines[i] = p = xmalloc( cols );
                    memset(p, ' ', cols );
		    buffer->attr[i]=ap=xcalloc( cols, sizeof (dabfAttr_t));
		    for(j=0; j < cols; j++, ap++ )
			*ap = buffer->t.ansi.a; /* white */
		}
	    }
	    else {
		buffer->t.std.next = 0; /* start at beginning */
		buffer->t.std.incomplete = 0;
	    }
	    Notify(*retHd, DABF_N_CREATE );
	    return 0;
	}
    LEAVECRIT();
    return E_DABF_NOHD; /* out of handles */
}


/****************
 *  Release a DataBuffer and all used resources
 */

int DaBfClose( int hd )
{
    buffer_t *buffer;
    ushort  n;

    if( hd < 0 || hd >= MAX_BUFFERS)
	return E_DABF_INVHD;
    buffer = bufferTbl + hd;
    if( !buffer->size )
	return E_DABF_NOHD;
    for( n=0; n < buffer->size ; n++ )
	FreePtr( buffer->lines[n] );
    free( buffer->lines );
    if( buffer->attr ) {
	for( n=0; n < buffer->size ; n++ )
	    FreePtr( buffer->attr[n] );
	free( buffer->attr );
    }
    buffer->size = 0;
    Notify(hd, DABF_N_CLOSE );
    return 0;
}



/****************
 * Retrieve a block of lines from a buffer.
 * Caller must pass an array that will get ptrs to
 * the lines, arraySize is the size of that array, This function
 * will place ptrs to the lines into that array, end of lines will be
 * marked with a NULL Ptr, so the maximum # of lines caller will get
 * back is one less than arraySize. The referenced Ptrs must be freed
 * with a call to DaBfFreePtrs !.
 * Mode = 0 : offset is from begin of buffer
 *	  1 : offset is from end of buffer
 * Offset gives the linenumber (starting with 0 for the first line)
 */

int DaBfGetLines( int hd, const char **retArray, ushort arraySize,
		       int mode, long offset )
{
    buffer_t *buffer;
    ushort  next, size, start, end;
    ushort index;
    const char *p;

    xassert( hd >= 0 && hd < MAX_BUFFERS);
    xassert( arraySize );
    buffer = bufferTbl + hd;
    if( buffer->type == DABF_T_ANSI )
	return DaBfGetALines( hd, retArray, NULL, arraySize, mode, offset );


    next = buffer->t.std.next;
    size = buffer->size;
    xassert( buffer->size );
    end = 0;
    if( !next ) {
	if( buffer->lines[0] )
	    start = 1, end = size-1; /* turnaround had occured: */
	else
	    start = size; /* speical marker for: no lines in buffer */
    }
    else if( next+1 < size ) { /* next between 1 and prelast element*/
	if( buffer->lines[next+1] )   /* buffer is full */
	    start = next+1;
	else
	    start = 0;
	end = next-1;
    }
    else { /* next < size */
	start = 0, end = next-1;    /* next is at last element */
    }


    arraySize--;    /* leave some room to append a NULL Ptr */
    index = 0;
    if( mode ) {    /* return in reverse order */
	if( start == size )  /* no data is in buffer */
	    ;
	else if( start <= end ) { /* thats simple */
	    /* start is only used as a counter; easier to compare unsigneds */
	    for(next=end; start <= end && index < arraySize;
						     start++, next-- )
		if( offset )
		    offset--;
		else
		    retArray[index++] = LockPtr( buffer->lines[next]) ;
	}
	else {	/* start > end */
	    for(next = end, start=0; start <= end && index < arraySize;
						     start++, next-- )
		if( offset )
		    offset--;
		else
		    retArray[index++] = LockPtr( buffer->lines[next]) ;
	    for(next =size-1; start < size && index < arraySize;
						     start++, next-- )
		if( offset )
		    offset--;
		else
		    retArray[index++] = LockPtr( buffer->lines[next]) ;
	}
	/* reverse the array */
	if( index > 1 ) /* reversing a one element array would be funny */
	    for( start=0, end=index-1; start < end; start++, end-- ) {
		 p = retArray[start];
		 retArray[start] = retArray[end];
		 retArray[end] = p;
	    }
    }
    else {
	if( start == size )  /* no data is in buffer */
	    ;
	else if( start <= end ) { /* thats simple */
	    for(; start <= end && index < arraySize; start++ )
		if( offset )
		    offset--;
		else
		    retArray[index++] = LockPtr( buffer->lines[start]) ;
	}
	else {	/* start > end */
	    for(; start < size && index < arraySize; start++ )
		if( offset )
		    offset--;
		else
		    retArray[index++] = LockPtr( buffer->lines[start]) ;
	    for(start=0; start <= end && index < arraySize; start++ )
		if( offset )
		    offset--;
		else
		    retArray[index++] = LockPtr( buffer->lines[start]) ;
	}
    }
    retArray[index] = NULL; /* append the NULL Ptr */

    return 0;
}



/****************
 * Wie Oben, es kann aber ein Array angegeben werden, welches
 * dann Ptr auf die Attribute erhaelt ( Oder NULL wenn nicht benoetigt )
 * Falls im AttributArray ein NULL pointer eingetragen ist, so
 * ist kein Attribut verfuegbar und es sollte das Standard attribut
 * benutzt werden.
 */

int DaBfGetALines( int hd, const char **retArray,
		   const dabfAttr_t **retAttr, ushort arraySize,
		       int mode, long offset )
{
    buffer_t *buffer;
    ushort index,n;

    xassert( hd >= 0 && hd < MAX_BUFFERS);
    xassert( arraySize );
    buffer = bufferTbl + hd;
    if( buffer->type != DABF_T_ANSI ) {
	DaBfGetLines( hd, retArray, arraySize, mode, offset );
	if( retAttr )
	    for( index=0; index < arraySize; index++ )
		retAttr[index] = NULL; /* use default attribute */
    }
    xassert( !mode );  /* offset from end not yet supported */

    arraySize--;    /* leave some room for terminating NULL Ptr */
    index = 0;
    for(n=offset; n < buffer->t.ansi.rows && index < arraySize; n++, index++)
	retArray[index] = LockPtr( buffer->lines[n]) ;
    retArray[index] = NULL; /* append the NULL Ptr */
    if( retAttr ) {
	index = 0;
	for(n=offset; n < buffer->t.ansi.rows && index<arraySize; n++, index++)
	    retAttr[index] = buffer->attr ? LockPtr( buffer->attr[n]):NULL ;
	retAttr[index]	= NULL; /* append the NULL Ptr */
    }

    return 0;
}



void DaBfUnlockPtrs( const void **array )
{
    for(; *array ; array++ )
	UnlockPtr( (void*)*array );
}


/****************
 * Put a new Line to the end of the Buffer
 */

int DaBfPutString( int hd, const char *string )
{
    return PutString( hd, string, 0 );
}


/****************
 * Ein einzelnes Zeichen ausgeben, funktioneier nur fr ANSI-Buffer
 */

int DaBfPutChar( int hd, int c)
{
    buffer_t *buffer;

    buffer = bufferTbl + hd;
    if( buffer->type == DABF_T_ANSI ) {
	PutAnsiChar( hd, buffer, c );
	Notify(hd,DABF_N_DATA);
    }
    return 0;
}


/****************
 * Nur auf ANSI-Terminal aktiv: String ausgeben und dabei Ctrl-Zeichen
 * die durch '^' gekennzeichnet sind uebersetzen; '^' selbst kann durch
 * Verdopplung ausgegeben werden. Der String sollte den CtrlString vollstaendig
 * enthalten, da sonst keine Ausgabe des Ctrl-Zeichens erfolgt (z.Zt jedenfalls)
 * Leere String werden ignoriert
 */

int DaBfPutCtrlString( int hd, const char *str)
{
    buffer_t *buffer;
    int esc;

    buffer = bufferTbl + hd;
    if( buffer->type == DABF_T_ANSI && *str ) {
	for(esc=0 ; *str; str++ )
	    if( esc ) {
                PutAnsiChar( hd, buffer, *str=='^'? '^':(*str&0x1f) );
		esc = 0;
	    }
            else if( *str == '^' )
		esc = 1;
	    else
		PutAnsiChar( hd, buffer, *str );
	Notify(hd,DABF_N_DATA);
    }
    return 0;
}


/****************
 *  mode 0 = complete line
 *	 1 = mark line for later append
 *	 2 = append to line and end line
 */

static int PutString( int hd, const char *string, int mode )
{
    buffer_t *buffer;
    char *line=NULL,*copy; /* avoid compiler warning about uninitialized use */

    xassert( hd >= 0 && hd < MAX_BUFFERS);
    buffer = bufferTbl + hd;
    xassert( buffer->size );
    if( buffer->type == DABF_T_ANSI ) {
	while( *string )
	    PutAnsiChar( hd, buffer, *string++ );
	Notify(hd,DABF_N_DATA);
	return 0;
    }


    if( mode && buffer->t.std.incomplete ) {
	if( *string ) { /* This is for better performance */
	    line = buffer->lines[buffer->t.std.last];
	    if( !(copy = xmalloc( strlen(line) + strlen(string) + 1 )) )
		return E_DABF_NOMEM;
	    strcpy( copy, line );
	    strcat( copy, string );
	    buffer->lines[buffer->t.std.last] = copy;
	}
	else
	    copy = NULL; /* to inhibit FreePtr for line */
    }
    else {
	line = buffer->lines[buffer->t.std.next];
	if( !(copy = strdup( string )))
	    return E_DABF_NOMEM;
	buffer->lines[buffer->t.std.last = buffer->t.std.next] = copy;
    }

    if( mode != 1 ) {
	buffer->t.std.incomplete = 0;
	if( ++buffer->t.std.next >= buffer->size )
	    buffer->t.std.next = 0;   /* turnaround */
    }
    else
	buffer->t.std.incomplete = 1;

    if( line && copy )
	FreePtr(line); /* remove old contents */
    Notify(hd,DABF_N_DATA);

    return 0;
}



int DaBfPrintf( int hd, const char *string, ... )
{
    va_list arg_ptr ;

    va_start( arg_ptr, string ) ;
    vsprintf(printfBuffer,string,arg_ptr) ;
    va_end( arg_ptr );
    PutLines( hd, printfBuffer );
    return 0;
}


int DaBfVprintf( int hd, const char *string, va_list arg_ptr )
{
    vsprintf(printfBuffer,string,arg_ptr) ;
    PutLines( hd, printfBuffer );
    return 0;
}



/****************
 * Put a string to the Databuffer, thereby splitting it
 * into lines.
 */

static void PutLines( int hd, char *string )
{

    char *p;

    while( (p = strchr( string, '\n' )) ) {
        *p = '\0' ;
	PutString( hd, string, 2 );
        *p = '\n' ;
	string = p+1;
    }
    if( *string )   /* a little optimization */
	PutString( hd, string, 1 );
}


/****************
 * Put a Pointer into the Lock Table
 */

static void *LockPtr( void *p )
{
    size_t i;

    if( !p )
	return p;
    for(i=0; i < ptrLockTblSize; i++ )
	if( ptrLockTbl[i].p == p ) {
	    if( ptrLockTbl[i].count == 255 )
                Bug("DaBf: PtrLockCount overflow");
	    ptrLockTbl[i].count++;
	    return p;
	}
    /* no locks yet - insert a new lock entry */
    for(i=0; i < ptrLockTblSize; i++ )
	if( !ptrLockTbl[i].p ) {
	    ptrLockTbl[i].p = p;
	    ptrLockTbl[i].count = 1;
	    ptrLockTbl[i].freeFlag = 0;
	    return p;
	}
    Bug("DaBf: ptrLockTbl is full");
    /*NOTREACHED*/ return 0;
}


/****************
 * Unlock a pointer or free him if its count is already 0
 */

static void UnlockPtr( void *p )
{
    size_t i;

    if( !p )
	return;
    for(i=0; i < ptrLockTblSize; i++ )
	if( ptrLockTbl[i].p == p ) { /* pointer found */
	    if( !ptrLockTbl[i].count )
                Bug("DaBf: UnlockPtr detected an lockcount of 0");
	    if( !--ptrLockTbl[i].count ) {
		if( ptrLockTbl[i].freeFlag )
		    free( ptrLockTbl[i].p );
		ptrLockTbl[i].p = NULL; /* remove from locktable */
	    }
	    return;
	}
    Bug("DaBf: Try to unlock a non locked ptr");
}



/****************
 * Free a pointer if its not in the lockTable or
 * decrement the lockcount
 */

static void FreePtr( void *p )
{
    size_t i;

    for(i=0; i < ptrLockTblSize; i++ )
	if( ptrLockTbl[i].p && ptrLockTbl[i].p == p ) {
	    if( ptrLockTbl[i].freeFlag )
                Bug("DaBf: FreePtr detected an already freed ptr");
	    ptrLockTbl[i].freeFlag = 1;
	    return; /* UnlockPtr will free him, if count is 0 */
	}
    free(p);
}


/****************
 * Notify all registered functions about an event
 * only used for standard notifies
 */

static void Notify( int hd, int reason )
{
    int i;

    for(i=0; i < MAX_NOTIFIES; i++ )
	if( notifyTbl[i] )
	    (notifyTbl[i])(hd, reason);
}


/****************
 * Use it for terminal events of type
 */

static void Notify2( int hd, int reason,
		     ushort x, ushort y, dabfAttr_t a, byte c )
{
    int i;

    for(i=0; i < MAX_NOTIFIES; i++ )
	if( notifyTbl[i] )
	    (notifyTbl[i])(hd, reason, x, y, a, c);
}




static void PutAnsiChar( int hd, buffer_t *buf, int c )
{
    ushort val1,val2;
    int i,j, more;
    dabfAttr_t *ap;

    switch( buf->t.ansi.state ) {
      case 0:
	if( c == 0x1b ) {
	    buf->t.ansi.state = 1;
	    buf->t.ansi.val1 = buf->t.ansi.val2 = 0;
	}
        else if( c == '\r' )
	    buf->t.ansi.x = 0;
        else if( c == '\n' ) {
	    if( buf->t.ansi.y+1 < buf->t.ansi.rows )
		buf->t.ansi.y++;
	    else {
		ScrollAnsi( buf );
		Notify(hd, DABF_N_SCROLL);
	    }
	}
	else { /* output */
	    val1= buf->t.ansi.x;
	    val2= buf->t.ansi.y;
	    buf->lines[val2][val1] = (byte)c;
	    if( buf->attr )
		buf->attr[val2][val1] = buf->t.ansi.a;
	    Notify2( hd, DABF_N_CHAR, val1, val2, buf->t.ansi.a, (byte)c );
	    if( val1+1 < buf->t.ansi.cols )
		buf->t.ansi.x++;
	    else {
		buf->t.ansi.x = 0;
		if( val2+1 < buf->t.ansi.rows )
		    buf->t.ansi.y++;
		else {
		    ScrollAnsi( buf );
		    Notify(hd, DABF_N_SCROLL);
		}
	    }
	}
	break;

      case 1:
        if( c == '[' )
	    buf->t.ansi.state = 2;
	else
	    buf->t.ansi.state = 0;
	break;

      case 2:
	if( isdigit(c) )
            buf->t.ansi.val1 = buf->t.ansi.val1*10 + (c-'0');
	else {
	    more = 0;
	    val1=buf->t.ansi.val1;
	    switch(c) {
              case 'A':
		if( !val1 )
		    val1++;
		if( buf->t.ansi.y > val1 )
		    buf->t.ansi.y -= val1;
		else
		    buf->t.ansi.y = 0;
		break;
              case 'B':
		if( !val1 )
		    val1++;
		if( buf->t.ansi.y + val1 < buf->t.ansi.rows )
		    buf->t.ansi.y += val1;
		else
		    buf->t.ansi.y = buf->t.ansi.rows-1;
		break;
              case 'C':
		if( !val1 )
		    val1++;
		if( buf->t.ansi.x + val1 < buf->t.ansi.cols )
		    buf->t.ansi.x += val1;
		else
		    buf->t.ansi.x = buf->t.ansi.cols-1;
		break;
              case 'D':
		if( !val1 )
		    val1++;
		if( buf->t.ansi.x > val1 )
		    buf->t.ansi.x -= val1;
		else
		    buf->t.ansi.x = 0;
		break;
              case 's':
		buf->t.ansi.savex = buf->t.ansi.x;
		buf->t.ansi.savey = buf->t.ansi.y;
		break;
              case 'u':
		buf->t.ansi.x = buf->t.ansi.savex;
		buf->t.ansi.y = buf->t.ansi.savey;
		break;
              case 'J':
		if( val1 == 2 ) {
		    buf->t.ansi.x = buf->t.ansi.y = 0;
		    for(i=0; i < buf->t.ansi.rows ; i++ )
                        memset(buf->lines[i], ' ', buf->t.ansi.cols);
		    if( buf->attr )
			for(i=0; i < buf->t.ansi.rows; i++ )
			    for(ap=buf->attr[i],j=0;
					j < buf->t.ansi.cols; j++, ap++ )
				*ap = buf->t.ansi.a;
                    Notify2( hd, DABF_N_CLS, 0, 0, buf->t.ansi.a, ' ' );
		}
		break;
              case 'K':
		val1= buf->t.ansi.x;
		val2= buf->t.ansi.y;
                memset( buf->lines[val2]+val1, ' ', buf->t.ansi.cols - val1);
		if( buf->attr )
		    for(ap=buf->attr[val2]+val1,j=val1;
				j < buf->t.ansi.cols; j++, ap++ )
			*ap = buf->t.ansi.a;
                Notify2( hd, DABF_N_CLEL, val1, val2,buf->t.ansi.a, ' ' );
		break;

              case 'H':
              case 'f':
		buf->t.ansi.y = val1 ? (val1-1): 0;
		buf->t.ansi.x = 0; /* kein 2. Parameter */
		break;

              case 'm':
		SetAnsiAttr( buf, val1 );
		break;

              case ';':
		buf->t.ansi.state = 3;
		more++;
		break;

	      default: break;
	    }
	    if( !more )
		buf->t.ansi.state = 0;
	}
	break;

      case 3:
      case 4:
	if( isdigit(c) )
            buf->t.ansi.val2 = buf->t.ansi.val2*10 + (c-'0');
	else {
	    more = 0;
	    val1=buf->t.ansi.val1;
	    val2=buf->t.ansi.val2;
	    switch(c) {
              case 'H':
              case 'f':
		buf->t.ansi.y = val1 ? (val1-1):0;
		buf->t.ansi.x = val2 ? (val2-1):0;
		break;

              case ';':  /* kann nur noch 'm' sein */
		more++;
		/* fall through */
              case 'm':
		if( buf->t.ansi.state == 3 ) {
		    SetAnsiAttr( buf, val1 );
		    buf->t.ansi.state = 4;
		}
		SetAnsiAttr( buf, val2 );
		break;


	      default: break;
	    }
	    if( more )
		buf->t.ansi.val2  = 0;
	    else
		buf->t.ansi.state = 0;
	}
	break;
    }

}


static void SetAnsiAttr( buffer_t *buf, ushort val )
{
    switch( val ) {
      case  0: buf->t.ansi.a=defaultAttr ; break;
      case  1: buf->t.ansi.a.bold     = 1; break;
      case  2: buf->t.ansi.a.faint    = 1; break;
      case  3: buf->t.ansi.a.italic   = 1; break;
      case  5: buf->t.ansi.a.blink    = 1; break;
      case  6: buf->t.ansi.a.rblink   = 1; break;
      case  7: buf->t.ansi.a.reverse  = 1; break;
      case  8: buf->t.ansi.a.concealed= 1; break;
      default:
	if( val >= 30 && val <= 37 )
	    buf->t.ansi.a.fore = val - 30;
	else if( val >= 40 && val <= 47 )
	    buf->t.ansi.a.back = val - 40;
	break;
    }
}

static void ScrollAnsi( buffer_t *buf )
{
    char *p;
    dabfAttr_t *ap;
    int n;

    p = buf->lines[0];
    for(n=1; n < buf->t.ansi.rows; n++ )
	buf->lines[n-1] = buf->lines[n];
    buf->lines[n-1] = p;
    memset( p, ' ', buf->t.ansi.cols);

    if( buf->attr ) {
	ap = buf->attr[0];
	for(n=1; n < buf->t.ansi.rows; n++ )
	    buf->attr[n-1] = buf->attr[n];
	buf->attr[n-1] = ap;
	for(n=0; n < buf->t.ansi.cols; n++, ap++ )
	    *ap = defaultAttr;
    }
}


#ifdef TEST
void main(int arg, char **argv)
{

    exit(0);
}
#endif /* TEST */
/**** bottom of file ****/
