/* [direc5.c wk 16.12.93] new filename functions
 *	Copyright (c) 1988-93 by Werner Koch (dd9jn)
 *  This file is part of WkLib.
 *
 *  WkLib is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WkLib is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * History:
 * 07.09.94 wk	added extname()
 */

#include <wk/tailor.h>
RCSID("$Id: direc5.c,v 1.7 1996/08/12 16:55:44 wk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#if __ZTC__
   #include <direct.h>
#endif
#include <wk/file.h>
#include <wk/string.h>
#include <wk/direc.h>
#include <wk/io.h>

/****** constants ********/
/****** types ********/
/****** globals ******/
/***** prototypes *****/
/***** functions ******/

#ifdef DOCUMENTATION
@Summary cwdname
 #include <wk/direc.h>

 char *cwdname( void );
@Description
 Returns the name of the current working directory in an allocated buffer.
 Buffer will be allocated with xmalloc().
 Any error will terminate the Programm with an fatal error message.
 On MSDOS filesystems this function includes the driveletter.
 All backslashes are converted to forward slashes.
@Return Value
 allocated buffer with the name of current directory.
#endif /*DOCUMENTATION*/

char *cwdname()
{
    char *p, *s;
    size_t n = 256;

    for(;;) {
	if( !(p = malloc(n)) )
	    Error(4,"out of memory in cwdname()");
	if( getcwd( p, n-1 ) ) {
	    for(s=p; *s; s++ )
		if( *s == '\\' )
		    *s = '/';
	    return p;
	}
	if( errno != ERANGE )
	    Error(1004,"error getting cwd");
	free(p);
	n += 256;
	if( n > 10000 )
	    Bug("getcwd() seems to be broken");
    }
    /*NOTREACHED*/
}


#ifdef DOCUMENTATION
@Summary basename
 #include <wk/direc.h>

 char *basename( const char *path );
@Description
 Extracts the basename, that is the filename without the directory-part,
 from path and returns it in an allocated buffer.
@Return Value
 allocated buffer with the basename part of path.
#endif /*DOCUMENTATION*/

char *basename( const char *path )
{
    const char *s;

    for(s=path+strlen(path)-1; s >= path; s-- )
      #if MSDOSFILESYSTEM
	if( *s == '/' || *s == '\\' || *s == ':' )
      #else
	if( *s == '/' )
      #endif
	    break;

    return xstrdup(s+1);
}


#ifdef DOCUMENTATION
@Summary extname
 #include <wk/direc.h>

 char *extname( const char *path );
@Description
 Extracts the extension from path and returns it in an allocated buffer.
 The extension includes the dot.
@Return Value
 Allocated buffer with the extension part of the filename.
 This may be an empty string if there is no extension.
#endif /*DOCUMENTATION*/

char *extname( const char *path )
{
    const char *s;

    for(s=path+strlen(path)-1; s >= path; s-- )
      #if MSDOSFILESYSTEM
	if( *s == '/' || *s == '\\' || *s == ':' )
      #else
	if( *s == '/' )
      #endif
	    break;
	else if( *s == '.' )  {
	    if( s > path ) {
		s--;
	      #if MSDOSFILESYSTEM
		if( *s == '/' || *s == '\\' || *s == ':' )
	      #else
		if( *s == '/' )
	      #endif
		    break; /* filename starts with a dot ... */
		return xstrdup(s+1);
	    }
	    break; /* filename starts with a dot -- this is not an extension*/
	}

    return xstrdup("");
}


#ifdef DOCUMENTATION
@Summary dirname
 #include <wk/direc.h>

 char *dirname( const char *path );
@Description
 Extracts the dirname, that is the filename without the basename-part,
 from path and returns it in an allocated buffer.
 there is no trailing slash at the resulting string, the resulting
 string may be an empty string.
 All backslashes are converted to forward slashes.
@Return Value
 allocated buffer with the dirname part of path.
#endif /*DOCUMENTATION*/

char *dirname( const char *path )
{
    const char *s;
    char *buf, *p;

    for(s=path+strlen(path)-1; s >= path; s-- )
      #if MSDOSFILESYSTEM
	if( *s == '/' || *s == '\\' )  {
	    p = buf = xmalloc(s-path+1);
	    while( path < s )
		if( *path == '\\' )
		    *p++ = '/', path++;
		else
		    *p++ = *path++;
	    *p=0;
	    return buf;
	}
	else if( *s == ':' ) {
	    /* Wird haben nur einen Doppelpunkt; muessen daraus aber unbedingt
	     * eine relative Pfadangabe machen, damit ein spaeteres joinen
	     * mit dem basename auch wieder eine relative Pfadangabe ergibt;
	     * "a:"+"/"+"myfile"  --> "a:/myfile"   ist falsch
	     * "a:."+"/"+"myfile" --> "a:./myfile"  is identisch mit "a:myfile"
	     */
	    p = buf = xmalloc(s-path+1+1);
	    while( path < s )
		if( *path == '\\' )
		    *p++ = '/', path++;
		else
		    *p++ = *path++;
	    *p++ = '.';
	    *p=0;
	    return buf;
	}
      #else
	if( *s == '/' )  {
	    if( s == path ) /* special handling for the root directory */
		return xstrdup("/");
	    p = buf = xmalloc(s-path+1);
	    while( path < s )
		*p++ = *path++;
	    *p=0;
	    return buf;
	}
      #endif

    return xstrdup("");
}



/*********************************************
 ********* Additional stuff ******************
 ********************************************/


#if GNU_C_LIB_OS2 && OS20
/* getcwd() is broken in my version (driveletter missing) */

#ifdef __GNUC__
ulong DosQueryCurrentDir()  asm("DosQueryCurrentDir");
ulong DosQueryCurrentDisk() asm("DosQueryCurrentDisk");
#endif

char *getcwd( char *xbuf, size_t size )
{
    ulong n, rc, drive, drivemap;
    char *buf;

    if( size < 5 ) {
	errno = ERANGE;
	return NULL;
    }

    buf = xbuf? xbuf : malloc(size=256);
    if( !buf ) {
	errno = ERANGE;
	return NULL;
    }

    n = size-3;
    if( !(rc=DosQueryCurrentDisk(&drive, &drivemap)) )
	rc = DosQueryCurrentDir(0, buf+3, &n );

    if( !rc ) {
	buf[0] = 'a'+ drive - 1;
	buf[1] = ':';
	buf[2] = '/';
	return buf;
    }

    if( !xbuf )
	free(buf);
    if( rc == 111 )	 /* ERROR_BUFFER_OVERFLOW */
	errno = ERANGE;
    else if( rc == 108 ) /* ERROR_DRIVE_LOCKED */
	errno = EACCES;
    else
	errno = EINVAL;
    return NULL;
}

#endif

/**** bottom of file ****/
