/* [lwpmain.c wk 15.01.94] LWP Kernel functions
 *	Copyright (c) 1994 by Werner Koch (dd9jn)
 * This file is part of WkLib.
 *
 * WkLib is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * WkLib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * This here are OS independent kernel functions.
 *
 * History:
 */

#include <wk/tailor.h>
RCSID("$Id: lwpmain.c,v 1.3 1995/03/08 16:56:39 wk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <wk/lwp.h>


/************************************************
 ************  local variables	*****************
 ************************************************/

static t_eventflag unusedEventFlags;

/************************************************
 ************  local functions	 ****************
 ************************************************/

/************************************************
 ************  public functions  ****************
 ************************************************/

/****************
 * Create an eventflag
 */

t_eventflag CreateEventFlag()
{
    t_eventflag e;

    if( e = unusedEventFlags )
	unusedEventFlags = e->next;
    else
	e = xmalloc( sizeof *e );
    e->next = NULL;
    e->value = 0;   /* not yet posted */
    e->lock = CreateSema(1);
    e->posted = CreateSema(0);
    return e;
}

void CloseEventFlag( t_eventflag e )
{
    CloseSema(e->lock);
    CloseSema(e->posted);
    e->next = unusedEventFlags;
    unusedEventFlags = e;
}


/****************
 * Post an eventflags; you may give a value, which the receiver
 * can obtain. If you specify a value of NULL this functions will
 * have no effect. The function returns NULL if the flag could be posted
 * or a non-zero value, which is the last posted value, when
 * the last posted flag was not removed by one of the Wait- or
 * Reset-Functions.
 */

void *PostEventFlag( t_eventflag e, void *value)
{
    void *ret;

    WaitSema(e->lock);
    if( e->value )
	ret = e->value;
    else {
	e->value = value;
	ret = NULL;
	SignalSema(e->posted);
    }
    SignalSema(e->lock);
    return ret;
}


/****************
 * Blocks until the Flag is posted; it returns the value
 * which was passed to PostEventFlag().
 * The returned value will never be NULL.
 */

void *WaitEventFlag( t_eventflag e )
{
    return WaitEventFlagTimed(e, 0);
}

/****************
 * Blocks until the Flag is posted; it returns the value
 * which was passed to PostEventFlag().
 * You may specify a non-zero value for timeout and this
 * function will time out after timeout milliseconds and
 * return a value of zero to indicate the timeout.
 */

void *WaitEventFlagTimed( t_eventflag e, ulong timeout )
{
    void *ret;

    do {
	if( WaitSemaTimed( e->posted, timeout ) )
	    return NULL;
	WaitSema(e->lock);
	ret = e->value;
	SignalSema(e->lock);
    } while( !ret && !timeout ); /* may have been resetted between */
				 /* Wait(posted) and Wait(lock) */
    return ret;
}


/****************
 * Clears the EventFlag  and returns its old value; a value of
 * zero means,that the flags was not posted.
 * This function will never block.
 */

void *ResetEventFlag( t_eventflag e )
{
    void *ret;

    WaitSema(e->lock);
    ret = e->value;
    ResetSema(e->posted);
    SignalSema(e->lock);
    return ret;
}



/****************
 * Create a queue with max size elements
 */

t_queue CreateQueue( size_t size )
{
    t_queue q;

    if( !size )
	size = 10; /* default value */
    q = xmalloc( sizeof *q + (size-1) * sizeof(void *) );
    q->size = size;
    q->head = 0;
    q->next = 0;
    q->space = CreateSema( (int)size );
    q->items = CreateSema( 0 );
    return q;
}

/****************
 * Close a queue and give all resources back to the OS.
 */

void CloseQueue( t_queue q )
{
    CloseSema(q->space);
    CloseSema(q->items);
    free(q);
}


/****************
 * add another buffer into the queue; return true on error.
 */

int Enqueue( t_queue q, void *element )
{
    if( TestSema(q->space) )
	return -1;
    q->d[q->next] = element;
    if( ++q->next == q->size )
	q->next = 0;
    SignalSema(q->items);
    return 0;
}

/****************
 * Get the first element from the queue or return NULL
 * if the queue is empty.
 */

void *Dequeue( t_queue q )
{
    void *element;

    if( TestSema(q->items) )
	return NULL;
    element = q->d[q->head];
    if( ++q->head == q->size )
	q->head = 0;
    SignalSema(q->space);
    return element;
}


/*** end of file ***/
